/*****************************************************************************\
 *  $Id: genders.h.in,v 1.16 2004/02/04 00:57:58 achu Exp $
 *****************************************************************************
 *  Copyright (C) 2001-2003 The Regents of the University of California.
 *  Produced at Lawrence Livermore National Laboratory (cf, DISCLAIMER).
 *  Written by Jim Garlick <garlick@llnl.gov> and Albert Chu <chu11@llnl.gov>.
 *  UCRL-CODE-2003-004.
 *
 *  This file is part of Genders, a cluster configuration database.
 *  For details, see <http://www.llnl.gov/linux/genders/>.
 *
 *  Genders is free software; you can redistribute it and/or modify it under
 *  the terms of the GNU General Public License as published by the Free
 *  Software Foundation; either version 2 of the License, or (at your option)
 *  any later version.
 *
 *  Genders is distributed in the hope that it will be useful, but WITHOUT ANY
 *  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 *  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with Genders; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.
\*****************************************************************************/

#ifndef _GENDERS_H
#define _GENDERS_H

#include <stdio.h>

/* Genders error codes */
#define GENDERS_ERR_SUCCESS       0 /* success */
#define GENDERS_ERR_NULLHANDLE    1 /* genders handle is null */
#define GENDERS_ERR_OPEN          2 /* error opening file */
#define GENDERS_ERR_READ          3 /* error reading file */
#define GENDERS_ERR_PARSE         4 /* parse error on genders file */
#define GENDERS_ERR_NOTLOADED     5 /* genders data not loaded */
#define GENDERS_ERR_ISLOADED      6 /* genders data is already loaded */
#define GENDERS_ERR_OVERFLOW      7 /* array is not big enough for result */
#define GENDERS_ERR_PARAMETERS    8 /* incorrect parameters passed in */
#define GENDERS_ERR_NULLPTR       9 /* null pointer in list */
#define GENDERS_ERR_NOTFOUND     10 /* node not found */
#define GENDERS_ERR_OUTMEM       11 /* out of memory */
#define GENDERS_ERR_MAGIC        12 /* magic number incorrect */
#define GENDERS_ERR_INTERNAL     13 /* internal system error */
#define GENDERS_ERR_ERRNUMRANGE  14 /* error number out of range */

#define GENDERS_DEFAULT_FILE     "/etc/genders"   

typedef struct genders *genders_t;

/* Creates & Initializes a genders handle 
 * Returns: NULL on memory allocation error
 */
genders_t genders_handle_create(void);

/* Destroys a genders handle
 * Returns: 0 on success, -1 on failure
 */
int genders_handle_destroy(genders_t handle);

/* Opens/reads/parses the specified genders file
 * If filename is NULL, attempts to read default genders file
 * Returns: 0 on success, -1 on failure  
 */
int genders_load_data(genders_t handle, const char *filename);

/* Returns an error code associated with a handle 
 */
int genders_errnum(genders_t handle);

/* Translates a genders error code to a text message  
 * Returns: pointer to NUL-terminated statically allocated string
 */
char *genders_strerror(int errnum);

/* Returns a pointer to a NUL-terminated statically allocated string
 * describing the most recent error that occurred
 */
char *genders_errormsg(genders_t handle);

/* Produces a message on standard error describing the most recent
 * error that occurred  
 */
void genders_perror(genders_t handle, const char *msg);
 
/* Get the number of nodes read from the genders file
 * Returns: number of nodes on success, -1 on failure 
 */
int genders_getnumnodes(genders_t handle);

/* Get the number of attributes read from the genders file
 * Returns: number of attributes on success, -1 on failure 
 */
int genders_getnumattrs(genders_t handle);

/* Get the max number of attributes read of any one node in the
 * genders file
 * Returns: number of attributes on success, -1 on failure 
 */
int genders_getmaxattrs(genders_t handle);

/* Get the max node name length of any one node in the genders file
 * Returns: maximum node length on success, -1 on failure 
 */
int genders_getmaxnodelen(genders_t handle);

/* Get the max attribute name length of any one attribute in the
 * genders file
 * Returns: maximum attribute length on success, -1 on failure 
 */
int genders_getmaxattrlen(genders_t handle);

/* Get the max value length of any one value in the genders file
 * Returns: maximum value length on success, -1 on failure 
 */
int genders_getmaxvallen(genders_t handle);

/* Allocate an array of character strings to store node names in
 * Returns: number of elements the list can store on succcess, -1 on failure
 */
int genders_nodelist_create(genders_t handle, char ***nodelist);

/* Clears the data stored in a previously created node list
 * Returns: 0 on success, -1 on failure
 */
int genders_nodelist_clear(genders_t handle, char **nodelist);

/* Frees memory of a previously created node list
 * Returns: 0 on success, -1 on failure
 */
int genders_nodelist_destroy(genders_t handle, char **nodelist);

/* Allocate an array of character strings to store attribute names in
 * Returns: number of elements the list can store on succcess, -1 on failure
 */
int genders_attrlist_create(genders_t handle, char ***attrlist);

/* Clears the data stored in a previously created attribute list
 * Returns: 0 on success, -1 on failure
 */
int genders_attrlist_clear(genders_t handle, char **attrlist);

/* Frees memory of a previously created attribute list
 * Returns: 0 on success, -1 on failure
 */
int genders_attrlist_destroy(genders_t handle, char **attrlist);

/* Allocate an array of character strings to store values in
 * Returns: number of elements the list can store on succcess, -1 on failure
 */
int genders_vallist_create(genders_t handle, char ***vallist);

/* Clears the data stored in a previously created value list
 * Returns: 0 on success, -1 on failure
 */
int genders_vallist_clear(genders_t handle, char **vallist);

/* Frees memory of a previously created value list
 * Returns: 0 on success, -1 on failure
 */
int genders_vallist_destroy(genders_t handle, char **vallist);

/* Get the name of the current node 
 * Node name returned is the shortened hostname
 * Returns: 0 on success, -1 on failure 
 */
int genders_getnodename(genders_t handle, char *node, int len);

/* Gets list of nodes with the specified attribute  
 * If attr is NULL, gets all nodes  
 * If val is non-NULL, get only nodes with attributes=val
 * Nodes are returned in genders file order
 * Returns: number of matches on success, -1 on failure
 */
int genders_getnodes(genders_t handle, char *nodes[], int len, 
                     const char *attr, const char *val);

/* Gets list of attributes for the specified node  
 * If node is NULL, gets all attributes for the current node  
 * If vals array is non-NULL, stores any attribute values in it
 * Returns: number of matches on success, -1 on failure
 */
int genders_getattr(genders_t handle, char *attrs[], char *vals[], 
                    int len, const char *node);

/* Gets all attributes stored in the genders file
 * Returns: number of attributes on success, -1 on failure
 */
int genders_getattr_all(genders_t handle, char *attrs[], int len);

/* Tests whether a node has an attribute
 * If node is NULL, tests the current node
 * If val is non-NULL, stores the attribute value in it
 * Returns: 1=true, 0=false, -1=failure
 */
int genders_testattr(genders_t handle, const char *node,
                     const char *attr, char *val, int len);

/* Tests whether node has an attribute=value pair
 * If node is NULL, tests the current node
 * If val is NULL, only the attribute is tested
 * Returns: 1=true, 0=false, -1=failure
 */
int genders_testattrval(genders_t handle, const char *node, 
                        const char *attr, const char *val);

/* Tests whether the node exists in the genders file
 * If node is NULL, tests the current node
 * Returns: 1=true , 0=false, -1=failure
 */
int genders_isnode(genders_t handle, const char *node);

/* Tests whether the attribute exists in the genders file
 * Returns: 1=true , 0=false, -1=failure
 */
int genders_isattr(genders_t handle, const char *attr);

/* Tests whether an attribute=value exists for some node in the
 * genders file
 * Returns: 1=true , 0=false, -1=failure
 */
int genders_isattrval(genders_t handle, const char *attr, const char *val);

/* Parses a genders file, and outputs parse debugging 
 * information to the file stream  
 * if filename is NULL, parses default genders file
 * if stream is NULL, outputs to stderr
 * Returns the number of parse errors (0 if no parse errors), -1 on error
 */
int genders_parse(genders_t handle, const char *filename, FILE *stream);

/* Set the errnum for a genders handle
 * - used for development of local genders libraries 
 */      
void genders_set_errnum(genders_t handle, int errnum);

#endif /* _GENDERS_H */
