// CLASSIFICATION: UNCLASSIFIED

/******************************************************************************
* Filename: CoordPanel.java
*
* Copyright BAE Systems Inc. 2012 ALL RIGHTS RESERVED
*
* MODIFICATION HISTORY
*
* DATE      NAME        DR#          DESCRIPTION
*
* 07/18/12  S. Gillis   MSP_00029550 Updated exception handling 
* 01/12/16  K. Chen     MSP_00030518 Add US Survey Feet Support
*****************************************************************************/
package geotrans3.gui;

import geotrans3.coordinates.*;
import geotrans3.enumerations.CoordinateType;
import geotrans3.exception.CoordinateConversionException;
import geotrans3.jni.*;
import geotrans3.misc.FormatOptions;
import geotrans3.misc.Source;
import geotrans3.misc.StringHandler;
import geotrans3.utility.Constants;
import geotrans3.utility.Platform;
 
public class CoordPanel extends javax.swing.JPanel
{
  private StringHandler stringHandler;
  private FormatOptions formatOptions;
  private boolean useNSEW;
  private boolean useMinutes;
  private boolean useSeconds;
  private char hemisphere;
  private long zone;
  // Flag to test if accuracy has been selected or is from previous conversion
  private boolean usePrevAccuracies;
  private boolean _3dConversion;
  private Accuracy currentAccuracy;

  private int HeightUnitType;
  
  public void setHeightType(int _heightHandler)
  {
	  HeightUnitType = _heightHandler;
  }

  
    /** Creates new form coordPanel */
  public CoordPanel(FormatOptions _formatOptions, StringHandler _stringHandler)
  {
    formatOptions = _formatOptions;
    stringHandler = _stringHandler;
    usePrevAccuracies = false;
    _3dConversion = false;
    currentAccuracy = new Accuracy();

    setFormat();
    initComponents();

    tempLabel1.setVisible(false);
    tempLabel2.setVisible(false);
    tempTextField1.setVisible(false);
    tempTextField2.setVisible(false);

    if(Platform.isJavaV1_3)
    {
      _3CoordFieldsLabelA.setForeground(java.awt.Color.black);
      _3CoordFieldsLabelB.setForeground(java.awt.Color.black);
      _3CoordFieldsLabelC.setForeground(java.awt.Color.black);
      _3CoordFieldsLabelUnit_C.setForeground(java.awt.Color.black);

      _2CoordFieldsLabelA.setForeground(java.awt.Color.black);
      _2CoordFieldsLabelB.setForeground(java.awt.Color.black);
      ce90Label.setForeground(java.awt.Color.black);
      le90Label.setForeground(java.awt.Color.black);
      se90Label.setForeground(java.awt.Color.black);
      sourcesLabel.setForeground(java.awt.Color.black);
    }

    setPanelSizes();
    fillAccuracyList();
    if(Platform.isUnix)
    {
      _3CoordFieldsLabelA.setFont(new java.awt.Font("Dialog", 1, 10));
      _3CoordFieldsLabelB.setFont(new java.awt.Font("Dialog", 1, 10));
      _3CoordFieldsLabelC.setFont(new java.awt.Font("Dialog", 1, 10));
      _3CoordFieldsLabelUnit_C.setFont(new java.awt.Font("Dialog", 1, 10));
      
      _2CoordFieldsLabelA.setFont(new java.awt.Font("Dialog", 1, 10));
      _2CoordFieldsLabelB.setFont(new java.awt.Font("Dialog", 1, 10));
      ce90Label.setFont(new java.awt.Font("Dialog", 1, 10));
      le90Label.setFont(new java.awt.Font("Dialog", 1, 10));
      se90Label.setFont(new java.awt.Font("Dialog", 1, 10));
      sourcesLabel.setFont(new java.awt.Font("Dialog", 1, 10));
      sourcesComboBox.setFont(new java.awt.Font("Dialog", 1, 10));
      sourcesComboBox.setMaximumSize(new java.awt.Dimension(265, 22));
      sourcesComboBox.setMinimumSize(new java.awt.Dimension(265, 22));
      sourcesComboBox.setPreferredSize(new java.awt.Dimension(265, 22));
      ce90TextField.setToolTipText("90% CE");
      le90TextField.setToolTipText("90% LE");
      se90TextField.setToolTipText("90% SE");
    }
  }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the FormEditor.
     */
    private void initComponents() {//GEN-BEGIN:initComponents
        java.awt.GridBagConstraints gridBagConstraints;

        coordLayeredPane = new javax.swing.JLayeredPane();
        _3CoordFieldsPanel = new javax.swing.JPanel();
        _3CoordFieldsLabelA = new javax.swing.JLabel();
        _3CoordFieldsLabelB = new javax.swing.JLabel();
        _3CoordFieldsLabelC = new javax.swing.JLabel();
        _3CoordFieldsLabelUnit_C = new javax.swing.JLabel();
        _3CoordFieldsLabelUnitFont_C = _3CoordFieldsLabelUnit_C.getFont();

        _3CoordFieldsTextFieldA = new javax.swing.JTextField();
        _3CoordFieldsTextFieldB = new javax.swing.JTextField();
        _3CoordFieldsTextFieldC = new javax.swing.JTextField();
        _2CoordFieldsPanel = new javax.swing.JPanel();
        _2CoordFieldsLabelA = new javax.swing.JLabel();
        _2CoordFieldsLabelB = new javax.swing.JLabel();
        _2CoordFieldsTextFieldA = new javax.swing.JTextField();
        _2CoordFieldsTextFieldB = new javax.swing.JTextField();
        tempLabel1 = new javax.swing.JLabel();
        tempLabel2 = new javax.swing.JLabel();
        tempTextField1 = new javax.swing.JTextField();
        tempTextField2 = new javax.swing.JTextField();
        errorsPanel = new javax.swing.JPanel();
        outputErrorPanel = new javax.swing.JPanel();
        ce90Label = new javax.swing.JLabel();
        le90Label = new javax.swing.JLabel();
        se90Label = new javax.swing.JLabel();
        ce90TextField = new javax.swing.JTextField();
        le90TextField = new javax.swing.JTextField();
        se90TextField = new javax.swing.JTextField();
        inputErrorPanel = new javax.swing.JPanel();
        sourcesLabel = new javax.swing.JLabel();
        sourcesComboBox = new javax.swing.JComboBox();

        setLayout(new java.awt.GridBagLayout());

        setMinimumSize(new java.awt.Dimension(439, 80));
        setPreferredSize(new java.awt.Dimension(439, 80));
        addComponentListener(new java.awt.event.ComponentAdapter() {
            public void componentResized(java.awt.event.ComponentEvent evt) {
                formComponentResized(evt);
            }
        });

        coordLayeredPane.setMaximumSize(new java.awt.Dimension(439, 36));
        coordLayeredPane.setMinimumSize(new java.awt.Dimension(439, 38));
        coordLayeredPane.setPreferredSize(new java.awt.Dimension(460, 38));
        coordLayeredPane.setOpaque(true);

        _3CoordFieldsPanel.setLayout(null);


        _3CoordFieldsPanel.setMaximumSize(new java.awt.Dimension(339, 36));
        _3CoordFieldsPanel.setMinimumSize(new java.awt.Dimension(339, 36));
        _3CoordFieldsPanel.setPreferredSize(new java.awt.Dimension(339, 36));
        _3CoordFieldsPanel.setOpaque(false);
        _3CoordFieldsLabelA.setText("X (m):");
        _3CoordFieldsLabelA.setMaximumSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelA.setMinimumSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelA.setPreferredSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelA.setOpaque(true);
        _3CoordFieldsPanel.add(_3CoordFieldsLabelA);

        _3CoordFieldsLabelA.setBounds(0, 0, 138, 18);

        _3CoordFieldsLabelB.setText("Y (m):");
        _3CoordFieldsLabelB.setMaximumSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelB.setMinimumSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelB.setPreferredSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelB.setOpaque(true);
        _3CoordFieldsPanel.add(_3CoordFieldsLabelB);

        _3CoordFieldsLabelB.setBounds(150, 0, 138, 18);


        _3CoordFieldsLabelC.setText("Z");        
        _3CoordFieldsLabelC.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);

        _3CoordFieldsLabelC.setMaximumSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelC.setMinimumSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelC.setPreferredSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelC.setOpaque(true);
        _3CoordFieldsPanel.add(_3CoordFieldsLabelC);

        _3CoordFieldsLabelC.setBounds(300, 0, 7, 18);

        _3CoordFieldsLabelUnit_C.setText("(m):");
        if (javax.swing.UIManager.getLookAndFeel().getName().equalsIgnoreCase("Metal"))
        	_3CoordFieldsLabelUnit_C.setFont(_3CoordFieldsLabelUnitFont_C);
        else
        	_3CoordFieldsLabelUnit_C.setFont(_3CoordFieldsLabelUnit_C.getFont().deriveFont(java.awt.Font.PLAIN));

        _3CoordFieldsLabelUnit_C.setMaximumSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelUnit_C.setMinimumSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelUnit_C.setPreferredSize(new java.awt.Dimension(105, 16));
        _3CoordFieldsLabelUnit_C.setOpaque(true);        
        _3CoordFieldsPanel.add(_3CoordFieldsLabelUnit_C);
        _3CoordFieldsLabelUnit_C.setBounds(310, 0, 30, 18);


        _3CoordFieldsTextFieldA.setText("0.0");
        _3CoordFieldsTextFieldA.setBorder(new javax.swing.border.BevelBorder(javax.swing.border.BevelBorder.LOWERED));
        _3CoordFieldsTextFieldA.setMinimumSize(new java.awt.Dimension(105, 20));
        _3CoordFieldsTextFieldA.setPreferredSize(new java.awt.Dimension(105, 20));
        _3CoordFieldsPanel.add(_3CoordFieldsTextFieldA);

        _3CoordFieldsTextFieldA.setBounds(0, 18, 138, 18);

        _3CoordFieldsTextFieldB.setText("0.0");
        _3CoordFieldsTextFieldB.setBorder(new javax.swing.border.BevelBorder(javax.swing.border.BevelBorder.LOWERED));
        _3CoordFieldsTextFieldB.setMinimumSize(new java.awt.Dimension(105, 20));
        _3CoordFieldsTextFieldB.setPreferredSize(new java.awt.Dimension(105, 20));
        _3CoordFieldsPanel.add(_3CoordFieldsTextFieldB);

        _3CoordFieldsTextFieldB.setBounds(150, 18, 138, 18);

        _3CoordFieldsTextFieldC.setText("0.0");
        _3CoordFieldsTextFieldC.setBorder(new javax.swing.border.BevelBorder(javax.swing.border.BevelBorder.LOWERED));
        _3CoordFieldsTextFieldC.setMinimumSize(new java.awt.Dimension(105, 20));
        _3CoordFieldsTextFieldC.setPreferredSize(new java.awt.Dimension(105, 20));
        _3CoordFieldsPanel.add(_3CoordFieldsTextFieldC);

        _3CoordFieldsTextFieldC.setBounds(300, 18, 138, 18);

        _3CoordFieldsPanel.setBounds(2, 2, 439, 37);
        coordLayeredPane.add(_3CoordFieldsPanel, javax.swing.JLayeredPane.DEFAULT_LAYER);

        _2CoordFieldsPanel.setLayout(new java.awt.GridBagLayout());

        _2CoordFieldsPanel.setMaximumSize(new java.awt.Dimension(229, 36));
        _2CoordFieldsPanel.setMinimumSize(new java.awt.Dimension(229, 36));
        _2CoordFieldsPanel.setPreferredSize(new java.awt.Dimension(229, 36));
        _2CoordFieldsPanel.setOpaque(false);
        _2CoordFieldsLabelA.setText("Easting / X (m):");
        _2CoordFieldsLabelA.setAlignmentX(1.0F);
        _2CoordFieldsLabelA.setMaximumSize(new java.awt.Dimension(106, 16));
        _2CoordFieldsLabelA.setMinimumSize(new java.awt.Dimension(106, 16));
        _2CoordFieldsLabelA.setPreferredSize(new java.awt.Dimension(106, 16));
        _2CoordFieldsLabelA.setOpaque(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        _2CoordFieldsPanel.add(_2CoordFieldsLabelA, gridBagConstraints);

        _2CoordFieldsLabelB.setText("Northing / Y (m):");
        _2CoordFieldsLabelB.setMaximumSize(new java.awt.Dimension(106, 16));
        _2CoordFieldsLabelB.setMinimumSize(new java.awt.Dimension(106, 16));
        _2CoordFieldsLabelB.setPreferredSize(new java.awt.Dimension(106, 16));
        _2CoordFieldsLabelB.setOpaque(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        _2CoordFieldsPanel.add(_2CoordFieldsLabelB, gridBagConstraints);

        _2CoordFieldsTextFieldA.setText("0.0");
        _2CoordFieldsTextFieldA.setAlignmentX(1.0F);
        _2CoordFieldsTextFieldA.setBorder(new javax.swing.border.BevelBorder(javax.swing.border.BevelBorder.LOWERED));
        _2CoordFieldsTextFieldA.setMargin(new java.awt.Insets(0, 0, 5, 0));
        _2CoordFieldsTextFieldA.setMinimumSize(new java.awt.Dimension(106, 20));
        _2CoordFieldsTextFieldA.setPreferredSize(new java.awt.Dimension(106, 20));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 1, 6);
        _2CoordFieldsPanel.add(_2CoordFieldsTextFieldA, gridBagConstraints);

        _2CoordFieldsTextFieldB.setText("0.0");
        _2CoordFieldsTextFieldB.setBorder(new javax.swing.border.BevelBorder(javax.swing.border.BevelBorder.LOWERED));
        _2CoordFieldsTextFieldB.setMargin(new java.awt.Insets(0, 0, 5, 0));
        _2CoordFieldsTextFieldB.setMinimumSize(new java.awt.Dimension(106, 20));
        _2CoordFieldsTextFieldB.setPreferredSize(new java.awt.Dimension(106, 20));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 1, 6);
        _2CoordFieldsPanel.add(_2CoordFieldsTextFieldB, gridBagConstraints);

        tempLabel1.setText("Easting / X (m):");
        tempLabel1.setAlignmentX(1.0F);
        tempLabel1.setMaximumSize(new java.awt.Dimension(106, 16));
        tempLabel1.setMinimumSize(new java.awt.Dimension(106, 16));
        tempLabel1.setPreferredSize(new java.awt.Dimension(106, 16));
        tempLabel1.setOpaque(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 6);
        _2CoordFieldsPanel.add(tempLabel1, gridBagConstraints);

        tempLabel2.setText("Northing / Y (m):");
        tempLabel2.setMaximumSize(new java.awt.Dimension(106, 16));
        tempLabel2.setMinimumSize(new java.awt.Dimension(106, 16));
        tempLabel2.setPreferredSize(new java.awt.Dimension(106, 16));
        tempLabel2.setOpaque(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 0);
        _2CoordFieldsPanel.add(tempLabel2, gridBagConstraints);

        tempTextField1.setText("0.0");
        tempTextField1.setAlignmentX(1.0F);
        tempTextField1.setBorder(new javax.swing.border.BevelBorder(javax.swing.border.BevelBorder.LOWERED));
        tempTextField1.setMargin(new java.awt.Insets(0, 0, 5, 0));
        tempTextField1.setMinimumSize(new java.awt.Dimension(106, 20));
        tempTextField1.setPreferredSize(new java.awt.Dimension(106, 20));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 1, 6);
        _2CoordFieldsPanel.add(tempTextField1, gridBagConstraints);

        tempTextField2.setText("0.0");
        tempTextField2.setBorder(new javax.swing.border.BevelBorder(javax.swing.border.BevelBorder.LOWERED));
        tempTextField2.setMargin(new java.awt.Insets(0, 0, 5, 0));
        tempTextField2.setMinimumSize(new java.awt.Dimension(106, 20));
        tempTextField2.setPreferredSize(new java.awt.Dimension(106, 20));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 1, 0);
        _2CoordFieldsPanel.add(tempTextField2, gridBagConstraints);

        _2CoordFieldsPanel.setBounds(4, 2, 460, 45);
        coordLayeredPane.add(_2CoordFieldsPanel, javax.swing.JLayeredPane.DEFAULT_LAYER);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(1, 2, 1, 2);
        add(coordLayeredPane, gridBagConstraints);

        errorsPanel.setLayout(new java.awt.GridBagLayout());

        errorsPanel.setMinimumSize(new java.awt.Dimension(460, 40));
        errorsPanel.setPreferredSize(new java.awt.Dimension(460, 40));
        errorsPanel.setOpaque(false);
        outputErrorPanel.setLayout(new java.awt.GridBagLayout());

        outputErrorPanel.setMinimumSize(new java.awt.Dimension(170, 40));
        outputErrorPanel.setPreferredSize(new java.awt.Dimension(170, 40));
        outputErrorPanel.setOpaque(false);
        ce90Label.setText("90% CE:");
        ce90Label.setMaximumSize(new java.awt.Dimension(48, 16));
        ce90Label.setMinimumSize(new java.awt.Dimension(48, 16));
        ce90Label.setPreferredSize(new java.awt.Dimension(48, 16));
        ce90Label.setOpaque(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 6);
        outputErrorPanel.add(ce90Label, gridBagConstraints);

        le90Label.setText("90% LE:");
        le90Label.setMaximumSize(new java.awt.Dimension(48, 16));
        le90Label.setMinimumSize(new java.awt.Dimension(48, 16));
        le90Label.setPreferredSize(new java.awt.Dimension(48, 16));
        le90Label.setOpaque(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        outputErrorPanel.add(le90Label, gridBagConstraints);

        se90Label.setText("90% SE:");
        se90Label.setMaximumSize(new java.awt.Dimension(48, 16));
        se90Label.setMinimumSize(new java.awt.Dimension(48, 16));
        se90Label.setPreferredSize(new java.awt.Dimension(48, 16));
        se90Label.setOpaque(true);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 0);
        outputErrorPanel.add(se90Label, gridBagConstraints);

        ce90TextField.setToolTipText("");
        ce90TextField.setBorder(new javax.swing.border.BevelBorder(javax.swing.border.BevelBorder.LOWERED));
        ce90TextField.setMinimumSize(new java.awt.Dimension(48, 20));
        ce90TextField.setPreferredSize(new java.awt.Dimension(48, 20));
        ce90TextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyTyped(java.awt.event.KeyEvent evt) {
                accuracyTextFieldKeyTyped(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 6);
        outputErrorPanel.add(ce90TextField, gridBagConstraints);

        le90TextField.setToolTipText("");
        le90TextField.setBorder(new javax.swing.border.BevelBorder(javax.swing.border.BevelBorder.LOWERED));
        le90TextField.setMinimumSize(new java.awt.Dimension(48, 20));
        le90TextField.setPreferredSize(new java.awt.Dimension(48, 20));
        le90TextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyTyped(java.awt.event.KeyEvent evt) {
                accuracyTextFieldKeyTyped(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 6);
        outputErrorPanel.add(le90TextField, gridBagConstraints);

        se90TextField.setToolTipText("");
        se90TextField.setBorder(new javax.swing.border.BevelBorder(javax.swing.border.BevelBorder.LOWERED));
        se90TextField.setMinimumSize(new java.awt.Dimension(48, 20));
        se90TextField.setPreferredSize(new java.awt.Dimension(48, 20));
        se90TextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyTyped(java.awt.event.KeyEvent evt) {
                accuracyTextFieldKeyTyped(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(0, 6, 0, 0);
        outputErrorPanel.add(se90TextField, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 3, 0, 5);
        errorsPanel.add(outputErrorPanel, gridBagConstraints);

        inputErrorPanel.setLayout(new java.awt.GridBagLayout());

        inputErrorPanel.setMinimumSize(new java.awt.Dimension(265, 40));
        inputErrorPanel.setPreferredSize(new java.awt.Dimension(265, 40));
        sourcesLabel.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
        sourcesLabel.setText("Sources:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        inputErrorPanel.add(sourcesLabel, gridBagConstraints);

        sourcesComboBox.setMinimumSize(new java.awt.Dimension(265, 23));
        sourcesComboBox.setPreferredSize(new java.awt.Dimension(265, 23));
        sourcesComboBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                sourcesComboBoxActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        inputErrorPanel.add(sourcesComboBox, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 5, 0, 3);
        errorsPanel.add(inputErrorPanel, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(1, 2, 1, 2);
        add(errorsPanel, gridBagConstraints);

    }//GEN-END:initComponents

  private void accuracyTextFieldKeyTyped(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_accuracyTextFieldKeyTyped
      int keyChar = evt.getKeyChar();
      if (keyChar != java.awt.event.KeyEvent.VK_ENTER)
          usePrevAccuracies = false;
  }//GEN-LAST:event_accuracyTextFieldKeyTyped

  private void sourcesComboBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_sourcesComboBoxActionPerformed
    updateAccuracy(false);
  }//GEN-LAST:event_sourcesComboBoxActionPerformed

  private void formComponentResized(java.awt.event.ComponentEvent evt) {//GEN-FIRST:event_formComponentResized
    setPanelSizes();
  }//GEN-LAST:event_formComponentResized

    private void fillAccuracyList()
    {
        try
        {
            long sourceCount = Source.count();

            for(int i = 0; i < sourceCount; i++)
            {
              sourcesComboBox.addItem(Source.name(i));
            }
        }
        catch(CoordinateConversionException e)
        {
            stringHandler.displayErrorMsg(new javax.swing.JFrame(), e.getMessage());
        }
    }


    private void setOneCoord()
    {
        _2CoordFieldsPanel.setVisible(false);
        _3CoordFieldsPanel.setVisible(true);
        coordLayeredPane.moveToFront(_3CoordFieldsPanel);

        // Hide optional fields.
        _3CoordFieldsLabelA.setVisible(false);
        _3CoordFieldsLabelC.setVisible(false);

        _3CoordFieldsLabelUnit_C.setVisible(false);

        _3CoordFieldsTextFieldA.setVisible(false);
        _3CoordFieldsTextFieldC.setVisible(false);

        _3CoordFieldsLabelB.setText("Coordinate String:");
    }

    private void setTwoCoords()
    {
        _3CoordFieldsPanel.setVisible(false);
        _2CoordFieldsPanel.setVisible(true);
        coordLayeredPane.moveToFront(_2CoordFieldsPanel);
        _2CoordFieldsLabelA.setText("Easting / X (m):");
        _2CoordFieldsLabelB.setText("Northing / Y (m):");
        _2CoordFieldsTextFieldA.setText( "0" );
        _2CoordFieldsTextFieldB.setText( "0" );
    }

    private void setThreeCoords()
    {
        _2CoordFieldsPanel.setVisible(false);
        _3CoordFieldsPanel.setVisible(true);
        coordLayeredPane.moveToFront(_3CoordFieldsPanel);

        // Show optional fields.
        _3CoordFieldsLabelA.setVisible(true);
        _3CoordFieldsLabelC.setVisible(true);

        _3CoordFieldsLabelUnit_C.setVisible(true);

        _3CoordFieldsTextFieldA.setVisible(true);
        _3CoordFieldsTextFieldC.setVisible(true);

        /* may be turned off by height type buttons. */
        setHeightFieldEditable();

        _3CoordFieldsLabelA.setText("X (m):");
        _3CoordFieldsLabelB.setText("Y (m):");

        _3CoordFieldsLabelC.setText("Z");
        _3CoordFieldsLabelC.setBounds(300, 0, 7, 18);
        _3CoordFieldsLabelUnit_C.setText("(m):");

        if (javax.swing.UIManager.getLookAndFeel().getName().equalsIgnoreCase("Metal"))
        	_3CoordFieldsLabelUnit_C.setFont(_3CoordFieldsLabelUnitFont_C);
        else
        	_3CoordFieldsLabelUnit_C.setFont(_3CoordFieldsLabelUnit_C.getFont().deriveFont(java.awt.Font.PLAIN));

        _3CoordFieldsLabelUnit_C.setBounds(310, 0, 30, 18);

        _3CoordFieldsTextFieldA.setText( "0" );
        _3CoordFieldsTextFieldB.setText( "0" );
        _3CoordFieldsTextFieldC.setText( "0" );
    }


    private void setFormat()
    {
        useNSEW = formatOptions.getUseNSEW();
        useMinutes = formatOptions.getUseMinutes();
        useSeconds = formatOptions.getUseSeconds();
        stringHandler.setPrecision(formatOptions.getPrecision());
    }

    // This function is needed to make the fields visible when the gui is displayed
    private void setPanelSizes()
    {
        java.awt.Dimension dsize = new java.awt.Dimension();
        dsize = coordLayeredPane.getSize();

        _2CoordFieldsPanel.setBounds (0, 0, dsize.width, dsize.height);
        _3CoordFieldsPanel.setBounds (0, 0, dsize.width, dsize.height);

        _2CoordFieldsPanel.updateUI();
        _3CoordFieldsPanel.updateUI();
    }

  /*
   *  The function initTargetCoordinates creates a new instance of the
   *  target coordinates.
   *
   *  @param    projType    : Type of target coordinates to create
   */
  public CoordinateTuple initTargetCoordinates(int projType) throws CoordinateConversionException
  {
    switch(projType)
    {
      case CoordinateType.ALBERS:
        return new MapProjectionCoordinates(CoordinateType.ALBERS);
      case CoordinateType.AZIMUTHAL:
        return new MapProjectionCoordinates(CoordinateType.AZIMUTHAL);
      case CoordinateType.BONNE:
        return new MapProjectionCoordinates(CoordinateType.BONNE);
      case CoordinateType.CASSINI:
        return new MapProjectionCoordinates(CoordinateType.CASSINI);
      case CoordinateType.CYLEQA:
        return new MapProjectionCoordinates(CoordinateType.CYLEQA);
      case CoordinateType.ECKERT4:
        return new MapProjectionCoordinates(CoordinateType.ECKERT4);
      case CoordinateType.ECKERT6:
        return new MapProjectionCoordinates(CoordinateType.ECKERT6);
      case CoordinateType.EQDCYL:
        return new MapProjectionCoordinates(CoordinateType.EQDCYL);
      case CoordinateType.GNOMONIC:
        return new MapProjectionCoordinates(CoordinateType.GNOMONIC);
      case CoordinateType.LAMBERT_1:
        return new MapProjectionCoordinates(CoordinateType.LAMBERT_1);
      case CoordinateType.LAMBERT_2:
        return new MapProjectionCoordinates(CoordinateType.LAMBERT_2);
      case CoordinateType.MERCATOR_SP:
        return new MapProjectionCoordinates(CoordinateType.MERCATOR_SP);
      case CoordinateType.MERCATOR_SF:
        return new MapProjectionCoordinates(CoordinateType.MERCATOR_SF);
      case CoordinateType.MILLER:
        return new MapProjectionCoordinates(CoordinateType.MILLER);
      case CoordinateType.MOLLWEIDE:
        return new MapProjectionCoordinates(CoordinateType.MOLLWEIDE);
      case CoordinateType.NEYS:
        return new MapProjectionCoordinates(CoordinateType.NEYS);
      case CoordinateType.NZMG:
        return new MapProjectionCoordinates(CoordinateType.NZMG);
      case CoordinateType.OMERC:
        return new MapProjectionCoordinates(CoordinateType.OMERC);
      case CoordinateType.ORTHOGRAPHIC:
        return new MapProjectionCoordinates(CoordinateType.ORTHOGRAPHIC);
      case CoordinateType.POLARSTEREO_SP:
        return new MapProjectionCoordinates(CoordinateType.POLARSTEREO_SP);
      case CoordinateType.POLARSTEREO_SF:
        return new MapProjectionCoordinates(CoordinateType.POLARSTEREO_SF);
      case CoordinateType.POLYCONIC:
        return new MapProjectionCoordinates(CoordinateType.POLYCONIC);
      case CoordinateType.SINUSOIDAL:
        return new MapProjectionCoordinates(CoordinateType.SINUSOIDAL);
      case CoordinateType.STEREOGRAPHIC:
        return new MapProjectionCoordinates(CoordinateType.STEREOGRAPHIC);
      case CoordinateType.TRCYLEQA:
        return new MapProjectionCoordinates(CoordinateType.TRCYLEQA);
      case CoordinateType.TRANMERC:
        return new MapProjectionCoordinates(CoordinateType.TRANMERC);
      case CoordinateType.GRINTEN:
        return new MapProjectionCoordinates(CoordinateType.GRINTEN);
      case CoordinateType.GEODETIC:
        return new GeodeticCoordinates(CoordinateType.GEODETIC);
      case CoordinateType.GEOCENTRIC:
        return new CartesianCoordinates(CoordinateType.GEOCENTRIC);
      case CoordinateType.LOCCART:
        return new CartesianCoordinates(CoordinateType.LOCCART);
      case CoordinateType.BNG:
        return new BNGCoordinates(CoordinateType.BNG, formatOptions.getPrecision());
      case CoordinateType.GARS:
        return new GARSCoordinates(CoordinateType.GARS, formatOptions.getPrecision());
      case CoordinateType.GEOREF:
        return new GEOREFCoordinates(CoordinateType.GEOREF, formatOptions.getPrecision());
      case CoordinateType.F16GRS:
      case CoordinateType.MGRS:
        return new MGRSorUSNGCoordinates(CoordinateType.MGRS, formatOptions.getPrecision());
      case CoordinateType.USNG:
        return new MGRSorUSNGCoordinates(CoordinateType.USNG, formatOptions.getPrecision());
      case CoordinateType.UPS:
        return new UPSCoordinates(CoordinateType.UPS);
      case CoordinateType.UTM:
        return new UTMCoordinates(CoordinateType.UTM);
      case CoordinateType.WEBMERCATOR:
        return new MapProjectionCoordinates(CoordinateType.WEBMERCATOR);
      default:
        throw new CoordinateConversionException("Invalid coordinate type");
    }
  }
  
  
  public boolean setCoordinates(int state, int direction, int projType, CoordinateTuple _coordinates)
  {
    boolean noErrors = true;

    setFormat();

    try
    {
      switch(projType)
      {
        case CoordinateType.ALBERS:
        case CoordinateType.AZIMUTHAL:
        case CoordinateType.BONNE:
        case CoordinateType.CASSINI:
        case CoordinateType.CYLEQA:
        case CoordinateType.ECKERT4:
        case CoordinateType.ECKERT6:
        case CoordinateType.EQDCYL:
        case CoordinateType.GNOMONIC:
        case CoordinateType.LAMBERT_1:
        case CoordinateType.LAMBERT_2:
        case CoordinateType.MERCATOR_SP:
        case CoordinateType.MERCATOR_SF:
        case CoordinateType.MILLER:
        case CoordinateType.MOLLWEIDE:
        case CoordinateType.NEYS:
        case CoordinateType.NZMG:
        case CoordinateType.OMERC:
        case CoordinateType.ORTHOGRAPHIC:
        case CoordinateType.POLARSTEREO_SP:
        case CoordinateType.POLARSTEREO_SF:
        case CoordinateType.POLYCONIC:
        case CoordinateType.SINUSOIDAL:
        case CoordinateType.STEREOGRAPHIC:
        case CoordinateType.TRCYLEQA:
        case CoordinateType.TRANMERC:
        case CoordinateType.GRINTEN:
        case CoordinateType.WEBMERCATOR:
        {
          MapProjectionCoordinates coordinates = (MapProjectionCoordinates) _coordinates;

          _2CoordFieldsTextFieldA.setText(stringHandler.meterToString(coordinates.getEasting()));
          _2CoordFieldsTextFieldB.setText(stringHandler.meterToString(coordinates.getNorthing()));

          break;
        }
        case CoordinateType.BNG:
        case CoordinateType.GARS:
        case CoordinateType.GEOREF:
        case CoordinateType.MGRS:
        case CoordinateType.USNG:
        {
          StringCoordinates coordinates = (StringCoordinates) _coordinates;

          _3CoordFieldsTextFieldB.setText(coordinates.getCoordinateString());

          break;
        }
        case CoordinateType.F16GRS:
        {
          StringCoordinates coordinates = (StringCoordinates) _coordinates;

          java.lang.String temp = coordinates.getCoordinateString();
          if(formatOptions.getPrecision() == formatOptions.MET100000)
            _3CoordFieldsTextFieldB.setText(temp + "00");
          else
            _3CoordFieldsTextFieldB.setText(temp);

          break;
        }
        case CoordinateType.GEOCENTRIC:
        case CoordinateType.LOCCART:
        {
          CartesianCoordinates coordinates = (CartesianCoordinates) _coordinates;

          _3CoordFieldsTextFieldA.setText(stringHandler.meterToString(coordinates.getX()));
          _3CoordFieldsTextFieldB.setText(stringHandler.meterToString(coordinates.getY()));
          _3CoordFieldsTextFieldC.setText(stringHandler.meterToString(coordinates.getZ()));

          break;
        }
        case CoordinateType.GEODETIC:
        {
          GeodeticCoordinates coordinates = (GeodeticCoordinates) _coordinates;

          _3CoordFieldsTextFieldA.setText(stringHandler.longitudeToString(coordinates.getLongitude() * Constants._180_OVER_PI, useNSEW, useMinutes, useSeconds));
          _3CoordFieldsTextFieldB.setText(stringHandler.latitudeToString(coordinates.getLatitude() * Constants._180_OVER_PI, useNSEW, useMinutes, useSeconds));

		  //converts the output from meter to feet
          if ( formatOptions.getHeightUnit() == FormatOptions.SURVEY_FEET)
          {
        	  double feetToMeterRatio = (double)1200/3937;;
        	  _3CoordFieldsTextFieldC.setText(stringHandler.surveyFeetToString((coordinates.getHeight())/feetToMeterRatio));
          }
          else
          {
        	  _3CoordFieldsTextFieldC.setText(stringHandler.meterToString((coordinates.getHeight())));        	  
          }

          break;
        }
        case CoordinateType.UPS:
        {
          UPSCoordinates coordinates = (UPSCoordinates) _coordinates;

          hemisphere = coordinates.getHemisphere();
          _2CoordFieldsTextFieldA.setText(stringHandler.meterToString(coordinates.getEasting()));
          _2CoordFieldsTextFieldB.setText(stringHandler.meterToString(coordinates.getNorthing()));

          break;
        }
        case CoordinateType.UTM:
        {
          UTMCoordinates coordinates = (UTMCoordinates) _coordinates;

          hemisphere = coordinates.getHemisphere();
          zone = coordinates.getZone();
          _2CoordFieldsTextFieldA.setText(stringHandler.meterToString(coordinates.getEasting()));
          _2CoordFieldsTextFieldB.setText(stringHandler.meterToString(coordinates.getNorthing()));

          break;
        }
        default:
          throw new CoordinateConversionException("Invalid coordinate type");
      }
    }
    catch(CoordinateConversionException e)
    {
      stringHandler.setErrorMessage(true, e.getMessage());
      noErrors = false;
    }

    return noErrors;
  }

    
  // Set the input coordinates in the ccs
  public CoordinateTuple getCoordinates(int state, int direction, int projType) throws CoordinateConversionException
  {
    switch(projType)
    {
        case CoordinateType.ALBERS:
        case CoordinateType.AZIMUTHAL:
        case CoordinateType.BONNE:
        case CoordinateType.CASSINI:
        case CoordinateType.CYLEQA:
        case CoordinateType.ECKERT4:
        case CoordinateType.ECKERT6:
        case CoordinateType.EQDCYL:
        case CoordinateType.GNOMONIC:
        case CoordinateType.LAMBERT_1:
        case CoordinateType.LAMBERT_2:
        case CoordinateType.MERCATOR_SP:
        case CoordinateType.MERCATOR_SF:
        case CoordinateType.MILLER:
        case CoordinateType.MOLLWEIDE:
        case CoordinateType.NEYS:
        case CoordinateType.NZMG:
        case CoordinateType.OMERC:
        case CoordinateType.ORTHOGRAPHIC:
        case CoordinateType.POLARSTEREO_SP:
        case CoordinateType.POLARSTEREO_SF:
        case CoordinateType.POLYCONIC:
        case CoordinateType.SINUSOIDAL:
        case CoordinateType.STEREOGRAPHIC:
        case CoordinateType.TRCYLEQA:
        case CoordinateType.TRANMERC:
        case CoordinateType.GRINTEN:
        case CoordinateType.WEBMERCATOR:
        {
            MapProjectionCoordinates coordinates = new MapProjectionCoordinates(projType,
                                                   stringHandler.stringToDouble(_2CoordFieldsTextFieldA.getText().trim(), "Invalid Easting"),
                                                   stringHandler.stringToDouble(_2CoordFieldsTextFieldB.getText().trim(), "Invalid Northing"));

            return coordinates;
        }
        case CoordinateType.MGRS:
        {
          MGRSorUSNGCoordinates coordinates = new MGRSorUSNGCoordinates(projType, _3CoordFieldsTextFieldB.getText().trim(), formatOptions.getPrecision());

          return coordinates;
        }
        case CoordinateType.BNG:
        {
          BNGCoordinates coordinates = new BNGCoordinates(projType, _3CoordFieldsTextFieldB.getText().trim(), formatOptions.getPrecision());

          return coordinates;
        }
        case CoordinateType.GARS:
        {
          GARSCoordinates coordinates = new GARSCoordinates(projType, _3CoordFieldsTextFieldB.getText().trim(), formatOptions.getPrecision());

          return coordinates;
        }
        case CoordinateType.GEOREF:
        {
          GEOREFCoordinates coordinates = new GEOREFCoordinates(projType, _3CoordFieldsTextFieldB.getText().trim(), formatOptions.getPrecision());

          return coordinates;
        }
        case CoordinateType.USNG:
        {
            MGRSorUSNGCoordinates coordinates = new MGRSorUSNGCoordinates(projType, _3CoordFieldsTextFieldB.getText().trim(), formatOptions.getPrecision());

            return coordinates;
        }
        case CoordinateType.F16GRS:
        {
            java.lang.String coordString = _3CoordFieldsTextFieldB.getText().trim();
            int length = coordString.length();
            MGRSorUSNGCoordinates coordinates;
            if ((coordString.charAt(length - 1) == ('0')) && (coordString.charAt(length - 2) == ('0')))
                coordinates = new MGRSorUSNGCoordinates(CoordinateType.MGRS, coordString.substring(0, length - 2), formatOptions.getPrecision());
            else
                coordinates = new MGRSorUSNGCoordinates(CoordinateType.MGRS, coordString, formatOptions.getPrecision());

            return coordinates;
        }
        case CoordinateType.GEOCENTRIC:
        case CoordinateType.LOCCART:
        {
            CartesianCoordinates coordinates = new CartesianCoordinates(projType, 
                                               stringHandler.stringToDouble(_3CoordFieldsTextFieldA.getText().trim(), "Invalid X"),
                                               stringHandler.stringToDouble(_3CoordFieldsTextFieldB.getText().trim(), "Invalid Y"),
                                               stringHandler.stringToDouble(_3CoordFieldsTextFieldC.getText().trim(), "Invalid Z"));

            return coordinates;
        }
        case CoordinateType.GEODETIC:
        {
        	double textFieldC_value = stringHandler.stringToDouble(_3CoordFieldsTextFieldC.getText().trim(), "Invalid Height");

        	//converts the input from feet to meter        	
            if ( formatOptions.getHeightUnit() == FormatOptions.SURVEY_FEET)
            {
				double meterToFeetRatio = (double)3937/1200;				
            	textFieldC_value = textFieldC_value / meterToFeetRatio;
            }
        	
            GeodeticCoordinates coordinates = new GeodeticCoordinates(projType, 
                                              stringHandler.stringToLongitude(_3CoordFieldsTextFieldA.getText().trim(), "Invalid Longitude"),
                                              stringHandler.stringToLatitude(_3CoordFieldsTextFieldB.getText().trim(), "Invalid Latitude"),
                                              textFieldC_value);

            return coordinates;
        }
        case CoordinateType.UPS:
        {
            UPSCoordinates coordinates = new UPSCoordinates(projType,
                                         hemisphere,
                                         stringHandler.stringToDouble(_2CoordFieldsTextFieldA.getText().trim(), "Invalid Easting"),
                                         stringHandler.stringToDouble(_2CoordFieldsTextFieldB.getText().trim(), "Invalid Northing"));

            return coordinates;
        }
        case CoordinateType.UTM:
        {
            UTMCoordinates coordinates = new UTMCoordinates(projType,
                                         zone, hemisphere,
                                         stringHandler.stringToDouble(_2CoordFieldsTextFieldA.getText().trim(), "Invalid Easting"),
                                         stringHandler.stringToDouble(_2CoordFieldsTextFieldB.getText().trim(), "Invalid Northing"));

            return coordinates;
        }
        default:
          throw new CoordinateConversionException("Invalid coordinate type");
    }
  }

  
    public void createCoordPanel(int projType)
    {
        setFormat();

        switch (projType)
        {
            case CoordinateType.GEOCENTRIC:
            {
                setThreeCoords();
                break;
            }
            case CoordinateType.GEODETIC:
            { // height type
                _2CoordFieldsPanel.setVisible(false);
                _3CoordFieldsPanel.setVisible(true);
                coordLayeredPane.moveToFront(_3CoordFieldsPanel);
                // Show optional fields.
                _3CoordFieldsLabelA.setVisible(true);
                _3CoordFieldsLabelC.setVisible(true);
                _3CoordFieldsLabelUnit_C.setVisible(true);

                _3CoordFieldsTextFieldA.setVisible(true);
                _3CoordFieldsTextFieldC.setVisible(true);
                /* may be turned off by height type buttons. */
                setHeightFieldEditable();

                _3CoordFieldsLabelA.setText("Longitude:");
                _3CoordFieldsLabelB.setText("Latitude:");
                _3CoordFieldsLabelC.setText("Height");
                _3CoordFieldsLabelC.setBounds(296, 0, 39, 18);

                if (formatOptions.getHeightUnit() == FormatOptions.METER)
                {
                   	//plain font
                	_3CoordFieldsLabelUnit_C.setText("(m):");
                	if (javax.swing.UIManager.getLookAndFeel().getName().equalsIgnoreCase("Metal"))
                		_3CoordFieldsLabelUnit_C.setFont(_3CoordFieldsLabelUnitFont_C);
                	else
                		_3CoordFieldsLabelUnit_C.setFont(_3CoordFieldsLabelUnit_C.getFont().deriveFont(java.awt.Font.PLAIN));
                }
                else
                {
                   	//italtic font
                	_3CoordFieldsLabelUnit_C.setText("(ft):");
                	if (javax.swing.UIManager.getLookAndFeel().getName().equalsIgnoreCase("Metal"))
                		_3CoordFieldsLabelUnit_C.setFont(_3CoordFieldsLabelUnit_C.getFont().deriveFont(java.awt.Font.BOLD | java.awt.Font.ITALIC));
                	else
                		_3CoordFieldsLabelUnit_C.setFont(_3CoordFieldsLabelUnit_C.getFont().deriveFont(java.awt.Font.PLAIN | java.awt.Font.ITALIC));
                }                

                _3CoordFieldsLabelUnit_C.setBounds(339, 0, 30, 18);

                try
                {
                    _3CoordFieldsTextFieldA.setText(stringHandler.longitudeToString(0, useNSEW, useMinutes, useSeconds));
                    _3CoordFieldsTextFieldB.setText(stringHandler.latitudeToString(0, useNSEW, useMinutes, useSeconds));
                }
                catch(CoordinateConversionException e)
                {
                    stringHandler.displayErrorMsg(this, e.getMessage());
                }

                _3CoordFieldsTextFieldC.setText( "0" );
                break;
            }
            case CoordinateType.GEOREF:
            {
                setOneCoord();
                _3CoordFieldsTextFieldB.setText( "NGAA0000000000" );
                break;
            }
            case CoordinateType.GARS:
            {
                setOneCoord();
                _3CoordFieldsTextFieldB.setText( "361HN37" );
                break;
            }
            case CoordinateType.F16GRS:
            {
                setOneCoord();
                _3CoordFieldsTextFieldB.setText( "31NEA00" );
                break;
            }
            case CoordinateType.MGRS:
            {
                setOneCoord();
                _3CoordFieldsTextFieldB.setText( "31NEA0000000000" );
                break;
            }
            case CoordinateType.BNG:
            {
                setOneCoord();
                _3CoordFieldsTextFieldB.setText( "SV 0000000000" );
                break;
            }
            case CoordinateType.NZMG:
            {
                setTwoCoords();
                _2CoordFieldsTextFieldA.setText("2510000");
                _2CoordFieldsTextFieldB.setText("6023150");
                break;
            }
            case CoordinateType.UPS:
            { // Hemisphere
                setTwoCoords();
                _2CoordFieldsTextFieldA.setText("2000000");
                _2CoordFieldsTextFieldB.setText("2000000");
                break;
            }
            case CoordinateType.USNG:
            {
                setOneCoord();
                _3CoordFieldsTextFieldB.setText( "31NEA0000000000" );
                break;
            }
            case CoordinateType.UTM:
            {
                // Hemisphere  // Zone
                setTwoCoords();
                _2CoordFieldsTextFieldA.setText("500000");
                break;
            }
            case CoordinateType.LOCCART:
            {
                setThreeCoords();
                break;
            }
            case CoordinateType.ALBERS:
            case CoordinateType.AZIMUTHAL:
            case CoordinateType.BONNE:
            case CoordinateType.CASSINI:
            case CoordinateType.CYLEQA:
            case CoordinateType.ECKERT4:
            case CoordinateType.ECKERT6:
            case CoordinateType.EQDCYL:
            case CoordinateType.GNOMONIC:
            case CoordinateType.LAMBERT_1:
            case CoordinateType.LAMBERT_2:
            case CoordinateType.MERCATOR_SP:
            case CoordinateType.MERCATOR_SF:
            case CoordinateType.MILLER:
            case CoordinateType.MOLLWEIDE:
            case CoordinateType.NEYS:
            case CoordinateType.OMERC:
            case CoordinateType.ORTHOGRAPHIC:
            case CoordinateType.POLARSTEREO_SP:
            case CoordinateType.POLARSTEREO_SF:
            case CoordinateType.POLYCONIC:
            case CoordinateType.SINUSOIDAL:
            case CoordinateType.STEREOGRAPHIC:
            case CoordinateType.TRCYLEQA:
            case CoordinateType.TRANMERC:
            case CoordinateType.GRINTEN:
            case CoordinateType.WEBMERCATOR:
            {
                setTwoCoords();
                break;
            }
            default:
            {
                break;
            }
        } /*    end of switch(projType)  */
    } /*   end of createCoordPanel()     */

    
   public void resetOutputErrors()
   {
      ce90TextField.setText("N/A");
      le90TextField.setText("N/A");
      se90TextField.setText("N/A");
   }


   public void setAccuracy(Accuracy accuracy, boolean applyPrecision )
   {
      currentAccuracy = accuracy;

      double ce90 = accuracy.getCE90();
      double le90 = accuracy.getLE90();
      double se90 = accuracy.getSE90();

      if( applyPrecision )
      {
         int precision = formatOptions.getPrecision();
         double sigma = 100000.0 / Math.sqrt( 12.0 );
         for( int i = 0; i < precision; i++ )
         {
            sigma /= 10.0;
         }

         if( le90 > 0.0 )
         {
            double lePrec = 1.6449 * sigma;
            le90 = Math.sqrt( le90 * le90 + lePrec * lePrec );
         }

         if( ce90 > 0.0 )
         {
            double cePrec = 2.146 * sigma;
            ce90 = Math.sqrt( ce90 * ce90 + cePrec * cePrec );
         }

         if( se90 > 0.0 )
         {
            double sePrec = 2.146 * sigma;
            se90 = Math.sqrt( se90 * se90 + sePrec * sePrec );
         }
      }

      //      usePrevAccuracies = true;

      if(ce90 == -1.0)
         ce90TextField.setText("Unk");
      else
         stringHandler.setNumberFormat(ce90TextField, ce90, 0);

      if(!_3dConversion)
      {
         le90TextField.setText("N/A");
         se90TextField.setText("N/A");
      }
      else
      {
         if(le90 == -1.0)
            le90TextField.setText("Unk");
         else
            stringHandler.setNumberFormat(le90TextField, le90, 0);
         
         if(se90 == -1.0)
            se90TextField.setText("Unk");
         else
            stringHandler.setNumberFormat(se90TextField, se90, 0);
      }
   }

  // Set the input accuracy values
  public Accuracy getAccuracy()
  {
    if(!usePrevAccuracies)
    {
      java.lang.String ce90String = ce90TextField.getText().trim();
      java.lang.String le90String = le90TextField.getText().trim();
      java.lang.String se90String = se90TextField.getText().trim();

      double ce90, le90, se90;
      if((ce90String.equals("Unk")) || (ce90String.equals("N/A")))
        ce90 = -1.0;
      else
        ce90 = stringHandler.stringToDouble(ce90String, "Invalid CE 90");
      if((le90String.equals("Unk")) || (le90String.equals("N/A")))
        le90 = -1.0;
      else
        le90 = stringHandler.stringToDouble(le90String, "Invalid LE 90");
      if((se90String.equals("Unk")) || (se90String.equals("N/A")))
        se90 = -1.0;
      else
        se90 = stringHandler.stringToDouble(se90String, "Invalid SE 90");

      currentAccuracy = new Accuracy(ce90, le90, se90);
    }

    return currentAccuracy;
  }

  // Set the source text fields editable/uneditable if it's a 3D/2D conversion
   public void updateSrcErrors(boolean _3dConv, boolean applyPrecision )
  {
    le90TextField.setEditable(_3dConv);
    se90TextField.setEditable(_3dConv);
    _3dConversion = _3dConv;
    updateAccuracy(applyPrecision);
  }

  // Set the source text fields with the values corresponding
  // to the current combo box selection
  public void updateAccuracy( boolean applyPrecision )
  {
     try
     {
        int index = sourcesComboBox.getSelectedIndex(); /// + 1;

           Accuracy accuracy = Source.accuracy(index);

           setAccuracy( accuracy, applyPrecision );
           
           usePrevAccuracies = false;
     }
     catch(CoordinateConversionException e)
     {
        stringHandler.displayErrorMsg(this, e.getMessage());
     }
  }

  public void setHeightText(String height)
  {
    _3CoordFieldsTextFieldC.setText(height);
  }

  public void setHeightFieldUneditable()
  {
    _3CoordFieldsTextFieldC.setEditable(false);
  }

  public void setHeightFieldEditable()
  {
    _3CoordFieldsTextFieldC.setEditable(true);
  }

  public void setHeightFieldEditable(boolean editable)
  {
    _3CoordFieldsTextFieldC.setEditable(editable);
  }

  public void setHemisphere(char hemi)
  {
    hemisphere = hemi;
  }

  public void setZone(int z)
  {
    zone = z;
  }

  public char getHemisphere()
  {
    return hemisphere;
  }

  public long getZone()
  {
    return zone;
  }

  public void setDefaults(FormatOptions _formatOptions, StringHandler _stringHandler)
  {
    formatOptions = _formatOptions;
    stringHandler = _stringHandler;
  }


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JLabel _2CoordFieldsLabelA;
    private javax.swing.JLabel _2CoordFieldsLabelB;
    private javax.swing.JPanel _2CoordFieldsPanel;
    private javax.swing.JTextField _2CoordFieldsTextFieldA;
    private javax.swing.JTextField _2CoordFieldsTextFieldB;
    private javax.swing.JLabel _3CoordFieldsLabelA;
    private javax.swing.JLabel _3CoordFieldsLabelB;
    private javax.swing.JLabel _3CoordFieldsLabelC;
    private javax.swing.JLabel _3CoordFieldsLabelUnit_C;
    private java.awt.Font _3CoordFieldsLabelUnitFont_C;

    private javax.swing.JPanel _3CoordFieldsPanel;
    private javax.swing.JTextField _3CoordFieldsTextFieldA;
    private javax.swing.JTextField _3CoordFieldsTextFieldB;
    private javax.swing.JTextField _3CoordFieldsTextFieldC;
    private javax.swing.JLabel ce90Label;
    private javax.swing.JTextField ce90TextField;
    private javax.swing.JLayeredPane coordLayeredPane;
    private javax.swing.JPanel errorsPanel;
    private javax.swing.JPanel inputErrorPanel;
    private javax.swing.JLabel le90Label;
    private javax.swing.JTextField le90TextField;
    private javax.swing.JPanel outputErrorPanel;
    private javax.swing.JLabel se90Label;
    private javax.swing.JTextField se90TextField;
    private javax.swing.JComboBox sourcesComboBox;
    private javax.swing.JLabel sourcesLabel;
    private javax.swing.JLabel tempLabel1;
    private javax.swing.JLabel tempLabel2;
    private javax.swing.JTextField tempTextField1;
    private javax.swing.JTextField tempTextField2;
    // End of variables declaration//GEN-END:variables


}// CLASSIFICATION: UNCLASSIFIED
