/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001 National Institute of Water and Atmospheric Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include <stdlib.h>
#include <math.h>

#include "config.h"
#include "boundary.h"
#include "simulation.h"
#include "adaptive.h"

static FttVector rpos[FTT_NEIGHBORS] = {
#ifdef FTT_2D
  {1.,0.,0.}, {-1.,0.,0.}, {0.,1.,0.}, {0.,-1.,0.}
#else  /* FTT_3D */
  {1.,0.,0.}, {-1.,0.,0.}, {0.,1.,0.}, {0.,-1.,0.}, {0.,0.,1.}, {0.,0.,-1.}
#endif /* FTT_3D */
};

/* GfsBoundary: Object */

static void gfs_boundary_destroy (GtsObject * object)
{
  GfsBoundary * boundary = GFS_BOUNDARY (object);

  if (boundary->root)
    ftt_cell_destroy (boundary->root, gfs_cell_cleanup);
  boundary->box->neighbor[FTT_OPPOSITE_DIRECTION (boundary->d)] = NULL;

  (* GTS_OBJECT_CLASS (gfs_boundary_class ())->parent_class->destroy) (object);
}

static void match (FttCell * cell, GfsBoundary * boundary)
{
  FttCell * neighbor = ftt_cell_neighbor (cell, boundary->d);
  guint level = ftt_cell_level (cell);

  cell->flags |= GFS_FLAG_BOUNDARY;
  if (neighbor == NULL || ftt_cell_level (neighbor) < level) {
    g_assert (!FTT_CELL_IS_ROOT (cell));
    ftt_cell_destroy (cell, gfs_cell_cleanup);
    return;
  }
  if (ftt_cell_level (neighbor) == level) {
    gdouble s = GFS_IS_MIXED (neighbor) ?
      GFS_STATE (neighbor)->solid->s[FTT_OPPOSITE_DIRECTION (boundary->d)] 
      : 1.;

    if (s == 0.) {
      g_assert (!FTT_CELL_IS_ROOT (cell));
      ftt_cell_destroy (cell, gfs_cell_cleanup);
      return;
    }
    if (s < 1.) {
      if (GFS_STATE (cell)->solid == NULL)
	GFS_STATE (cell)->solid = g_malloc0 (sizeof (GfsSolidVector));
      GFS_STATE (cell)->solid->s[boundary->d] = s;
    }
    if (FTT_CELL_IS_LEAF (cell) && !FTT_CELL_IS_LEAF (neighbor))
      ftt_cell_refine_single (cell, (FttCellInitFunc) gfs_cell_init,
			      gfs_box_domain (boundary->box));
  }
  else
    g_assert_not_reached ();
  if (!FTT_CELL_IS_LEAF (cell))
    level++;
  if (level > boundary->depth)
    boundary->depth = level;
}

static void boundary_pressure (FttCellFace * face,
			       GfsBoundary * boundary)
{
  GFS_VARIABLE (face->cell, boundary->v->i) = 
    GFS_VARIABLE (face->neighbor, boundary->v->i);
}

static void boundary_center (FttCellFace * face,
			     GfsBoundary * boundary)
{
  if (boundary->v->i == GFS_VELOCITY_INDEX (face->d/2) ||
      boundary->v->i == GFS_GRADIENT_INDEX (face->d/2)) {
    GFS_VARIABLE (face->cell, boundary->v->i) = 
      - GFS_VARIABLE (face->neighbor, boundary->v->i);
    return;
  }

  GFS_VARIABLE (face->cell, boundary->v->i) =
    GFS_VARIABLE (face->neighbor, boundary->v->i);
}

static void boundary_face (FttCellFace * face,
			   GfsBoundary * boundary)
{
  GFS_STATE (face->cell)->f[face->d].v = 
    GFS_STATE (face->neighbor)->f[FTT_OPPOSITE_DIRECTION (face->d)].v;
}

static void boundary_match (GfsBoundary * boundary)
{
  guint l = 0;

  while (l <= boundary->depth) {
    ftt_cell_traverse_boundary (boundary->root, boundary->d,
				FTT_PRE_ORDER, FTT_TRAVERSE_LEVEL, l,
				(FttCellTraverseFunc) match, boundary);
    l++;
  }
  ftt_cell_flatten (boundary->root, boundary->d, NULL);
}

static void gfs_boundary_class_init (GfsBoundaryClass * klass)
{
  klass->pressure = boundary_pressure;
  klass->center = boundary_center;
  klass->face = boundary_face;
  klass->match = boundary_match;

  GTS_OBJECT_CLASS (klass)->destroy = gfs_boundary_destroy;
}

static void gfs_boundary_init (GfsBoundary * b)
{
  b->type = GFS_BOUNDARY_CENTER_VARIABLE;
}

GfsBoundaryClass * gfs_boundary_class (void)
{
  static GfsBoundaryClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_boundary_info = {
      "GfsBoundary",
      sizeof (GfsBoundary),
      sizeof (GfsBoundaryClass),
      (GtsObjectClassInitFunc) gfs_boundary_class_init,
      (GtsObjectInitFunc) gfs_boundary_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (gts_object_class (), &gfs_boundary_info);
  }

  return klass;
}

/**
 * gfs_boundary_new:
 * @klass: a #GfsBoundaryClass.
 * @box: a #GfsBox.
 * @d: a direction.
 *
 * Creates a new boundary of type @klass for @box in direction @d.
 *
 * This function fails if @box has already a boundary in direction @d.
 *
 * Returns: a new #GfsBoundary.
 */
GfsBoundary * gfs_boundary_new (GfsBoundaryClass * klass,
				GfsBox * box,
				FttDirection d)
{
  GfsBoundary * boundary;
  GfsDomain * domain;
  FttVector pos;
  gdouble size;

  g_return_val_if_fail (box != NULL, NULL);
  g_return_val_if_fail (d < FTT_NEIGHBORS, NULL);
  g_return_val_if_fail (box->neighbor[d] == NULL, NULL);

  boundary = GFS_BOUNDARY (gts_object_new (GTS_OBJECT_CLASS (klass)));
  domain = gfs_box_domain (box);
  boundary->root = ftt_cell_new ((FttCellInitFunc) gfs_cell_init, domain);
  boundary->box = box;
  box->neighbor[d] = GTS_OBJECT (boundary);
  boundary->d = FTT_OPPOSITE_DIRECTION (d);
  ftt_cell_set_level (boundary->root, ftt_cell_level (box->root));
  ftt_cell_set_neighbor_match (boundary->root, box->root, boundary->d, 
			       (FttCellInitFunc) gfs_cell_init, domain);
  ftt_cell_pos (box->root, &pos);
  size = ftt_cell_size (box->root);
  pos.x += rpos[d].x*size;
  pos.y += rpos[d].y*size;
  pos.z += rpos[d].z*size;
  ftt_cell_set_pos (boundary->root, &pos);

  boundary_match (boundary);

  return boundary;
}

/**
 * gfs_boundary_match:
 * @boundary: a #GfsBoundary.
 *
 * Calls the @match() method of @boundary.
 */
void gfs_boundary_match (GfsBoundary * boundary)
{
  g_return_if_fail (boundary != NULL);

  g_assert (GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass)->match);
  (* GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass)->match) (boundary);
}

/**
 * gfs_boundary_send:
 * @boundary: a #GfsBoundary.
 *
 * Calls the @send() method of @boundary.
 */
void gfs_boundary_send (GfsBoundary * boundary)
{
  g_return_if_fail (boundary != NULL);

  if (GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass)->send)
    (* GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass)->send) (boundary);
}

/**
 * gfs_boundary_receive:
 * @boundary: a #GfsBoundary.
 * @flags: the traversal flags.
 * @max_depth: the maximum depth of the traversal.
 *
 * Calls the @receive() method of @boundary.
 */
void gfs_boundary_receive (GfsBoundary * boundary,
			   FttTraverseFlags flags,
			   gint max_depth)
{
  g_return_if_fail (boundary != NULL);

  if (GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass)->receive)
    (* GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass)->receive) 
      (boundary, flags, max_depth);
}

/**
 * gfs_boundary_synchronize:
 * @boundary: a #GfsBoundary.
 *
 * Calls the @synchronize() method of @boundary.
 */
void gfs_boundary_synchronize (GfsBoundary * boundary)
{
  g_return_if_fail (boundary != NULL);

  if (GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass)->synchronize)
    (* GFS_BOUNDARY_CLASS (GTS_OBJECT (boundary)->klass)->synchronize) 
      (boundary);
}

/* GfsBoundaryInflow: Object */

static void center_inflow (FttCellFace * face,
			   GfsBoundary * boundary)
{
  if (boundary->v->i == GFS_VELOCITY_INDEX (face->d/2))
    return;

  if (boundary->v->i == GFS_GRADIENT_INDEX (face->d/2)) {
    GFS_VARIABLE (face->cell, boundary->v->i) = 
      - GFS_VARIABLE (face->neighbor, boundary->v->i);
    return;
  }

  GFS_VARIABLE (face->cell, boundary->v->i) =
    GFS_VARIABLE (face->neighbor, boundary->v->i);
}

static void face_inflow (FttCellFace * face,
			 GfsBoundary * boundary)
{
  GFS_STATE (face->cell)->f[face->d].v = 
    GFS_VARIABLE (face->cell, boundary->v->i);
}

static GtsColor inflow_color (GtsObject * o)
{
  GtsColor c = { 0., 0., 1. }; /* blue */

  return c;
}

static void gfs_boundary_inflow_class_init (GfsBoundaryClass * klass)
{
  klass->center =   center_inflow;
  klass->face =     face_inflow;

  GTS_OBJECT_CLASS (klass)->color = inflow_color;
}

GfsBoundaryInflowClass * gfs_boundary_inflow_class (void)
{
  static GfsBoundaryInflowClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_boundary_inflow_info = {
      "GfsBoundaryInflow",
      sizeof (GfsBoundary),
      sizeof (GfsBoundaryInflowClass),
      (GtsObjectClassInitFunc) gfs_boundary_inflow_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_boundary_class ()),
				  &gfs_boundary_inflow_info);
  }

  return klass;
}

/* GfsBoundaryInflowConstant: Object */

static void inflow_constant_init (FttCell * cell, 
				  GfsBoundaryInflowConstant * b)
{
  FttComponent c = GFS_BOUNDARY (b)->d/2;

  GFS_VARIABLE (cell, GFS_VELOCITY_INDEX (c)) = b->un;
}

static void inflow_constant_match (GfsBoundary * b)
{
  g_assert (GFS_BOUNDARY_CLASS (gfs_boundary_inflow_class ())->match);
  (* GFS_BOUNDARY_CLASS (gfs_boundary_inflow_class ())->match) (b);

  ftt_cell_traverse_boundary (b->root, b->d,
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttCellTraverseFunc) inflow_constant_init, b);
}

static void inflow_constant_write (GtsObject * o, FILE * fp)
{
  fprintf (fp, " %g", GFS_BOUNDARY_INFLOW_CONSTANT (o)->un);
}

static void inflow_constant_read (GtsObject ** o, GtsFile * fp)
{
  GfsBoundaryInflowConstant * b = GFS_BOUNDARY_INFLOW_CONSTANT (*o);  

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (un)");
    return;
  }
  b->un = atof (fp->token->str);
  gts_file_next_token (fp);
  gfs_boundary_match (GFS_BOUNDARY (b));
}

static void gfs_boundary_inflow_constant_class_init 
  (GfsBoundaryInflowConstantClass * klass)
{
  GFS_BOUNDARY_CLASS (klass)->match = inflow_constant_match;
  GTS_OBJECT_CLASS (klass)->read   = inflow_constant_read;
  GTS_OBJECT_CLASS (klass)->write  = inflow_constant_write;
}

static void gfs_boundary_inflow_constant_init 
  (GfsBoundaryInflowConstant * object)
{
  object->un = 0.;
}

GfsBoundaryInflowConstantClass * gfs_boundary_inflow_constant_class (void)
{
  static GfsBoundaryInflowConstantClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_boundary_inflow_constant_info = {
      "GfsBoundaryInflowConstant",
      sizeof (GfsBoundaryInflowConstant),
      sizeof (GfsBoundaryInflowConstantClass),
      (GtsObjectClassInitFunc) gfs_boundary_inflow_constant_class_init,
      (GtsObjectInitFunc) gfs_boundary_inflow_constant_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = 
      gts_object_class_new (GTS_OBJECT_CLASS (gfs_boundary_inflow_class ()),
			    &gfs_boundary_inflow_constant_info);
  }

  return klass;
}

/* GfsBoundaryInflowConstantTracer: Object */

static void center_constant_tracer (FttCellFace * face,
				    GfsBoundary * boundary)
{
  if (boundary->v->i == GFS_GRADIENT_INDEX (face->d/2))
    GFS_VARIABLE (face->cell, boundary->v->i) = 
      - GFS_VARIABLE (face->neighbor, boundary->v->i);
}

static void constant_tracer_cell_init (FttCell * cell, 
				       GfsBoundaryInflowConstantTracer * b)
{
  FttComponent c = FTT_ORTHOGONAL_COMPONENT (GFS_BOUNDARY (b)->d/2);
  FttVector p;

  ftt_cell_pos (cell, &p);
  if (fabs (((gdouble *)&p)[c]) < b->width)
    GFS_STATE (cell)->c = 1.;
  else
    GFS_STATE (cell)->c = 0.;
}

static void constant_tracer_match (GfsBoundary * b)
{
  g_assert (GFS_BOUNDARY_CLASS (gfs_boundary_inflow_constant_class ())->match);
  (* GFS_BOUNDARY_CLASS (gfs_boundary_inflow_constant_class ())->match) (b);

  ftt_cell_traverse_boundary (b->root, b->d,
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttCellTraverseFunc) constant_tracer_cell_init,
			      b);
}

static void constant_tracer_read (GtsObject ** o, GtsFile * fp)
{
  if (GTS_OBJECT_CLASS (gfs_boundary_inflow_constant_tracer_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_boundary_inflow_constant_tracer_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;
  
  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (width)");
    return;
  }
  GFS_BOUNDARY_INFLOW_CONSTANT_TRACER (*o)->width = atof (fp->token->str);
  gts_file_next_token (fp);
}

static void constant_tracer_write (GtsObject * o, FILE * fp)
{
  if (GTS_OBJECT_CLASS (gfs_boundary_inflow_constant_tracer_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_boundary_inflow_constant_tracer_class ())->parent_class->write) 
      (o, fp);
  fprintf (fp, " %g", GFS_BOUNDARY_INFLOW_CONSTANT_TRACER (o)->width);
}

static void constant_tracer_class_init (GfsBoundaryInflowConstantClass * klass)
{
  GFS_BOUNDARY_CLASS (klass)->center = center_constant_tracer;
  GFS_BOUNDARY_CLASS (klass)->match = constant_tracer_match;
  GTS_OBJECT_CLASS (klass)->read = constant_tracer_read;
  GTS_OBJECT_CLASS (klass)->write = constant_tracer_write;
}

static void constant_tracer_init (GfsBoundaryInflowConstantTracer * object)
{
  object->width = 0.;
}

GfsBoundaryInflowConstantClass * 
gfs_boundary_inflow_constant_tracer_class (void)
{
  static GfsBoundaryInflowConstantClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo constant_tracer_info = {
      "GfsBoundaryInflowConstantTracer",
      sizeof (GfsBoundaryInflowConstantTracer),
      sizeof (GfsBoundaryInflowConstantClass),
      (GtsObjectClassInitFunc) constant_tracer_class_init,
      (GtsObjectInitFunc) constant_tracer_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_boundary_inflow_constant_class ()),
				  &constant_tracer_info);
  }

  return klass;
}

/* GfsBoundaryOutflow: Object */

static void pressure_outflow (FttCellFace * face, 
			      GfsBoundary * boundary)
{
  GFS_VARIABLE (face->cell, boundary->v->i) = 
    - GFS_VARIABLE (face->neighbor, boundary->v->i);
}

static void center_outflow (FttCellFace * face, 
			    GfsBoundary * boundary)
{
  switch (boundary->v->i) {
  case GFS_GX: case GFS_GY: /* pressure gradient */
#ifndef FTT_2D
  case GFS_GZ:
#endif /* FTT_3D */
    GFS_VARIABLE (face->cell, boundary->v->i) = 
      - GFS_VARIABLE (face->neighbor, boundary->v->i);
    break;
  default: 
    GFS_VARIABLE (face->cell, boundary->v->i) = 
      GFS_VARIABLE (face->neighbor, boundary->v->i);
  }
}

static GtsColor outflow_color (GtsObject * o)
{
  GtsColor c = { 0., 1., 0. }; /* green */

  return c;
}

static void gfs_boundary_outflow_class_init (GfsBoundaryClass * klass)
{
  klass->pressure = pressure_outflow;
  klass->center   = center_outflow;

  GTS_OBJECT_CLASS (klass)->color = outflow_color;
}

GfsBoundaryOutflowClass * gfs_boundary_outflow_class (void)
{
  static GfsBoundaryOutflowClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_boundary_outflow_info = {
      "GfsBoundaryOutflow",
      sizeof (GfsBoundary),
      sizeof (GfsBoundaryOutflowClass),
      (GtsObjectClassInitFunc) gfs_boundary_outflow_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_boundary_class ()),
				  &gfs_boundary_outflow_info);
  }

  return klass;
}

/* GfsBoundaryOutflowSource: Object */

static void pressure_outflow_source (FttCellFace * face, 
				     GfsBoundary * boundary)
{
  switch (boundary->v->i) {
  case GFS_DP:
    GFS_VARIABLE (face->cell, boundary->v->i) = 
      - GFS_VARIABLE (face->neighbor, boundary->v->i);
    break;
  case GFS_P: {
    GfsDomain * domain = gfs_box_domain (boundary->box);
    gdouble cval = GFS_STATE (face->neighbor)->c;
    gdouble dt = GFS_SIMULATION (domain)->advection_params.dt;
    gdouble rho = GFS_SIMULATION (domain)->advection_params.rho;
    FttVector p;

    ftt_cell_pos (face->neighbor, &p);
    GFS_VARIABLE (face->cell, boundary->v->i) = 
      2.*dt*(1. + (cval > 1. ? 1. : cval < 0. ? 0. : cval)*(rho - 1.))*p.y*
      GFS_BOUNDARY_OUTFLOW_SOURCE (boundary)->intensity
      - GFS_VARIABLE (face->neighbor, boundary->v->i);
    break;
  }
  default:
    g_assert_not_reached ();
  }
}

static void gfs_boundary_outflow_source_read (GtsObject ** o, GtsFile * fp)
{
  if (GTS_OBJECT_CLASS (gfs_boundary_outflow_source_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_boundary_outflow_source_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (intensity)\n");
    return;
  }
  GFS_BOUNDARY_OUTFLOW_SOURCE (*o)->intensity = atof (fp->token->str);
  gts_file_next_token (fp);
}

static void gfs_boundary_outflow_source_write (GtsObject * o, FILE * fp)
{
  if (GTS_OBJECT_CLASS (gfs_boundary_outflow_source_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_boundary_outflow_source_class ())->parent_class->write) 
      (o, fp);
  fprintf (fp, " %g", GFS_BOUNDARY_OUTFLOW_SOURCE (o)->intensity);
}

static GtsColor outflow_source_color (GtsObject * o)
{
  GtsColor c = { 1., 0., 1. }; /* purple */

  return c;
}

static void 
gfs_boundary_outflow_source_class_init (GfsBoundaryOutflowClass * klass)
{
  GTS_OBJECT_CLASS (klass)->read = gfs_boundary_outflow_source_read;
  GTS_OBJECT_CLASS (klass)->write = gfs_boundary_outflow_source_write;
  GTS_OBJECT_CLASS (klass)->color = outflow_source_color;

  GFS_BOUNDARY_CLASS (klass)->pressure = pressure_outflow_source;
}

GfsBoundaryOutflowClass * gfs_boundary_outflow_source_class (void)
{
  static GfsBoundaryOutflowClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_boundary_outflow_source_info = {
      "GfsBoundaryOutflowSource",
      sizeof (GfsBoundaryOutflowSource),
      sizeof (GfsBoundaryOutflowClass),
      (GtsObjectClassInitFunc) gfs_boundary_outflow_source_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = 
      gts_object_class_new (GTS_OBJECT_CLASS (gfs_boundary_outflow_class ()),
			    &gfs_boundary_outflow_source_info);
  }

  return klass;
}

/* GfsGEdge: Object */

static void gfs_gedge_write (GtsObject * object, FILE * fp)
{
  fprintf (fp, " %s", ftt_direction_name [GFS_GEDGE (object)->d]);
}

static void gfs_gedge_read (GtsObject ** o, GtsFile * fp)
{
  GfsGEdge * e = GFS_GEDGE (*o);

  if (fp->type != GTS_STRING) {
    gts_file_error (fp, "expecting a string (direction)");
    return;
  }
  e->d = ftt_direction_from_name (fp->token->str);
  if (e->d >= FTT_NEIGHBORS) {
    gts_file_error (fp, "unknown direction `%s'", fp->token->str);
    e->d = 0;
    return;
  }
  gts_file_next_token (fp);
}

static void gfs_gedge_class_init (GtsObjectClass * klass)
{
  klass->write = gfs_gedge_write;
  klass->read = gfs_gedge_read;
}

static void gfs_gedge_init (GfsGEdge * object)
{
  object->d = -1;
}

GfsGEdgeClass * gfs_gedge_class (void)
{
  static GfsGEdgeClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gedge_info = {
      "GfsGEdge",
      sizeof (GfsGEdge),
      sizeof (GfsGEdgeClass),
      (GtsObjectClassInitFunc) gfs_gedge_class_init,
      (GtsObjectInitFunc) gfs_gedge_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gts_gedge_class ()),
				  &gfs_gedge_info);
  }

  return klass;
}

/**
 * gfs_gedge_link_boxes:
 * @edge: a #GfsGEdge.
 *
 * Links the two boxes connected by @edge. The boxes are set as their
 * respective neighbors in the direction defined by @edge and their
 * relative positions are set accordingly.
 */
void gfs_gedge_link_boxes (GfsGEdge * edge)
{
  GfsBox * b1, * b2;

  g_return_if_fail (edge != NULL);
  g_return_if_fail (GTS_GEDGE (edge)->n1 != NULL);
  g_return_if_fail (GTS_GEDGE (edge)->n2 != NULL);
  g_return_if_fail (edge->d >= 0 && edge->d < FTT_NEIGHBORS);

  b1 = GFS_BOX (GTS_GEDGE (edge)->n1);
  b2 = GFS_BOX (GTS_GEDGE (edge)->n2);

  g_return_if_fail (b1->neighbor[edge->d] == NULL);
  g_return_if_fail (b2->neighbor[FTT_OPPOSITE_DIRECTION (edge->d)] == NULL);

  ftt_cell_set_neighbor (b1->root, b2->root, edge->d, 
			 (FttCellInitFunc) gfs_cell_init, gfs_box_domain (b1));
  b1->neighbor[edge->d] = GTS_OBJECT (b2);
  b2->neighbor[FTT_OPPOSITE_DIRECTION (edge->d)] = GTS_OBJECT (b1);
  if (b1 != b2)
    gfs_box_set_relative_pos (b2, b1, edge->d);
}

/**
 * gfs_gedge_new:
 * @klass: a #GfsGEdgeClass.
 * @b1: a #GfsBox.
 * @b2: another #GfsBox.
 * @d: a direction.
 *
 * Returns: a new #GfsGEdge linking @b1 to @b2 in direction @d. The
 * boxes are linked using gfs_gedge_link_boxes().
 */
GfsGEdge * gfs_gedge_new (GfsGEdgeClass * klass,
			GfsBox * b1, GfsBox * b2,
			FttDirection d)
{
  GfsGEdge * edge;

  g_return_val_if_fail (klass != NULL, NULL);
  g_return_val_if_fail (b1 != NULL, NULL);
  g_return_val_if_fail (b2 != NULL, NULL);
  g_return_val_if_fail (d >= 0 && d < FTT_NEIGHBORS, NULL);

  edge = GFS_GEDGE (gts_gedge_new (GTS_GEDGE_CLASS (klass),
				  GTS_GNODE (b1), GTS_GNODE (b2)));
  edge->d = d;
  
  gfs_gedge_link_boxes (edge);

  return edge;
}

/* GfsBox: Object */

static void gfs_box_destroy (GtsObject * object)
{
  GfsBox * box = GFS_BOX (object);
  FttDirection d;

  if (box->root)
    ftt_cell_destroy (box->root, gfs_cell_cleanup);
  for (d = 0; d < FTT_NEIGHBORS; d++)
    if (GFS_IS_BOUNDARY (box->neighbor[d]))
      gts_object_destroy (box->neighbor[d]);
    else if (GFS_IS_BOX (box->neighbor[d])) {
      g_assert (GFS_BOX (box->neighbor[d])->neighbor[FTT_OPPOSITE_DIRECTION (d)] == GTS_OBJECT (box));
      GFS_BOX (box->neighbor[d])->neighbor[FTT_OPPOSITE_DIRECTION (d)] = NULL;
    }

  (* GTS_OBJECT_CLASS (gfs_box_class ())->parent_class->destroy) (object);
}

static void box_size (FttCell * cell, guint * size)
{
  (*size)++;
}

static void gfs_box_write (GtsObject * object, FILE * fp)
{
  GfsBox * box = GFS_BOX (object);
  FttDirection d;
  guint size = 0;
  GfsDomain * domain = gfs_box_domain (box);

  ftt_cell_traverse (box->root, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
		     (FttCellTraverseFunc) box_size, &size);
  fprintf (fp, "%s { id = %u pid = %d size = %u", 
	   object->klass->info.name, box->id, box->pid, size);
  for (d = 0; d < FTT_NEIGHBORS; d++)
    if (GFS_IS_BOUNDARY (box->neighbor[d])) {
      fprintf (fp, " %s = %s",
	       ftt_direction_name[d],
	       box->neighbor[d]->klass->info.name);
      if (box->neighbor[d]->klass->write)
	(* box->neighbor[d]->klass->write) (box->neighbor[d], fp);
    }
  fputs (" }", fp);
  if (domain != NULL && domain->max_depth_write > -2) {
    fputs (" {\n", fp);
    ftt_cell_write (box->root, domain->max_depth_write, fp, 
		    (FttCellWriteFunc) gfs_cell_write, 
		    domain->variables_io);
    fputc ('}', fp);
  }
}

static void gfs_box_read (GtsObject ** o, GtsFile * fp)
{
  GfsBox * b = GFS_BOX (*o);
  GtsObjectClass * klass;
  gboolean class_changed = FALSE;
  GtsFileVariable var[] = {
    {GTS_UINT, "id",     TRUE},
    {GTS_INT,  "pid",    TRUE},
    {GTS_UINT, "size",   TRUE},
    {GTS_FILE, "right",  TRUE},
    {GTS_FILE, "left",   TRUE},
    {GTS_FILE, "top",    TRUE},
    {GTS_FILE, "bottom", TRUE},
#ifndef FTT_2D
    {GTS_FILE, "front",  TRUE},
    {GTS_FILE, "back",   TRUE},
#endif /* 3D */
    {GTS_NONE}
  };
  GtsFileVariable * v;
  gfloat weight;
  GfsDomain * domain;

  g_assert (GTS_SLIST_CONTAINEE (b)->containers &&
	    !GTS_SLIST_CONTAINEE (b)->containers->next);
  domain = GFS_DOMAIN (GTS_SLIST_CONTAINEE (b)->containers->data);

  if (fp->type != GTS_STRING) {
    gts_file_error (fp, "expecting a string (GfsBoxClass)");
    return;
  }
  klass = gts_object_class_from_name (fp->token->str);
  if (klass == NULL) {
    gts_file_error (fp, "unknown class `%s'", fp->token->str);
    return;
  }
  if (!gts_object_class_is_from_class (klass, gfs_box_class ())) {
    gts_file_error (fp, "`%s' is not a GfsBox", fp->token->str);
    return;
  }
  if (klass != (*o)->klass) {
    *o = gts_object_new (klass);
    gts_object_destroy (GTS_OBJECT (b));
    b = GFS_BOX (*o);
    gts_container_add (GTS_CONTAINER (domain), GTS_CONTAINEE (b));
    class_changed = TRUE;
  }
  gts_file_next_token (fp);

  g_assert (b->root == NULL);
  b->root = ftt_cell_new ((FttCellInitFunc) gfs_cell_init, domain);

  weight = gts_gnode_weight (GTS_GNODE (b));
  var[0].data = &b->id;
  var[1].data = &b->pid;
  var[2].data = &b->size;
  gts_file_assign_start (fp, var);
  if (fp->type == GTS_ERROR)
    return;
  while ((v = gts_file_assign_next (fp, var)))
    if (v->type == GTS_FILE) {
      GtsObjectClass * boundary_class = 
	gts_object_class_from_name (fp->token->str);
      GtsObject * boundary;
	
      if (boundary_class == NULL) {
	gts_file_error (fp, "unknown class `%s'", fp->token->str);
	return;
      }
      if (!gts_object_class_is_from_class (boundary_class,
					   gfs_boundary_class ())) {
	gts_file_error (fp, "`%s' is not a GfsBoundary", fp->token->str);
	return;
      }
      boundary = 
	GTS_OBJECT (gfs_boundary_new (GFS_BOUNDARY_CLASS (boundary_class),
				      b, ftt_direction_from_name (v->name)));
      gts_file_next_token (fp);
      if (fp->type != '\n' && boundary_class->read)
	(* boundary_class->read) (&boundary, fp);
    }
  
  if (fp->type == '{') {
    FttDirection d;

    ftt_cell_destroy (b->root, gfs_cell_cleanup);
    fp->scope_max++;
    gts_file_first_token_after (fp, '\n');
    b->root = ftt_cell_read (fp, (FttCellReadFunc) gfs_cell_read, domain);
    fp->scope_max--;
    if (fp->type == GTS_ERROR)
      return;
    if (fp->type != '}') {
      gts_file_error (fp, "expecting a closing brace");
      return;
    }
    gts_file_next_token (fp);

    for (d = 0; d < FTT_NEIGHBORS; d++)
      if (GFS_IS_BOUNDARY (b->neighbor[d])) {
	GfsBoundary * boundary = GFS_BOUNDARY (b->neighbor[d]);

	ftt_cell_set_neighbor_match (boundary->root, b->root, boundary->d, 
				     (FttCellInitFunc) gfs_cell_init, domain);
      }
  }

  if (ftt_cell_level (b->root) != domain->rootlevel)
    ftt_cell_set_level (b->root, domain->rootlevel);
  /* updates weight of domain */
  GTS_WGRAPH (domain)->weight += gts_gnode_weight (GTS_GNODE (b)) - weight;

  if (class_changed && klass->read)
    (* klass->read) (o, fp);
}

static gfloat gfs_box_weight (GtsGNode * node)
{
  GfsBox * box = GFS_BOX (node);

  if (box->size >= 0)
    return box->size;
  else {
    guint size = 0;

    if (box->root)
      ftt_cell_traverse (box->root, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			 (FttCellTraverseFunc) box_size, &size);
    return size;
  }
}

static void gfs_box_class_init (GfsBoxClass * klass)
{
  GTS_GNODE_CLASS (klass)->weight = gfs_box_weight;

  GTS_OBJECT_CLASS (klass)->destroy = gfs_box_destroy;
  GTS_OBJECT_CLASS (klass)->write = gfs_box_write;
  GTS_OBJECT_CLASS (klass)->read = gfs_box_read;
}

static void gfs_box_init (GfsBox * box)
{
  static guint id = 1;

  box->id = id++;
  box->pid = -1;
  box->size = -1;
}

GfsBoxClass * gfs_box_class (void)
{
  static GfsBoxClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_box_info = {
      "GfsBox",
      sizeof (GfsBox),
      sizeof (GfsBoxClass),
      (GtsObjectClassInitFunc) gfs_box_class_init,
      (GtsObjectInitFunc) gfs_box_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gts_gnode_class ()),
				  &gfs_box_info);
  }

  return klass;
}

GfsBox * gfs_box_new (GfsBoxClass * klass)
{
  GfsBox * object;

  object = GFS_BOX (gts_object_new (GTS_OBJECT_CLASS (klass)));

  return object;
}

/* GfsBoxNotAdapt: Object */

static void gfs_box_not_adapt_read (GtsObject ** o, GtsFile * fp)
{
  GfsSimulation * sim;
  GfsAdaptNotBox * a;
  GfsBoxNotAdapt * b = GFS_BOX_NOT_ADAPT (*o);
  GfsDomain * domain = gfs_box_domain (GFS_BOX (b));

  g_assert (GFS_IS_SIMULATION (domain));
  sim = GFS_SIMULATION (domain);

  g_assert (gts_container_size (GTS_CONTAINER (b->c)) == 0);
  a = gfs_adapt_not_box_new (gfs_adapt_not_box_class (), GFS_BOX (b));
  gts_container_add (GTS_CONTAINER (b->c), GTS_CONTAINEE (a));
  gts_container_add (GTS_CONTAINER (sim->adapts), GTS_CONTAINEE (a));
}

static void gfs_box_not_adapt_destroy (GtsObject * object)
{
  GfsBoxNotAdapt * b = GFS_BOX_NOT_ADAPT (object);

  gts_container_foreach (GTS_CONTAINER (b->c),
			 (GtsFunc) gts_object_destroy, NULL);
  gts_object_destroy (GTS_OBJECT (b->c));

  (* GTS_OBJECT_CLASS (gfs_box_not_adapt_class ())->parent_class->destroy) 
    (object);
}

static void gfs_box_not_adapt_class_init (GfsBoxClass * klass)
{
  GTS_OBJECT_CLASS (klass)->read = gfs_box_not_adapt_read;
  GTS_OBJECT_CLASS (klass)->destroy = gfs_box_not_adapt_destroy;
}

static void gfs_box_not_adapt_init (GfsBoxNotAdapt * object)
{
  object->c = GTS_SLIST_CONTAINER (gts_container_new 
				   (GTS_CONTAINER_CLASS 
				    (gts_slist_container_class ())));
}

GfsBoxClass * gfs_box_not_adapt_class (void)
{
  static GfsBoxClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_box_not_adapt_info = {
      "GfsBoxNotAdapt",
      sizeof (GfsBoxNotAdapt),
      sizeof (GfsBoxClass),
      (GtsObjectClassInitFunc) gfs_box_not_adapt_class_init,
      (GtsObjectInitFunc) gfs_box_not_adapt_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_box_class ()),
				  &gfs_box_not_adapt_info);
  }

  return klass;
}

static void box_set_pos (GfsBox * box, FttVector * pos, 
			 GHashTable * set,
			 FttDirection dold)
{
  FttVector p;
  static FttDirection id[FTT_NEIGHBORS][FTT_NEIGHBORS] = 
#ifdef FTT_2D
  {
    {0,1,2,3},
    {1,0,3,2},
    {2,3,1,0},
    {3,2,0,1},
  };
#else  /* 3D */
  {
    {0,1,2,3,5,4},
    {1,0,3,2,4,5},
    {2,3,1,0,5,4},
    {3,2,0,1,4,5},
    {4,5,2,3,0,1},
    {5,4,3,2,1,0}
  };
#endif /* 3D */

  if (g_hash_table_lookup (set, box))
    return;
  g_hash_table_insert (set, box, box);

  ftt_cell_pos (box->root, &p);
  if (pos->x != p.x || pos->y != p.y || pos->z != p.z) {
    FttDirection i;
    gdouble size = ftt_cell_size (box->root);

    ftt_cell_set_pos (box->root, pos);
    for (i = 0; i < FTT_NEIGHBORS; i++) {
      FttDirection d = id[dold][i];

      p.x = pos->x + rpos[d].x*size;
      p.y = pos->y + rpos[d].y*size;
      p.z = pos->z + rpos[d].z*size;
      if (GFS_IS_BOX (box->neighbor[d]))
	box_set_pos (GFS_BOX (box->neighbor[d]), &p, set, d);
      else if (GFS_IS_BOUNDARY (box->neighbor[d]))
	ftt_cell_set_pos (GFS_BOUNDARY (box->neighbor[d])->root, &p);
    }
  }
}

/**
 * gfs_box_set_pos:
 * @box: a #GfsBox.
 * @pos: the new position of the center of the box.
 *
 * Recursively sets the position of the center of @box and of its
 * neighbors.  
 */
void gfs_box_set_pos (GfsBox * box, FttVector * pos)
{
  GHashTable * set;

  g_return_if_fail (box != NULL);
  g_return_if_fail (pos != NULL);

  set = g_hash_table_new (NULL, NULL);
  box_set_pos (box, pos, set, FTT_RIGHT);
  g_hash_table_destroy (set);
}

/**
 * gfs_box_set_relative_pos:
 * @box: a #GfsBox.
 * @reference: a reference #GfsBox.
 * @d: the direction in which @box is found relative to @reference.
 *
 * Recursively sets the position of the center of @box and of its
 * neighbors relative to the position of @reference in direction @d.
 */
void gfs_box_set_relative_pos (GfsBox * box, GfsBox * reference, FttDirection d)
{
  FttVector pos;
  gdouble size;

  g_return_if_fail (box != NULL);
  g_return_if_fail (reference != NULL);
  g_return_if_fail (d >= 0 && d < FTT_NEIGHBORS);

  ftt_cell_pos (reference->root, &pos);
  size = ftt_cell_size (reference->root);
  pos.x += rpos[d].x*size;
  pos.y += rpos[d].y*size;
  pos.z += rpos[d].z*size;
  gfs_box_set_pos (box, &pos);
}

#ifndef G_CAN_INLINE
/**
 * gfs_box_domain:
 * @box: a #GfsBox.
 *
 * Returns: the #GfsDomain to which @box belongs or %NULL if @box does not
 * belong to any domain.
 */ 
GfsDomain * gfs_box_domain (GfsBox * box)
{
  g_return_val_if_fail (box != NULL, NULL);
  
  if (GTS_SLIST_CONTAINEE (box)->containers) {
    g_assert (!GTS_SLIST_CONTAINEE (box)->containers->next);
    return GTS_SLIST_CONTAINEE (box)->containers->data;
  }
  return NULL;
}
#endif /* not G_CAN_INLINE */
