/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001 National Institute of Water and Atmospheric Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#ifndef __BOUNDARY_H__
#define __BOUNDARY_H__

#include "fluid.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef struct _GfsDomain         GfsDomain;

typedef enum {
  GFS_BOUNDARY_CENTER_VARIABLE,
  GFS_BOUNDARY_FACE_VARIABLE,
  GFS_BOUNDARY_MATCH_VARIABLE,
  GFS_BOUNDARY_VARIABLE_NUMBER
} GfsBoundaryVariableType;

typedef struct _GfsBox                    GfsBox;
typedef struct _GfsBoxClass               GfsBoxClass;
typedef struct _GfsBoundary               GfsBoundary;
typedef struct _GfsBoundaryClass          GfsBoundaryClass;

struct _GfsBoundary {
  /*< private >*/
  GtsObject parent;

  FttCell * root;
  GfsBox * box;
  FttDirection d;
  guint depth;

  /*< public >*/
  GfsVariable * v;
  GfsBoundaryVariableType type;
};

struct _GfsBoundaryClass {
  GtsObjectClass parent_class;

  void (* pressure)          (FttCellFace * face, 
			      GfsBoundary * boundary);
  void (* center)            (FttCellFace * face, 
			      GfsBoundary * boundary);
  void (* face)              (FttCellFace * face, 
			      GfsBoundary * boundary);
  void (* match)             (GfsBoundary * boundary);

  void (* send)              (GfsBoundary * boundary);
  void (* receive)           (GfsBoundary * boundary,
			      FttTraverseFlags flags,
			      gint max_depth);
  void (* synchronize)       (GfsBoundary * boundary);
};

#define GFS_BOUNDARY(obj)            GTS_OBJECT_CAST (obj,\
					           GfsBoundary,\
					           gfs_boundary_class ())
#define GFS_BOUNDARY_CLASS(klass)    GTS_OBJECT_CLASS_CAST (klass,\
						   GfsBoundaryClass,\
						   gfs_boundary_class())
#define GFS_IS_BOUNDARY(obj)         (gts_object_is_from_class (obj,\
						   gfs_boundary_class ()))
     
GfsBoundaryClass * gfs_boundary_class                (void);
GfsBoundary *      gfs_boundary_new                  (GfsBoundaryClass * klass,
						    GfsBox * box,
						    FttDirection d);
void              gfs_boundary_match                (GfsBoundary * boundary);
void              gfs_boundary_send                 (GfsBoundary * boundary);
void              gfs_boundary_receive              (GfsBoundary * boundary,
						    FttTraverseFlags flags,
						    gint max_depth);
void              gfs_boundary_synchronize          (GfsBoundary * boundary);

typedef struct _GfsBoundaryInflowClass    GfsBoundaryInflowClass;

struct _GfsBoundaryInflowClass {
  GfsBoundaryClass parent_class;
};

#define GFS_BOUNDARY_INFLOW_CLASS(klass)    GTS_OBJECT_CLASS_CAST (klass,\
						   GfsBoundaryInflowClass,\
						   gfs_boundary_inflow_class())
#define GFS_IS_BOUNDARY_INFLOW(obj)         (gts_object_is_from_class (obj,\
					    gfs_boundary_inflow_class ()))
     
GfsBoundaryInflowClass * gfs_boundary_inflow_class    (void);

typedef struct _GfsBoundaryInflowConstant         GfsBoundaryInflowConstant;
typedef struct _GfsBoundaryInflowConstantClass    GfsBoundaryInflowConstantClass;

struct _GfsBoundaryInflowConstant {
  GfsBoundary parent;

  gdouble un;
};

struct _GfsBoundaryInflowConstantClass {
  GfsBoundaryInflowClass parent_class;
};

#define GFS_BOUNDARY_INFLOW_CONSTANT(obj)  GTS_OBJECT_CAST (obj,\
					 GfsBoundaryInflowConstant,\
					 gfs_boundary_inflow_constant_class ())
#define GFS_BOUNDARY_INFLOW_CONSTANT_CLASS(klass) GTS_OBJECT_CLASS_CAST (klass,\
					 GfsBoundaryInflowConstantClass,\
					 gfs_boundary_inflow_constant_class())
#define GFS_IS_BOUNDARY_INFLOW_CONSTANT(obj) (gts_object_is_from_class (obj,\
					 gfs_boundary_inflow_constant_class ()))
     
GfsBoundaryInflowConstantClass * gfs_boundary_inflow_constant_class  (void);

/* GfsBoundaryInflowConstantTracer: Header */

typedef struct _GfsBoundaryInflowConstantTracer GfsBoundaryInflowConstantTracer;

#define GFS_BOUNDARY_INFLOW_CONSTANT_TRACER(obj)   GTS_OBJECT_CAST (obj,\
				 GfsBoundaryInflowConstantTracer,\
				 gfs_boundary_inflow_constant_tracer_class ())

struct _GfsBoundaryInflowConstantTracer {
  /*< private >*/
  GfsBoundaryInflowConstant parent;

  /*< public >*/
  gdouble width;
};

GfsBoundaryInflowConstantClass * 
gfs_boundary_inflow_constant_tracer_class  (void);

typedef struct _GfsBoundaryOutflowClass    GfsBoundaryOutflowClass;

struct _GfsBoundaryOutflowClass {
  GfsBoundaryClass parent_class;
};

#define GFS_BOUNDARY_OUTFLOW_CLASS(klass)    GTS_OBJECT_CLASS_CAST (klass,\
						   GfsBoundaryOutflowClass,\
						   gfs_boundary_outflow_class())
#define GFS_IS_BOUNDARY_OUTFLOW(obj)         (gts_object_is_from_class (obj,\
					    gfs_boundary_outflow_class ()))
     
GfsBoundaryOutflowClass * gfs_boundary_outflow_class    (void);

/* GfsBoundaryOutflowSource: Header */

typedef struct _GfsBoundaryOutflowSource         GfsBoundaryOutflowSource;

struct _GfsBoundaryOutflowSource {
  /*< private >*/
  GfsBoundary parent;

  /*< public >*/
  gdouble intensity;
};

#define GFS_BOUNDARY_OUTFLOW_SOURCE(obj)            GTS_OBJECT_CAST (obj,\
					GfsBoundaryOutflowSource,\
					gfs_boundary_outflow_source_class ())
#define GFS_IS_BOUNDARY_OUTFLOW_SOURCE(obj) (gts_object_is_from_class (obj,\
					gfs_boundary_outflow_source_class ()))

GfsBoundaryOutflowClass * gfs_boundary_outflow_source_class  (void);

typedef struct _GfsGEdge         GfsGEdge;
typedef struct _GfsGEdgeClass    GfsGEdgeClass;

struct _GfsGEdge {
  GtsGEdge parent;

  FttDirection d;
};

struct _GfsGEdgeClass {
  GtsGEdgeClass parent_class;
};

#define GFS_GEDGE(obj)            GTS_OBJECT_CAST (obj,\
					          GfsGEdge,\
					          gfs_gedge_class ())
#define GFS_GEDGE_CLASS(klass)    GTS_OBJECT_CLASS_CAST (klass,\
						  GfsGEdgeClass,\
						  gfs_gedge_class())
#define GFS_IS_GEDGE(obj)         (gts_object_is_from_class (obj,\
						  gfs_gedge_class ()))
     
GfsGEdgeClass * gfs_gedge_class          (void);
GfsGEdge *      gfs_gedge_new            (GfsGEdgeClass * klass,
					GfsBox * b1, GfsBox * b2,
					FttDirection d);
void           gfs_gedge_link_boxes     (GfsGEdge * edge);

struct _GfsBox {
  GtsGNode parent;

  FttCell * root;
  GtsObject * neighbor[FTT_NEIGHBORS];
  guint id;
  int pid;
  gint size;
};

struct _GfsBoxClass {
  GtsGNodeClass parent_class;
};

#define GFS_BOX(obj)            GTS_OBJECT_CAST (obj,\
					        GfsBox,\
					        gfs_box_class ())
#define GFS_BOX_CLASS(klass)    GTS_OBJECT_CLASS_CAST (klass,\
						      GfsBoxClass,\
						      gfs_box_class())
#define GFS_IS_BOX(obj)         (gts_object_is_from_class (obj,\
						          gfs_box_class ()))
     
GfsBoxClass *    gfs_box_class                (void);
GfsBox *         gfs_box_new                  (GfsBoxClass * klass);
void             gfs_box_set_pos              (GfsBox * box, 
					       FttVector * pos);
void             gfs_box_set_relative_pos     (GfsBox * box, 
					       GfsBox * reference, 
					       FttDirection d);
G_INLINE_FUNC
GfsDomain *      gfs_box_domain               (GfsBox * box);

#ifdef G_CAN_INLINE
G_INLINE_FUNC
GfsDomain * gfs_box_domain (GfsBox * box)
{
  g_return_val_if_fail (box != NULL, NULL);
  
  if (GTS_SLIST_CONTAINEE (box)->containers) {
    g_assert (!GTS_SLIST_CONTAINEE (box)->containers->next);
    return GTS_SLIST_CONTAINEE (box)->containers->data;
  }
  return NULL;
}
#endif /* G_CAN_INLINE */

/* GfsBoxNotAdapt: Header */

typedef struct _GfsBoxNotAdapt         GfsBoxNotAdapt;

struct _GfsBoxNotAdapt {
  /*< private >*/
  GfsBox parent;
  GtsSListContainer * c;
};

#define GFS_BOX_NOT_ADAPT(obj)            GTS_OBJECT_CAST (obj,\
					         GfsBoxNotAdapt,\
					         gfs_box_not_adapt_class ())
#define GFS_IS_BOX_NOT_ADAPT(obj)         (gts_object_is_from_class (obj,\
						 gfs_box_not_adapt_class ()))

GfsBoxClass * gfs_box_not_adapt_class  (void);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __BOUNDARY_H__ */
