/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001 National Institute of Water and Atmospheric Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include <math.h>
#include "poisson.h"

static void relax (FttCell * cell, gpointer * data)
{
  guint u = ((GfsVariable *) data[0])->i;
  guint rhs = ((GfsVariable *) data[1])->i;
  gint max_level = *((gint *) data[2]);
  GfsGradient g = { 0., 0. };
  FttCellNeighbors neighbor;
  FttCellFace f;
  GfsGradient ng;

  f.cell = cell;
  ftt_cell_neighbors (cell, &neighbor);
  for (f.d = 0; f.d < FTT_NEIGHBORS; f.d++) {
    f.neighbor = neighbor.c[f.d];
    if (f.neighbor) {
      gfs_face_weighted_gradient (&f, &ng, u, max_level);
      g.a += ng.a;
      g.b += ng.b;
    }
  }
  if (g.a > 0.)
    GFS_VARIABLE (cell, u) = (g.b - GFS_VARIABLE (cell, rhs))/g.a;
  else
    GFS_VARIABLE (cell, u) = 0.;
}

/**
 * gfs_relax:
 * @domain: the domain to relax.
 * @max_depth: the maximum depth of the domain to relax.
 * @u: the variable to use as left-hand side.
 * @rhs: the variable to use as right-hand side.
 *
 * Apply one pass of a Jacobi relaxation to all the leaf cells of
 * @domain with a level inferior or equal to @max_depth and to all the
 * cells at level @max_depth. The relaxation should converge (if the
 * right-hand-side @rhs verifies the solvability conditions) toward
 * the solution of a Poisson equation for @u at the maximum depth.
 */
void gfs_relax (GfsDomain * domain,
		gint max_depth,
		GfsVariable * u, 
		GfsVariable * rhs)
{
  gpointer data[3];

  g_return_if_fail (domain != NULL);

  data[0] = u;
  data[1] = rhs;
  data[2] = &max_depth;

  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, 
			   FTT_TRAVERSE_LEVEL | FTT_TRAVERSE_LEAFS,
			   max_depth, 
			   (FttCellTraverseFunc) relax, data);
}

static void residual_set (FttCell * cell, gpointer * data)
{
  guint u = ((GfsVariable *) data[0])->i;
  guint rhs = ((GfsVariable *) data[1])->i;
  guint res = ((GfsVariable *) data[2])->i;
  GfsGradient g = { 0., 0. };
  FttCellNeighbors neighbor;
  FttCellFace f;
  GfsGradient ng;

  f.cell = cell;
  ftt_cell_neighbors (cell, &neighbor);
  for (f.d = 0; f.d < FTT_NEIGHBORS; f.d++) {
    f.neighbor = neighbor.c[f.d];
    if (f.neighbor) {
      gfs_face_weighted_gradient (&f, &ng, u, -1);
      g.a += ng.a;
      g.b += ng.b;
    }
  }
  GFS_VARIABLE (cell, res) = GFS_VARIABLE (cell, rhs) - 
    (g.b - GFS_VARIABLE (cell, u)*g.a);
}

/**
 * gfs_residual:
 * @domain: a domain.
 * @flags: which types of cells are to be visited.
 * @max_depth: maximum depth of the traversal.
 * @u: the variable to use as left-hand side.
 * @rhs: the variable to use as right-hand side.
 * @res: the variable to use to store the residual.
 *
 * For each cell of @domain, computes the sum of the residual over
 * the volume of the cell for a Poisson equation with @u as
 * left-hand-side and @rhs as right-hand-side. Stores the result in
 * @res.  
 */
void gfs_residual (GfsDomain * domain,
		   FttTraverseFlags flags,
		   gint max_depth,
		   GfsVariable * u, GfsVariable * rhs, GfsVariable * res)
{
  gpointer data[3];

  g_return_if_fail (domain != NULL);

  data[0] = u;
  data[1] = rhs;
  data[2] = res;

  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, flags, max_depth,
			   (FttCellTraverseFunc) residual_set, data);
}

static void reset_coeff (FttCell * cell)
{
  FttDirection d;
  GfsFaceStateVector * f = GFS_STATE (cell)->f;
  
  for (d = 0; d < FTT_NEIGHBORS; d++)
    f[d].v = 0.;
}

static void poisson_coeff (FttCellFace * face, gdouble * lambda2)
{
  GfsStateVector * s = GFS_STATE (face->cell);
  gdouble v = lambda2[face->d/2];

  if (GFS_IS_MIXED (face->cell))
    v *= s->solid->s[face->d];
  s->f[face->d].v = v;

  switch (ftt_face_type (face)) {
  case FTT_FINE_FINE:
    GFS_STATE (face->neighbor)->f[FTT_OPPOSITE_DIRECTION (face->d)].v = v;
    break;
  case FTT_FINE_COARSE:
    GFS_STATE (face->neighbor)->f[FTT_OPPOSITE_DIRECTION (face->d)].v +=
      v/(FTT_CELLS/2);
    break;
  default:
    g_assert_not_reached ();
  }
}

static void poisson_density_coeff (FttCellFace * face,
				   gpointer * data)
{
  GfsVariable * c = data[0];
  gdouble * rho = data[1];
  gdouble * lambda2 = data[2];
  gdouble v = lambda2[face->d/2], cval;
  GfsStateVector * s = GFS_STATE (face->cell);

  if (GFS_IS_MIXED (face->cell))
    v *= s->solid->s[face->d];
  cval = gfs_face_interpolated_value (face, c->i);
  v /= 1. + (cval > 1. ? 1. : cval < 0. ? 0. : cval)*(*rho - 1.);
  s->f[face->d].v = v;

  switch (ftt_face_type (face)) {
  case FTT_FINE_FINE:
    GFS_STATE (face->neighbor)->f[FTT_OPPOSITE_DIRECTION (face->d)].v = v;
    break;
  case FTT_FINE_COARSE:
    GFS_STATE (face->neighbor)->f[FTT_OPPOSITE_DIRECTION (face->d)].v +=
      v/(FTT_CELLS/2);
    break;
  default:
    g_assert_not_reached ();
  }
}

static void poisson_coeff_from_below (FttCell * cell)
{
  FttDirection d;
  GfsFaceStateVector * f = GFS_STATE (cell)->f;

  for (d = 0; d < FTT_NEIGHBORS; d++) {
    FttCellChildren child;
    guint i;

    f[d].v = 0.;
    ftt_cell_children_direction (cell, d, &child);
    for (i = 0; i < FTT_CELLS/2; i++)
      if (child.c[i])
	f[d].v += GFS_STATE (child.c[i])->f[d].v;
    f[d].v /= FTT_CELLS/2;
  }
}

/**
 * gfs_poisson_coefficients:
 * @domain: a #GfsDomain.
 * @c: the volume fraction.
 * @rho: the relative density.
 *
 * Initializes the face coefficients for the Poisson equation.
 */
void gfs_poisson_coefficients (GfsDomain * domain,
			       GfsVariable * c,
			       gdouble rho)
{
  gdouble lambda2[FTT_DIMENSION];
  FttComponent i;

  g_return_if_fail (domain != NULL);

  for (i = 0; i < FTT_DIMENSION; i++) {
    gdouble lambda = (&domain->lambda.x)[i];

    lambda2[i] = lambda*lambda;
  }
  gfs_domain_cell_traverse (domain,
			    FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) reset_coeff, NULL);
  if (c == NULL || rho == 1.)
    gfs_domain_face_traverse (domain, FTT_XYZ, 
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttFaceTraverseFunc) poisson_coeff, lambda2);
  else {
    gpointer data[3];

    data[0] = c;
    data[1] = &rho;
    data[2] = lambda2;
    gfs_domain_face_traverse (domain, FTT_XYZ, 
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttFaceTraverseFunc) poisson_density_coeff, 
			      data);
  }
  gfs_domain_cell_traverse (domain,
			    FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
			    (FttCellTraverseFunc) poisson_coeff_from_below, 
			    NULL);
}

static void correct (FttCell * cell, GfsVariable * u) 
{
  GFS_VARIABLE (cell, u->i) += GFS_STATE (cell)->dp;
}

/**
 * gfs_poisson:
 * @domain: the domain on which to solve the Poisson equation.
 * @levelmin: the top level of the multigrid hierarchy.
 * @depth: the total depth of the domain.
 * @nrelax: the number of relaxations to apply at each level.
 * @u: the variable to use as left-hand side.
 * @rhs: the variable to use as right-hand side.
 *
 * Apply one multigrid iteration to the Poisson equation defined by @u
 * and @rhs.
 *
 * The initial value of %GFS_RES on the leaves of @root must be set to
 * the residual of the Poisson equation (using gfs_residual()).
 *
 * The face coefficients must be set using gfs_poisson_coefficients().
 *
 * The values of @u on the leaf cells are updated as well as the values
 * of %GFS_RES (i.e. the cell tree is ready for another iteration).
 */
void gfs_poisson (GfsDomain * domain,
		  guint levelmin,
		  guint depth,
		  guint nrelax,
		  GfsVariable * u,
		  GfsVariable * rhs)
{
  guint n, l;

  g_return_if_fail (domain != NULL);

  /* compute residual on non-leafs cells */
  gfs_domain_cell_traverse (domain, 
			    FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
      (FttCellTraverseFunc) gfs_get_from_below_extensive, 
			    gfs_res);
  /* relax top level */
  gfs_domain_cell_traverse (domain, 
			    FTT_PRE_ORDER, FTT_TRAVERSE_LEVEL, levelmin,
			    (FttCellTraverseFunc) gfs_cell_reset, gfs_dp);
  for (n = 0; n < 10*nrelax; n++) {
    gfs_domain_pressure_bc (domain, 
			    FTT_TRAVERSE_LEVEL | FTT_TRAVERSE_LEAFS,
			    levelmin, gfs_dp);
    gfs_relax (domain, levelmin, gfs_dp, gfs_res);
  }
  /* relax from top to bottom */
  for (l = levelmin + 1; l <= depth; l++) {
    /* get initial guess from coarser grid */ 
    gfs_domain_cell_traverse (domain, 
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEVEL, l,
			      (FttCellTraverseFunc) gfs_get_from_above, 
			      gfs_dp);
    for (n = 0; n < nrelax; n++) {
      gfs_domain_pressure_bc (domain, 
			      FTT_TRAVERSE_LEVEL | FTT_TRAVERSE_LEAFS,
			      l, gfs_dp);
      gfs_relax (domain, l, gfs_dp, gfs_res);
    }
  }
  /* correct on leaf cells */
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) correct, u);
  gfs_domain_pressure_bc (domain, FTT_TRAVERSE_LEAFS, -1, u);
  /* compute new residual on leaf cells */
  gfs_residual (domain, FTT_TRAVERSE_LEAFS, -1, u, rhs, gfs_res);
}

