/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001 National Institute of Water and Atmospheric Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include <stdlib.h>
#include <math.h>
#include "source.h"
#include "simulation.h"

/**
 * gfs_variable_source:
 * @v: a #GfsVariable.
 * @cell: a #FttCell.
 *
 * Returns: the sum of all the sources for variable @v in @cell.
 */
gdouble gfs_variable_source (GfsVariable * v, FttCell * cell)
{
  gdouble sum;
  GSList * i;

  g_return_val_if_fail (v != NULL, 0.);
  g_return_val_if_fail (cell != NULL, 0.);

  if (v->sources == NULL)
    return 0.;

  sum = 0.;
  i = GTS_SLIST_CONTAINER (v->sources)->items;
  while (i) {
    GtsObject * o = i->data;

    if (GFS_SOURCE_CLASS (o->klass)->value)
      sum += (* GFS_SOURCE_CLASS (o->klass)->value) (GFS_SOURCE (o), cell);
    i = i->next;
  }
  return sum;
}

static void add_sources (FttCell * cell, gpointer * data)
{
  GfsAdvectionParams * par = data[0];
  GfsVariable * v = data[1];

  GFS_VARIABLE (cell, v->i) += par->dt*gfs_variable_source (v, cell);
}

/**
 * gfs_variable_sources:
 * @domain: a #GfsDomain.
 * @v: a #GfsVariable.
 * @par: the #GfsAdvectionParams.
 *
 * Adds the corresponding source terms to @v.
 */
void gfs_variable_sources (GfsDomain * domain, 
			   GfsVariable * v,
			   GfsAdvectionParams * par)
{
  g_return_if_fail (domain != NULL);
  g_return_if_fail (v != NULL);
  g_return_if_fail (par != NULL);

  if (v->sources) {
    gpointer data[2];

    data[0] = par;
    data[1] = v;
    gfs_domain_cell_traverse (domain, 
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttCellTraverseFunc) add_sources, data);    
  }
}

/**
 * gfs_velocity_sources:
 * @domain: a #GfsDomain.
 * @par: the #GfsAdvectionParams.
 *
 * Adds the corresponding source term to each component of the velocity.
 */
void gfs_velocity_sources (GfsDomain * domain, GfsAdvectionParams * par)
{
  FttComponent c;
  GfsVariable * v;

  g_return_if_fail (domain != NULL);
  g_return_if_fail (par != NULL);
  
  v = gfs_variable_from_name (domain->variables, "U");
  for (c = 0; c < FTT_DIMENSION; c++, v = v->next)
    gfs_variable_sources (domain, v, par);
}

/* GfsSource: Object */

static void gfs_source_write (GtsObject * o, FILE * fp)
{
  if (GTS_OBJECT_CLASS (gfs_source_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_source_class ())->parent_class->write) 
      (o, fp);

  g_assert (GFS_SOURCE (o)->v);
  fprintf (fp, " %s", GFS_SOURCE (o)->v->name);
}

static void gfs_source_read (GtsObject ** o, GtsFile * fp)
{
  GfsSource * source;
  GfsDomain * domain;

  if (GTS_OBJECT_CLASS (gfs_source_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_source_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  source = GFS_SOURCE (*o);
  domain = gfs_object_simulation (source);
  if (fp->type != GTS_STRING) {
    gts_file_error (fp, "expecting a string (GfsVariable)");
    return;
  }
  source->v = gfs_variable_from_name (domain->variables, 
				      fp->token->str);
  if (source->v == NULL) {
    gts_file_error (fp, "unknown variable `%s'", fp->token->str);
    return;
  }
  if (source->v->sources == NULL)
    source->v->sources = 
      gts_container_new (GTS_CONTAINER_CLASS (gts_slist_container_class ()));
  gts_container_add (source->v->sources, GTS_CONTAINEE (source));
  
  gts_file_next_token (fp);
}

static void gfs_source_class_init (GfsSourceClass * klass)
{
  GTS_OBJECT_CLASS (klass)->read =  gfs_source_read;
  GTS_OBJECT_CLASS (klass)->write = gfs_source_write;
}

static void gfs_source_init (GfsSource * s)
{
  GFS_EVENT (s)->istep = 1;
}

GfsSourceClass * gfs_source_class (void)
{
  static GfsSourceClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_source_info = {
      "GfsSource",
      sizeof (GfsSource),
      sizeof (GfsSourceClass),
      (GtsObjectClassInitFunc) gfs_source_class_init,
      (GtsObjectInitFunc) gfs_source_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_event_class ()),
				  &gfs_source_info);
  }

  return klass;
}

/* GfsSourceConstant: Object */

static void gfs_source_constant_read (GtsObject ** o, GtsFile * fp)
{
  if (GTS_OBJECT_CLASS (gfs_source_constant_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_source_constant_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  if (fp->type != GTS_INT && fp->type != GTS_FLOAT) {
    gts_file_error (fp, "expecting a number (intensity)");
    return;
  }
  GFS_SOURCE_CONSTANT (*o)->intensity = atof (fp->token->str);

  gts_file_next_token (fp);
}

static void gfs_source_constant_write (GtsObject * o, FILE * fp)
{
  if (GTS_OBJECT_CLASS (gfs_source_constant_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_source_constant_class ())->parent_class->write) 
      (o, fp);
  fprintf (fp, " %g", GFS_SOURCE_CONSTANT (o)->intensity);
}

static gdouble gfs_source_constant_value (GfsSource * s, FttCell * cell)
{
  return GFS_SOURCE_CONSTANT (s)->intensity;
}

static void gfs_source_constant_class_init (GfsSourceClass * klass)
{
  GTS_OBJECT_CLASS (klass)->read = gfs_source_constant_read;
  GTS_OBJECT_CLASS (klass)->write = gfs_source_constant_write;

  klass->value = gfs_source_constant_value;
}

GfsSourceClass * gfs_source_constant_class (void)
{
  static GfsSourceClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_source_constant_info = {
      "GfsSourceConstant",
      sizeof (GfsSourceConstant),
      sizeof (GfsSourceClass),
      (GtsObjectClassInitFunc) gfs_source_constant_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_source_class ()),
				  &gfs_source_constant_info);
  }

  return klass;
}

/* GfsSourceGaussian: Object */

static void gfs_source_gaussian_read (GtsObject ** o, GtsFile * fp)
{
  GfsSourceGaussian * g;
  GtsFileVariable var[] = {
    {GTS_DOUBLE, "x", TRUE},
    {GTS_DOUBLE, "y", TRUE},
    {GTS_DOUBLE, "z", TRUE},
    {GTS_DOUBLE, "r", TRUE},
    {GTS_NONE}
  };

  if (GTS_OBJECT_CLASS (gfs_source_gaussian_class ())->parent_class->read)
    (* GTS_OBJECT_CLASS (gfs_source_gaussian_class ())->parent_class->read) 
      (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  g = GFS_SOURCE_GAUSSIAN (*o);
  var[0].data = &g->p.x;
  var[1].data = &g->p.y;
  var[2].data = &g->p.z;
  var[3].data = &g->r;
  gts_file_assign_variables (fp, var);
}

static void gfs_source_gaussian_write (GtsObject * o, FILE * fp)
{
  GfsSourceGaussian * g = GFS_SOURCE_GAUSSIAN (o);

  if (GTS_OBJECT_CLASS (gfs_source_gaussian_class ())->parent_class->write)
    (* GTS_OBJECT_CLASS (gfs_source_gaussian_class ())->parent_class->write) 
      (o, fp);  
  fprintf (fp, " { x = %g y = %g z = %g r = %g }", 
	   g->p.x, g->p.y, g->p.z, g->r);
}

static gdouble gfs_source_gaussian_value (GfsSource * s, 
					  FttCell * cell)
{
  GfsSourceGaussian * g = GFS_SOURCE_GAUSSIAN (s);
  FttVector p;
  gdouble r2;

  if (g->r > 0.) {
    ftt_cell_pos (cell, &p);
    r2 = ((p.x - g->p.x)*(p.x - g->p.x) + 
	  (p.y - g->p.y)*(p.y - g->p.y) +
	  (p.z - g->p.z)*(p.z - g->p.z));
    return GFS_SOURCE_CONSTANT (s)->intensity*exp (-r2/(g->r*g->r));
  }
  return 0.;
}

static void gfs_source_gaussian_class_init (GfsSourceClass * klass)
{
  GTS_OBJECT_CLASS (klass)->read = gfs_source_gaussian_read;
  GTS_OBJECT_CLASS (klass)->write = gfs_source_gaussian_write;

  klass->value = gfs_source_gaussian_value;
}

GfsSourceClass * gfs_source_gaussian_class (void)
{
  static GfsSourceClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_source_gaussian_info = {
      "GfsSourceGaussian",
      sizeof (GfsSourceGaussian),
      sizeof (GfsSourceClass),
      (GtsObjectClassInitFunc) gfs_source_gaussian_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_source_constant_class ()),
				  &gfs_source_gaussian_info);
  }

  return klass;
}
