/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001 National Institute of Water and Atmospheric Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "config.h"
#ifdef HAVE_GETOPT_H
#  include <getopt.h>
#endif /* HAVE_GETOPT_H */
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include "init.h"
#include "simulation.h"
#include "graphic.h"

static gdouble local_size_ratio (GtsSegment * s, GfsDomain * domain)
{
  GtsPoint * p1 = GTS_POINT (s->v1);
  GtsPoint * p2 = GTS_POINT (s->v2);
  gdouble l = gts_point_distance (p1, p2);
  FttVector p;
  FttCell * cell;
  gdouble size = G_MAXDOUBLE;

  p.x = p1->x;
  p.y = p1->y;
  p.z = p1->z;
  cell = gfs_domain_locate (domain, p, -1);
  if (cell)
    size = ftt_cell_size (cell);

  p.x = p2->x;  
  p.y = p2->y;
  p.z = p2->z;
  cell = gfs_domain_locate (domain, p, -1);
  if (cell) {
    gdouble s = ftt_cell_size (cell);
    
    if (size == G_MAXDOUBLE || s > size)
      size = s;
  }
  
  return size/l;
}

static gboolean stop (gdouble cost, guint nedge)
{
  if (cost >= 1. || nedge > 50000)
    return TRUE;
  return FALSE;
}

static void draw_vector (FttCell * cell, gdouble * scale)
{
  FILE * fp = stdout;
  FttVector pos, f;
  
  ftt_cell_pos (cell, &pos);
  
  f.x = GFS_STATE (cell)->u*(*scale);
  f.y = GFS_STATE (cell)->v*(*scale);
#ifdef FTT_2D
  f.z = 0.;
#else
  f.z = GFS_STATE (cell)->w*(*scale);
#endif
  fprintf (fp, "VECT 1 3 0 3 0 %g %g %g %g %g %g %g %g %g\n",
	   pos.x + f.x - (f.x - f.y/2.)/5.,
	   pos.y + f.y - (f.x/2. + f.y)/5.,
	   pos.z + f.z,
	   pos.x + f.x,
	   pos.y + f.y,
	   pos.z + f.z,
	   pos.x + f.x - (f.x + f.y/2.)/5.,
	   pos.y + f.y + (f.x/2. - f.y)/5.,
	   pos.z + f.z);
  fprintf (fp, "VECT 1 2 0 2 0 %g %g %g %g %g %g\n",
	   pos.x, pos.y, pos.z,
	   pos.x + f.x,
	   pos.y + f.y,
	   pos.z + f.z);
}

int main (int argc, char * argv[])
{
  int c = 0;
  GfsVariable * v, * var = NULL;
  GtsFile * fp;
  GtsSurface * surface = NULL;
  gboolean draw_surface = FALSE;

  gboolean verbose = FALSE;
  gboolean refine = FALSE;

  gdouble vector = -1.;

  FILE * stream = NULL;
  gchar * streamname = NULL;
  gboolean ribbon = FALSE, squares = FALSE;

  GtsBBox * box = NULL;

  gdouble min = 0., max = 0.;
#ifdef FTT_2D
  gboolean gnuplot = FALSE;
#endif /* FTT_2D */


  gfs_init (&argc, &argv);

  /* parse options using getopt */
  while (c != EOF) {
#ifdef HAVE_GETOPT_LONG
    static struct option long_options[] = {
      {"min", required_argument, NULL, 'm'},
      {"max", required_argument, NULL, 'M'},
      {"squares", no_argument, NULL, 'S'},
#ifdef FTT_2D
      {"gnuplot", no_argument, NULL, 'g'},
#endif /* FTT_2D */
      {"sx", required_argument, NULL, 'x'},
      {"sy", required_argument, NULL, 'y'},
      {"sz", required_argument, NULL, 'z'},
      {"color", required_argument, NULL, 'c'},
      {"cylinder", required_argument, NULL, 'C'},
      {"ribbon", required_argument, NULL, 'R'},
      {"refine", no_argument, NULL, 'r'},
      {"surface", required_argument, NULL, 's'},
      {"vector", required_argument, NULL, 'V'},
      {"help", no_argument, NULL, 'h'},
      {"verbose", no_argument, NULL, 'v'},
    };
    int option_index = 0;
    switch ((c = getopt_long (argc, argv, "hvs:rV:C:R:c:x:y:z:Sm:M:"
#ifdef FTT_2D
			      "g"
#endif /* FTT_2D */
			      , long_options, &option_index))) {
#else /* not HAVE_GETOPT_LONG */
    switch ((c = getopt (argc, argv, "hvs:rV:C:R:c:x:y:z:Sm:M:g"))) {
#endif /* not HAVE_GETOPT_LONG */
    case 'M': /* max */
      max = atof (optarg);
      break;
    case 'm': /* min */
      min = atof (optarg);
      break;
#ifdef FTT_2D
    case 'g': /* gnuplot */
      gnuplot = TRUE;
      break;
#endif /* FTT_2D */
    case 'S': /* squares */
      squares = TRUE;
      break;
    case 'x': /* sx */
      box = gts_bbox_new (gts_bbox_class (), NULL,
			  atof (optarg), -G_MAXDOUBLE/2., -G_MAXDOUBLE/2.,
			  atof (optarg), G_MAXDOUBLE/2., G_MAXDOUBLE/2.);
      break;
    case 'y': /* sy */
      box = gts_bbox_new (gts_bbox_class (), NULL,
			  -G_MAXDOUBLE/2., atof (optarg), -G_MAXDOUBLE/2.,
			  G_MAXDOUBLE/2., atof (optarg), G_MAXDOUBLE/2.);
      break;
    case 'z': /* sz */
      box = gts_bbox_new (gts_bbox_class (), NULL,
			  -G_MAXDOUBLE/2., -G_MAXDOUBLE/2., atof (optarg),
			  G_MAXDOUBLE/2., G_MAXDOUBLE/2., atof (optarg));
      break;
    case 's': /* surface */
      draw_surface = TRUE;
      if (strcmp (optarg, "solid")) {
	FILE * fp = fopen (optarg, "rt");
	GtsFile * f;
	
	if (fp == NULL) {
	  fprintf (stderr, 
		   "gfs2oogl: cannot open file `%s'\n"
		   "Try `gfs2oogl --help' for more information.\n",
		   optarg);
	  return 1; /* failure */
	}
	f = gts_file_new (fp);
	surface = gts_surface_new (gts_surface_class (),
				   gts_face_class (),
				   gts_edge_class (),
				   gts_vertex_class ());
	if (gts_surface_read (surface, f)) {
	  fprintf (stderr, "gfs2oogl: file `%s' is not a valid GTS file\n", 
		   optarg);
	  fprintf (stderr, "%s:%d:%d: %s\n",
		   optarg, f->line, f->pos, f->error);
	  return 1; /* failure */
	}
	gts_file_destroy (f);
	fclose (fp);
      }
      break;
    case 'R': /* ribbon */
      ribbon = TRUE;
      /* fall through */
    case 'C': /* cylinder */
      stream = fopen (optarg, "rt");
      streamname = g_basename (optarg); 
      if (stream == NULL) {
	fprintf (stderr, 
		 "gfs2oogl: cannot open file `%s'\n"
		 "Try `gfs2oogl --help' for more information.\n",
		 optarg);
	return 1; /* failure */
      }
      break;
    case 'V': /* vector */
      vector = atof (optarg);
      break;
    case 'r': /* refine */
      refine = TRUE;
      break;
    case 'c': /* color */
      if ((var = gfs_variable_from_name (gfs_derived_first, optarg)) == NULL) {
	fprintf (stderr, 
		 "gfs2oogl: unknown variable `%s'\n"
		 "Try `gfs2oogl --help' for more information.\n",
		 optarg);
	return 1; /* failure */
      }
      break;
    case 'v': /* verbose */
      verbose = TRUE;
      break;
    case 'h': /* help */
      fprintf (stderr,
     "Usage: gfs2oogl [OPTION] < GFS_FILE\n"
     "Converts a Gerris simulation file to other (graphical) formats.\n"
     "\n"
     "  -S      --squares     draw (colored) squares\n"
#ifdef FTT_2D
     "  -g      --gnuplot     output gnuplot data\n"
#endif /* FTT_2D */
     "  -x VAL  --sx=VAL      outputs a GTS surface, cross section for x = VAL\n"
     "                        of the scalar variable\n"
     "  -y VAL  --sy=VAL      outputs a GTS surface, cross section for y = VAL\n"
     "                        of the scalar variable\n"
     "  -z VAL  --sz=VAL      outputs a GTS surface, cross section for z = VAL\n"
     "                        of the scalar variable\n"
     "  -s S    --surface=S   outputs the surface defined by file S (or the solid\n"
     "                        surface is S is equal to `solid')\n"
     "  -V S    --vector=S    output an OOGL representation of the velocity vector\n"
     "                        field in the mixed cells\n"
     "  -C F    --cylinder=F  draw stream cylinders starting from each point defined\n"
     "                        in file F\n"
     "  -R F    --ribbon=F    draw stream ribbons starting from each point defined\n"
     "                        in file F\n"
     "  -r                    refines the solid surface according to the local\n"
     "                        resolution\n"
     "  -c V    --color=V     color surfaces, streamlines etc... according to the\n"
     "  -m V    --min=V       set minimum scalar value to V\n"
     "  -M V    --max=V       set maximum scalar value to V\n"
     "  -v      --verbose     display statistics and other info\n"
     "  -h      --help        display this help and exit\n"
     "\n"
     "Reports bugs to %s\n",
	       FTT_MAINTAINER);
      return 0; /* success */
      break;
    case '?': /* wrong options */
      fprintf (stderr, "Try `gfs2oogl --help' for more information.\n");
      return 1; /* failure */
    }
  }

  fp = gts_file_new (stdin);

  while (fp->type == GTS_INT) {
    GfsSimulation * simulation;
    GfsDomain * domain;
    GtsRange stats;
      
    simulation = gfs_simulation_new (gfs_simulation_class ());
    if (gfs_simulation_read (simulation, fp)) {
      fprintf (stderr, 
	       "gfs2oogl: file on standard input is not a valid simulation file\n"
	       "<stdin>:%d:%d: %s\n",
	       fp->line, fp->pos, fp->error);
      return 1;
    }

    if (verbose)
      fprintf (stderr, "gfs2oogl: processing t = %10e\n", simulation->time.t);

    domain = GFS_DOMAIN (simulation);
    
    gfs_domain_match (domain);
    v = domain->variables;
    while (v) {
      gfs_domain_center_bc (domain, v);
      v = v->next;
    }

    if (var != NULL) {
      if (var->derived) {
	gfs_variable_set_parent (var, GTS_OBJECT (domain));
	gfs_domain_cell_traverse (domain, 
				  FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
				  (FttCellTraverseFunc) var->derived, var);
      }
      gfs_domain_cell_traverse (domain,
				FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
				(FttCellTraverseFunc) 
				gfs_get_from_below_intensive, var);
      if (min == max) {
	stats = gfs_domain_stats_variable (domain, var, FTT_TRAVERSE_ALL, -1);
	if (verbose)
	  fprintf (stderr, 
		   "min: %g avg: %g| %g max: %g n: %7d\n",
		   stats.min, stats.mean, stats.stddev, stats.max, stats.n);
      }
      else {
	stats.min = min;
	stats.max = max;
      }	
    }
    else
      stats.min = stats.max = 0.;

    if (box && var != NULL) {
      if (squares)
	gfs_write_squares (domain, var, stats.min, stats.max, 
			   FTT_TRAVERSE_LEAFS, -1, box, stdout);
#ifdef FTT_2D
      else if (gnuplot)
	gfs_write_gnuplot (domain, var, 
			   FTT_TRAVERSE_LEAFS, -1, box, stdout);
#endif /* FTT_2D */
      else
	gfs_write_gts (domain, var, FTT_TRAVERSE_LEAFS, -1, box, stdout);
    }
    else if (stream) {
      FttVector p;

      rewind (stream);
      printf ("(geometry \"%s-%g\" = LIST {\n",
	      streamname,
	      simulation->time.t);
      while (fscanf (stream, "%lf %lf %lf", &p.x, &p.y, &p.z) == 3) {
#ifdef FTT_2D
	gfs_draw_streamline (domain, p, stdout);
#else /* 3D */
	if (ribbon)
	  gfs_draw_stream_ribbon (domain, p, 2e-3,
				  var, stats.min, stats.max, stdout);
	else
	  gfs_draw_stream_cylinder (domain, p, 5e-4, 
				    var, stats.min, stats.max, stdout);
#endif /* 3D */
      }
      printf ("})\n");
    }
    else if (vector > 0.) {
      GtsRange stats;
      gdouble scale = 1.;

      gfs_domain_cell_traverse (domain,
				FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
				(FttCellTraverseFunc) gfs_velocity_norm, 
				gfs_div);
      stats = gfs_domain_stats_variable (domain, gfs_div, 
					 FTT_TRAVERSE_LEAFS, -1);
      if (verbose)
	fprintf (stderr, 
		 "min: %g avg: %g| %g max: %g n: %7d\n",
		 stats.min, stats.mean, stats.stddev, stats.max, stats.n);
      if (stats.max > 0.)
	scale = vector*ftt_level_size (gfs_domain_depth (domain))/stats.max;
      printf ("(geometry \"vector-%g\" = LIST {\n", simulation->time.t);
#ifdef FTT_2D
      if (box == NULL)
	box = gts_bbox_new (gts_bbox_class (), NULL,
			    0., -G_MAXDOUBLE/2., -G_MAXDOUBLE/2.,
			    0., G_MAXDOUBLE/2., G_MAXDOUBLE/2.);
#else /* 3D */
      if (box == NULL)
	gfs_domain_traverse_mixed (domain,
				  (FttCellTraverseFunc) draw_vector, &scale);
      else
#endif /* 3D */
      gfs_domain_cell_traverse_box (domain, box,
				    FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
				    (FttCellTraverseFunc) draw_vector, &scale);
      printf ("})\n");
    }
    else if (draw_surface) {
      GtsSurface * s = surface;

      if (s == NULL)
	s = simulation->surface;

      if (s) {
	if (refine)
	  gts_surface_refine (s, 
			      (GtsKeyFunc) local_size_ratio, domain,
			      NULL, NULL,
			      (GtsStopFunc) stop, NULL);
	gfs_draw_surface (domain, s, 
			  var, stats.min, stats.max,
			  stdout);
      }
    }
    else {
      gfs_draw_refined_boundaries (domain, stdout);
      gfs_draw_solid_boundaries (domain, stdout);
      gfs_draw_boundary_conditions (domain, stdout);
    }

    gts_object_destroy (GTS_OBJECT (simulation));
  }

  gts_file_destroy (fp);

  return 0;
}
