/* 
 * File: message.h
 * Author: Jason Short
 * Project: GGZCards Server
 * Date: 06/20/2001
 * Desc: Functions and data for messaging system
 * $Id: message.h 4484 2002-09-09 04:09:04Z jdorje $
 *
 * Right now GGZCards uses a super-generic "messaging" system.  Text
 * messages are generated by the server and sent to the client for
 * direct display.  This makes i18n basically impossible.  It also
 * makes really cool client displays basically impossible.  However,
 * it makes it really easy to add new games - you don't even have to
 * change the client!
 *
 * A more complicated alternate messaging system is in planning.
 *
 * Copyright (C) 2001-2002 Brent Hendricks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-130
 */

#include <ggz.h>		/* libggz */

#include "types.h"

#ifndef __MESSAGE_H
#define __MESSAGE_H

/* Note the difference between set_player_message and set_global_message.
   They have similar names, but behave quite differently.  This is because
   global and player messages are handled quite differently.  Global messages 
   are set at specific points in the code, and sent out to the clients as-is. 
   Player messages are set by a specific function in the code, so the rest of 
   the code just calls set_player_message() to have them automatically set up 
   and sent out. */


/* 
 * player messages
 */

/** @brief Sends the seat's current message to one player
 *  @param s The seat of the message to be sent.
 *  @param p The player to send the message to.
 *  @todo Should this be send_seat_message? */
void send_player_message(seat_t s, player_t p);

/** @brief Sends the seat's current message to everyone
 *  @param s The seat of the message to be sent. */
void send_player_message_toall(seat_t s);

/** @brief Builds and sends the player's message.
 *
 *  This function calls the appropriate game set_player_message
 *  function to _build_ the player's message, then sends it
 *  to everyone.
 *  @param p The player (note: player) of the message. */
void set_player_message(player_t p);

/** @brief Broadcast a player's message to everyone.
 *
 *  Broadcasts the seat's existing player message to all
 *  connected players and spectators.
 *  @param s The seat whose message to broadcast. */
void broadcast_player_message(seat_t s);

/** @brief Calls set_player_message for all players.
 *  @see set_player_message */
void set_all_player_messages(void);

/** @brief Clears the seat's message.
 *  @param s The seat (note: seat) of the message.
 *  @todo Rename to clear_seat_message */
void clear_player_message(seat_t s);

/** @brief Sets the seat's message, printf-style.
 *  @param s The seat of the message.
 *  @param fmt A printf-style format string
 *  @param ... A printf-style list of arguments
 *  @note This erases any existing message.
 *  @todo Rename to put_seat_message */
void put_player_message(seat_t s, char *fmt, ...)
			ggz__attribute((format(printf, 2, 3)));

/** @brief Adds to the seat's message, printf-style.
 *  @param s The seat of the message
 *  @param fmt A printf-style format string
 *  @param ... A printf-style list of arguments
 *  @todo Rename to add_seat_message */
void add_player_message(seat_t s, char *fmt, ...)
			ggz__attribute((format(printf, 2, 3)));


/* 
 * global messages
 */

/** @brief Sends an existing global message to one player
 *  @param mark The "mark" of the message to be sent.
 *  @param p The player to send the message to */
void send_global_message(const char *mark, player_t p);

/** @brief Sends an existing message to everyone
 *  @param mark The "mark" of the message to be sent */
void send_global_message_toall(const char *mark);

/** @brief sends all existing global messages to one player
 *  @param p The player to send the messages to */
void send_all_global_messages(player_t p);

/** @brief sets and sends one global message, printf-stype
 *
 *  This function sets the designated message according to
 *  the printf-style list of arguments, then sends it to
 *  all players.
 *  @param mark The "mark" of the message to be set.
 *  @param message A printf-style format string
 *  @param ... A printf-stye list of arguments
 *  @note This should be the most commonly used global message function. */
void set_global_message(char *mark, char *message, ...)
			ggz__attribute((format(printf, 2, 3)));

/** @brief Get a global message
 *  @param mark The "mark" of the message to be found
 *  @return The message for the given mark, or NULL if none
 *  @note This shouldn't be needed much. */
const char *get_global_message(const char *mark);


/* the following are automated player messages */
void add_player_score_message(player_t p);
void add_player_bid_message(player_t p);
void add_player_tricks_message(player_t p);
void add_player_action_message(player_t p);
void add_player_rating_message(player_t p);

/*
 * the following are automated global messages
 */

/* sends a listing of the cards in the last hand */
void send_last_hand(void);

/* sends a listing of the cards on the last trick */
void send_last_trick(void);

/* initialized the "cumulative scores" tracking */
void init_cumulative_scores(void);

/* updates the "cumulative scores", and sends new one */
void update_cumulative_scores(void);

/* sends the bid history from the current hand */
void send_bid_history(void);

#endif /* __MESSAGE_H */
