--------------------------------------------------------------------------------
-- |
-- Module      :  Sound.OpenAL.ALC.Context
-- Copyright   :  (c) Sven Panne 2003
-- License     :  BSD-style (see the file libraries/OpenAL/LICENSE)
-- 
-- Maintainer  :  sven_panne@yahoo.com
-- Stability   :  provisional
-- Portability :  portable
--
-- Managing Rendering Contexts
--
--------------------------------------------------------------------------------

module Sound.OpenAL.ALC.Context (
   ContextAttribute(..), Context, createContext, destroyContext, currentContext,
   processContext, suspendContext,

   contextsDevice, allAttributes
) where


import Control.Monad ( liftM )
import Foreign.Marshal.Array ( withArray )
import Foreign.Ptr ( Ptr, nullPtr )
import Graphics.Rendering.OpenGL.GL.StateVar (
   StateVar, makeStateVar, GettableStateVar, makeGettableStateVar )
import Sound.OpenAL.AL.BasicTypes ( ALint )
import Sound.OpenAL.ALC.Device ( Device )
import Sound.OpenAL.ALC.Queries ( IntQuery(..), getInteger, getIntegerv )
import Sound.OpenAL.ALC.BasicTypes (
   ALCenum, isDevice,
   ContextAttribute(..), marshalContextAttribute, unmarshalContextAttribute )
import Sound.OpenAL.ALC.Errors ( checkError )

--------------------------------------------------------------------------------

type ALCcontext = ()

newtype Context = Context (Ptr ALCcontext)
   deriving ( Eq, Ord, Show )

isContext :: Context -> Bool
isContext (Context ctx) = ctx /= nullPtr

--------------------------------------------------------------------------------

-- | Allocates, initializes, and returns a context for the given device with the
-- requested attributes. If any attribute cannot have the required value met or
-- exceeded, an 'IOError' is thrown. If the device is not valid, an
-- 'InvalidDevice' error is set and an 'IOError' is thrown. Note that
-- 'createContext' does /not/ set the current context, this must be done
-- separately via 'currentContext'.

createContext :: Device -> [ContextAttribute] -> IO Context
createContext device attributes =
   let pairToList (key, value) = [key, value]
       attrs = concatMap (pairToList . marshalContextAttribute) attributes
   in checkError isContext (withArray (attrs ++ [0]) $ alcCreateContext device)

foreign import CALLCONV unsafe "alcCreateContext"
   alcCreateContext :: Device -> Ptr ALint -> IO Context

--------------------------------------------------------------------------------

-- | Destroys the given context.

destroyContext :: Context -> IO ()
destroyContext = ignore . alcDestroyContext

foreign import CALLCONV unsafe "alcDestroyContext"
   alcDestroyContext :: Context -> IO ALCenum

--------------------------------------------------------------------------------

-- | 'Just' the current context, 'Nothing' means paused.

currentContext :: StateVar (Maybe Context)
currentContext = makeStateVar getCurrentContext makeContextCurrent

-- | Returns 'Just' the current context or 'Nothing' if there is none.

getCurrentContext :: IO (Maybe Context)
getCurrentContext = do
   context <- alcGetCurrentContext
   return $ if isContext context then Nothing else Just context

foreign import CALLCONV unsafe "alcGetCurrentContext"
   alcGetCurrentContext :: IO Context  

-- | Makes the given context the current context or pauses when given 'Nothing'.

makeContextCurrent :: Maybe Context -> IO ()
makeContextCurrent mbContext = 
   ignore $ alcMakeContextCurrent (maybe (Context nullPtr) id mbContext)

foreign import CALLCONV unsafe "alcMakeContextCurrent"
   alcMakeContextCurrent :: Context -> IO ALCenum

--------------------------------------------------------------------------------

-- | Performs processing on a synced context, nop on an asynchronous context.

processContext :: Context -> IO ()
processContext = ignore . alcProcessContext

foreign import CALLCONV unsafe "alcProcessContext"
   alcProcessContext :: Context -> IO Context

-- | Suspends processing on an asynchronous context. This is a legal nop on a
-- synced context.

foreign import CALLCONV unsafe "alcSuspendContext"
   suspendContext :: Context -> IO ()

--------------------------------------------------------------------------------

-- ALC's error handling is highly irregular: Somtimes a success/failure value is
-- returned in addition to setting the internal error state. We don't do the
-- same in our Haskell API, effectively ignoring the returned value.

ignore :: IO a -> IO ()
ignore action = do
   action
   return ()

--------------------------------------------------------------------------------

-- | 'Just' the device of the given context or 'Nothing' if the context is
-- invalid.

contextsDevice :: Context -> GettableStateVar (Maybe Device)
contextsDevice context = makeGettableStateVar $ do
   device <- alcGetContextsDevice context
   return $ if isDevice device then Nothing else Just device

foreign import CALLCONV unsafe "alcGetContextsDevice"
   alcGetContextsDevice :: Context -> IO Device

--------------------------------------------------------------------------------

-- | Note that we need a current context here!

allAttributes :: Device -> GettableStateVar [ContextAttribute]
allAttributes device = makeGettableStateVar $ do
   numALints <- liftM fromIntegral $ getInteger device AttributesSize
   liftM toContextAttributes $ getIntegerv device AllAttributes numALints

toContextAttributes :: [ALint] -> [ContextAttribute]
toContextAttributes xs = case xs of
   [] -> []  -- should only happen when device and/or current context is invalid
   (0:_) -> []
   (x:y:rest) -> unmarshalContextAttribute (x,y) : toContextAttributes rest
   _ -> error ("toContextAttributes: illegal value " ++ show xs)
