--------------------------------------------------------------------------------
-- |
-- Module      :  Sound.OpenAL.ALUT.WAV
-- Copyright   :  (c) Sven Panne 2003
-- License     :  BSD-style (see the file libraries/OpenAL/LICENSE)
-- 
-- Maintainer  :  sven_panne@yahoo.com
-- Stability   :  provisional
-- Portability :  portable
--
--------------------------------------------------------------------------------

module Sound.OpenAL.ALUT.WAV (
   withWAVFile, withWAVMemory
) where

import Control.Monad ( liftM )
import Foreign.C.String ( withCString )
import Foreign.Marshal.Alloc ( alloca )
import Foreign.Storable ( Storable(peek) )
import Foreign.Ptr ( Ptr, castPtr )
import Sound.OpenAL.AL.BasicTypes ( ALbyte, ALsizei, ALenum )
import Sound.OpenAL.AL.ALboolean ( ALboolean, unmarshalALboolean )
import Sound.OpenAL.AL.Buffer ( BufferData(..) )
import Sound.OpenAL.AL.Format ( marshalFormat, unmarshalFormat )
import Sound.OpenAL.ALUT.Exception ( finally )

--------------------------------------------------------------------------------

withWAVFile :: String -> (BufferData -> Bool -> IO a) -> IO a
withWAVFile fileName action =
   withCString fileName $ \fileNameBuf ->
      withWAVInternal (alutLoadWAVFile (castPtr fileNameBuf)) action

foreign import CALLCONV unsafe "alutLoadWAVFile"
   alutLoadWAVFile :: Ptr ALbyte -> Loader

withWAVMemory :: Ptr ALbyte -> (BufferData -> Bool -> IO a) -> IO a
withWAVMemory mem action = withWAVInternal (alutLoadWAVMemory mem) action

foreign import CALLCONV unsafe "alutLoadWAVMemory"
   alutLoadWAVMemory :: Ptr ALbyte -> Loader

type Loader = Ptr ALenum -> Ptr (Ptr ()) -> Ptr ALsizei -> Ptr ALsizei
           -> Ptr ALboolean -> IO ()

withWAVInternal :: Loader -> (BufferData -> Bool -> IO a) -> IO a
withWAVInternal loader action = do
   (bufferData, loop) <- alloca $ \formatBuf ->
      alloca $ \rawBuf ->
       alloca $ \sizeBuf ->
          alloca $ \frequencyBuf ->
             alloca $ \loopBuf -> do
                loader formatBuf rawBuf sizeBuf frequencyBuf loopBuf
                format <- liftM unmarshalFormat $ peek formatBuf
                raw <- peek rawBuf
                size <- peek sizeBuf
                frequency <- peek frequencyBuf
                loop <- liftM unmarshalALboolean $ peek loopBuf
                return (BufferData format raw size frequency, loop)
   action bufferData loop `finally` unloadWAV bufferData

--------------------------------------------------------------------------------

unloadWAV :: BufferData -> IO ()
unloadWAV (BufferData format raw size frequency) =
   alutUnloadWAV (marshalFormat format) raw size frequency

foreign import CALLCONV unsafe "alutUnloadWAV"
   alutUnloadWAV :: ALenum -> Ptr () -> ALsizei -> ALsizei -> IO  ()
