// MM1TAB9.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "mm1tab9.h"

#include <iomanip>
#include <fstream>
#include <strstream>
#include <algorithm>
using namespace std;

#define RAND_f64 ((f64) rand() / (f64) RAND_MAX - 0.5)

/*################################################################################################*/

input_at::input_at(void)
{
	typerule = NULL;
	description = NULL;
}

input_at::input_at(ifstream & ifile)
{
	char buffer[1024];
	
	ifile >> atomtype[0]; atomtype[1] = 0;
	
	while (ifile.peek() != '(') ifile.get();
	typerule = new mm1_typerule(& ifile, & cout);
	
	while (ifile.get() != '\"');
	ifile.getline(buffer, sizeof(buffer), '\"');
	description = new char[strlen(buffer) + 1];
	strcpy(description, buffer);
	
	ifile >> formal_charge;
	ifile >> flags;
	
	cout << "found " << description << endl;
	
	atomic_number = (atomtype[0] >> 8);
	number_of_hits = 0;
}

input_at::input_at(const input_at & p1)
{
	atomtype[0] = p1.atomtype[0]; atomtype[1] = p1.atomtype[1];
	
	if (p1.typerule != NULL) typerule = new mm1_typerule(* p1.typerule);
	else typerule = NULL;
	
	if (p1.description != NULL)
	{
		description = new char[strlen(p1.description) + 1];
		strcpy(description, p1.description);
	}
	else description = NULL;
	
	formal_charge = p1.formal_charge;
	flags = p1.flags;
	
	// ff-related...
	// ff-related...
	// ff-related...
	
	atomic_number = p1.atomic_number;
	number_of_hits = p1.number_of_hits;
}

input_at::~input_at(void)
{
	if (typerule != NULL) delete typerule;
	if (description != NULL) delete description;
}

input_at & input_at::operator=(const input_at & p1)
{
	if (this != & p1)
	{
		if (typerule != NULL) delete typerule;
		if (description != NULL) delete description;
		
		// make copy...
		// make copy...
		// make copy...
		
		atomtype[0] = p1.atomtype[0]; atomtype[1] = p1.atomtype[1];
		
		if (p1.typerule != NULL) typerule = new mm1_typerule(* p1.typerule);
		else typerule = NULL;
		
		if (p1.description != NULL)
		{
			description = new char[strlen(p1.description) + 1];
			strcpy(description, p1.description);
		}
		else description = NULL;
		
		formal_charge = p1.formal_charge;
		flags = p1.flags;
		
		// ff-related...
		// ff-related...
		// ff-related...
		
		atomic_number = p1.atomic_number;
		number_of_hits = p1.number_of_hits;
	}
	
	return (* this);
}

bool input_at::operator<(const input_at & p1) const
{
	if (atomic_number != p1.atomic_number) return (atomic_number < p1.atomic_number);
	else return (number_of_hits > p1.number_of_hits);	// inverted logic; generic types first!!!
}

/*################################################################################################*/

prmfit_tables::prmfit_tables(const char * p1)
{
	path = new char[strlen(p1) + 1];
	strcpy(path, p1);
	
	char fn[1024];
	ostream * ostr = & cout;	// print output.
//	ostream * ostr = NULL;		// do not print output.

	ifstream file;
	file.unsetf(ios::dec | ios::oct | ios::hex);
	
	char buffer[1024];
	
/*##############################################*/
/*##############################################*/

	ostrstream fn_strt(fn, sizeof(fn));
	fn_strt << path << "/atomtypes.txt.out" << ends;
	file.open(fn, ios::in);
	
	if (ostr != NULL) (* ostr) << "reading file \"" << fn << "\": ";
	
	while (file.peek() != '#')		// #end
	{
		if (file.peek() == '0')		// 0x????
		{
			prmfit_at newat;
			file >> newat.atomtype[0];
			file >> newat.atomtype[1];
			
			while (file.peek() != '(') file.get();
			newat.typerule = new mm1_typerule(& file, ostr);
			
			while (file.get() != '\"');
			file.getline(buffer, sizeof(buffer), '\"');
			newat.description = new char[strlen(buffer) + 1];
			strcpy(newat.description, buffer);
			
			file >> newat.lj_r >> newat.lj_e;
			file >> newat.formal_charge;
			file >> newat.flags;
			
			at2_vector.push_back(newat);
		}
		
		file.getline(buffer, sizeof(buffer));
	}
	
	file.close();
	
	if (ostr != NULL) (* ostr) << "found " << at2_vector.size() << " atomtypes." << endl;
	
/*##############################################*/
/*##############################################*/
	
	ostrstream fn_str1(fn, sizeof(fn));
	fn_str1 << path << "/parameters1.txt" << ends;
	file.open(fn, ios::in);
	
	if (ostr != NULL) (* ostr) << "reading file \"" << fn << "\": ";
	
	while (file.peek() != '#')		// #end
	{
		if (file.peek() == '0')		// 0x????
		{
			prmfit_bs tmp; char bt[16];
			file >> tmp.atmtp[0] >> tmp.atmtp[1] >> bt;
			file >> tmp.opt >> tmp.fc >> tmp.cid;
			
			tmp.bndtp = bondtype(bt[0]);
			bs_vector.push_back(tmp);
		}
		
		file.getline(buffer, sizeof(buffer));
	}
	
	file.close();
	
	if (ostr != NULL) (* ostr) << "found " << bs_vector.size() << " bs-terms." << endl;
	
/*##############################################*/
/*##############################################*/
	
	ostrstream fn_str2(fn, sizeof(fn));
	fn_str2 << path << "/parameters2.txt" << ends;
	file.open(fn, ios::in);
	
	if (ostr != NULL) (* ostr) << "reading file \"" << fn << "\": ";
	
	while (file.peek() != '#')		// #end
	{
		if (file.peek() == '0')		// 0x????
		{
			prmfit_ab tmp; char bt[16];
			file >> tmp.atmtp[0] >> tmp.atmtp[1] >> tmp.atmtp[2] >> bt;
			file >> tmp.opt >> tmp.fc;
			
			for (i32s n1 = 0;n1 < 2;n1++) tmp.bndtp[n1] = bondtype(bt[n1]);
			ab_vector.push_back(tmp);
		}
		
		file.getline(buffer, sizeof(buffer));
	}
	
	file.close();
	
	if (ostr != NULL) (* ostr) << "found " << ab_vector.size() << " ab-terms." << endl;
	
/*##############################################*/
/*##############################################*/
	
	ostrstream fn_str3(fn, sizeof(fn));
	fn_str3 << path << "/parameters3.txt" << ends;
	file.open(fn, ios::in);
	
	if (ostr != NULL) (* ostr) << "reading file \"" << fn << "\": ";
	
	while (file.peek() != '#')		// #end
	{
		if (file.peek() == '0')		// 0x????
		{
			prmfit_tr tmp; char bt[16];
			file >> tmp.atmtp[0] >> tmp.atmtp[1] >> tmp.atmtp[2] >> tmp.atmtp[3] >> bt;
			file >> tmp.k[0] >> tmp.k[1] >> tmp.k[2] >> tmp.t[0] >> tmp.t[1] >> tmp.t[2];
			
			for (i32s n1 = 0;n1 < 3;n1++) tmp.bndtp[n1] = bondtype(bt[n1]);
			tr_vector.push_back(tmp);
		}
		
		file.getline(buffer, sizeof(buffer));
	}
	
	file.close();
	
	if (ostr != NULL) (* ostr) << "found " << tr_vector.size() << " tr-terms." << endl;

/*##############################################*/
/*##############################################*/

	ostrstream fn_str4(fn, sizeof(fn));
	fn_str4 << path << "/parameters4.txt" << ends;
	file.open(fn, ios::in);
	
	if (ostr != NULL) (* ostr) << "reading file \"" << fn << "\": ";
	
	while (file.peek() != '#')		// #end
	{
		if (file.peek() == '0')		// 0x????
		{
			prmfit_op tmp; char bt[16];
			file >> tmp.atmtp[0] >> tmp.atmtp[1] >> tmp.atmtp[2] >> tmp.atmtp[3] >> bt;
			file >> tmp.opt >> tmp.fc;
			
			for (i32s n1 = 0;n1 < 3;n1++) tmp.bndtp[n1] = bondtype(bt[n1]);
			op_vector.push_back(tmp);
		}
		
		file.getline(buffer, sizeof(buffer));
	}
	
	file.close();
	
	if (ostr != NULL) (* ostr) << "found " << op_vector.size() << " op-terms." << endl;
	
/*##############################################*/
/*##############################################*/

}

prmfit_tables::~prmfit_tables(void)
{
	for (i32u n1 = 0;n1 < at2_vector.size();n1++)
	{
		delete at2_vector[n1].typerule;
		delete[] at2_vector[n1].description;
	}
	
	delete[] path;
}

const prmfit_at * prmfit_tables::GetAtomType(i32s p1)
{
	i32u index = 0;
	while (index < at2_vector.size())
	{
		if (at2_vector[index].atomtype[0] == p1) return (& at2_vector[index]);
		else index++;
	}
	
	// could not find the requested type -> return a NULL pointer instead...
	
	return NULL;
}

void prmfit_tables::DoParamSearch(prmfit_bs_query & query, ostream * ostr)
{
	for (i32u n1 = 0;n1 < bs_vector.size();n1++)
	{
		if (bs_vector[n1].bndtp.GetValue() != query.bndtp.GetValue()) continue;
		
		bool flag = false; i32s dir;
		for (dir = 0;dir < 2;dir++)
		{
			bool test1 = (bs_vector[n1].atmtp[0] == query.atmtp[dir]);
			bool test2 = (bs_vector[n1].atmtp[1] == query.atmtp[!dir]);
			
			if (test1 && test2) flag = true;
			
			bool wc1 = (bs_vector[n1].atmtp[0] == 0xffff);
			bool wc2 = (bs_vector[n1].atmtp[1] == 0xffff);
			
			if (wc1 && test2) flag = true;
			if (test1 && wc2) flag = true;
			if (wc1 && wc2) flag = true;
			
			if (flag) break;
		}
		
		if (flag)
		{
			query.index = n1;
			query.dir = dir;
			
			query.opt = bs_vector[n1].opt;
			query.fc = bs_vector[n1].fc;
			
			query.cid = bs_vector[n1].cid;
			
			return;		// success, return the parameters...
		}
	}
	
// recursive search?!?!?! now this is always "strict"...
// recursive search?!?!?! now this is always "strict"...
// recursive search?!?!?! now this is always "strict"...

	if (ostr != NULL)
	{
		(* ostr) << "unknown bs: ";
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[0] << dec << " ";
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[1] << dec << " ";
		(* ostr) << query.bndtp.GetValue() << " ";
		(* ostr) << endl;
	}
	
	// the search failed, return default parameters...
	// the search failed, return default parameters...
	// the search failed, return default parameters...
	
	query.index = NOT_DEFINED;
	query.dir = false;
	
	query.opt = 0.140;
	query.fc = 60.0e+03;
	
	query.cid = 0.0;
}

void prmfit_tables::DoParamSearch(prmfit_ab_query & query, ostream * ostr)
{
	for (i32u n1 = 0;n1 < ab_vector.size();n1++)
	{
		if (ab_vector[n1].atmtp[1] != query.atmtp[1]) continue;
		
		// bondtype checking not yet implemented....
		// it will be basically similar to the above, but must be moved into the dir-loop!!!
		// it will be basically similar to the above, but must be moved into the dir-loop!!!
		// it will be basically similar to the above, but must be moved into the dir-loop!!!
		
		bool flag = false; i32s dir;
		for (dir = 0;dir < 2;dir++)
		{
			i32s index[2];
			index[0] = (!dir ? 0 : 2);
			index[1] = (!dir ? 2 : 0);
			
			bool test1 = (ab_vector[n1].atmtp[0] == query.atmtp[index[0]]);
			bool test2 = (ab_vector[n1].atmtp[2] == query.atmtp[index[1]]);
			
			if (test1 && test2) flag = true;
			
			bool wc1 = (ab_vector[n1].atmtp[0] == 0xffff);
			bool wc2 = (ab_vector[n1].atmtp[2] == 0xffff);
			
			if (wc1 && test2) flag = true;
			if (test1 && wc2) flag = true;
			if (wc1 && wc2) flag = true;
			
			if (flag) break;
		}
		
		if (flag)
		{
			query.index = n1;
			query.dir = dir;
			
			query.opt = ab_vector[n1].opt;
			query.fc = ab_vector[n1].fc;
			
			return;		// success, return the parameters...
		}
	}
	
	if (ostr != NULL)
	{
		(* ostr) << "unknown ab: " << hex;
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[0] << dec << " ";
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[1] << dec << " ";
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[2] << dec << " ";
		(* ostr) << query.bndtp[0].GetValue() << " ";
		(* ostr) << query.bndtp[1].GetValue() << " ";
		(* ostr) << endl;
	}

	// the search failed, return default parameters...
	// the search failed, return default parameters...
	// the search failed, return default parameters...
	
	query.index = NOT_DEFINED;
	query.dir = false;
	
	query.opt = 2.10;
	query.fc = 250.0;
}

void prmfit_tables::DoParamSearch(prmfit_tr_query & query, ostream * ostr)
{
	for (i32u n1 = 0;n1 < tr_vector.size();n1++)
	{
		if (tr_vector[n1].bndtp[1].GetValue() != query.bndtp[1].GetValue()) continue;

		// proper bondtype checking not yet implemented....
		// it will be basically similar to the above, but must be moved into the dir-loop!!!
		// it will be basically similar to the above, but must be moved into the dir-loop!!!
		// it will be basically similar to the above, but must be moved into the dir-loop!!!
		
		bool flag = false; i32s dir;
		for (dir = 0;dir < 2;dir++)
		{
			i32s index[4];
			index[0] = (!dir ? 0 : 3);
			index[1] = (!dir ? 1 : 2);
			index[2] = (!dir ? 2 : 1);
			index[3] = (!dir ? 3 : 0);
			
			bool test1 = (tr_vector[n1].atmtp[0] == query.atmtp[index[0]]);
			bool test2 = (tr_vector[n1].atmtp[1] == query.atmtp[index[1]]);
			bool test3 = (tr_vector[n1].atmtp[2] == query.atmtp[index[2]]);
			bool test4 = (tr_vector[n1].atmtp[3] == query.atmtp[index[3]]);
			
			if (test1 && test2 && test3 && test4) flag = true;
			
			bool wc1 = (tr_vector[n1].atmtp[0] == 0xffff);
			bool wc2 = (tr_vector[n1].atmtp[3] == 0xffff);
			
			if (wc1 && test2 && test3 && test4) flag = true;
			if (test1 && test2 && test3 && wc2) flag = true;
			if (wc1 && test2 && test3 && wc2) flag = true;
			
			if (flag) break;
		}
		
		if (flag)
		{
			query.index = n1;
			query.dir = dir;
			
			for (i32s n2 = 0;n2 < 3;n2++)
			{
				query.k[n2] = tr_vector[n1].k[n2];
				query.t[n2] = tr_vector[n1].t[n2];
			}
			
			return;		// success, return the parameters...
		}
	}
	
	if (ostr != NULL)
	{
		(* ostr) << "unknown tr: " << hex;
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[0] << dec << " ";
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[1] << dec << " ";
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[2] << dec << " ";
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[3] << dec << " ";
		(* ostr) << query.bndtp[0].GetValue() << " ";
		(* ostr) << query.bndtp[1].GetValue() << " ";
		(* ostr) << query.bndtp[2].GetValue() << " ";
		(* ostr) << endl;
	}
	
	// the search failed, return default parameters...
	// the search failed, return default parameters...
	// the search failed, return default parameters...
	
	query.index = NOT_DEFINED;
	query.dir = false;
	
	for (i32s n2 = 0;n2 < 3;n2++)
	{
		query.k[n2] = 0.0;
		query.t[n2] = 0.0;
	}
}

void prmfit_tables::DoParamSearch(prmfit_op_query & query, ostream * ostr)
{
	for (i32u n1 = 0;n1 < op_vector.size();n1++)
	{
		// the atomtypes are defined in the following way:
		// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		//
		//   3  	the idea is to measure how much atom 3
		//   |  	is bend from the plane defined by atoms
		//   1  	0, 1 and 2.
		//  / \ 
		// 0   2	order of atoms 0 and 2 is not relevant,
		//		but atoms 1 and 3 must match exactly.
		
		if (op_vector[n1].atmtp[1] != query.atmtp[1]) continue;
		if (op_vector[n1].atmtp[3] != query.atmtp[3]) continue;
		if (op_vector[n1].bndtp[2].GetValue() != query.bndtp[2].GetValue()) continue;
		
		// proper bondtype checking not yet implemented....
		// it will be basically similar to the above, but must be moved into the dir-loop!!!
		// it will be basically similar to the above, but must be moved into the dir-loop!!!
		// it will be basically similar to the above, but must be moved into the dir-loop!!!
		
		bool flag = false; i32s dir;
		for (dir = 0;dir < 2;dir++)
		{
			i32s index[2];
			index[0] = (!dir ? 0 : 2);
			index[1] = (!dir ? 2 : 0);
			
			bool test1 = (op_vector[n1].atmtp[0] == query.atmtp[index[0]]);
			bool test2 = (op_vector[n1].atmtp[2] == query.atmtp[index[1]]);
			
			if (test1 && test2) flag = true;
			
			bool wc1 = (op_vector[n1].atmtp[0] == 0xffff);
			bool wc2 = (op_vector[n1].atmtp[2] == 0xffff);
			
			if (wc1 && test2) flag = true;
			if (test1 && wc2) flag = true;
			if (wc1 && wc2) flag = true;
			
			if (flag) break;
		}
		
		if (flag)
		{
			query.index = n1;
			
			query.opt = op_vector[n1].opt;
			query.fc = op_vector[n1].fc;
			
			return;		// success, return the parameters...
		}
	}
	
	if (ostr != NULL)
	{
		(* ostr) << "unknown op: " << hex;
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[0] << dec << " ";
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[1] << dec << " ";
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[2] << dec << " ";
		(* ostr) << "0x" << hex << setw(4) << setfill('0') << query.atmtp[3] << dec << " ";
		(* ostr) << query.bndtp[0].GetValue() << " ";
		(* ostr) << query.bndtp[1].GetValue() << " ";
		(* ostr) << query.bndtp[2].GetValue() << " ";
		(* ostr) << endl;
	}
	
	// the search failed, return default parameters...
	// the search failed, return default parameters...
	// the search failed, return default parameters...
	
	query.index = NOT_DEFINED;
	
	query.opt = 0.0;
	query.fc = 0.0;
}

i32s prmfit_tables::UpdateTypes(mm1_mdl & mdl, ostream * ostr)
{
	i32s errors = 0;
	if (ostr != NULL) (* ostr) << "setting up atom types..." << endl;
	for (iter_mm1al it1 = mdl.GetAtomsBegin();it1 != mdl.GetAtomsEnd();it1++)
	{
		if (ostr != NULL && !((* it1).index % 10)) (* ostr) << "*" << flush;
		
		i32u at_range[2];
		
		at_range[0] = 0;
		while (true)
		{
			if (at_range[0] == at2_vector.size()) break;
			if ((at2_vector[at_range[0]].atomtype[0] >> 8) == (* it1).el.GetAtomicNumber()) break;
			
			at_range[0]++;
		}
		
		at_range[1] = at_range[0];
		while (true)
		{
			if (at_range[1] == at2_vector.size()) break;
			if ((at2_vector[at_range[1]].atomtype[0] >> 8) != (* it1).el.GetAtomicNumber()) break;
			
			at_range[1]++;
		}
		
		i32s index = NOT_DEFINED;
		for (i32u n1 = at_range[0];n1 < at_range[1];n1++)
		{
			bool flag = at2_vector[n1].typerule->Check(& mdl, & (* it1), 0);
			if (flag) index = n1;
		}
		
		if (index != NOT_DEFINED)
		{
			(* it1).atmtp = at2_vector[index].atomtype[0];
		}
		else
		{
			char mbuff1[256];
			ostrstream str1(mbuff1, sizeof(mbuff1));
			str1 << "WARNING : could not determine atomtype (atom index = " << (* it1).index << ")." << endl << ends;
			mdl.PrintToLog(mbuff1);
			
			(* it1).atmtp = NOT_DEFINED; (* it1).selected = true; errors++;
		}
	}
	
	if (ostr != NULL) (* ostr) << endl;
	return errors;
}

void prmfit_tables::PrintAllTypeRules(ostream & p1)
{
	for (i32u n1 = 0;n1 < at2_vector.size();n1++)
	{
		p1 << (n1 + 1) << ": 0x" << hex << setw(4) << setfill('0') << at2_vector[n1].atomtype << dec;
		p1 << " (" << (* at2_vector[n1].typerule) << ") \"" << at2_vector[n1].description << "\"" << endl;
	}
	
	p1 << at2_vector.size() << " entries." << endl;
}

// extensions start here...
// extensions start here...
// extensions start here...

void prmfit_tables::InitSTAGE1(void)
{
	// read in the initial atomtypes for STAGE1...
	
	char filename[1024];
	ostrstream str1(filename, sizeof(filename));
	str1 << path << "/atomtypes.txt.in" << ends;
	
	ifstream ifile;
	ifile.unsetf(ios::dec | ios::oct | ios::hex);
	ifile.open(filename, ios::in);
	
	char buffer[1024];
	
	while (ifile.peek() != '#')	// look for "#end"...
	{
		char first_char = ifile.peek();
		
		if (first_char == '0')		// read a record in...
		{
			input_at newat(ifile);
			at1_vector.push_back(newat);
		}
		
		ifile.getline(buffer, sizeof(buffer));
	}
	
	ifile.close();
	
	cout << "FOUND TOTAL OF " << at1_vector.size() << " INITIAL ATOMTYPES." << endl;
}

void prmfit_tables::AddCaseSTAGE1a(file_io_handler * handler)
{
	// test all possible atom types, and increment counters for matching ones.
	
	for (iter_mm1al it1 = handler->mdl->GetAtomsBegin();it1 != handler->mdl->GetAtomsEnd();it1++)
	{
		i32s nhits = 0;
		
		for (i32u n1 = 0;n1 < at1_vector.size();n1++)
		{
			if (at1_vector[n1].atomic_number != (* it1).el.GetAtomicNumber()) continue;
			
			bool flag = at1_vector[n1].typerule->Check(handler->mdl, & (* it1), 0);
			if (flag)
			{
				at1_vector[n1].number_of_hits++;
				nhits++;
			}
		}
		
		if (!nhits) cout << "an atom found with no matching atomtype!!!" << endl;
	}
}

void prmfit_tables::ValidateAtomTypesSTAGE1a(void)
{
	cout << "starting validation (1a)..." << endl;
	
	// remove those atomtypes that didn't match to any atom in the database.
	
	while (true)
	{
		i32u n1 = 0;
		while (n1 < at1_vector.size())
		{
			if (!at1_vector[n1].number_of_hits) break;
			else n1++;
		}
		
		if (n1 == at1_vector.size()) break;
		else
		{
			cout << "must discard atomtype " << at1_vector[n1].description << " due to lack of data." << endl;
			at1_vector.erase(at1_vector.begin() + n1);
		}
	}
	
	// sort the atomtype table according to element and number of hits.
	// sort() could be used as well, but stable_sort() retains the old order if possible (if ordering is non-unique).
	// the use of stable_sort is useful since it guarantees that default type is always first; DO NOT CHANGE!!!
	
	stable_sort(at1_vector.begin(), at1_vector.end());
	
	// update the atomtype indexing.
	
	i32s element = NOT_DEFINED;
	i32s type_counter = NOT_DEFINED;
	for (i32u n1 = 0;n1 < at1_vector.size();n1++)
	{
		if (at1_vector[n1].atomic_number != element)
		{
			element = at1_vector[n1].atomic_number;
			type_counter = 0;
		}
		
		at1_vector[n1].atomtype[0] = (element << 8) + type_counter++;
		if (type_counter >= (1 << 8))
		{
			cout << "PROBLEM : too many atom types; id overflow!!!" << endl;
			exit(EXIT_FAILURE);
		}
	}
	
	// allocate and clean the "secondary" memory blocks for the next validation round...
	
	for (i32u n1 = 0;n1 < at1_vector.size();n1++)
	{
		at1_vector[n1].secondary = new i32u[at1_vector.size()];
		for (i32u n2 = 0;n2 < at1_vector.size();n2++)
		{
			at1_vector[n1].secondary[n2] = 0;
		}
	}
}

void prmfit_tables::AddCaseSTAGE1b(file_io_handler * handler)
{
	// test all possible atom types, and increment counters for cases all other than the best one
	// (the best-matching case is now the last one in our sorted atomtype table).
	
	for (iter_mm1al it1 = handler->mdl->GetAtomsBegin();it1 != handler->mdl->GetAtomsEnd();it1++)
	{
		vector<i32u> hits_vector;
		
		for (i32u n1 = 0;n1 < at1_vector.size();n1++)
		{
			if (at1_vector[n1].atomic_number != (* it1).el.GetAtomicNumber()) continue;
			
			bool flag = at1_vector[n1].typerule->Check(handler->mdl, & (* it1), 0);
			if (flag)
			{
				hits_vector.push_back(n1);
			}
		}
		
		if (hits_vector.size() > 1)
		{
			i32u best = hits_vector.back();
			for (i32s n1 = 0;n1 < ((i32s) hits_vector.size() - 1);n1++)
			{
				at1_vector[best].secondary[hits_vector[n1]]++;
			}
		}
	}
}

void prmfit_tables::ValidateAtomTypesSTAGE1b(void)
{
	cout << "starting validation (1b)..." << endl;
	
	// determine the nearest secondary atomtypes; this is the last secondary type that matches...
	
	for (i32s n1 = 0;n1 < (i32s) at1_vector.size();n1++)
	{
		for (i32s n2 = 0;n2 < (i32s) at1_vector.size();n2++)
		{
			if (at1_vector[n1].secondary[n2] != 0)
			{
				at1_vector[n1].atomtype[1] = at1_vector[n2].atomtype[0];
			}
		}
		
		// now also free the "secondary" memory blocks...
		
		delete[] at1_vector[n1].secondary;
	}
	
	// copy the current at1_vector to at2_vector (using default values
	// for lj_r and lj_e), and then write the atomtypes.txt.out file.
	
	while (!at2_vector.empty()) at2_vector.pop_back();	// a minor memory leak???
	for (i32u n1 = 0;n1 < at1_vector.size();n1++)
	{
		prmfit_at newat;
		newat.atomtype[0] = at1_vector[n1].atomtype[0];
		newat.atomtype[1] = at1_vector[n1].atomtype[1];
		
		newat.typerule = new mm1_typerule(* at1_vector[n1].typerule);
		
		newat.description = new char[strlen(at1_vector[n1].description) + 1];
		strcpy(newat.description, at1_vector[n1].description);
		
		newat.lj_r = 0.150;	// default value (hydrogen)...
		newat.lj_e = 0.175;	// default value (hydrogen)...
		
		newat.formal_charge = at1_vector[n1].formal_charge;
		newat.flags = at1_vector[n1].flags;
		
		at2_vector.push_back(newat);
	}
	
	WriteAtomTypes();
}

#define MAX_RECURSION_DEPTH_STAGE1	2

#define SECONDARY_TYPE_INDEX	1	// this must be 1; do not change!!!

void prmfit_tables::AddCaseSTAGE1c(file_io_handler * handler)
{
	// create an eng-object of mm1_eng_prmfit type.
	// then calculate energy to calculate bond lengths and angles.
	
	mm1_eng_prmfit * eng = new mm1_eng_prmfit(* handler->mdl, * this);
	
	CopyCRD(handler->mdl, eng, 0);
	eng->Compute(0);
	
	// add all exactly matching terms, and all combinations using secondary types.
	
	// bs-terms...
	// bs-terms...
	// bs-terms...
	
	for (i32u n1 = 0;n1 < eng->bt1_vector.size();n1++)
	{
		prmfit_bs_query query; query.strict = true;
		query.atmtp[0] = eng->index[eng->bt1_vector[n1].atmi[0]]->atmtp;
		query.atmtp[1] = eng->index[eng->bt1_vector[n1].atmi[1]]->atmtp;
		query.bndtp = bondtype(eng->bt1_vector[n1].bt);
		
		AddCaseSTAGE1c(& query, eng->bt1_vector[n1].x, 0);
	}
	
	// ab-terms...
	// ab-terms...
	// ab-terms...
	
	for (i32u n1 = 0;n1 < eng->bt2_vector.size();n1++)
	{
		prmfit_ab_query query; query.strict = true;
		query.atmtp[0] = eng->index[eng->bt2_vector[n1].atmi[0]]->atmtp;
		query.atmtp[1] = eng->index[eng->bt2_vector[n1].atmi[1]]->atmtp;
		query.atmtp[2] = eng->index[eng->bt2_vector[n1].atmi[2]]->atmtp;
		query.bndtp[0] = bondtype(eng->bt2_vector[n1].bt[0]);
		query.bndtp[1] = bondtype(eng->bt2_vector[n1].bt[1]);

		AddCaseSTAGE1c(& query, eng->bt2_vector[n1].x, 0);
	}
	
	// tr-terms...
	// tr-terms...
	// tr-terms...
	
	for (i32u n1 = 0;n1 < eng->bt3_vector.size();n1++)
	{
		prmfit_tr_query query; query.strict = true;
		query.atmtp[0] = eng->index[eng->bt3_vector[n1].atmi[0]]->atmtp;
		query.atmtp[1] = eng->index[eng->bt3_vector[n1].atmi[1]]->atmtp;
		query.atmtp[2] = eng->index[eng->bt3_vector[n1].atmi[2]]->atmtp;
		query.atmtp[3] = eng->index[eng->bt3_vector[n1].atmi[3]]->atmtp;
		query.bndtp[0] = bondtype(eng->bt3_vector[n1].bt[0]);
		query.bndtp[1] = bondtype(eng->bt3_vector[n1].bt[1]);
		query.bndtp[2] = bondtype(eng->bt3_vector[n1].bt[2]);
		
		AddCaseSTAGE1c(& query, 0);
	}
	
	// op-terms...
	// op-terms...
	// op-terms...
	
	for (i32u n1 = 0;n1 < eng->bt4_vector.size();n1++)
	{
		prmfit_op_query query; query.strict = true;
		query.atmtp[0] = eng->index[eng->bt4_vector[n1].atmi[0]]->atmtp;
		query.atmtp[1] = eng->index[eng->bt4_vector[n1].atmi[1]]->atmtp;
		query.atmtp[2] = eng->index[eng->bt4_vector[n1].atmi[2]]->atmtp;
		query.atmtp[3] = eng->index[eng->bt4_vector[n1].atmi[3]]->atmtp;
		query.bndtp[0] = bondtype(eng->bt4_vector[n1].bt[0]);
		query.bndtp[1] = bondtype(eng->bt4_vector[n1].bt[1]);
		query.bndtp[2] = bondtype(eng->bt4_vector[n1].bt[2]);
		
		AddCaseSTAGE1c(& query, eng->bt4_vector[n1].x, 0);
	}
	
	// no need to collect lj-params here : one for each atomtype!!!
	// no need to collect lj-params here : one for each atomtype!!!
	// no need to collect lj-params here : one for each atomtype!!!
	
	// no need to collect ci-params here : one for each bs-term!!!
	// no need to collect ci-params here : one for each bs-term!!!
	// no need to collect ci-params here : one for each bs-term!!!
	
	// ok, everything is ready, and we can move on...
	
	delete eng;
}

void prmfit_tables::AddCaseSTAGE1c(const prmfit_bs_query * ref, f64 obs, i32s depth)
{
	prmfit_bs_query query = (* ref);
	
	DoParamSearch(query, & cout);
	i32s index = query.index;
	
	if (index == NOT_DEFINED)
	{
		// add a new bs-record...
		
		bool dir = !(query.atmtp[0] < query.atmtp[1]);
		
		prmfit_bs newbs;
		newbs.atmtp[0] = query.atmtp[dir];
		newbs.atmtp[1] = query.atmtp[!dir];
		newbs.bndtp = query.bndtp;
		
		newbs.opt = 0.0;		// collect observed data here!!!
		newbs.obs_count = 0;
		
		newbs.fc = 5.8e+05;		// default value; C-C bond fc.
		newbs.cid = 0.0;		// default value; always zero...
		
		index = bs_vector.size();
		bs_vector.push_back(newbs);
	}
	
	// add the new observed value...
	
	bs_vector[index].obs_count++;
	bs_vector[index].opt += obs;
	
	// handle the secondary types recursively...
	// handle the secondary types recursively...
	// handle the secondary types recursively...
	
	if (depth++ >= MAX_RECURSION_DEPTH_STAGE1) return;
	
	for (i32s t1 = 0;t1 < 2;t1++)
	{
		for (i32s t2 = 0;t2 < 2;t2++)
		{
			if (!t1 && !t2) continue;	// this case is already added...
			
			if (!t1) query.atmtp[0] = (* ref).atmtp[0];
			else
			{
				const prmfit_at * at = GetAtomType((* ref).atmtp[0]);
				if (!at) { cout << "GetAtomType() failed!!!" << endl; exit(EXIT_FAILURE); }
				query.atmtp[0] = at->atomtype[SECONDARY_TYPE_INDEX];	// the secondary type...
				if (query.atmtp[0] == 0x0000) continue;
			}
			
			if (!t2) query.atmtp[1] = (* ref).atmtp[1];
			else
			{
				const prmfit_at * at = GetAtomType((* ref).atmtp[1]);
				if (!at) { cout << "GetAtomType() failed!!!" << endl; exit(EXIT_FAILURE); }
				query.atmtp[1] = at->atomtype[SECONDARY_TYPE_INDEX];	// the secondary type...
				if (query.atmtp[1] == 0x0000) continue;
			}
			
			AddCaseSTAGE1c(& query, obs, depth);
		}
	}
}

void prmfit_tables::AddCaseSTAGE1c(const prmfit_ab_query * ref, f64 obs, i32s depth)
{
	prmfit_ab_query query = (* ref);
	
	DoParamSearch(query, & cout);
	i32s index = query.index;
	
	if (index == NOT_DEFINED)
	{
		// add a new ab-record...
		
		bool dir = !(query.atmtp[0] < query.atmtp[2]);
		
		prmfit_ab newab;
		newab.atmtp[0] = query.atmtp[dir ? 2 : 0];
		newab.atmtp[1] = query.atmtp[1];
		newab.atmtp[2] = query.atmtp[!dir ? 2 : 0];
		newab.bndtp[0] = query.bndtp[dir];
		newab.bndtp[1] = query.bndtp[!dir];
		
		newab.opt = 0.0;		// collect observed data here!!!
		newab.obs_count = 0;
		
		newab.fc = 550.0;		// default value; C-C-C angle fc.
		
		index = ab_vector.size();
		ab_vector.push_back(newab);
	}
	
	// add the new observed value...
	
	ab_vector[index].obs_count++;
	ab_vector[index].opt += obs;
	
	// handle the secondary types recursively...
	// handle the secondary types recursively...
	// handle the secondary types recursively...
	
	if (depth++ >= MAX_RECURSION_DEPTH_STAGE1) return;
	
	// first and third types of ab will be handled using wildcards later...
	// first and third types of ab will be handled using wildcards later...
	// first and third types of ab will be handled using wildcards later...
	
	const prmfit_at * at = GetAtomType((* ref).atmtp[1]);
	if (!at) { cout << "GetAtomType() failed!!!" << endl; exit(EXIT_FAILURE); }
	query.atmtp[1] = at->atomtype[SECONDARY_TYPE_INDEX];	// the secondary type...
	if (query.atmtp[1] == 0x0000) return;
	
	AddCaseSTAGE1c(& query, obs, depth);
}

void prmfit_tables::AddCaseSTAGE1c(const prmfit_tr_query * ref, i32s depth)
{
	prmfit_tr_query query = (* ref);
	
	DoParamSearch(query, & cout);
	i32s index = query.index;
	
	if (index == NOT_DEFINED)
	{
		// add a new tr-record...
		
		bool dir;
		if (query.atmtp[1] != query.atmtp[2]) dir = !(query.atmtp[1] < query.atmtp[2]);
		else dir = !(query.atmtp[0] < query.atmtp[3]);
		
		prmfit_tr newtr;
		newtr.atmtp[0] = query.atmtp[dir ? 3 : 0];
		newtr.atmtp[1] = query.atmtp[dir ? 2 : 1];
		newtr.atmtp[2] = query.atmtp[!dir ? 2 : 1];
		newtr.atmtp[3] = query.atmtp[!dir ? 3 : 0];
		newtr.bndtp[0] = query.bndtp[dir ? 2 : 0];
		newtr.bndtp[1] = query.bndtp[1];
		newtr.bndtp[2] = query.bndtp[!dir ? 2 : 0];
		
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			newtr.k[n2] = 0.0;	// default value???
			newtr.t[n2] = 0.0;	// default value???
		}
		
		// for ???-planar-planar-??? cases set different default values...
		// for ???-planar-planar-??? cases set different default values...
		// for ???-planar-planar-??? cases set different default values...
		
		const prmfit_at * at;
		at = GetAtomType((* ref).atmtp[1]); bool planar1 = (at != NULL && (at->flags & 3) == 2);
		at = GetAtomType((* ref).atmtp[2]); bool planar2 = (at != NULL && (at->flags & 3) == 2);
		if (planar1 && planar2) newtr.k[1] = +10.0;
		
		index = tr_vector.size();
		tr_vector.push_back(newtr);
	}
	
	// tr-data is not collected here...
	
	// handle the secondary types recursively...
	// handle the secondary types recursively...
	// handle the secondary types recursively...
	
	if (depth++ >= MAX_RECURSION_DEPTH_STAGE1) return;
	
	// first and fourth types of tr will be handled using wildcards later...
	// first and fourth types of tr will be handled using wildcards later...
	// first and fourth types of tr will be handled using wildcards later...
	
	for (i32s t1 = 0;t1 < 2;t1++)
	{
		for (i32s t2 = 0;t2 < 2;t2++)
		{
			if (!t1 && !t2) continue;	// this case is already added...
			
			if (!t1) query.atmtp[1] = (* ref).atmtp[1];
			else
			{
				const prmfit_at * at = GetAtomType((* ref).atmtp[1]);
				if (!at) { cout << "GetAtomType() failed!!!" << endl; exit(EXIT_FAILURE); }
				query.atmtp[1] = at->atomtype[SECONDARY_TYPE_INDEX];	// the secondary type...
				if (query.atmtp[1] == 0x0000) continue;
			}
			
			if (!t2) query.atmtp[2] = (* ref).atmtp[2];
			else
			{
				const prmfit_at * at = GetAtomType((* ref).atmtp[2]);
				if (!at) { cout << "GetAtomType() failed!!!" << endl; exit(EXIT_FAILURE); }
				query.atmtp[2] = at->atomtype[SECONDARY_TYPE_INDEX];	// the secondary type...
				if (query.atmtp[2] == 0x0000) continue;
			}
			
			AddCaseSTAGE1c(& query, depth);
		}
	}
}

void prmfit_tables::AddCaseSTAGE1c(const prmfit_op_query * ref, f64 obs, i32s depth)
{
	prmfit_op_query query = (* ref);
	
	DoParamSearch(query, & cout);
	i32s index = query.index;
	
	if (index == NOT_DEFINED)
	{
		// add a new op-record...
		
		bool dir = !(query.atmtp[0] < query.atmtp[2]);
		
		prmfit_op newop;
		newop.atmtp[0] = query.atmtp[dir ? 2 : 0];
		newop.atmtp[1] = query.atmtp[1];
		newop.atmtp[2] = query.atmtp[!dir ? 2 : 0];
		newop.atmtp[3] = query.atmtp[3];
		newop.bndtp[0] = query.bndtp[dir ? 1 : 0];
		newop.bndtp[1] = query.bndtp[!dir ? 1 : 0];
		newop.bndtp[2] = query.bndtp[2];
		
		newop.opt = 0.0;		// collect observed data here!!!
	//	newop.obs_count = 0;
		
		newop.fc = 125.0;		// default value; see the default for ab...
		
		index = op_vector.size();
		op_vector.push_back(newop);
	}
	
	// add the new observed value...
	
// collect data for non-planar cases?!?!?!
//op_vector[index].obs_count++; REMEMBER ZEROING ABOVE!!!
//op_vector[index].opt += obs;

	// handle the secondary types recursively...
	// handle the secondary types recursively...
	// handle the secondary types recursively...
	
	if (depth++ >= MAX_RECURSION_DEPTH_STAGE1) return;
	
	// all types other than second of op will be handled using wildcards later...
	// all types other than second of op will be handled using wildcards later...
	// all types other than second of op will be handled using wildcards later...
	
	const prmfit_at * at = GetAtomType((* ref).atmtp[1]);
	if (!at) { cout << "GetAtomType() failed!!!" << endl; exit(EXIT_FAILURE); }
	query.atmtp[1] = at->atomtype[SECONDARY_TYPE_INDEX];	// the secondary type...
	if (query.atmtp[1] == 0x0000) return;
	
	at = GetAtomType(query.atmtp[1]);			// also test if the term...
	if (!at || (at != NULL && !(at->flags & 16))) return;	// ...is requested at all.
	
	AddCaseSTAGE1c(& query, obs, depth);
}

void prmfit_tables::CalcParamSTAGE1(void)
{
	// here we just calculate averages using data collected at prmfit_tables::AddCaseSTAGE1b()...
	
	// bs-terms...
	// bs-terms...
	// bs-terms...
	
	sort(bs_vector.begin(), bs_vector.end());
	
	for (i32u n1 = 0;n1 < bs_vector.size();n1++)
	{
		f64 tmp1 = bs_vector[n1].opt;
		f64 tmp2 = bs_vector[n1].obs_count;
		
		if (tmp2 < 1.0)
		{
			cout << "FATAL ERROR at prmfit_tables::CalcParamSTAGE1():" << endl;
			cout << "obs_count with value of zero was found..." << endl;
			exit(EXIT_FAILURE);
		}
		
		bs_vector[n1].opt = tmp1 / tmp2;
	}
	
	// ab-terms...
	// ab-terms...
	// ab-terms...
	
// HOW TO HANDLE WILDCARDED TERMS??? try to cluster non-wildcarded ones and average?!?!?!
// HOW TO HANDLE WILDCARDED TERMS??? try to cluster non-wildcarded ones and average?!?!?!
// HOW TO HANDLE WILDCARDED TERMS??? try to cluster non-wildcarded ones and average?!?!?!

	sort(ab_vector.begin(), ab_vector.end());
	
	for (i32u n1 = 0;n1 < ab_vector.size();n1++)
	{
		f64 tmp1 = ab_vector[n1].opt;
		f64 tmp2 = ab_vector[n1].obs_count;
		
		if (tmp2 < 1.0)
		{
			cout << "FATAL ERROR at prmfit_tables::CalcParamSTAGE1():" << endl;
			cout << "obs_count with value of zero was found..." << endl;
			exit(EXIT_FAILURE);
		}
		
		ab_vector[n1].opt = tmp1 / tmp2;
	}
	
	// ok, no more data is collected at this stage. now just sort the rest of the tables...
	
	sort(tr_vector.begin(), tr_vector.end());
	sort(op_vector.begin(), op_vector.end());
	
	// set initial values for lj_r and lj_e to atomtype records...
	
	for (i32u n1 = 0;n1 < at2_vector.size();n1++)
	{
		// how to get some reasonable estimates for lj-parameters?!?!?!
		// how to get some reasonable estimates for lj-parameters?!?!?!
		// how to get some reasonable estimates for lj-parameters?!?!?!
		
		f64 r; f64 e;
		switch (at2_vector[n1].atomtype[0] >> 8)
		{
			case 1:		r = 0.150;	e = 0.175;	break;
			case 6:		r = 0.170;	e = 0.450;	break;
			case 7:		r = 0.155;	e = 0.400;	break;
			case 8:		r = 0.152;	e = 0.485;	break;
			default:	r = 0.170;	e = 0.450;
		}
		
		at2_vector[n1].lj_r = r;
		at2_vector[n1].lj_e = e;
	}
}

void prmfit_tables::DistortStructureSTAGE3(mm1_eng_prmfit * eng)
{
	// what is the best strategy of distorting structures here???
	
	// how much higher energy the resulting structures should be compared to starting structures?
	// generally, it would be good to avoid coupling between terms (for example, between len and ang).
	// keeping the modification frequencies low helps avoiding the coupling between terms...
	
	i32s number_of_modifications = 0;
	start_of_modifications_loop:
	
	for (i32u n1 = 0;n1 < eng->bt1_vector.size();n1++)
	{
		if ((RAND_f64 + 0.5) > 0.10) continue;					// 10 % modified
		eng->bt1_vector[n1].opt *= (1.0 + 0.40 * RAND_f64);			// plus/minus 20 %
		number_of_modifications++;
	}
	
	for (i32u n1 = 0;n1 < eng->bt2_vector.size();n1++)
	{
		if ((RAND_f64 + 0.5) > 0.10) continue;					// 10 % modified
		eng->bt2_vector[n1].opt += ((80.0 * M_PI / 180.0) * RAND_f64);		// plus/minus 40 deg
		eng->bt2_vector[n1].fc *= 10.0;		// we also need to override the connected bt2-terms...
		number_of_modifications++;
		
		if (eng->bt2_vector[n1].opt > M_PI) eng->bt2_vector[n1].opt = 2.0 * M_PI - eng->bt2_vector[n1].opt;
		if (eng->bt2_vector[n1].opt < 0.0) eng->bt2_vector[n1].opt = -eng->bt2_vector[n1].opt;
	}
	
	// torsions???
	// torsions???
	// torsions???
	
	for (i32u n1 = 0;n1 < eng->bt4_vector.size();n1++)
	{
		if ((RAND_f64 + 0.5) > 0.10) continue;					// 10 % modified
		eng->bt4_vector[n1].opt += ((80.0 * M_PI / 180.0) * RAND_f64);		// plus/minus 40 deg
		eng->bt4_vector[n1].fc *= 10.0;		// we also need to override the connected bt4-terms...
		number_of_modifications++;
		
		if (eng->bt4_vector[n1].opt > M_PI) eng->bt4_vector[n1].opt = 2.0 * M_PI - eng->bt4_vector[n1].opt;
		if (eng->bt4_vector[n1].opt < 0.0) eng->bt4_vector[n1].opt = -eng->bt4_vector[n1].opt;
	}
	
	// since the modifications are random, with low frequencies, here we make
	// sure that at least one modification is done for each of the structures...
	
	if (!number_of_modifications) goto start_of_modifications_loop;
}

void prmfit_tables::WriteAtomTypes(void)
{
	char fn[1024];
	ostrstream fn_str(fn, sizeof(fn));
	fn_str << path << "/atomtypes.txt.out" << ends;
	
	cout << "writing parameter file:" << endl;
	cout << fn << endl << endl;
	
	ofstream file;
	file.open(fn, ios::out);
	
	file << "// an automatically generated atomtype file - do not edit." << endl;
	file << "// [primary_type] [secondary_type] [typerule] [description] [lj_r] [lj_e] [formal_charge] [flags]" << endl;
	file << endl;
	file << "// [lj_r] = nm" << endl;
	file << "// [lj_e] = kJ/mol" << endl;
	file << "// [formal_charge] = electron charge" << endl;
	file << endl;
	
	file.setf(ios::fixed);
	
	for (i32u n1 = 0;n1 < at2_vector.size();n1++)
	{
		file << "0x" << hex << setw(4) << setfill('0') << at2_vector[n1].atomtype[0] << dec << " ";
		file << "0x" << hex << setw(4) << setfill('0') << at2_vector[n1].atomtype[1] << dec << " ";
		
		file << "(" << (* at2_vector[n1].typerule) << ") ";
		file << "\"" << at2_vector[n1].description << "\" ";
		
		file.precision(6);
		file << at2_vector[n1].lj_r << " ";
		file << at2_vector[n1].lj_e << " ";
		
		file.setf(ios::showpos); file.precision(4);
		file << at2_vector[n1].formal_charge << " ";
		file.unsetf(ios::showpos);
		
		file << at2_vector[n1].flags << " ";
		file << endl;
	}
	
	file << "#end" << endl;
	file.close();
}

void prmfit_tables::WriteParamFiles(void)
{
	char fn[1024]; ofstream file;
	
/*##############################################*/
/*##############################################*/
	
	ostrstream fn_str1(fn, sizeof(fn));
	fn_str1 << path << "/parameters1.txt" << ends;
	file.open(fn, ios::out);
	
	cout << "writing parameter file:" << endl;
	cout << fn << endl << endl;
	
	file << "// an automatically generated parameter file - do not edit." << endl;
	file << "// bs : [id1] [id2] [bt1] [opt] [fc] [cid]" << endl;
	file << endl;
	file << "// [opt] = nm" << endl;
	file << "// [fc] = kJ/(mol*nm^2)" << endl;
	file << "// [cid] = electron charge" << endl;
	file << endl;
	
	file.setf(ios::scientific); file.precision(6);
	
	for (i32u n1 = 0;n1 < bs_vector.size();n1++)
	{
		file << "0x" << hex << setw(4) << setfill('0') << bs_vector[n1].atmtp[0] << dec << " ";
		file << "0x" << hex << setw(4) << setfill('0') << bs_vector[n1].atmtp[1] << dec << " ";
		file << bs_vector[n1].bndtp.GetSymbol1() << " ";
		
		file << bs_vector[n1].opt << " ";
		file << bs_vector[n1].fc << " ";
		
		file << bs_vector[n1].cid << " ";
		
		file << endl;
	}
	
	file << "#end" << endl;
	file.close();
	
/*##############################################*/
/*##############################################*/

	ostrstream fn_str2(fn, sizeof(fn));
	fn_str2 << path << "/parameters2.txt" << ends;
	file.open(fn, ios::in);
	
	cout << "writing parameter file:" << endl;
	cout << fn << endl << endl;
	
	file << "// an automatically generated parameter file - do not edit." << endl;
	file << "// ab : [id1] [id2] [id3] [bt1][bt2] [opt] [fc]" << endl;
	file << endl;
	file << "// [opt] = rad" << endl;
	file << "// [fc] = kJ/(mol*rad^2)" << endl;
	file << endl;
	
	file.setf(ios::scientific); file.precision(6);
	
	for (i32u n1 = 0;n1 < ab_vector.size();n1++)
	{
		file << "0x" << hex << setw(4) << setfill('0') << ab_vector[n1].atmtp[0] << dec << " ";
		file << "0x" << hex << setw(4) << setfill('0') << ab_vector[n1].atmtp[1] << dec << " ";
		file << "0x" << hex << setw(4) << setfill('0') << ab_vector[n1].atmtp[2] << dec << " ";
		file << ab_vector[n1].bndtp[0].GetSymbol1();	// no space!!!
		file << ab_vector[n1].bndtp[1].GetSymbol1() << " ";
		
		file << ab_vector[n1].opt << " ";
		file << ab_vector[n1].fc << " ";
		
		file << endl;
	}
	
	file << "#end" << endl;
	file.close();
	
/*##############################################*/
/*##############################################*/

	ostrstream fn_str3(fn, sizeof(fn));
	fn_str3 << path << "/parameters3.txt" << ends;
	file.open(fn, ios::in);
	
	cout << "writing parameter file:" << endl;
	cout << fn << endl << endl;
	
	file << "// an automatically generated parameter file - do not edit." << endl;
	file << "// tr : [id1] [id2] [id3] [id4] [bt1][bt2][bt3] [k1] [k2] [k3] [t1] [t2] [t3]" << endl;
	file << endl;
	file << "// [k?] = kJ/mol" << endl;
	file << "// [t?] = rad" << endl;
	file << endl;
	
	file.setf(ios::scientific); file.precision(6);
	
	for (i32u n1 = 0;n1 < tr_vector.size();n1++)
	{
		file << "0x" << hex << setw(4) << setfill('0') << tr_vector[n1].atmtp[0] << dec << " ";
		file << "0x" << hex << setw(4) << setfill('0') << tr_vector[n1].atmtp[1] << dec << " ";
		file << "0x" << hex << setw(4) << setfill('0') << tr_vector[n1].atmtp[2] << dec << " ";
		file << "0x" << hex << setw(4) << setfill('0') << tr_vector[n1].atmtp[3] << dec << " ";
		file << tr_vector[n1].bndtp[0].GetSymbol1();	// no space!!!
		file << tr_vector[n1].bndtp[1].GetSymbol1();	// no space!!!
		file << tr_vector[n1].bndtp[2].GetSymbol1() << " ";
		
		file << tr_vector[n1].k[0] << " ";
		file << tr_vector[n1].k[1] << " ";
		file << tr_vector[n1].k[2] << " ";
		
		file << tr_vector[n1].t[0] << " ";
		file << tr_vector[n1].t[1] << " ";
		file << tr_vector[n1].t[2] << " ";
		
		file << endl;
	}
	
	file << "#end" << endl;
	file.close();
	
/*##############################################*/
/*##############################################*/

	ostrstream fn_str4(fn, sizeof(fn));
	fn_str4 << path << "/parameters4.txt" << ends;
	file.open(fn, ios::in);
	
	cout << "writing parameter file:" << endl;
	cout << fn << endl << endl;
	
	file << "// an automatically generated parameter file - do not edit." << endl;
	file << "// op : [id1] [id2] [id3] [id4] [bt1][bt2][bt3] [opt] [fc]" << endl;
	file << endl;
	file << "// [opt] = rad" << endl;
	file << "// [fc] = kJ/(mol*rad^2)" << endl;
	file << endl;
	
	file.setf(ios::scientific); file.precision(6);
	
	for (i32u n1 = 0;n1 < op_vector.size();n1++)
	{
		file << "0x" << hex << setw(4) << setfill('0') << op_vector[n1].atmtp[0] << dec << " ";
		file << "0x" << hex << setw(4) << setfill('0') << op_vector[n1].atmtp[1] << dec << " ";
		file << "0x" << hex << setw(4) << setfill('0') << op_vector[n1].atmtp[2] << dec << " ";
		file << "0x" << hex << setw(4) << setfill('0') << op_vector[n1].atmtp[3] << dec << " ";
		file << op_vector[n1].bndtp[0].GetSymbol1();	// no space!!!
		file << op_vector[n1].bndtp[1].GetSymbol1();	// no space!!!
		file << op_vector[n1].bndtp[2].GetSymbol1() << " ";
		
		file << op_vector[n1].opt << " ";
		file << op_vector[n1].fc << " ";
		
		file << endl;
	}
	
	file << "#end" << endl;
	file.close();
	
/*##############################################*/
/*##############################################*/

}

/*################################################################################################*/

prmfit_cg_optim::prmfit_cg_optim(const char * p1) : prmfit_tables(p1), conjugate_gradient(100, 0.001)
{
	debug_level = 0;
	
	for (i32u n1 = 0;n1 < bs_vector.size();n1++)
	{
		AddVar(& bs_vector[n1].fc, & bs_vector[n1].d_fc);
	}
}

prmfit_cg_optim::~prmfit_cg_optim(void)
{
}

void prmfit_cg_optim::Check(i32s)
{
	const f64 delta = 0.000001;	// the finite difference step...
	f64 tmp1 = GetGradient();
	f64 tmp2; f64 old;
	
	for (i32u n1 = 0;n1 < bs_vector.size();n1++)
	{
		old = bs_vector[n1].fc;
		bs_vector[n1].fc = old + delta;
		tmp2 = (GetValue() - tmp1) / delta;
		bs_vector[n1].fc = old;
		
		cout << "bs " << n1 << " : ";
		cout << "a = " << bs_vector[n1].d_fc << " ";
		cout << "n = " << tmp2 << endl;
	}
}

f64 prmfit_cg_optim::GetValue(void)
{
	Calculate(0);
	return value;
}

f64 prmfit_cg_optim::GetGradient(void)
{
	Calculate(1);
	return value;
}

void prmfit_cg_optim::Calculate(i32s p1)
{
	value = 0.0;
	
	if (p1 > 0)
	{
		for (i32u n1 = 0;n1 < bs_vector.size();n1++)
		{
			bs_vector[n1].d_fc = 0.0;
		}
	}
	
	// the evaluation starts...
	// the evaluation starts...
	// the evaluation starts...
	
	const char files_stage1[] = "/home/thassine/DATABASE-mm1/organic_molecules/output_files/STAGE1.txt";
	
	const char ipath1[] = "/home/thassine/DATABASE-mm1/organic_molecules/input_files/ALLFILES";
	const char ipath2[] = "/home/thassine/DATABASE-mm1/organic_molecules/output_files/STAGE1";
	
	ProcessFiles(p1, files_stage1, ipath1, ipath2, 1.0, 1.0);
	
	// ok, all cases have been handled.
	// ok, all cases have been handled.
	// ok, all cases have been handled.
	
	if (debug_level > 1) cout << "VALUE = " << value << endl;
}

void prmfit_cg_optim::ProcessFiles(i32s p1, const char * ifiles, const char * ip1, const char * ip2, f64 w0, f64 w1)
{
	ifstream ifile1;
	
	char buffer[65536];
	
	char * str1; char * str2;
	i32s grp_counter; i32s max_counter;
	
	// read in the STAGE1 data...
	// read in the STAGE1 data...
	// read in the STAGE1 data...
	
	ifile1.open(ifiles, ios::in);
	grp_counter = 0; str1 = NULL; vector<i32s> s1_gv;
	
	while (true)
	{
		if (ifile1.peek() == '#') break;	// #end
		
		char filename1[1024];
		
		ifile1 >> filename1;
		ifile1.getline(buffer, sizeof(buffer));
		
		i32s search = strlen(filename1);
		while (search >= 0 && filename1[search] != '#') search--;
		if (search < 0) { cout << "search failed!!!" << endl; exit(EXIT_FAILURE); }
		filename1[search] = 0;
		
		str2 = new char[strlen(filename1) + 1]; strcpy(str2, filename1);
		
		bool new_grp = false;
		if (str1 == NULL) new_grp = true;
		if (str1 != NULL && strcmp(str1, str2) != 0) new_grp = true;
		if (new_grp) grp_counter++;
		
		s1_gv.push_back(grp_counter);
		
		if (str1 != NULL) delete[] str1; str1 = str2;
	}	if (str1 != NULL) delete[] str1;
	
	ifile1.close();
	ifile1.open(ifiles, ios::in);
	
	max_counter = s1_gv.size();
	s1_gv.push_back(-1);
	
	grp_counter = 0;
	while (grp_counter < max_counter)
	{
		i32s current = s1_gv[grp_counter];
		vector<file_io_handler *> iohv;
		
		while (s1_gv[grp_counter] == current)
		{
			char filename1[1024];
			
			ifile1 >> filename1;
			ifile1.getline(buffer, sizeof(buffer));
			
			if (debug_level > 1) cout << "grp = " << grp_counter << " ";
			if (debug_level > 0) cout << filename1 << " : ";
			
			const char * short_fn1 = filename1;
			for (i32u sfn = 0;sfn < strlen(filename1);sfn++)
			{
				if (filename1[sfn] == '/') short_fn1 = & filename1[sfn + 1];
			}
			
			char filename2[1024];
			ostrstream str2(filename2, sizeof(filename2)); str2 << ip1 << "/";
			i32s fn2 = 0; while (short_fn1[fn2] != '#') str2 << short_fn1[fn2++];
			str2 << short_fn1[fn2 + 0] << short_fn1[fn2 + 1] << short_fn1[fn2 + 2];
			str2 << ".gpr" << ends;
			
		//	mm1_mdl * mdl = new mm1_mdl(& cout, * console_class_factory::GetInstance());
			mm1_mdl * mdl = new mm1_mdl(NULL, * console_class_factory::GetInstance());
			
			ifstream ifile2;
			ifile2.open(filename2, ios::in);
			if (ifile2.fail())
			{
				cout << "could not open project file!!!" << endl;
				ifile2.close(); delete mdl; exit(EXIT_FAILURE);
			}
			
			mdl->ReadStream(ifile2, false);
			ifile2.close();
			
		// run a small geometry optimization?!?!?!
		// only read groups with more than 1 structure?!?!?!
			
			i32s search = strlen(filename1);
			while (search >= 0 && filename1[search] != '.') search--;
			if (search < 0) { cout << "search failed!!!" << endl; exit(EXIT_FAILURE); }
			
			file_io_handler * handler = NULL;
			switch (filename1[search + 1])
			{
				case 'g':	if (debug_level > 0) cout << "gaussian";
				handler = new gaussian_io_handler(* mdl);
				break;
				
				case 'm':	if (debug_level > 0) cout << "mpqc";
				handler = new mpqc_io_handler(* mdl);
				break;
				
				default:
				cout << "unknown output type id " << filename1[search + 1] << endl;
				exit(EXIT_FAILURE);
			}
			
			char filename3[1024];
			ostrstream str3(filename3, sizeof(filename3));
			str3 << ip2 << "/" << filename1 << ends;
			
			bool success = handler->ReadOutput(filename3, true, true, true);
			if (!success && (debug_level > 0)) cout << " FAILED!!!";
			if (debug_level > 0) cout << endl;
			
			iohv.push_back(handler);
			grp_counter++;
		}
		
		if (debug_level > 1) cout << "PROCESS!!!" << endl;
		
		for (i32s n1 = 0;n1 < (i32s) iohv.size() - 1;n1++)
		{
			for (i32s n2 = n1 + 1;n2 < (i32s) iohv.size();n2++)
			{
				CompareEnergyDiffs(p1, iohv[n1], iohv[n2], w0);
			}
		}
		
		while (!iohv.empty())
		{
			mm1_mdl * mdl = iohv.back()->mdl;
			delete iohv.back(); delete mdl;
			iohv.pop_back();
		}
	}
	
	ifile1.close();
}

void prmfit_cg_optim::CompareEnergyDiffs(i32s p1, file_io_handler * h1, file_io_handler * h2, f64 weight)
{
	if (debug_level > 1) cout << "CompareEnergyDiffs() called, weight = " << weight << endl;
	
	mm1_eng_prmfit * eng1 = new mm1_eng_prmfit(* h1->mdl, * this);
	CopyCRD(h1->mdl, eng1, 0); eng1->Compute(1);
	
	mm1_eng_prmfit * eng2 = new mm1_eng_prmfit(* h2->mdl, * this);
	CopyCRD(h2->mdl, eng2, 0); eng2->Compute(1);
	
	// f = ((Emm2 - Emm1) - (Eqm2 - Eqm1)) ^ 2
	// df = ???
	
	f64 de_mm = eng2->energy - eng1->energy;
	f64 de_qm = h2->e - h1->e;
	
	f64 delta = de_mm - de_qm;
	value += weight * delta * delta;
	
	if (p1 > 0)
	{
		f64 tmp9 = 2.0 * delta;
	//	const file_io_handler * htab[2] = { h1, h2 };
		const mm1_eng_prmfit * etab[2] = { eng1, eng2 };
		
		if (eng1->bt1_vector.size() != eng2->bt1_vector.size())
		{
			cout << "bt1 size error." << endl;
			if (h1->debug != NULL) cout << "h1 = " << h1->debug << endl;
			if (h2->debug != NULL) cout << "h2 = " << h2->debug << endl;
			exit(EXIT_FAILURE);
		}
		
		for (i32s loop = 0;loop < 2;loop++)
		{
			for (i32u n1 = 0;n1 < etab[loop]->bt1_vector.size();n1++)
			{
				i32s qi = etab[loop]->bt1_vector[n1].qi;
				
				f64 opt = etab[loop]->bt1_vector[n1].opt;
				f64 x = etab[loop]->bt1_vector[n1].x;
				
				f64 tmp1 = x - opt;
				f64 tmp2 = tmp1 * tmp1;
				if (!loop) tmp2 = -tmp2;
				
				bs_vector[qi].d_fc += tmp2 * tmp9;
			}
		}
	}
	
	delete eng1;
	delete eng2;
}

void prmfit_cg_optim::CompareGradients(i32s p1, file_io_handler * h, f64 weight)
{
	if (debug_level > 1) cout << "CompareGradients() called, weight = " << weight << endl;
}

/*################################################################################################*/

file_io_handler::file_io_handler(mm1_mdl & p1)
{
	mdl = & p1;
	
	has_e = false;
	has_d1 = false; d1 = NULL;
	has_crd = false; crd = NULL;
	
	debug = NULL;
}

file_io_handler::~file_io_handler(void)
{
	if (d1 != NULL) delete[] d1;
	if (crd != NULL) delete[] crd;
	
	if (debug != NULL) delete[] debug;
}

/*################################################################################################*/

gaussian_io_handler::gaussian_io_handler(mm1_mdl & p1) : file_io_handler(p1)
{
}

gaussian_io_handler::~gaussian_io_handler(void)
{
}

void gaussian_io_handler::WriteSettings(ofstream & ofile)
{
	ofile << "0 1" << endl;		// write total charge and spin state
}

void gaussian_io_handler::WriteMolecule(ofstream & ofile)
{
	for (iter_mm1al it1 = mdl->GetAtomsBegin();it1 != mdl->GetAtomsEnd();it1++)
	{
		const f64 cf1 = 10.0;	// conversion factor for [nm] -> []
		
		ofile << "  " << (* it1).el.GetAtomicNumber();
		ofile << "  " << setprecision(6) << setw(12) << ((* it1).crd_vector[0][0] * cf1);
		ofile << "  " << setprecision(6) << setw(12) << ((* it1).crd_vector[0][1] * cf1);
		ofile << "  " << setprecision(6) << setw(12) << ((* it1).crd_vector[0][2] * cf1);
		ofile << endl;
	}
}

void gaussian_io_handler::WriteInput(const char * fn, bool optimize, bool get_energy, bool get_gradient)
{
	ofstream ofile;
	ofile.open(fn, ios::out);

	ofile << "%chk=checkpoint1" << endl;
	
	if (optimize)
	{
		ofile << "#t AM1 Opt" << endl;
		ofile << endl;
		ofile << fn << endl;		// kommentti
		ofile << endl;
		WriteSettings(ofile);
		WriteMolecule(ofile);
		
		ofile << endl;
		
		ofile << "--link1--" << endl;
		ofile << "%chk=checkpoint1" << endl;
		ofile << "#t HF/6-31G* Opt=readfc" << endl;
		ofile << "geom=check" << endl;
		ofile << endl;
		ofile << fn << endl;		// kommentti
		ofile << endl;
		WriteSettings(ofile);
		
		ofile << endl;
		
		ofile << "--link1--" << endl;
		ofile << "%chk=checkpoint1" << endl;
		ofile << "#t MP2/6-31G* Opt=readfc" << endl;
		ofile << "geom=check guess=read maxdisk=50000000" << endl;
		ofile << endl;
		ofile << fn << endl;		// kommentti
		ofile << endl;
		WriteSettings(ofile);
		
		ofile << endl;
		
		ofile << "--link1--" << endl;
		ofile << "%chk=checkpoint1" << endl;
		ofile << "# MP2/6-31G* Force scf=tight" << endl;
		ofile << "geom=check guess=read maxdisk=50000000" << endl;
		ofile << endl;
		ofile << fn << endl;		// kommentti
		ofile << endl;
		WriteSettings(ofile);
		
		ofile << endl;
		ofile << endl;
		ofile << endl;
	}
	else
	{
		ofile << "# MP2/6-31G* Force scf=tight" << endl;
		ofile << "maxdisk=50000000" << endl;
		ofile << endl;
		ofile << fn << endl;		// kommentti
		ofile << endl;
		WriteSettings(ofile);
		WriteMolecule(ofile);
		
		ofile << endl;
		ofile << endl;
		ofile << endl;
	}
	
	ofile.close();
}

bool gaussian_io_handler::ReadOutput(const char * fn, bool optimize, bool get_energy, bool get_gradient)
{
if (debug != NULL) delete[] debug;
debug = new char[strlen(fn) + 1]; strcpy(debug, fn);

	// first initialize the arrays that we will need.
	
	if (get_gradient) d1 = new f64[mdl->GetAtomCount() * 3];
	if (optimize) crd = new f64[mdl->GetAtomCount() * 3];
	
	const double cf1 = 18.897162;	// conversion factor for [nm] -> [bohr]
	const double cf2 = 2625.5;	// conversion factor for [Hartree] -> [kJ/mol]
	
	// start reading the file...
	
	ifstream ifile; ifile.open(fn, ios::in);
	if (ifile.fail()) { ifile.close(); return false; }	// could not open the file...
	
	char buffer1[1024]; char buffer2[1024]; i32s line_number = 0;
	
	// when a requested result is found from the file, change the corresponding flag to true.
	// cycle the loop until all results are found, or until we reach end of file...
	
	while (true)
	{
		bool we_have_found_all_results = true;
		if (get_energy && !has_e) we_have_found_all_results = false;
		if (get_gradient && !has_d1) we_have_found_all_results = false;
		if (optimize && !has_crd) we_have_found_all_results = false;
		if (we_have_found_all_results) break;
		
		ifile.getline(buffer1, sizeof(buffer1)); line_number++;
		if (ifile.eof()) { ifile.close(); return false; }		// end of file was reached...
		
		strcpy(buffer2, buffer1);					// gradient.
		if (!strcmp(buffer2, " Center     Atomic                   Forces (Hartrees/Bohr)"))
		{
//			cout << "d1 found " << line_number << endl;
			ifile.getline(buffer2, sizeof(buffer2)); line_number++;
			ifile.getline(buffer2, sizeof(buffer2)); line_number++;
			
			for (i32s n1 = 0;n1 < mdl->GetAtomCount();n1++)
			{
				ifile.getline(buffer2, sizeof(buffer2)); line_number++;
				istrstream str1(buffer2, sizeof(buffer2));
				
				i32s t1; i32s t2; f64 tx; f64 ty; f64 tz;
				str1 >> t1 >> t2 >> tx >> ty >> tz;
				
				d1[n1 * 3 + 0] = tx * cf2 * cf1;
				d1[n1 * 3 + 1] = ty * cf2 * cf1;
				d1[n1 * 3 + 2] = tz * cf2 * cf1;
			}
			
			has_d1 = true; continue;
		}
		
	/*	strcpy(buffer2, buffer1);					// energy if optimized.
		if (!strcmp(buffer2, "  The optimization has converged."))
		{
//			cout << "e found " << line_number << endl;
			ifile.getline(buffer2, sizeof(buffer2)); line_number++;
			ifile.getline(buffer2, sizeof(buffer2), ':'); line_number++;
			
			f64 t1; ifile >> t1; e = t1 * cf2;
			
			has_e = true; continue;
		}	*/	has_e = true;
		
		strcpy(buffer2, buffer1);					// coordinates if optimized.
		if (!strcmp(buffer2, "                          Input orientation:                          "))
		{
//			cout << "crd found " << line_number << endl;
			ifile.getline(buffer2, sizeof(buffer2)); line_number++;
			ifile.getline(buffer2, sizeof(buffer2)); line_number++;
			ifile.getline(buffer2, sizeof(buffer2)); line_number++;
			ifile.getline(buffer2, sizeof(buffer2)); line_number++;

			for (i32s n1 = 0;n1 < mdl->GetAtomCount();n1++)
			{
				ifile.getline(buffer2, sizeof(buffer2)); line_number++;
				istrstream str1(buffer2, sizeof(buffer2));
				
				i32s t1; i32s t2; i32s t3; f64 tx; f64 ty; f64 tz;
				str1 >> t1 >> t2 >> t3 >> tx >> ty >> tz;
				
				crd[n1 * 3 + 0] = tx / cf1;
				crd[n1 * 3 + 1] = ty / cf1;
				crd[n1 * 3 + 2] = tz / cf1;
			}
			
			has_crd = true; continue;
		}
	}
	
	ifile.close();
	return true;
}

/*################################################################################################*/

mpqc_io_handler::mpqc_io_handler(mm1_mdl & p1) : file_io_handler(p1)
{
}

mpqc_io_handler::~mpqc_io_handler(void)
{
}

void mpqc_io_handler::WriteInput(const char * fn, bool optimize, bool get_energy, bool get_gradient)
{
	ofstream ofile;
	ofile.open(fn, ios::out);
	ofile.setf(ios::fixed);
	
	const fGL cf1 = 10.0;		// conversion factor for [nm] -> []
	
	// molecule-record...
	// molecule-record...
	// molecule-record...
	
	ofile << "molecule<Molecule>: (" << endl;
	ofile << "  symmetry = C1" << endl;
	ofile << "  unit = angstrom" << endl;
	ofile << "  { atoms geometry } = {" << endl;
	
	for (iter_mm1al it1 = mdl->GetAtomsBegin();it1 != mdl->GetAtomsEnd();it1++)
	{
		fGL * crd = (* it1).crd_vector[0].data;
		
		ofile << "    " << (* it1).el.GetSymbol() << "\t[ ";
		ofile << setprecision(6) << setw(12) << (crd[0] * cf1) << " ";
		ofile << setprecision(6) << setw(12) << (crd[1] * cf1) << " ";
		ofile << setprecision(6) << setw(12) << (crd[2] * cf1) << " ";
		ofile << "]" << endl;
	}
	
	ofile << "  }" << endl;
	ofile << ")" << endl;
	
	// basis-record...
	// basis-record...
	// basis-record...
	
	ofile << "basis<GaussianBasisSet>: (" << endl;
	
//	ofile << "  name = \"STO-3G\"" << endl;
//	ofile << "  name = \"STO-6G\"" << endl;
//	ofile << "  name = \"3-21G\"" << endl;
//	ofile << "  name = \"3-21G*\"" << endl;
//	ofile << "  name = \"4-31G\"" << endl;
	ofile << "  name = \"4-31G*\"" << endl;
//	ofile << "  name = \"4-31G**\"" << endl;
//	ofile << "  name = \"6-31G\"" << endl;
//	ofile << "  name = \"6-31G*\"" << endl;
//	ofile << "  name = \"6-31G**\"" << endl;
	
	ofile << "  molecule = $:molecule" << endl;
	ofile << ")" << endl;
	
	// mpqc-record...
	// mpqc-record...
	// mpqc-record...
	
	ofile << "mpqc: (" << endl;
	
	if (get_energy) ofile << "  do_energy = yes" << endl;
	else ofile << "  do_energy = no" << endl;
	
	if (get_gradient) ofile << "  do_gradient = yes" << endl;
	else ofile << "  do_gradient = no" << endl;
	
	ofile << "  mole<CLHF>: (" << endl;
//	ofile << "    total_charge = 0" << endl;
	ofile << "    molecule = $:molecule" << endl;
	ofile << "    basis = $:basis" << endl;
	ofile << "    coor = $..:coor" << endl;
	ofile << "  )" << endl;
	
	if (optimize)
	{
		ofile << "  opt<QNewtonOpt>: (" << endl;
		ofile << "    max_iterations = 100" << endl;
		ofile << "    function = $..:mole" << endl;
		ofile << "    update<BFGSUpdate>: ()" << endl;
		ofile << "    convergence<MolEnergyConvergence>: (" << endl;
		ofile << "      cartesian = yes" << endl;
		ofile << "      energy = $..:..:mole" << endl;
		ofile << "    )" << endl;
		ofile << "  )" << endl;
	}
	
	ofile << ")" << endl;
	ofile.close();
}

bool mpqc_io_handler::ReadOutput(const char * fn, bool optimize, bool get_energy, bool get_gradient)
{
if (debug != NULL) delete[] debug;
debug = new char[strlen(fn) + 1]; strcpy(debug, fn);

	// first initialize the arrays that we will need.
	
	if (get_gradient) d1 = new f64[mdl->GetAtomCount() * 3];
	if (optimize) crd = new f64[mdl->GetAtomCount() * 3];
	
	const double cf1 = 18.897162;	// conversion factor for [nm] -> [bohr]
	const double cf2 = 2625.5;	// conversion factor for [Hartree] -> [kJ/mol]
	
	// start reading the file...
	
	ifstream ifile; ifile.open(fn, ios::in);
	if (ifile.fail()) { ifile.close(); return false; }	// could not open the file...
	
	char buffer1[1024]; char buffer2[1024]; i32s line_number = 0;
	
	// when a requested result is found from the file, change the corresponding flag to true.
	// cycle the loop until all results are found, or until we reach end of file...
	
	while (true)
	{
		bool we_have_found_all_results = true;
		if (get_energy && !has_e) we_have_found_all_results = false;
		if (get_gradient && !has_d1) we_have_found_all_results = false;
		if (optimize && !has_crd) we_have_found_all_results = false;
		if (we_have_found_all_results) break;
		
		ifile.getline(buffer1, sizeof(buffer1)); line_number++;
		if (ifile.eof()) { ifile.close(); return false; }		// end of file was reached...
		
		strcpy(buffer2, buffer1);					// gradient.
		if (!strcmp(buffer2, "  Total Gradient:"))
		{
//			cout << "d1 found " << line_number << endl;
			for (i32s n1 = 0;n1 < mdl->GetAtomCount();n1++)
			{
				ifile.getline(buffer2, sizeof(buffer2)); line_number++;
				istrstream str1(buffer2, sizeof(buffer2));
				
				i32s t1; char t2[16]; f64 tx; f64 ty; f64 tz;
				str1 >> t1 >> t2 >> tx >> ty >> tz;
				
				d1[n1 * 3 + 0] = tx * cf2 * cf1;
				d1[n1 * 3 + 1] = ty * cf2 * cf1;
				d1[n1 * 3 + 2] = tz * cf2 * cf1;
			}
			
			has_d1 = true; continue;
		}
		
		strcpy(buffer2, buffer1);					// energy if optimized.
		if (!strcmp(buffer2, "  The optimization has converged."))
		{
//			cout << "e found " << line_number << endl;
			ifile.getline(buffer2, sizeof(buffer2)); line_number++;
			ifile.getline(buffer2, sizeof(buffer2), ':'); line_number++;
			
			f64 t1; ifile >> t1; e = t1 * cf2;
			
			has_e = true; continue;
		}
		
		strcpy(buffer2, buffer1);					// coordinates if optimized.
		if (!strcmp(buffer2, "      {  n atoms                        geometry                     }={"))
		{
//			cout << "crd found " << line_number << endl;
			for (i32s n1 = 0;n1 < mdl->GetAtomCount();n1++)
			{
				ifile.getline(buffer2, sizeof(buffer2)); line_number++;
				istrstream str1(buffer2, sizeof(buffer2));
				
				i32s t1; char t2[16]; char t3[16]; f64 tx; f64 ty; f64 tz;
				str1 >> t1 >> t2 >> t3 >> tx >> ty >> tz;
				
				crd[n1 * 3 + 0] = tx / cf1;
				crd[n1 * 3 + 1] = ty / cf1;
				crd[n1 * 3 + 2] = tz / cf1;
			}
			
			has_crd = true; continue;
		}
	}
	
	ifile.close();
	return true;
}

/*################################################################################################*/

// eof
