/*
 * $Id: xml.c,v 1.6 2003/12/22 02:47:21 hipnod Exp $
 *
 * Copyright (C) 2001-2003 giFT project (gift.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include "gt_gnutella.h"

#include <zlib.h>

#ifdef USE_LIBXML2
#include <libxml/parser.h>         /* xmlParseMemory() */
#include <libxml/xmlerror.h>       /* xmlSetGenericErrorFunc() */
#endif /* USE_LIBXML2 */

#include "xml.h"

/*****************************************************************************/

#ifndef USE_LIBXML2
BOOL gt_xml_parse (const char *xml, Dataset **ret)
{
	return FALSE;
}

BOOL gt_xml_parse_indexed (const char *xml, size_t bin_len, Share **shares,
                           size_t shares_len)
{
	return FALSE;
}
#endif /* !USE_LIBXML2 */

/*****************************************************************************/

/* the rest of this file is conditional on using libxml */
#ifdef USE_LIBXML2

/*****************************************************************************/

#define XML_BUFSIZE  65536

static char      xml_buf[XML_BUFSIZE];    /* for decompressing xml */

/*****************************************************************************/

static void print_nodes (xmlNodePtr node, Dataset **ret)
{
	while (node != NULL)
	{
		/*
		 * If this node has no children, it is a leaf node,
		 * so set the metadata from it.
		 */
		if (node->xmlChildrenNode)
			print_nodes (node->xmlChildrenNode, ret);
		else
			GT->DBGFN (GT, "name=%s", node->name);

		node = node->next;
	}
}

BOOL gt_xml_parse (const char *xml, Dataset **ret)
{
	xmlDocPtr doc;

	/* disable for now because it doesn't work anyway: need to share
	 * code with parse_indexed */
	if (!XML_DEBUG)
		return FALSE;

	/* only parse documents starting with '<' */
	if (!xml || xml[0] != '<')
		return FALSE;

	if (!(doc = xmlParseMemory (xml, strlen (xml))))
		return FALSE;

	print_nodes (doc->xmlChildrenNode, ret);

	xmlFreeDoc (doc);

	return TRUE;
}

static void add_child (Dataset **children, char *key, char *value)
{
	BOOL need_free = FALSE;

	if (!key || !value)
		return;

	/*
	 * Hack to map some of the attributes from XML documents found
	 * on Gnutella to ones peddled by giFT.
	 */
	if (!strcasecmp (key, "bitrate"))
	{
		need_free = TRUE;
		value = stringf_dup ("%s000", value);
	}
	else if (!strcasecmp (key, "seconds"))
	{
		key = "duration";
	}

	dataset_insertstr (children, key, value);

	if (need_free)
		free (value);
}

static Dataset *collect_attributes (xmlNode *node)
{
	xmlAttr  *attr;
	Dataset  *children = NULL;
	BOOL      do_log;

	attr = (xmlAttr *)node->properties;

	do_log = XML_DEBUG;

	while (attr != NULL)
	{
		xmlChar *str;

		/* is there an easier way to get attribute content? */
		str = xmlGetProp (node, attr->name);

		if (do_log)
		{
			GT->dbg (GT, "name=%s content=%s",
			         (char *)attr->name, (char *)str);
		}

		/* add the key->value pair to the dataset */
		add_child (&children, (char *)attr->name, (char *)str);

		/* xmlGetProp() allocates memory */
		free (str);

		attr = (xmlAttr *)attr->next;
	}

	return children;
}

static void set_meta_foreach (ds_data_t *key, ds_data_t *value, Share *share)
{
	char *meta_key = key->data;
	char *meta_val = value->data;

	share_set_meta (share, meta_key, meta_val);
}

static void set_share_meta (Share **shares, size_t shares_len,
                            Dataset *children)
{
	char      *index_str;
	size_t     index;

	/*
	 * Lookup the "index" attribute, and use that to determine
	 * which Share the XML applies to.
	 */
	if (!(index_str = dataset_lookupstr (children, "index")))
		return;

	index = gift_strtoul (index_str);

	if (index >= shares_len)
		return;

	if (!shares[index])
		return;

	/* skip the index attribute */
	dataset_removestr (children, "index");

	dataset_foreach (children, DS_FOREACH(set_meta_foreach), shares[index]);
}

static void set_metadata_from_indexed_xml (Share **shares, size_t shares_len,
                                           xmlDoc *doc)
{
	xmlNode *node;

	node = xmlDocGetRootElement (doc);

	if (!node)
		return;

	node = node->xmlChildrenNode;

	while (node != NULL)
	{
		Dataset *children;

		children = collect_attributes (node);

		set_share_meta (shares, shares_len, children);
		dataset_clear (children);

		node = node->next;
	}
}

static const char *inflate_xml (const char *xml, size_t bin_len)
{
	z_streamp z;
	size_t    xml_len;

	if (!(z = NEW (z_stream)))
		return NULL;

	if (inflateInit (z) != Z_OK)
	{
		free (z);
		return NULL;
	}

	/* set the input parameters */
	z->next_in   = (char *)xml;
	z->avail_in  = bin_len;

	/* set the output parameters */
	z->next_out  = xml_buf;
	z->avail_out = sizeof (xml_buf) - 1;

	if (inflate (z, Z_FINISH) != Z_STREAM_END)
	{
		GT->DBGFN (GT, "error inflating xml");
		return NULL;
	}

	/* null terminate (the hopefully plaintext) XML */
	xml_len = (sizeof (xml_buf) - 1) - z->avail_out;
	xml_buf[xml_len] = 0;

	if (XML_DEBUG)
		GT->dbg (GT, "inflated xml: %s", xml_buf);

	inflateEnd (z);
	free (z);

	return xml_buf;
}

BOOL gt_xml_parse_indexed (const char *xml, size_t bin_len, Share **shares,
                           size_t shares_len)
{
	xmlDoc    *doc;
	size_t     xml_len;

	if (!xml || (xml_len = strlen (xml)) <= 4)
		return FALSE;

	/*
	 * Look for the encoding type, currently possible
	 * encoding values are: "{}" meaning plain text, "{plaintext}",
	 * and "{deflate}".
	 */

	if (!strncmp (xml, "{}", 2))
	{
		xml += 2;
	}
	else if (!strncasecmp (xml, "{plaintext}", sizeof ("{plaintext}") - 1))
	{
		xml += sizeof ("{plaintext}") - 1;
	}
	else if (!strncasecmp (xml, "{deflate}", sizeof ("{deflate}") - 1))
	{
		xml = inflate_xml (xml + sizeof ("{deflate}") - 1, bin_len);

		if (XML_DEBUG)
			assert (xml != NULL);    /* assume valid input */

		/* reset the length since the compressed xml was binary
		 * (but guaranteed null-terminated) */
		xml_len = STRLEN (xml);
	}

	if (!xml)
		return FALSE;

	if (xml[0] != '<')
		return FALSE;

	if (!(doc = xmlParseMemory (xml, xml_len)))
		return FALSE;

	set_metadata_from_indexed_xml (shares, shares_len, doc);
	xmlFreeDoc (doc);

	return TRUE;
}

/* gets called when there are parsing errors */
void error_handler_func (void *udata, const char *msg, ...)
{
	char     buf[4096];
	va_list  args;

	/* this is here until i figure out why i get a message about
	 * namespace errors (but it still seems to work... */
	if (!XML_DEBUG)
		return;

	va_start (args, msg);
	vsnprintf (buf, sizeof (buf) - 1, msg, args);
	va_end (args);

	GT->DBGFN (GT, "xml parse error: %s", buf);
}

/*****************************************************************************/

#endif /* USE_LIBXML2 */

/*****************************************************************************/

void gt_xml_init (void)
{
#ifdef USE_LIBXML2
	/* so libxml doesn't print messages on stderr */
	xmlSetGenericErrorFunc (NULL, error_handler_func);
#endif /* USE_LIBXML2 */
}

void gt_xml_cleanup (void)
{
}
