/* giFTui
 * Copyright (C) 2003 the giFTui team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "main.h"

#include <string.h>

#include "event.h"
#include "util.h"

#define GIFTUI_REGISTRED(arg) ((GiftuiRegistred_t *) (arg))

typedef struct _GiftuiRegistred_t
{
	gboolean sent;
	GiftuiEventType_t type;
	guint id;
	gpointer instance;
	GiftuiRegistredCB_t callback;
} GiftuiRegistred_t;

/* Each command sent by giFT is read in io.c and for each we create an
 * event. Those functions deal events to registred widgets (according
 * to ids and event type's...). All events are destroyed after has
 * been sent to widgets. */

static GSList *instances = NULL;

/**/

void
giftui_event_free (GiftuiEvent_t *ev)
{
	g_return_if_fail (ev != NULL);
	
	if (ev->iface != NULL)
		interface_free (ev->iface);
	g_free (ev);
	
	return;
}

/*
  Create event from interface.
*/
GiftuiEvent_t *
giftui_event_new_from_interface (Interface *in)
{
	GiftuiEvent_t *ev = g_new0 (GiftuiEvent_t, 1);
       	
	g_return_val_if_fail (in != NULL, NULL);
	g_return_val_if_fail (ev != NULL, NULL);
	
	if (!strcasecmp (in->command, "ITEM"))
		ev->type = EVENT_ITEM;
	else if (!strcasecmp (in->command, "ADDDOWNLOAD"))
		ev->type = EVENT_ADDDOWNLOAD;
	else if (!strcasecmp (in->command, "CHGDOWNLOAD"))
		ev->type = EVENT_CHGDOWNLOAD;
	else if (!strcasecmp (in->command, "DELDOWNLOAD"))
		ev->type = EVENT_DELDOWNLOAD;
	else if (!strcasecmp (in->command, "ADDUPLOAD"))
		ev->type = EVENT_ADDUPLOAD;
	else if (!strcasecmp (in->command, "CHGUPLOAD"))
		ev->type = EVENT_CHGUPLOAD;
	else if (!strcasecmp (in->command, "DELUPLOAD"))
		ev->type = EVENT_DELUPLOAD;
	else if (!strcasecmp (in->command, "ADDSOURCE"))
		ev->type = EVENT_ADDSOURCE;
	else if (!strcasecmp (in->command, "DELSOURCE"))
		ev->type = EVENT_DELSOURCE;
	else if (!strcasecmp (in->command, "SHARE"))
		ev->type = EVENT_SHARE;
	else if (!strcasecmp (in->command, "STATS"))
		ev->type = EVENT_STATS;
	else if (!strcasecmp (in->command, "OPEN"))
		ev->type = EVENT_OPEN;
	else if (!strcasecmp (in->command, "ATTACH"))
		ev->type = EVENT_ATTACH;
	else if (!strcasecmp (in->command, "CLOSE"))
		ev->type = EVENT_CLOSE;
	else if (!strcasecmp (in->command, "DISCONNECT"))
		ev->type = EVENT_DISCONNECT;
	else if (!strcasecmp (in->command, "MESSAGE"))
		ev->type = EVENT_MESSAGE;
	else
		ev->type = EVENT_OTHER;
	
	/* No id for those kind of event. */
	if (in->value != NULL && ev->type != EVENT_MESSAGE &&
	    ev->type != EVENT_OTHER && ev->type != EVENT_DISCONNECT &&
	    ev->type != EVENT_OPEN && ev->type != EVENT_CLOSE)
		ev->id = atoi (in->value);
	
	ev->iface = in;
	
	return ev;
}

static void
giftui_registred_free (GiftuiRegistred_t *re)
{
	g_free (re);
	
	return;
}

static GSList *
giftui_widget_find_all (gconstpointer instance)
{
	GSList *l = instances;
	GSList *r = NULL;
	
	while (l != NULL)
	{
		if (((GiftuiRegistred_t *) (l->data))->instance == instance)
			r = g_slist_prepend (r, l->data);
		l = g_slist_next (l);
	}
	
	return r;
}

static GSList *
giftui_widget_find (gconstpointer instance, GiftuiEventType_t type, guint id)
{
	GSList *l = instances;
	GSList *r = NULL;
	GiftuiRegistred_t *w;
	
	while (l != NULL)
	{
		w = (GiftuiRegistred_t *) (l->data);
		if (w->instance == instance && w->type == type && w->id == id)
			r = g_slist_prepend (r, l->data);
		l = g_slist_next (l);
	}
	
	return r;
}

/* When you want a widget receives events, you must tell which type,
 * which id (0 means no id), which widget send event and with which
 * function. Status and transfert widgets do not use ids, they receive
 * all event from the type they requested. Browse and search widgets
 * use ids. */
void
giftui_event_register (GiftuiEventType_t type, guint id, gpointer instance,
		       GiftuiRegistredCB_t callback)
{
	GiftuiRegistred_t *r;
	
#ifdef HAVE_DEBUG
	GIFTUI_PRINT_INFO (("register 0x%x with id=%u type=%i\n",
			    (unsigned int) instance, id, type));
#endif
	
	g_return_if_fail (callback != NULL);
	
	r = g_new0 (GiftuiRegistred_t, 1);
	g_return_if_fail (r != NULL);
	
	r->type = type;
	r->id = id;
	r->instance = instance;
	r->callback = callback;
	
	instances = g_slist_prepend (instances, (gpointer) r);
	
	return;
}

void
giftui_event_unregister (gconstpointer instance, GiftuiEventType_t type, guint id)
{
	GSList *r;
	
#ifdef HAVE_DEBUG
	GIFTUI_PRINT_INFO (("unregister 0x%x with id=%u type=%i\n",
			    (unsigned int) instance, id, type));
#endif
	
	r = giftui_widget_find (instance, type, id);
	while (r != NULL)
	{
		instances = g_slist_remove (instances, r->data);
		giftui_registred_free (GIFTUI_REGISTRED(r->data));
		r->data = NULL;
		r = g_slist_next (r);
	}
	
	return;
}

void
giftui_event_unregister_all (gconstpointer instance)
{
	GSList *r, *t;
	
#ifdef HAVE_DEBUG
	GIFTUI_PRINT_INFO (("unregister all for 0x%x\n", (unsigned int) instance));
#endif
	
	r = giftui_widget_find_all (instance);
	t = r;
	while (t != NULL)
	{
		instances = g_slist_remove (instances, t->data);
		giftui_registred_free (GIFTUI_REGISTRED(t->data));
		r->data = NULL;
		t = g_slist_next (t);
	}
	g_slist_free (r);
	
	return;
}

/* When you call this function, the event you give as argument is
 * always destroyed, so don't use it after. */
void
giftui_event_send_to_widgets (GiftuiEvent_t *ev)
{
	GSList *l = instances, *t;
	GiftuiRegistred_t *w;

#if 0	
	GIFTUI_PRINT_INFO (("send type=%i id=%u\n", ev->type, ev->id));
#endif	
	
	while (l != NULL)
	{
		l = instances;
		while (l != NULL && GIFTUI_REGISTRED (l->data)->sent)
			l = g_slist_next (l);
		if (l != NULL)
		{
			w = GIFTUI_REGISTRED (l->data);
			w->sent = TRUE;
			t = g_slist_next (l);
			if (w->type == ev->type && (!w->id || w->id == ev->id))
				w->callback (w->instance, ev);
			l = t;
		}
	}
	
	l = instances;
	
	while (l != NULL)
	{
		w = GIFTUI_REGISTRED (l->data);
		w->sent = FALSE;
		l = g_slist_next (l);
	}

#if 0	
	GIFTUI_PRINT_INFO (("sent type=%i id=%u\n", ev->type, ev->id));
#endif	
	
	return;
}
