/* giFTui
 * Copyright (C) 2003 the giFTui team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "main.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "event.h"
#include "io.h"
#include "configure.h"
#include "util.h"

#include "ui.h"
#include "ui_util.h"

/**/

enum
{
	OPTION_COLUMN = 0,
	NUMBER_COLUMN,
	N_COLUMNS
};

static gpointer parent_class = NULL;

static void giftui_pref_init (GiftuiPref *pref);
static void giftui_pref_class_init (GiftuiPrefClass *class);
static void giftui_pref_dispose (GObject *object);

/**/

GType
giftui_pref_get_type (void)
{
	static GType pref_type = 0;
	
	if (!pref_type)
	{
		static const GTypeInfo pref_type_info =
			{
				sizeof (GiftuiPrefClass),
				NULL,		/* base_init */
				NULL,		/* base_finalize */
				(GClassInitFunc) giftui_pref_class_init,
				NULL,		/* class_finalize */
				NULL,		/* class_data */
				sizeof (GiftuiPref),
				0,		/* n_preallocs */
				(GInstanceInitFunc) giftui_pref_init,
				NULL
			};
		
		pref_type = g_type_register_static (GIFTUI_TYPE_CHILD, "GiftuiPref",
						    &pref_type_info, 0);
	}
	
	return pref_type;
}

static void
giftui_pref_class_init (GiftuiPrefClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);
	
	object_class->dispose = giftui_pref_dispose;
	
	return;
}

/**/

static gchar *color_props[] = { PREFS_TRANSFERS_ACTIVE_COLOR,
				PREFS_TRANSFERS_COMPLETED_COLOR,
				PREFS_TRANSFERS_PAUSED_COLOR,
				PREFS_TRANSFERS_CANCELED_COLOR };

static void
giftui_pref_load (GiftuiPref *pref)
{
	gint i;
	
	/* giFT */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->cbLaunch),
				      giftui_config_get_bool (PREFS_DAEMON_LAUNCH));
	gtk_option_menu_set_history (GTK_OPTION_MENU (pref->omKill),
				      giftui_config_get_int (PREFS_DAEMON_KILL));
	gtk_entry_set_text (GTK_ENTRY (pref->txtHost),
			    giftui_config_get_str (PREFS_DAEMON_HOST));
	widget_spin_set_int (GTK_SPIN_BUTTON (pref->txtPort),
			     giftui_config_get_int (PREFS_DAEMON_PORT));
	
	/* giFTui */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->chcomplete),
				      giftui_config_get_bool (PREFS_TRANSFERS_CLEAR_COMPLETED));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pref->chcancel),
				      giftui_config_get_bool (PREFS_TRANSFERS_CLEAR_CANCELED));
	
	for (i = 0 ; i < 4 ; i++)
		str_to_color (&pref->colors[i], giftui_config_get_str (color_props[i]));
	
	return;
}

static void
giftui_pref_apply (GiftuiPref *pref)
{
	gint i;
	gchar *str;
		
	/* giFT */
	giftui_config_set_bool (PREFS_DAEMON_LAUNCH,
				gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->cbLaunch)));
	giftui_config_set_int (PREFS_DAEMON_KILL,
			       gtk_option_menu_get_history (GTK_OPTION_MENU (pref->omKill)));
	giftui_config_set_str (PREFS_DAEMON_HOST,
			       gtk_entry_get_text (GTK_ENTRY (pref->txtHost)));
	giftui_config_set_int (PREFS_DAEMON_PORT,
			       gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (pref->txtPort)));
	
	/* giFTui */
	giftui_config_set_bool (PREFS_TRANSFERS_CLEAR_COMPLETED,
			       gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->chcomplete)));
	giftui_config_set_bool (PREFS_TRANSFERS_CLEAR_CANCELED,
			       gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->chcancel)));
	
	for (i = 0 ; i < 4 ; i++)
	{
		str = color_to_str (&pref->colors[i]);
		giftui_config_set_str (color_props[i], str);
		g_free (str);
	}
	
	return;
}

/**/

static void
giftui_pref_shares_set (GiftuiPref *pref, gboolean state)
{
	gtk_widget_set_sensitive (pref->shsShow, !state);
	gtk_widget_set_sensitive (pref->shsHide, state);
		
	return;
}

static void
giftui_pref_shares_updated (GiftuiPref *pref, GiftuiEvent_t *ev)
{
	gchar *s, *s2, tmp[256];
	
	g_return_if_fail (pref != NULL);
	g_return_if_fail (ev != NULL);
	
	s2 = interface_get (ev->iface, "action");
	if (s2)
	{
		if (!strcasecmp (s2, "sync"))
		{
			s = interface_get (ev->iface, "status");
			if (s && strcasecmp (s, "Done"))
			{
				sprintf (tmp, _("%s file(s) synced"), s);
				gtk_label_set_text (GTK_LABEL (pref->lblSync), tmp);
			}
			else
				pref->resyncing = FALSE;
		}
		else if (!strcasecmp (s2, "hide"))
			giftui_pref_shares_set (pref, FALSE);
		else if (!strcasecmp (s2, "show"))
			giftui_pref_shares_set (pref, TRUE);
	}
	
	return;
}

static void
giftui_pref_attach (GiftuiPref *pref, GiftuiEvent_t *ev)
{
	gift_send_str ("SHARE action;");
	
	return;
}

/**/

static void
giftui_pref_launch (GiftuiPref *pref)
{
	gtk_widget_set_sensitive (pref->txtHost,
				  !gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pref->cbLaunch)));
	
	return;
}

static void
giftui_pref_selection_changed (GtkTreeSelection *selection, GtkNotebook *notebook)
{
	gint index;
	GtkTreeIter iter;
	GtkTreeView *treeview;
	GtkTreeModel *model;
	
	if (!gtk_tree_selection_get_selected (selection, NULL, &iter))
		return;
	
	treeview = gtk_tree_selection_get_tree_view (selection);
	model = gtk_tree_view_get_model (treeview);
	
	gtk_tree_model_get (model, &iter, NUMBER_COLUMN, &index, -1);
	gtk_notebook_set_page (notebook, index - 1);
	
	return;
}

static void
giftui_pref_shares_resync (GtkWidget *button, GiftuiPref *pref)
{
	if (!gift_connected () || pref->resyncing)
		return;
	
	pref->resyncing = TRUE;
	
	gift_send_str ("SHARE action(sync);");
	
	return;
}

static void
giftui_pref_shares_set_clicked (GiftuiPref *pref, GtkWidget *button)
{
	if (gift_connected ())
	{
		if (pref->shsShow == button)
			gift_send_str ("SHARE action(show);");
		else
			gift_send_str ("SHARE action(hide);");
	}
	
	return;
}

/**/

static GtkWidget *
giftui_pref_add_menu_prop (GtkWidget *vbox, gchar *text, GtkTooltips *tips, gchar *help, GtkWidget *menu)
{
	GtkWidget *label, *omProp;
	GtkWidget *hbox;
	
	hbox = gtk_hbox_new (FALSE, 0);
	label = gtk_label_new (text);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 5);
	
	omProp = gtk_option_menu_new ();
	gtk_option_menu_set_menu (GTK_OPTION_MENU (omProp), menu);
	gtk_box_pack_start (GTK_BOX (hbox), omProp, FALSE, FALSE, 5);
	
	gtk_tooltips_set_tip (tips, omProp, help, NULL);
	
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 5);
	
	return omProp;
}

static GtkWidget *
giftui_pref_notebook_append_page (GtkTreeView *treeview, GtkTreeIter *parent, GtkTreeIter *child,
				  gchar *name, gint index, GtkNotebook *nbConf)
{
	GtkTreeStore *store = GTK_TREE_STORE (gtk_tree_view_get_model (treeview));
	GtkWidget *bvbox, *vbox;
	GtkWidget *frame;
	GtkWidget *label;
	PangoFontDescription *font;
		
	gtk_tree_store_append (store, child, parent);
	gtk_tree_store_set (store, child, OPTION_COLUMN, name, NUMBER_COLUMN, index, -1);
	
	bvbox = gtk_vbox_new (FALSE, 0);
	gtk_notebook_append_page (nbConf, bvbox, NULL);
	
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
	gtk_box_pack_start (GTK_BOX (bvbox), frame, FALSE, TRUE, 0);
	
	label = gtk_label_new (name);
	font = pango_font_description_new ();
	pango_font_description_set_weight (font, PANGO_WEIGHT_HEAVY);
	pango_font_description_set_size (font, 12 * PANGO_SCALE);
	gtk_widget_modify_font (label, font);
	pango_font_description_free (font);
	
	gtk_misc_set_alignment (GTK_MISC (label), 0.95, 0.5);
	gtk_misc_set_padding (GTK_MISC (label), 2, 1);
	gtk_container_add (GTK_CONTAINER (frame), label);
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start_defaults (GTK_BOX (bvbox), vbox);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 3);
	
	return vbox;
}

static void
giftui_pref_init (GiftuiPref *pref)
{
	gint i, index = 1;
	gchar *color_text[] = { _("Active color :"),
				_("Complete color :"),
				_("Pause color :"),
				_("Cancel color :") };
	GtkWidget *nbConf;
	GtkWidget *hbox, *nbox, *vbox;
	GtkWidget *paned;
	GtkWidget *button;
	GtkWidget *treeview;
	GtkWidget *menu;
	GtkTooltips *tips = gtk_tooltips_new ();
	GtkTreeStore *store;
	GtkTreeViewColumn *column;
	GtkTreeIter iter1, iter2;
	GiftuiChild *child;
	
	child = GIFTUI_CHILD (pref);
	vbox = GTK_WIDGET (pref);
	pref->resyncing = FALSE;
	child->type = GIFTUI_CHILD_PREF;
	g_object_set (G_OBJECT (pref), "label", _("Configuration"),
		      "stock", GTK_STOCK_PREFERENCES, NULL);
	
	paned = gtk_hpaned_new ();
	store = gtk_tree_store_new (N_COLUMNS, G_TYPE_STRING, G_TYPE_INT);
	treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
	g_object_unref (store);
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (treeview), FALSE);
	gtk_paned_pack1 (GTK_PANED (paned), treeview, FALSE, FALSE);
	
	column = gtk_tree_view_column_new_with_attributes (NULL, gtk_cell_renderer_text_new (),
							   "text", 0, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
	gtk_tree_view_column_set_resizable (column, FALSE);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
	
	nbConf = gtk_notebook_new ();
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nbConf), FALSE);
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nbConf), FALSE);
	
	g_signal_connect (gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview)), "changed",
			  G_CALLBACK (giftui_pref_selection_changed), G_OBJECT (nbConf));
	
	/* giFT */
	nbox = giftui_pref_notebook_append_page (GTK_TREE_VIEW (treeview), NULL, &iter1,
						 _("giFT"), index++, GTK_NOTEBOOK (nbConf));
	/*   host   */
	nbox = giftui_pref_notebook_append_page (GTK_TREE_VIEW (treeview), &iter1, &iter2,
						 _("host"), index++, GTK_NOTEBOOK (nbConf));
	
	
	pref->cbLaunch = widget_check_add (GTK_BOX (nbox), _("Launch giFT on localhost at giftui startup (if it isn't already)"));
	g_signal_connect_swapped (pref->cbLaunch, "toggled", GTK_SIGNAL_FUNC (giftui_pref_launch), pref);
	
	menu = gtk_menu_new ();
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), gtk_menu_item_new_with_label (_("No")));
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), gtk_menu_item_new_with_label (_("Yes")));
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), gtk_menu_item_new_with_label (_("Ask")));
	pref->omKill = giftui_pref_add_menu_prop (nbox, _("Kill giFT daemon on close ?"),
						  tips, NULL, menu);
	
	widget_vhseparator_add (GTK_BOX (nbox), HORIZONTAL);
	
	pref->txtHost = widget_entry_add_chars (GTK_BOX (nbox), _("Host :"), NULL, 256);
	widget_tooltips_set_tip (tips, pref->txtHost,
				 _("Host of the giFT daemon that is listening for interface connections"));
	
	pref->txtPort = widget_spin_add_int (GTK_BOX (nbox), _("Port :"), 0, 65535, 1, 1);
	widget_tooltips_set_tip (tips, pref->txtPort,
				 _("Interface port which the [possibly] remote giFT daemon is listening on"));
	
	/*   shares   */
	nbox = giftui_pref_notebook_append_page (GTK_TREE_VIEW (treeview), &iter1, &iter2,
						 _("shares"), index++, GTK_NOTEBOOK (nbConf));
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (nbox), hbox, FALSE, FALSE, 0);
	
	button = gtk_button_new_with_label (_("Sync shares"));
	gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	g_signal_connect (button, "clicked", G_CALLBACK (giftui_pref_shares_resync), pref);
	
	pref->lblSync = gtk_label_new (NULL);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), pref->lblSync);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (nbox), hbox, FALSE, FALSE, 0);
	
	pref->shsShow = gtk_button_new_with_label (_("Show shares"));
	gtk_box_pack_start (GTK_BOX (hbox), pref->shsShow, FALSE, FALSE, 0);
	g_signal_connect_swapped (pref->shsShow, "clicked",
				  G_CALLBACK (giftui_pref_shares_set_clicked), pref);
	
	pref->shsHide = gtk_button_new_with_label (_("Hide shares"));
	gtk_box_pack_start (GTK_BOX (hbox), pref->shsHide, FALSE, FALSE, 0);
	g_signal_connect_swapped (pref->shsHide, "clicked",
				  G_CALLBACK (giftui_pref_shares_set_clicked), pref);
	
	/*   plugins (planned for 0.3.0 or later) */
	
	/* giFTui */
	nbox = giftui_pref_notebook_append_page (GTK_TREE_VIEW (treeview), NULL, &iter1,
						 PACKAGE, index++, GTK_NOTEBOOK (nbConf));
	/*   transfers   */
	nbox = giftui_pref_notebook_append_page (GTK_TREE_VIEW (treeview), &iter1, &iter2,
						 _("transfers"), index++, GTK_NOTEBOOK (nbConf));
	
	for (i = 0 ; i < 4 ; i++)
	{
		pref->wcolors[i] = widget_button_color_sel_add_from_str (GTK_BOX (nbox), color_text[i]);
		g_object_set_data (G_OBJECT (pref->wcolors[i]), "color", &pref->colors[i]);
	}
	
	widget_vhseparator_add (GTK_BOX (nbox), HORIZONTAL);
	
	pref->chcomplete = widget_check_add (GTK_BOX (nbox), _("Clear completed downloads"));
	pref->chcancel = widget_check_add (GTK_BOX (nbox), _("Clear canceled downloads"));
	
	gtk_paned_pack2 (GTK_PANED (paned), nbConf, FALSE, FALSE);
	gtk_box_pack_start (GTK_BOX (vbox), paned, TRUE, TRUE, 5);
	
	hbox = gtk_hbox_new (FALSE, 5);
	gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, FALSE, 5);
	
	/*button = gtk_button_new_from_stock (GTK_STOCK_SAVE);
	  g_signal_connect_swapped (button, "clicked",
	  GTK_SIGNAL_FUNC (giftui_pref_write), pref);
	  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 5);*/
	
	button = gtk_button_new_from_stock (GTK_STOCK_APPLY);
	g_signal_connect_swapped (button, "clicked",
				  GTK_SIGNAL_FUNC (giftui_pref_apply), pref);
	gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 5);
	
	return;
}

/**/

GtkWidget *
giftui_pref_new (void)
{
	GiftuiPref *pref;
	
	pref = g_object_new (GIFTUI_TYPE_PREF, "close", FALSE, NULL);
	
	giftui_pref_load (pref);
	
	giftui_event_register (EVENT_ATTACH, 0, pref,
			       GIFTUI_REGISTRED_CB (giftui_pref_attach));
	giftui_event_register (EVENT_SHARE, 0, pref,
			       GIFTUI_REGISTRED_CB (giftui_pref_shares_updated));
	
	gtk_widget_show_all (GTK_WIDGET (pref));
	
	return GTK_WIDGET (pref);
}

static void
giftui_pref_dispose (GObject *object)
{
	GiftuiPref *pref = GIFTUI_PREF (object);
	
	giftui_event_unregister_all (pref);
	
	G_OBJECT_CLASS (parent_class)->dispose (object);
	
	return;
}
