import datetime
import glob
import os
import re
import thread
from xml.parsers.expat import ExpatError
from gettext import gettext as _
from xml.dom.minidom import parse

import gtk
import W3CDate

from gimmie_base import Item, ItemSource
from gimmie_util import FileMonitor, launcher

# FIXME:  This should really just use Beagle.


class NoteItem(Item):
    def __init__(self, uri):
        Item.__init__(self, uri=uri, icon="stock_notes", mimetype="x-tomboy/note")
        self.do_reload()

    def do_open(self):
        note_uri = "note://tomboy/%s" % os.path.split(self.get_uri())[1]
        launcher.launch_uri(note_uri, self.get_mimetype())

    def do_reload(self):
        note_doc = parse(self.get_uri())

        title_node = note_doc.getElementsByTagName("title")[0]
        self.title = title_node.childNodes[0].data

        try:
            # Parse the ISO timestamp format .NET's XmlConvert class uses:
            # yyyy-MM-ddTHH:mm:ss.fffffffzzzzzz, where f* is a 7-digit partial
            # second, and z* is the timezone offset from UTC in the form -08:00.
            changed_node = note_doc.getElementsByTagName("last-change-date")[0]
            changed_str = changed_node.childNodes[0].data
            changed_str = re.sub("\.[0-9]*", "", changed_str) # W3Date chokes on partial seconds
            self.timestamp = W3CDate.W3CDate(changed_str).getSeconds()
        except(ValueError, IndexError, AttributeError):
            self.timestamp = 0

        self.content_text = None
        content_node = note_doc.getElementsByTagName("note-content")[0]
        if content_node:
            self.content_text = self._get_text_from_node(content_node).lower()

        note_doc.unlink()

    def _get_text_from_node(self, node):
        if node.nodeType == node.TEXT_NODE:
            return node.data
        else:
            return "".join([self._get_text_from_node(x) for x in node.childNodes])

    def get_timestamp(self):
        return self.timestamp

    def get_comment(self):
        return self.pretty_print_time_since(self.get_timestamp())

    def get_name(self):
        return self.title or os.path.split(self.get_uri())[1] or self.get_uri()

    def matches_text(self, text):
        return Item.matches_text(self, text) or \
               (self.content_text and self.content_text.find(text) > -1)


class TomboySource(ItemSource):
    def __init__(self, note_path=None):
        ItemSource.__init__(self,
                            name=_("Notes"),
                            icon="stock_notes",
                            uri="source:///Documents/Tomboy")

        self.new_note_item = Item(name=_("Create New Note"),
                                  comment=_("Make a new Tomboy note"),
                                  icon=gtk.STOCK_NEW,
                                  special=True)
        self.new_note_item.do_open = lambda: self._make_new_note()

        if not note_path:
            if os.environ.has_key("TOMBOY_PATH"):
                note_path = os.environ["TOMBOY_PATH"]
            else:
                note_path = "~/.tomboy"
            note_path = os.path.expanduser(note_path)
        self.note_path = note_path

        self.note_path_monitor = FileMonitor(self.note_path)
        self.note_path_monitor.connect("created", lambda x, s: self.emit("reload"))
        self.note_path_monitor.connect("deleted", lambda x, s: self.emit("reload"))
        self.note_path_monitor.connect("changed", lambda x, s: self.emit("reload"))
        self.note_path_monitor.open()

        thread.start_new_thread(self.do_reload, ())

    def _make_new_note(self):
        launcher.launch_command("tomboy --new-note")

    def do_reload(self):
        items = [self.new_note_item] # Allow creating a new note

        for notepath in glob.glob("%s/*.note" % self.note_path):
            try:
                items.append(NoteItem(notepath))
            except (IOError, ExpatError), err:
                if isinstance(err, IOError):
                    err = err[1]
                print " !!! Error parsing %s: %s" % (self.get_uri(), err)
                continue

        self.set_items(items)

