/* GIMP Dimage color plugin
 * Copyright (C) 2002 Laurent HOUDARD <lhoudard@netcourrier.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <string.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <libgimp/gimpmatrix.h>

#include "dimage_color.h"
#include "profile.h"
#include "transform.h"

/* Input gamma lookup table */
static double input_gamma[256];

/* Output gamma lookup tables */
static guchar srgb_output_gamma[1024];
static guchar inverted_dimage_gamma[1024];

/* Color gamut transformation matrix */
static GimpMatrix3 shaper;

void
build_transform (void)
{
  GimpMatrix3 srgb_i;
  gint i;

  /* Build color gamut transformation: t = (sRGB)-1 * dimage */
  gimp_matrix3_invert(srgb_prim, srgb_i);
  memcpy (shaper, dimage_prim, sizeof (GimpMatrix3));
  gimp_matrix3_mult(srgb_i, shaper);

  /* Build input gamma lookup table */
  for (i = 0; i < 256; i++) 
    {
      input_gamma[i] = dimage_gamma[(i + 1) * 2 - 1];
    }

  /* Build output gamma lookup tables */

  /* sRGB */
  build_srgb_output_gamma(srgb_output_gamma, 1024);

  /* inverted Dimage gamma, used when gamma correction is disabled */
  build_invert_gamma(dimage_gamma, 512, 
		     inverted_dimage_gamma, 1024);
}

void
transform_point (guchar *src, guchar *dst, gboolean visualize_out)
{
  gdouble p1[3], p2[3];
  gint i;
  guchar *output_gamma;
  double *itrc;

  /* Input gamma */
  p1[0] = input_gamma[src[0]];
  p1[1] = input_gamma[src[1]];
  p1[2] = input_gamma[src[2]];

  /* Color gamut transform */
  p2[0] = p1[0] * shaper[0][0] + p1[1] * shaper[0][1] + p1[2] *shaper[0][2];
  p2[1] = p1[0] * shaper[1][0] + p1[1] * shaper[1][1] + p1[2] *shaper[1][2];
  p2[2] = p1[0] * shaper[2][0] + p1[1] * shaper[2][1] + p1[2] *shaper[2][2];
  
  /* Clipping / Out of gamut */
  for (i = 0; i < 3; i++) 
    {
      if (visualize_out == TRUE)
	{
	  if (p2[i] < 0.0 || p2[i] > 1.0) 
	    { 
	      dst[0] = 255; 
	      dst[1] = 0; 
	      dst[2] = 255; 
	      return; 
	    }
	}
      else
	{
	  if (p2[i] < 0.0) { p2[i] = 0.0; }; 
	  if (p2[i] > 1.0) { p2[i] = 1.0; };
	}
    }

  /* Non standard tone correction curve (iTRC tag) */
  if (dc_vals.gamma == TRUE)
    {
      switch (dc_vals.tone_curve) 
	{
	case CURVE_DEFAULT:
	  itrc = dimage_itrc;
	  break;
	  
	  /* Add custom curves here */
	  
	default:
	  itrc = NULL;
	  break;
	}
      
      if (itrc) 
	{
	  p2[0] = itrc[(gint) (p2[0] * (512 - 1))];
	  p2[1] = itrc[(gint) (p2[1] * (512 - 1))];
	  p2[2] = itrc[(gint) (p2[2] * (512 - 1))];
	}
    }
  

  /* Output gamma */
  if (dc_vals.gamma == TRUE) 
    output_gamma = srgb_output_gamma;
  else 
    output_gamma = inverted_dimage_gamma;

  dst[0] = output_gamma[(gint) (p2[0] * (1024 - 1))];
  dst[1] = output_gamma[(gint) (p2[1] * (1024 - 1))];
  dst[2] = output_gamma[(gint) (p2[2] * (1024 - 1))];
}

void
build_srgb_output_gamma (guchar gamma_table[], gint size)
{
  gint i;
  gdouble x, y;
  
  for (i = 0; i < size; i++) 
    {
      x = (gdouble) i / (gdouble) (size - 1);
      
      if (x <= 0.00304)
	y = x * 12.92;
      else
	y = 1.055 * (pow (x, 1.0 / 2.4)) - 0.055;

      gamma_table[i] = (guchar) (y * 255.0 + 0.5);
    }
}

void
build_invert_gamma (double gamma_in[], gint size_in,
		    guchar gamma_out[], gint size_out)
{
  gint i, j;
  gdouble x, y;

  gamma_out[0] = 0;

  j = 1;
  for (i = 1; i < size_out; i++)
    {
      y = (gdouble) i / (gdouble) (size_out - 1);

      while (gamma_in[j] < y)
	j++;
      
      x = 
	((gdouble) (j-1) / (gdouble) (size_in - 1)) + 
	((y - gamma_in[j-1]) * 
	 (1 / (gdouble) (size_in - 1)) / (gamma_in[j] - gamma_in[j-1]));
      
      gamma_out[i] = (guchar) (x * 255 + 0.5);
    }
}

