/*
 *  
 *  $Id: icontrato.h 3964 2011-07-11 22:17:46Z carlos $
 *  Ginkgo CADx Project
 *
 *  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
 *  http://ginkgo-cadx.com
 *
 *  This file is licensed under LGPL v3 license.
 *  See License.txt for details
 *
 */
#pragma once
#include "ivista.h"
#include <map>
#include <list>
#include <cstddef>


namespace GNC {
	namespace GCS {

//region "Helpers"
		// <editor-fold defaultstate="collapsed" desc="lista prioritaria">
		template<class T>
		class priority_list {
			std::list<T> lst;
		public:
			class iterator;

			friend class iterator;

			class iterator : public std::iterator< std::bidirectional_iterator_tag,T,ptrdiff_t> {
				typename std::list<T>* r;
				typename std::list<T>::iterator it;
			public:
				iterator(priority_list<T>& otro) { *this = otro; }
				iterator(std::list<T>& lst, const typename std::list<T>::iterator& i) : r(&lst), it(i) {}

				bool operator==(const iterator& x) const {
					return it == x.it;
				}
				bool operator!=(const iterator& x) const {
					return !(*this == x);
				}
				typename std::list<T>::reference operator*() const {
					return *it;
				}
				iterator& operator=(const iterator& otro) {
					r = otro.r;
					it = otro.it;
				}
				iterator& operator++() {
					++it;
					return *this;
				}
				iterator operator++(int) {
					iterator tmp = *this;
					++*this;
					return tmp;
				}
				iterator& operator--() {
					--it;
					return *this;
				}
				iterator operator--(int) {
					iterator tmp = *this;
					--*this;
					return tmp;
				}
				iterator insert(const T& x){
					return iterator(*r, r->insert(it));
				}
				iterator erase() {
					return iterator(*r, r->erase(it));
				}
			};
			void insertar(const T& x) {
				bool stop = false;
				for (typename std::list<T>::iterator it1 = lst.begin(); !stop && it1 != lst.end(); it1++) {
					if ((*it1) > x) {
						lst.insert(it1, x);
						stop = true;
					}
				}
				if (!stop) {
					lst.push_back(x);
				}
			}
			void eliminar(const T& x) {
				bool stop = false;
				for (typename std::list<T>::iterator it1 = lst.begin(); !stop && it1 != lst.end(); it1++) {
					if ((*it1) == x) {
						lst.erase(it1);
						stop = true;
					}
				}
			}
			iterator begin() {
				return iterator(lst, lst.begin());
			}
			iterator end() {
				return iterator(lst, lst.end());
			}
			int size() { return lst.size(); }
		};
		// </editor-fold>
//endregion

		class EXTAPI ContratoException : public std::exception {
		public:

			ContratoException(const char* msg)
			{
				m_Mensaje.assign(msg);
			}

			ContratoException(std::string& mensaje)
			{
				m_Mensaje = mensaje;
			}

			ContratoException(std::string mensaje)
			{
				m_Mensaje = mensaje;
			}

			~ContratoException() throw ()
			{
			}

			std::string& getCause()
			{
				return m_Mensaje;
			}

		protected:
			std::string m_Mensaje;
		};

		//------------------------------------------------------------------------
		/* Interfaz generica de contratos.
		 * Cuidado con la herencia multiple aqui
		 */
		class EXTAPI IContrato {
		public:

			typedef enum {
				TC_Obligatorio,
				TC_Opcional
			} TipoContrato;

			IContrato(TipoContrato tipo = TC_Obligatorio, int prioridad = 0)
			{
				m_Tipo = tipo;
				m_Prioridad = prioridad;
			};

			~IContrato()
			{
				;
			};

			void Ping()
			{
			};

			bool operator<(const IContrato& otro) const {
				if (m_Prioridad < otro.m_Prioridad) {
					return true;
				}
				return false;
			};
			bool operator<(const IContrato* otro) const {
				if (m_Prioridad < otro->m_Prioridad) {
					return true;
				}
				return false;
			};


		public:
			TipoContrato m_Tipo;
			int          m_Prioridad;
		};

		//------------------------------------------------------------------------
		/* Clase de gestion de contratos */
		template <class TContrato>
		class IContratable {

//region "Interfaz pública de gestion de contratos"

		public:
			typedef TContrato TipoContrato;
			typedef priority_list<TContrato*> ListaContratos;
			typedef typename ListaContratos::iterator IteradorListaContratos;

		public:

			/* Registra un contrato asociado a la lista. Los objetos apuntados por pContrato no son nunca liberados por esta clase */
			void Subscribir(GNC::GCS::IVista* pVista, TContrato* pContrato)
			{
				/* solo para mostrar errores legibles en tiempo de compilación */
				GTRACE("IContrato::Subscribir( Herramienta = " << this << ", Vista = " << pVista << ", Contrato = " << pContrato << ")");
				GNC::GCS::IContrato* c = static_cast<GNC::GCS::IContrato*> (pContrato);
				if (c != NULL) {
					c->Ping();
					IteradorMapaContratos it = m_Contratos.find(pVista);
					if (it == m_Contratos.end()) { // Insertamos una nueva lista
						ListaContratos* pLista = new ListaContratos();
						pLista->insertar(pContrato);
						m_Contratos[pVista] = pLista;

					}
					else {
						(*it).second->insertar(pContrato);
					}
				}
			};
			/* DesRegistra un contrato asociado a la lista. El objeto apuntado por pContrato no es liberado nunca por esta clase. */
			void DesSubscribir(GNC::GCS::IVista* pVista, TContrato* pContrato)
			{
				GTRACE("IContrato::DesSubscribir( Herramienta = " << this << ", Vista = " << pVista << ", Contrato = " << pContrato << ")");
				IteradorMapaContratos it = m_Contratos.find(pVista);
				if (it != m_Contratos.end()) {
					ListaContratos* pLista = (*it).second;
					if (pLista != NULL) {
						pLista->eliminar(pContrato);
					}
				}
				if (pVista == m_pVistaActiva) {
					m_pVistaActiva = NULL;
					m_pListaActiva = NULL;
				}

			};

			void DesSubscribirsLosDeLaVista(GNC::GCS::IVista* pVista)
			{
				GTRACE("IContratable::DesSubscribirLosDeLaVista ( this = " << this << ", Vista = " << pVista << ")");
				if (pVista != NULL) {
					IteradorMapaContratos it = m_Contratos.find(pVista);
					if (it != m_Contratos.end()) {
						delete (*it).second;
						m_Contratos.erase(it);
					}
					if (pVista == m_pVistaActiva) {
						m_pVistaActiva = NULL;
						m_pListaActiva = NULL;
					}
				}
			};

			bool ValidaContrato(GNC::GCS::IVista* pVista) {
				ListaContratos* lista = NULL;
				if (pVista != NULL) {
					lista = ObtenerContratos(pVista);
				}
				return (lista != NULL && lista->size() > 0);
			}

			bool ValidaContratoLaVistaActiva() {
				return m_pListaActiva != NULL && m_pListaActiva->size() > 0;
			}
//endregion


//region "Interfaz privada de gestion de contratos"
		protected:
			typedef std::map<IVista*, ListaContratos* > MapaContratos;
			typedef typename MapaContratos::iterator IteradorMapaContratos;

			void EstablecerVistaActiva(GNC::GCS::IVista* pVista)
			{
				m_pVistaActiva = pVista;
				if (pVista != NULL) {
					IteradorMapaContratos it = m_Contratos.find(pVista);
					if (it != m_Contratos.end()) {
						m_pListaActiva = (*it).second;
					}
					else {
						m_pListaActiva = NULL;
					}
				} else {
					m_pListaActiva = NULL;
				}
			};

			ListaContratos* ObtenerContratos(GNC::GCS::IVista* pVista)
			{
				ListaContratos* lista = NULL;
				if (pVista != NULL) {
					IteradorMapaContratos it = m_Contratos.find(pVista);
					if (it != m_Contratos.end()) {
						lista = (*it).second;
					}
				}
				return lista;
			};

			ListaContratos* ObtenerContratosDeActiva()
			{
				return m_pListaActiva;
			};

			void PurgarTodo()
			{
				for (IteradorMapaContratos it = m_Contratos.begin(); it != m_Contratos.end(); it++) {
					delete (*it).second;
					m_Contratos.erase(it);
				}
			}

			IContratable() {
				m_pListaActiva = NULL;
				m_pVistaActiva = NULL;
			}

			~IContratable() {
				m_Contratos.clear();
				m_pListaActiva = NULL;
			}

//endregion

//region "Atributos"

		protected:
			MapaContratos m_Contratos;
			GNC::GCS::IVista* m_pVistaActiva;
			ListaContratos* m_pListaActiva;

//endregion
		};
		#if defined(_WINDOWS)
		EXTAPI_TEMPLATE template IContratable<IContrato>;
		#endif
	};

};
