#include <stdio.h>
#include <dirent.h>

#include <gtk/gtk.h>
#include <glade/glade.h>

#include "gkdial.h"
#include "defines.h"
#include "util.h"
#include "prefs.h"
#include "peers_list.h"

GtkTreeStore *
create_tree_model (void)
{
  GtkTreeStore *tree_model;

  tree_model = gtk_tree_store_new (NUM_COLUMNS, G_TYPE_BOOLEAN, G_TYPE_STRING);	/* peer 
										 */

  return tree_model;
}

void
prefs_peers_list_select (GkDial * gkdial, GtkTreeIter * iter)
{
  GtkTreeView *tview = gkdial->prefs_peers_list;
  GtkTreeSelection *sel = gtk_tree_view_get_selection (tview);

  gtk_tree_selection_select_iter (sel, iter);
}

gboolean
peers_list_get_selected (GkDial * gkdial)
{
  GtkTreeView *tview = gkdial->prefs_peers_list;
  GtkTreeModel *tmodel = GTK_TREE_MODEL (gkdial->prefs_peers_model);
  GtkTreeSelection *tselection = gtk_tree_view_get_selection (tview);
  GtkTreeIter ti;

  GkConnection *connection = gkdial->connection;

  /*
   * FIXME: show message asking the user to select a peer 
   */
  if (!gtk_tree_selection_get_selected (tselection, &tmodel, &ti))
    {
#ifdef DEBUG
      fprintf (stderr, "(peers_list_get_selected): no connection "
	       "selected!!\n");
#endif
      gk_dialog (GTK_MESSAGE_INFO,
		 _("You need to select a connection "
		   "to perform this action."));

      if (connection->name)
	{
	  g_free (connection->name);
	  connection->name = NULL;
	}
      return FALSE;
    }

  if (connection->name)
    {
      g_free (connection->name);
      connection->name = g_malloc (sizeof (gchar) * 256);
    }

  gtk_tree_model_get (tmodel, &ti, PEERS_COLUMN, &(connection->name), -1);

  return TRUE;
}

/*
 * finds something in the prefs_peers_list
 * if find_str == NULL it will return the default peer
 * iter
 */
GtkTreeIter *
prefs_peers_list_find (GkDial * gkdial, const gchar * find_str)
{
  GtkTreeModel *tree_model = GTK_TREE_MODEL (gkdial->prefs_peers_model);
  GtkTreeIter *iter = g_malloc (sizeof (GtkTreeIter));
  GtkTreeIter *iter_found = NULL;

  if (!gtk_tree_model_get_iter_root (tree_model, iter))
    {
      gtk_tree_iter_free (iter);
      return;
    }

  do
    {
      if (find_str)
	{
	  gchar *peer = NULL;

	  gtk_tree_model_get (tree_model, iter, PEERS_COLUMN, &peer, -1);

	  if (peer && !strcmp (peer, find_str))
	    {
	      iter_found = gtk_tree_iter_copy (iter);
	      g_free (peer);
	      break;
	    }
	}
      else
	{
	  gboolean is_default = FALSE;

	  gtk_tree_model_get (tree_model, iter,
			      DEFAULT_COLUMN, &is_default, -1);

	  if (is_default)
	    {
	      iter_found = gtk_tree_iter_copy (iter);
	      break;
	    }
	}
    }
  while (gtk_tree_model_iter_next (tree_model, iter));

  if (iter)
    gtk_tree_iter_free (iter);

  return iter_found;
}

/*
 * toggles the radio button for the new default and unsets 
 * the last default
 */
void
default_toggle_set_peers_list (GkDial * gkdial, GtkTreeIter * ti)
{
  GtkTreeModel *tree_model = GTK_TREE_MODEL (gkdial->prefs_peers_model);
  GtkTreeIter *old_default;
  gboolean is_default;
  gchar *default_peer = NULL;

  gtk_tree_model_get (GTK_TREE_MODEL (tree_model),
		      ti, DEFAULT_COLUMN, &is_default, -1);

  if (is_default)
    return;

  old_default = prefs_peers_list_find (gkdial, NULL);
  if (old_default)
    {
      gtk_tree_store_set (GTK_TREE_STORE (tree_model), old_default,
			  DEFAULT_COLUMN, FALSE, -1);
      gtk_tree_iter_free (old_default);
    }

  gtk_tree_store_set (GTK_TREE_STORE (tree_model), ti,
		      DEFAULT_COLUMN, TRUE, -1);
}

/*
 * sets the configuration into gconf
 */
void
default_toggle_set_gconf (GkDial * gkdial, gchar * path_str)
{
  GtkTreeModel *tree_model = GTK_TREE_MODEL (gkdial->prefs_peers_model);
  GtkTreeIter ti;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
  gboolean is_default;
  gchar *default_peer = NULL;

  gtk_tree_model_get_iter (GTK_TREE_MODEL (tree_model), &ti, path);
  gtk_tree_model_get (GTK_TREE_MODEL (tree_model), &ti, 0, &is_default, -1);

  if (is_default)
    {
      gtk_tree_path_free (path);
      return;
    }

  gtk_tree_model_get (GTK_TREE_MODEL (tree_model),
		      &ti, PEERS_COLUMN, &default_peer, -1);

  gkdial_conf_set_string (gkdial, "preferences/default_peer", default_peer);

  gtk_tree_path_free (path);
}

/*
 * default_toggle_set_gconf causes a notification to
 * be emited by gconf, so gkdial_conf_changed will be
 * responsible for calling default_toggle_set_peers_list
 */
void
default_toggle_cb (GtkCellRendererToggle * cr, gchar * path_str,
		   gpointer data)
{
  GkDial *gkdial = (GkDial *) data;

  default_toggle_set_gconf (gkdial, path_str);

#ifdef DEBUG
  printf ("default_toggle_cb: Hey!\n");
#endif
}

void
prefs_peers_set_default (GkDial * gkdial, gchar * peer_to_select)
{
  GtkTreeModel *tree_model = GTK_TREE_MODEL (gkdial->prefs_peers_model);
  GtkTreeIter *default_iter;

  default_iter = prefs_peers_list_find (gkdial, peer_to_select);
  if (!default_iter)
    return;

  gtk_tree_store_set (GTK_TREE_STORE (tree_model), default_iter,
		      DEFAULT_COLUMN, TRUE, -1);
}

void
create_view_columns (GkDial * gkdial)
{
  GtkTreeView *tview = gkdial->prefs_peers_list;
  GtkTreeModel *tree_model = GTK_TREE_MODEL (gkdial->prefs_peers_model);
  GtkCellRenderer *cr;
  GtkTreeViewColumn *column;

  /*
   * toggle for 'default' 
   */
  cr = gtk_cell_renderer_toggle_new ();
  gtk_cell_renderer_toggle_set_radio (GTK_CELL_RENDERER_TOGGLE (cr), TRUE);
  g_signal_connect (G_OBJECT (cr), "toggled",
		    G_CALLBACK (default_toggle_cb), gkdial);

  /*
   * column for toggles 
   */
  column = gtk_tree_view_column_new_with_attributes (_("Default"),
						     cr,
						     "active",
						     DEFAULT_COLUMN, NULL);
  gtk_tree_view_append_column (tview, column);

  /*
   * column for description 
   */
  cr = gtk_cell_renderer_text_new ();
  gtk_tree_view_insert_column_with_attributes (tview,
					       -1,
					       _("Peers"),
					       cr,
					       "text", PEERS_COLUMN, NULL);
}

void
prefs_peers_list_append (GkDial * gkdial, const gchar * name)
{
  GtkTreeIter iter;

  gtk_tree_store_append (gkdial->prefs_peers_model, &iter, NULL);
  gtk_tree_store_set (gkdial->prefs_peers_model, &iter,
		      PEERS_COLUMN, name, -1);
}

/*
 * main combo stuff 
 */
void
combo_set_selection (GkDial * gkdial, const gchar * choice)
{
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (gkdial->main_combo)->entry),
		      choice);
}

/*
 * fills/updates the main combo box and the preferences' peers list 
 */
gint
peers_list_fill (GkDial * gkdial, gboolean starting)
{
  DIR *peers_dir;
  GList *peers_list = NULL;
  struct dirent *files_peers;
  gchar *tmp = NULL;
  int peers_found = 0;

  GtkWidget *combo = gkdial->main_combo;

  peers_dir = opendir (PEERS_DIR);
  if (!peers_dir)
    {
      gk_dialog (GTK_MESSAGE_ERROR,
		 _("Unable to open directory %s...\n"
		   "You need read acces to this directory at least\n"
		   "to run this program, try adding your user to\n"
		   "the `dip' group. "), PEERS_DIR);
      exit (1);
    }

  /*
   * clear the prefs list 
   */
  gtk_tree_store_clear (gkdial->prefs_peers_model);

  while ((files_peers = readdir (peers_dir)))
    {
      if (((files_peers->d_name[0]) != '.'))
	{
	  tmp = strrchr (files_peers->d_name, '~');
	  if (tmp)
	    continue;

	  tmp = strrchr (files_peers->d_name, '.');
	  if (tmp)
	    if (!strcmp (tmp, ".bak"))
	      continue;

	  /*
	   * for the prefs peers_list 
	   */
	  prefs_peers_list_append (gkdial, (gchar *) files_peers->d_name);
	  /*
	   * for the main combo box 
	   */
	  peers_list = g_list_append (peers_list,
				      (gchar *) files_peers->d_name);
	  peers_found++;
	}
    }

  if (peers_found)
    {
      gchar *peer_to_select = NULL;

      if (starting)
	{
	  peer_to_select = gkdial_conf_get_string (gkdial,
						   "preferences/default_peer");
#ifdef DEBUG
	  printf ("peer_to_select[GConf]: %s\n", peer_to_select);
#endif
	}
      else
	{
	  peer_to_select =
	    gtk_editable_get_chars (GTK_EDITABLE
				    (GTK_COMBO (combo)->entry), 0, -1);
#ifdef DEBUG
	  printf ("peer_to_select[cur_sel]: %s\n", peer_to_select);
#endif
	}

      gtk_combo_set_popdown_strings (GTK_COMBO (combo), peers_list);

      if (peer_to_select && strcmp ("", peer_to_select))
	{
#ifdef DEBUG
	  printf ("setting combo to: %s\n", peer_to_select);
#endif
	  prefs_peers_set_default (gkdial, peer_to_select);
	  combo_set_selection (gkdial, peer_to_select);
	}
      else
	{
	  GList *first_peer;

	  first_peer = g_list_first (peers_list);
#ifdef DEBUG
	  printf ("setting combo to first of the list: %s\n",
		  (gchar *) first_peer->data);
#endif
	  combo_set_selection (gkdial, (gchar *) first_peer->data);
	}
    }

  return peers_found;
}

/*
 * initializes the main combo box and the preferences' peers list when
 * gkdial starts 
 */
void
gkdial_peers_list_init (GkDial * gkdial)
{
  /*
   * prepare the tree stuff to be filled initially 
   */
  gkdial->prefs_peers_model = create_tree_model ();
  gtk_tree_view_set_model (gkdial->prefs_peers_list,
			   GTK_TREE_MODEL (gkdial->prefs_peers_model));
  create_view_columns (gkdial);

  peers_list_fill (gkdial, TRUE);
}
