/* gkrellm-thinkbat.c - gkrellm-ThinkBat a Thinkpad battery monitor for Gkrellm2
 *
 * Copyright (C) 2006 Rasto Sramek <rasto@ksp.sk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gkrellm2/gkrellm.h>
#include <string.h>

#include "gkrellm-thinkbat.h"
#include "collect_data.h"

static gint panel_top_expose_event(GtkWidget *widget, GdkEventExpose *ev)
{
	gdk_draw_pixmap(widget->window,
		widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		panel_top->pixmap, ev->area.x, ev->area.y, ev->area.x, ev->area.y,
		ev->area.width, ev->area.height);

	return(FALSE);
}

static gint panel_bottom_expose_event(GtkWidget *widget, GdkEventExpose *ev)
{
	gdk_draw_pixmap(widget->window,
		widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		panel_bottom->pixmap, ev->area.x, ev->area.y, ev->area.x, ev->area.y,
		ev->area.width, ev->area.height);

	return(FALSE);
}

static gint panel_click_event(GtkWidget * widget, GdkEventButton * ev)
{
	if (ev->button == 3)
		gkrellm_open_config_window(monitor);
	return 1;
}

static void create_plugin(GtkWidget *vbox, gint first_create) 
{
	GkrellmStyle *style;
	GkrellmMargin *margin;
	GkrellmTextstyle *ts, *ts_alt;
	GdkPixmap *png_pixmap = gkrellm_decal_misc_pixmap();
	GdkBitmap *mask = gkrellm_decal_misc_mask();

	if (first_create) {
		panel_top = gkrellm_panel_new0();
		panel_bottom = gkrellm_panel_new0();
	}

	style = gkrellm_meter_style(style_id);
	ts = gkrellm_meter_textstyle(style_id);
	ts_alt = gkrellm_meter_alt_textstyle(style_id);
	margin = gkrellm_get_style_margins(style);

    	decal_pixmap = gkrellm_create_decal_pixmap(panel_top, png_pixmap, mask, 12, style, margin->left, -1);
	gkrellm_draw_decal_pixmap(panel_top, decal_pixmap, PIC_BATT);

	decal_percent = gkrellm_create_decal_text(panel_top, "XX%%", ts, style, margin->left, -1, -1);

	krell = gkrellm_create_krell(panel_top, gkrellm_krell_meter_piximage(style_id), style);
	gkrellm_monotonic_krell_values(krell, 0);
	gkrellm_set_krell_full_scale(krell, 100, 1);

	
	decal_time = gkrellm_create_decal_text(panel_bottom, "X:XX", ts_alt, style, margin->left, margin->top, -1);
	decal_watt = gkrellm_create_decal_text(panel_bottom, "XX.XW", ts_alt, style, margin->left, margin->top, -1);

	gkrellm_panel_configure(panel_top, NULL, style);
	gkrellm_panel_create(vbox, monitor, panel_top);
	
	gkrellm_panel_configure(panel_bottom, NULL, style);
	gkrellm_panel_create(vbox, monitor, panel_bottom);

	if (first_create) {
		g_signal_connect(G_OBJECT (panel_top->drawing_area), "expose_event", G_CALLBACK (panel_top_expose_event), NULL);
		g_signal_connect(G_OBJECT (panel_bottom->drawing_area), "expose_event", G_CALLBACK (panel_bottom_expose_event), NULL);
		gtk_signal_connect(GTK_OBJECT(panel_top->drawing_area), "button_release_event", (GtkSignalFunc) panel_click_event, NULL);
		gtk_signal_connect(GTK_OBJECT(panel_bottom->drawing_area), "button_release_event", (GtkSignalFunc) panel_click_event, NULL);

	}
	if(!show_all_button_val)
		gkrellm_panel_hide(panel_bottom);
}

static void create_plugin_config(GtkWidget *vbox)
{
	GtkWidget *notebook, *front_page, *front_box, *front_hbox, *info_page, *about_page;
	GtkWidget *front_hbox2, *about_box, *about, *label;
	GtkAdjustment *invoke;
	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);

	gtk_box_pack_start(GTK_BOX(vbox), notebook, 1, 1, 0);
	// Battery - first page
	front_page = gkrellm_gtk_notebook_page(notebook, _("Battery"));
	front_box = gkrellm_gtk_framed_vbox(front_page, 0, 4, 1, 0, 0);
	gkrellm_gtk_check_button(front_box, &show_all_button, show_all_button_val, 0, 0, "Show estimated remaining time and power consumption");

	front_hbox = gtk_hbox_new(0, 0);
	gtk_box_pack_start(GTK_BOX(front_box), front_hbox, 0, 0, 5);
	
	invoke = (GtkAdjustment*) gtk_adjustment_new(0, 1, 60, 1.0, 0, 0);
	invoke_adj = gtk_spin_button_new(invoke, 1.0, 0);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(invoke_adj), invoke_time);
	gtk_box_pack_start(GTK_BOX(front_hbox), invoke_adj, 0, 1, 0);
	label = gtk_label_new(" Seconds between updates");
	gtk_box_pack_start(GTK_BOX(front_hbox), label, 0, 1, 0);
	
	front_hbox2 = gtk_hbox_new(0, 0);
	gtk_box_pack_start(GTK_BOX(front_box), front_hbox2, 1, 0, 5);
	
	gkrellm_gtk_alert_button(front_hbox2, 0, 0, 0, 4, 1, set_alert, NULL);
	
	// About page
	about_page = gkrellm_gtk_notebook_page(notebook, _("About"));
	about_box = gkrellm_gtk_framed_vbox(about_page, 0, 4, 1, 0, 0);
	//gtk_box_pack_start(GTK_BOX(about_page), about_box, 1, 1, 0);
	about = gtk_label_new(_(plugin_desc));
	gtk_box_pack_start(GTK_BOX(about_box), about, 1, 1, 0);
}

static void apply_plugin_config(void)
{
	show_all_button_val = GTK_TOGGLE_BUTTON(show_all_button)->active;
	invoke_time = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(invoke_adj));
	if(show_all_button_val) {
		gkrellm_panel_show(panel_bottom);
	} else if(!show_all_button_val) {
		gkrellm_panel_hide(panel_bottom);
	}
}

static void save_plugin_config(FILE *file)
{
	fprintf(file, "%s show_all_button_val %d\n", CONFIG_NAME, show_all_button_val);
	fprintf(file, "%s invoke_time %d\n", CONFIG_NAME, invoke_time);
	gkrellm_save_alertconfig(file, alert, CONFIG_NAME, 0);
}

static void load_plugin_config(gchar *arg)
{
	gchar config[32], item[CFG_BUFSIZE];
	if (sscanf(arg, "%31s %[^\n]", config, item) == 2) {
		if (!strcmp(config, "show_all_button_val")) {
			sscanf(item, "%d", &show_all_button_val);
		} else if (!strcmp(config, "invoke_time")) {
			sscanf(item, "%d", &invoke_time);
		} else if (!strcmp(config, GKRELLM_ALERTCONFIG_KEYWORD)) {
			create_alert();
			gkrellm_load_alertconfig(&alert, item);
		}
	}
}

static void alert_trigger_callback(GkrellmAlert *alert, gpointer data)
{
	GkrellmAlertdecal *decal;

	alert->panel = panel_top;
	decal = &alert->ad;
	gkrellm_render_default_alert_decal(alert);
}

static void create_alert(void)
{
	if (!alert) {
		alert = gkrellm_alert_create(NULL, _("ThinkBat"), _("Battery Remaining Limits"), 0, 1, 1, 99, 0, 1, 10, 0);
		gkrellm_alert_delay_config(alert, 1, 600, 0);
	}
	gkrellm_alert_trigger_connect(alert, alert_trigger_callback, 0);
}

static void set_alert(GtkWidget *entry, gpointer data)
{
	create_alert();
	gkrellm_alert_config_window(&alert);
}



static void update_plugin()
{
	gchar text[20];
	int hours = 0, minutes, percent;

	if(invoke_time < 1 || invoke_time > 60) invoke_time = 5;
	if ((GK.timer_ticks % (20*invoke_time)) != 0) return;

	// Read all data from /sys
	read_battery_data();

	percent = get_percent_remaining();
	
	// Place correct pixmap
	if(get_ac_power()) {
		gkrellm_draw_decal_pixmap(panel_top, decal_pixmap, PIC_AC);
		if(!power_old) {
			gkrellm_reset_alert(alert);
		}
		power_old=1;
	} else {
		// Check alert
		gkrellm_check_alert(alert, (gfloat)percent);
		if(power_old)
			time_offline = 1;
		gkrellm_draw_decal_pixmap(panel_top, decal_pixmap, PIC_BATT);
		power_old=0;
	}

	// Write percent left
	snprintf(text, 20, "%d%%", percent);
	decal_percent->x_off = (decal_percent->w - gdk_string_width((GdkFont *)decal_percent->text_style.font, text))/2 + 8;
	gkrellm_draw_decal_text(panel_top, decal_percent, text, 0);

	// Update krell
	gkrellm_update_krell(panel_top, krell, percent);

	gkrellm_draw_panel_layers(panel_top);

	// If configured
	if(show_all_button_val) {
		// Write time left
		if(!power_old && time_offline < TIME_POWER_NOW)
			minutes = get_time_remaining(1);
		else 
			minutes = get_time_remaining(0);
		while(minutes >= 60) {
			minutes -= 60;
			hours++;
		}
		snprintf(text, 20, "%d:%.2d", hours, minutes);
		gkrellm_draw_decal_text(panel_bottom, decal_time, text, 0);
		
		// Write power consumption/loading
		if(!power_old && time_offline < TIME_POWER_NOW) {
			snprintf(text, 20, "%.1fW", get_power_change(1));
			time_offline++;
		} else {
			snprintf(text, 20, "%.1fW", get_power_change(0));
		}
		decal_watt->x_off = decal_watt->w - gdk_string_width((GdkFont *)decal_watt->text_style.font, text);
		gkrellm_draw_decal_text(panel_bottom, decal_watt, text, 0);
		gkrellm_draw_panel_layers(panel_bottom);
	}

}

GkrellmMonitor *gkrellm_init_plugin()
{
    style_id = gkrellm_add_meter_style(&plugin_mon, STYLE_NAME);
    monitor = &plugin_mon;
    return &plugin_mon;
}
