/* GKrellMSS - GKrellM Sound Scope
|  Copyright (C) 2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <sys/asoundlib.h>

/* snd_pcm calls here are for ALSA 0.5.
*/


#define BROKEN_NONBLOCKING	1

#if defined(BROKEN_NONBLOCKING)
#include <errno.h>
#endif


static gint		alsa_card = 0,
				alsa_device = 0;


#if defined(BROKEN_NONBLOCKING)
  /* code lifted out of ALSA 0.5.  I modify it so snd_pcm_loopback_read()
  |  will return after reading each header+buffer of data because non-blocking
  |  mode appears to be broken.
  */
struct snd_pcm_loopback {
    int card;
    int device;
    int fd;
    long mode;
    size_t buffer_size;
    char *buffer;
};

ssize_t _snd_pcm_loopback_read(snd_pcm_loopback_t *lb,
				snd_pcm_loopback_callbacks_t *callbacks)
{
	ssize_t result = 0, res, count;
	size_t size;
	char *buf;
	snd_pcm_loopback_header_t header;

	if (!lb || !callbacks)
		return -EINVAL;
	if (callbacks->max_buffer_size == 0)
		size = 64 * 1024;
	else
		size = callbacks->max_buffer_size < 16 ? 16 : callbacks->max_buffer_size;
	while (1)
		{
		if (lb->mode == SND_PCM_LB_STREAM_MODE_RAW)
			{
			header.size = size;
			header.type = SND_PCM_LB_TYPE_DATA;
			}
		else
			{
			res = read(lb->fd, &header, sizeof(header));
			if (res < 0)
				return -errno;
			if (res == 0)
				break;
			if (res != sizeof(header))
				return -EBADFD;
			result += res;
			}
		switch (header.type)
			{
			case SND_PCM_LB_TYPE_DATA:
				if (lb->buffer_size < size)
					{
					buf = (char *) realloc(lb->buffer, size);
					if (buf == NULL)
						return -ENOMEM;
					lb->buffer = buf;
					lb->buffer_size = size;
					}
				else
					{
					buf = lb->buffer;
					}
				while (header.size > 0)
					{
					count = header.size;
					if (count > (ssize_t)size)
						count = size;
					res = read(lb->fd, buf, count);
					if (res < 0)
						return -errno;
					result += res;
					if (lb->mode == SND_PCM_LB_STREAM_MODE_PACKET && res != count)
						return -EBADFD;
					if (res == 0)
						break;
					if (callbacks->data)
						callbacks->data(callbacks->private_data, buf, res);
					if (res < count && lb->mode == SND_PCM_LB_STREAM_MODE_RAW)
						break;
					header.size -= res;
					}
				break;
			case SND_PCM_LB_TYPE_FORMAT:
				{
				snd_pcm_format_t format;
				
				res = read(lb->fd, &format, sizeof(format));
				if (res < 0)
					return -errno;
				result += res;
				if (res != sizeof(format))
					return -EBADFD;
				if (callbacks->format_change)
					callbacks->format_change(callbacks->private_data, &format);
				}
				break;
			case SND_PCM_LB_TYPE_POSITION:
				{
				unsigned int pos;
				
				res = read(lb->fd, &pos, sizeof(pos));
				if (res < 0)
					return -errno;
				result += res;
				if (res != sizeof(pos))
					return -EBADFD;
				if (callbacks->position_change)
					callbacks->position_change(callbacks->private_data, pos);
				}
				break;
			case SND_PCM_LB_TYPE_SILENCE:
				if (callbacks->silence)
					callbacks->silence(callbacks->private_data, header.size);
				break;
			}
		break;  /* Added in to force snd_pcm_loopback_read() to return */
		}
	return result;
	}
#endif	/* BROKEN_NONBLOCKING */



static void
sound_input_read(void *private_data, char *buffer, size_t count)
	{
	if (count > 0)
		{
		memcpy(gkrellmss->buffer, buffer, count);
		process_sound_samples(count);
		}
	}

static void
alsa_loopback_ready(gpointer data, gint source, GdkInputCondition condition)
	{
	static snd_pcm_loopback_callbacks_t	callbacks;

	callbacks.data = sound_input_read;
	callbacks.max_buffer_size = gkrellmss->buf_len * sizeof(SoundSample);

	/* Should read data from the loopback until no more bytes to read, and
	|  then should return.
	*/
#if defined(BROKEN_NONBLOCKING)
	_snd_pcm_loopback_read((snd_pcm_loopback_t *) gkrellmss->handle, &callbacks);
#else
	snd_pcm_loopback_read((snd_pcm_loopback_t *) gkrellmss->handle, &callbacks);
#endif
	}

void
gkrellmss_sound_open_stream(gchar *host)
	{
	snd_pcm_loopback_t	*handle = NULL;
	snd_pcm_format_t	*format;
	gint				code;

	/* use card and device of zero */
	if ((code = snd_pcm_loopback_open(&handle, alsa_card, alsa_device,
				0, SND_PCM_LB_OPEN_PLAYBACK)) < 0)
		{
		fprintf (stderr, "snd_pcm_loopback_open failed: %s\n",
					snd_strerror(code));
		gkrellmss_sound_close_stream();
		return;
		}
	gkrellmss->handle = (gpointer) handle;

	/* Set loopback mode to non blocking so snd_pcm_loopback_read() will
	|  return if no data to read.  Then I can select on the read fd.
	|  (XXX This appears to be broken on ALSA 0.5).
	*/
	if ((code = snd_pcm_loopback_block_mode(handle, 0)) < 0)
		{
		fprintf(stderr, "snd_pcm_loopback_block_mode failed: %s\n",
					snd_strerror(code));
		gkrellmss_sound_close_stream();
		return;
		}

	format = g_new0(snd_pcm_format_t, 1);
	format->format = SND_PCM_SFMT_S16_LE;
	format->rate = SAMPLE_RATE;
	format->voices = 2;
	if ((code = snd_pcm_loopback_format(handle, format)) < 0)
		fprintf(stderr, "snd_pcm_loopback_format failed: %s\n",
					snd_strerror(code));
	else
		gkrellmss->stream_open = TRUE;
	g_free(format);

	gkrellmss->fd = snd_pcm_loopback_file_descriptor(handle);
	gkrellmss->input_id = gdk_input_add(gkrellmss->fd, GDK_INPUT_READ,
						(GdkInputFunction) alsa_loopback_ready, NULL);
	}

void
gkrellmss_sound_close_stream(void)
	{
	if (gkrellmss->handle)
		snd_pcm_loopback_close((snd_pcm_loopback_t *) gkrellmss->handle);
	reset_sound_data();
	}
