/*
 *  (GLABELS) Label and Business Card Creation program for GNOME
 *
 *  bc.c:  GLabels barcode module
 *
 *  Copyright (C) 2001-2003  Jim Evins <evins@snaught.com>.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */
#include <config.h>

#include <libgnome/libgnome.h>

#include "bc.h"
#include "bc-postnet.h"
#include "bc-gnubarcode.h"

#include "debug.h"

/*========================================================*/
/* Private macros and constants.                          */
/*========================================================*/

/*========================================================*/
/* Private types.                                         */
/*========================================================*/

typedef struct {
	gchar            *id;
	gchar            *name;
	glBarcodeNewFunc  new;
	gboolean          can_text;
	gboolean          text_optional;
	gboolean          can_checksum;
	gboolean          checksum_optional;
	gchar            *default_digits;
} Backend;


/*========================================================*/
/* Private globals.                                       */
/*========================================================*/

Backend backends[] = {

	{ "POSTNET", N_("POSTNET (any)"), gl_barcode_postnet_new,
	  FALSE, FALSE, TRUE, FALSE, "000000000"},

	{ "POSTNET-5", N_("POSTNET-5 (ZIP only)"), gl_barcode_postnet_new,
	  FALSE, FALSE, TRUE, FALSE, "00000"},

	{ "POSTNET-9", N_("POSTNET-9 (ZIP+4)"), gl_barcode_postnet_new,
	  FALSE, FALSE, TRUE, FALSE, "000000000"},

	{ "POSTNET-11", N_("POSTNET-11 (DPBC)"), gl_barcode_postnet_new,
	  FALSE, FALSE, TRUE, FALSE, "00000000000"},

	{ "EAN", N_("EAN (any)"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "000000000000 00000"},

	{ "EAN-8", N_("EAN-8"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "0000000"},

	{ "EAN-8+2", N_("EAN-8 +2"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "0000000 00"},

	{ "EAN-8+5", N_("EAN-8 +5"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "0000000 00000"},

	{ "EAN-13", N_("EAN-13"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "000000000000"},

	{ "EAN-13+2", N_("EAN-13 +2"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "000000000000 00"},

	{ "EAN-13+5", N_("EAN-13 +5"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "000000000000 00000"},

	{ "UPC", N_("UPC (UPC-A or UPC-E)"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "00000000000 00000"},

	{ "UPC-A", N_("UPC-A"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "00000000000"},

	{ "UPC-A+2", N_("UPC-A +2"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "00000000000 00"},

	{ "UPC-A+5", N_("UPC-A +5"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "00000000000 00000"},

	{ "UPC-E", N_("UPC-E"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "000000"},

	{ "UPC-E+2", N_("UPC-E +2"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "000000 00"},

	{ "UPC-E+5", N_("UPC-E +5"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, FALSE, "000000 00000"},

	{ "ISBN", N_("ISBN"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, TRUE, "0-00000-000-0"},

	{ "ISBN+5", N_("ISBN +5"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, TRUE, "0-00000-000-0 00000"},

	{ "Code39", N_("Code 39"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, TRUE, "0000000000"},

	{ "Code128", N_("Code 128"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, TRUE, "0000000000"},

	{ "Code128C", N_("Code 128C"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, TRUE, "0000000000"},

	{ "Code128B", N_("Code 128B"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, TRUE, "0000000000"},

	{ "I25", N_("Interleaved 2 of 5"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, TRUE, "0000000000"},

	{ "CBR", N_("Codabar"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, TRUE, "0000000000"},

	{ "MSI", N_("MSI"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, TRUE, "0000000000"},

	{ "PLS", N_("Plessey"), gl_barcode_gnubarcode_new,
	  TRUE, TRUE, TRUE, TRUE, "0000000000"},

	{ NULL, NULL, NULL, FALSE, FALSE, FALSE, FALSE, NULL}

};

/*========================================================*/
/* Private function prototypes.                           */
/*========================================================*/


/*---------------------------------------------------------------------------*/
/* Convert id to index into above table.                                     */
/*---------------------------------------------------------------------------*/
static gint
id_to_index (const gchar *id)
{
	gint i;

	if (id == 0) {
		return 0; /* NULL request default. I.e., the first element. */
	}

	for (i=0; backends[i].id != NULL; i++) {
		if (g_strcasecmp (id, backends[i].id) == 0) {
			return i;
		}
	}

	g_warning( "Unknown barcode id \"%s\"", id );
	return 0;
}

/*---------------------------------------------------------------------------*/
/* Convert name to index into above table.                                   */
/*---------------------------------------------------------------------------*/
static gint
name_to_index (const gchar *name)
{
	gint i;

	g_return_val_if_fail (name!=NULL, 0);

	for (i=0; backends[i].id != NULL; i++) {
		if (g_strcasecmp (name, backends[i].name) == 0) {
			return i;
		}
	}

	g_warning( "Unknown barcode name \"%s\"", name );
	return 0;
}

/*****************************************************************************/
/* Call appropriate barcode backend to create barcode in intermediate format.*/
/*****************************************************************************/
glBarcode *
gl_barcode_new (const gchar    *id,
		gboolean        text_flag,
		gboolean        checksum_flag,
		gdouble         w,
		gdouble         h,
		const gchar    *digits)
{
	glBarcode *gbc;
	gint       i;

	g_return_val_if_fail (digits!=NULL, NULL);

	i = id_to_index (id);
	gbc = backends[i].new (backends[i].id,
			       text_flag,
			       checksum_flag,
			       w,
			       h,
			       digits);

	return gbc;
}

/*****************************************************************************/
/* Free previously created barcode.                                          */
/*****************************************************************************/
void
gl_barcode_free (glBarcode **gbc)
{
	GList *p;

	if (*gbc != NULL) {

		for (p = (*gbc)->lines; p != NULL; p = p->next) {
			g_free (p->data);
			p->data = NULL;
		}
		g_list_free ((*gbc)->lines);
		(*gbc)->lines = NULL;

		for (p = (*gbc)->chars; p != NULL; p = p->next) {
			g_free (p->data);
			p->data = NULL;
		}
		g_list_free ((*gbc)->chars);
		(*gbc)->chars = NULL;

		g_free (*gbc);
		*gbc = NULL;
	}
}

/*****************************************************************************/
/* Get a list of names for valid barcode styles.                             */
/*****************************************************************************/
GList *
gl_barcode_get_styles_list  (void)
{
	gint   i;
	GList *list = NULL;

	for (i=0; backends[i].id != NULL; i++) {
		list = g_list_append (list, g_strdup (backends[i].name));
	}

	return list;
}

/*****************************************************************************/
/* Free up a previously allocated list of style names.                       */
/*****************************************************************************/
void
gl_barcode_free_styles_list (GList *styles_list)
{
	GList *p;

	for (p=styles_list; p != NULL; p=p->next) {
		g_free (p->data);
		p->data = NULL;
	}

	g_list_free (styles_list);
}


/*****************************************************************************/
/* Return an appropriate set of digits for the given barcode style.          */
/*****************************************************************************/
gchar *
gl_barcode_default_digits (const gchar *id)
{
	return g_strdup (backends[id_to_index (id)].default_digits);
}

/*****************************************************************************/
/* Query text capabilities.                                                  */
/*****************************************************************************/
gboolean
gl_barcode_can_text (const gchar *id)
{
	return backends[id_to_index (id)].can_text;
}

gboolean
gl_barcode_text_optional (const gchar *id)
{
	return backends[id_to_index (id)].text_optional;
}

/*****************************************************************************/
/* Query checksum capabilities.                                              */
/*****************************************************************************/
gboolean
gl_barcode_can_csum (const gchar *id)
{
	return backends[id_to_index (id)].can_checksum;
}

gboolean
gl_barcode_csum_optional (const gchar *id)
{
	return backends[id_to_index (id)].checksum_optional;
}

/*****************************************************************************/
/* Convert style to text.                                                    */
/*****************************************************************************/
const gchar *
gl_barcode_id_to_name (const gchar *id)
{
	return backends[id_to_index (id)].name;
}

/*****************************************************************************/
/* Convert name to style.                                                    */
/*****************************************************************************/
const gchar *
gl_barcode_name_to_id (const gchar *name)
{
	g_return_val_if_fail (name!=NULL, backends[0].id);

	return backends[name_to_index (name)].id;
}
