/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <guichaz@yahoo.fr>
 */

/************************************
 * The Images and Directories menus *
 ************************************/

#include <string.h>             /* strlen() */

#include "gliv.h"
#include "images_menus.h"
#include "options.h"
#include "menus.h"
#include "messages.h"
#include "next_image.h"
#include "mnemonics.h"
#include "files_list.h"
#include "loading.h"
#include "thumbnails.h"
#include "callbacks.h"
#include "tree.h"
#include "timestamp.h"
#include "dirty_gtk.h"

extern options_struct *options;

/* The progress indicator. */
static GtkMenuItem *rebuilding_entry;

/* The menu entry to cancel the rebuilding. */
static GtkMenuItem *cancel_menu_item;

/* The number of files in the current tree. */
static gint tree_count;

/*** Functions for both menus. ***/

/* Called at menu creation time. */
void set_rebuilding_entry(GtkMenuItem * item)
{
    rebuilding_entry = item;
}

/* Refresh the percent indicator. */
static void set_menu_indicator(gchar * name, gint percent)
{
    static gchar *rebuilding = NULL;

    if (name == NULL)
        set_menu_label(rebuilding_entry, "", FALSE);
    else {
        gchar *label;

        if (rebuilding == NULL)
            /* First time. */
            rebuilding = _("Rebuilding:");

        label = g_strdup_printf("%s %s (%d%%)", rebuilding, name, percent);
        set_menu_label(rebuilding_entry, label, FALSE);

        g_free(label);
    }
}

/* Update the percent indicator. */
static void set_progress(gchar * menu, gint * percent, gint number)
{
    if ((*percent + 1) * tree_count <= 100 * number) {
        *percent = (100 * number) / tree_count;
        set_menu_indicator(menu, *percent);
    }
}

/* We don't want to perform the signal lookup for each file. */
static void connect_activate(GtkMenuItem * instance, const gchar * filename)
{
    static guint signal_id = 0;
    GClosure *closure;

    if (signal_id == 0)
        /* First time. */
        signal_id = g_signal_lookup("activate", G_TYPE_FROM_INSTANCE(instance));

    closure = g_cclosure_new_swap(G_CALLBACK(menu_load),
                                  (gpointer) filename, NULL);

    g_signal_connect_closure_by_id(instance, signal_id, 0, closure, FALSE);
}

typedef enum {
    MENU_FILE,                  /* From both menus. */
    MENU_DIR,                   /* From the Images menu. */
    MENU_SUBMENU                /* From the Directories menu. */
} menu_type;

/* Add also a mnemonic and a thumbnail. */
static GtkMenuItem *add_menu_item(DirtyGtkMenu * menu,
                                  tree_item * item, menu_type type)
{
    const gchar *name;
    GtkMenuItem *menu_item;
    GtkImage *thumbnail = NULL;

    if (options->thumbnails && type == MENU_FILE && fill_thumbnail(item))
        thumbnail = GTK_IMAGE(gtk_image_new_from_pixbuf(item->thumb));
    else
        /* Simulate the do_threaded() effect of fill_thumbnail(). */
        process_events();

    if (type == MENU_DIR)
        name = item->path;
    else
        name = item->name;

    if (options->mnemonics && type != MENU_DIR) {
        name = add_mnemonic(name);

        if (thumbnail)
            menu_item =
                GTK_MENU_ITEM(gtk_image_menu_item_new_with_mnemonic(name));
        else
            menu_item = GTK_MENU_ITEM(gtk_menu_item_new_with_mnemonic(name));

    } else {
        if (thumbnail)
            menu_item = GTK_MENU_ITEM(gtk_image_menu_item_new_with_label(name));
        else
            menu_item = GTK_MENU_ITEM(gtk_menu_item_new_with_label(name));
    }

    if (thumbnail)
        gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menu_item),
                                      GTK_WIDGET(thumbnail));

    dirty_gtk_menu_append(menu, GTK_WIDGET(menu_item));

    switch (type) {
    case MENU_FILE:
        connect_activate(menu_item, item->path);
        break;

    case MENU_DIR:
        gtk_widget_set_sensitive(GTK_WIDGET(menu_item), FALSE);
        break;

    case MENU_SUBMENU:
        break;
    }

    return menu_item;
}

/*** Directories menu. ***/

static void add_file_from_tree(DirtyGtkMenu * menu, tree_item * item)
{
    static gchar *directories = NULL;
    static gint number = 0, percent = 0;

    if (canceled_using_tree())
        return;

    if (menu == NULL) {
        /* First time for this menu. */

        if (directories == NULL)
            /* First time. */
            directories = _("Directories");

        number = 0;
        percent = 0;
        return;
    }

    add_menu_item(menu, item, MENU_FILE);

    set_progress(directories, &percent, number);
    number++;
}

static DirtyGtkMenu *add_sub_menu(DirtyGtkMenu * parent, tree_item * item)
{
    DirtyGtkMenu *menu;
    GtkMenuItem *menu_item;

    menu_item = add_menu_item(parent, item, MENU_SUBMENU);

    menu = dirty_gtk_menu_new();
    gtk_menu_item_set_submenu(menu_item, GTK_WIDGET(menu->gtk_menu_shell));

    dirty_gtk_menu_append(menu, gtk_tearoff_menu_item_new());

    return menu;
}

static void make_menu_from_tree_rec(GNode * tree, DirtyGtkMenu * parent)
{
    tree_item *item;
    DirtyGtkMenu *menu;

    if (canceled_using_tree())
        return;

    if (G_NODE_IS_LEAF(tree)) {
        add_file_from_tree(parent, tree->data);
        return;
    }

    item = tree->data;
    menu = add_sub_menu(parent, item);

    push_mnemonics();

    g_node_children_foreach(tree, G_TRAVERSE_ALL,
                            (GNodeForeachFunc) make_menu_from_tree_rec, menu);

    dirty_gtk_menu_release(menu);
    pop_mnemonics();
}

/*** Images menu. ***/

static gboolean add_file_item(GNode * tree, DirtyGtkMenu * parent_menu)
{
    static gchar *images = NULL;
    static gint number = 0, percent = 0;
    GNode *sibling;

    if (canceled_using_tree())
        return TRUE;

    if (parent_menu == NULL) {
        /* First time for this menu. */

        if (images == NULL)
            /* First time. */
            images = _("Images");

        number = 0;
        percent = 0;
        return FALSE;
    }

    /* Check if it is the first image in its directory. */
    sibling = g_node_prev_sibling(tree);
    if (sibling == NULL && tree->parent != NULL)
        add_menu_item(parent_menu, tree->parent->data, MENU_DIR);

    add_menu_item(parent_menu, tree->data, MENU_FILE);

    set_progress(images, &percent, number);
    number++;

    return FALSE;
}

/*** Menus builders. ***/

static DirtyGtkMenu *begin_rebuild(GtkMenuItem * root, GCallback func)
{
    static DirtyGtkMenu *menu;
    GtkWidget *rebuilder;

    if (root != NULL)
        gtk_menu_item_deselect(root);

    menu = dirty_gtk_menu_new();
    dirty_gtk_menu_append(menu, gtk_tearoff_menu_item_new());

    rebuilder =
        gtk_menu_item_new_with_mnemonic(add_mnemonic(_("Rebuild this menu")));

    g_signal_connect_swapped(rebuilder, "activate", func, NULL);
    dirty_gtk_menu_append(menu, rebuilder);

    gtk_menu_item_set_submenu(root, GTK_WIDGET(menu->gtk_menu_shell));

    return menu;
}

static GNode *get_tree(void)
{
    if (is_loading())
        /*
         * There is no problem in rebuilding the images menus during
         * a loading, it would just make the loading too long.
         */
        return NULL;

    return make_tree();
}

#define REBUILD_START(name, func)                                            \
    do {                                                                     \
        if (root != NULL) {                                                  \
            menu_item = root;                                                \
            return TRUE;                                                     \
        }                                                                    \
                                                                             \
        if (more_recent_than_tree(timestamp))                                \
            /* The menu is already up to date. */                            \
            return TRUE;                                                     \
                                                                             \
        tree = get_tree();                                                   \
        if (tree == NULL)                                                    \
            return FALSE;                                                    \
                                                                             \
        tree_count = tree_count_files(tree);                                 \
        gtk_widget_set_sensitive(GTK_WIDGET(cancel_menu_item), TRUE);        \
        reset_mnemonics();                                                   \
        menu = begin_rebuild(menu_item, G_CALLBACK(func));                   \
        set_menu_indicator(name, 0);                                         \
    } while (0)

#define REBUILD_END()                                                        \
    do {                                                                     \
        if (root != NULL)                                                    \
            gtk_menu_item_deselect(root);                                    \
                                                                             \
        set_menu_indicator(NULL, 0);                                         \
        gtk_widget_set_sensitive(GTK_WIDGET(cancel_menu_item), FALSE);       \
        if (canceled_using_tree())                                           \
            reset_timestamp(&timestamp);                                     \
        else                                                                 \
            touch(&timestamp);                                               \
        end_using_tree();                                                    \
        reset_mnemonics();                                                   \
        gtk_widget_show_all(GTK_WIDGET(menu_item));                          \
        dirty_gtk_menu_release(menu);                                        \
        return timestamp != 0;                                               \
    } while (0)

gboolean rebuild_directories(GtkMenuItem * root)
{
    static GtkMenuItem *menu_item;
    static DECLARE_TIMESTAMP(timestamp);
    DirtyGtkMenu *menu;
    GNode *tree, *child;
    gchar *prefix, *old_name;
    tree_item *item;

    REBUILD_START(_("Directories"), rebuild_directories);

    /* Build the menu. */
    add_file_from_tree(NULL, NULL);

    item = tree->data;
    prefix = item->path;

    if (prefix[0] == '\0' || prefix[1] == '\0') {
        for (child = g_node_first_child(tree); child; child = child->next) {
            item = child->data;
            old_name = item->name;
            item->name = g_build_filename(prefix, item->name, NULL);

            make_menu_from_tree_rec(child, menu);

            g_free(item->name);
            item->name = old_name;
        }
    } else
        make_menu_from_tree_rec(tree, menu);

    REBUILD_END();
}

gboolean rebuild_images(GtkMenuItem * root)
{
    static GtkMenuItem *menu_item;
    static DECLARE_TIMESTAMP(timestamp);
    DirtyGtkMenu *menu;
    GNode *tree;

    REBUILD_START(_("Images"), rebuild_images);

    /* Build the menu. */
    add_file_item(NULL, NULL);
    g_node_traverse(tree, G_PRE_ORDER, G_TRAVERSE_LEAFS, -1,
                    (GNodeTraverseFunc) add_file_item, menu);

    REBUILD_END();
}

/* Rebuild both menus. */
gboolean rebuild_images_menus(void)
{
    return rebuild_directories(NULL) && rebuild_images(NULL);
}

void set_stop_rebuilding_menu(GtkMenuItem * item)
{
    cancel_menu_item = item;
    gtk_widget_set_sensitive(GTK_WIDGET(item), FALSE);
}
