/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <guichaz@yahoo.fr>
 */

/********************
 * Main source file *
 ********************/

#include <unistd.h>             /* close(), dup(), dup2(), STDERR_FILENO */
#include <stdlib.h>             /* exit() */
#include <gtk/gtkgl.h>

#include "gliv.h"
#include "main.h"
#include "cmdline.h"
#include "options.h"
#include "gliv-image.h"
#include "rcfile.h"
#include "files_list.h"
#include "messages.h"
#include "windows.h"
#include "ipc.h"

rt_struct *rt;
options_struct *options;
GlivImage *current_image = NULL;
GtkWidget *gl_widget;
GtkMenuBar *menu_bar;

/*
 * This is borrowed from gtk+-2, without the gtk_grab_notify() calls.
 * They slow down things a lot when the images menus are built with many files.
 */
GtkWindowGroup *_gtk_window_get_group(GtkWindow * window)
{
    if (window && window->group)
        return window->group;
    else {
        static GtkWindowGroup *default_group = NULL;

        if (!default_group)
            default_group = gtk_window_group_new();

        return default_group;
    }
}

static GtkWindowGroup *gtk_main_get_window_group(GtkWidget * widget)
{
    GtkWidget *toplevel = NULL;

    if (widget)
        toplevel = gtk_widget_get_toplevel(widget);

    if (toplevel && GTK_IS_WINDOW(toplevel))
        return _gtk_window_get_group(GTK_WINDOW(toplevel));
    else
        return _gtk_window_get_group(NULL);
}

void gtk_grab_add(GtkWidget * widget)
{
    GtkWindowGroup *group;

    g_return_if_fail(widget != NULL);

    if (!GTK_WIDGET_HAS_GRAB(widget) && GTK_WIDGET_IS_SENSITIVE(widget)) {
        GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_GRAB);

        group = gtk_main_get_window_group(widget);

        g_object_ref(widget);
        group->grabs = g_slist_prepend(group->grabs, widget);

/*      gtk_grab_notify (group, widget, FALSE); */
    }
}

void gtk_grab_remove(GtkWidget * widget)
{
    GtkWindowGroup *group;

    g_return_if_fail(widget != NULL);

    if (GTK_WIDGET_HAS_GRAB(widget)) {
        GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_GRAB);

        group = gtk_main_get_window_group(widget);
        group->grabs = g_slist_remove(group->grabs, widget);

        g_object_unref(widget);

/*      gtk_grab_notify (group, widget, TRUE); */
    }
}

void destroy_ggo_inputs(struct gengetopt_args_info *ggo)
{
    gint i;

    for (i = 0; i < ggo->inputs_num; i++)
        g_free(ggo->inputs[i]);

    g_free(ggo->inputs);
    ggo->inputs = NULL;
    ggo->inputs_num = 0;
}

void destroy_ggo_flags(struct gengetopt_args_info *ggo)
{
    g_free(ggo->menu_arg);
    g_free(ggo->client_arg);
    g_free(ggo->alpha_checks_arg);
    g_free(ggo->build_menus_arg);
    g_free(ggo->force_load_arg);
    g_free(ggo->full_screen_arg);
    g_free(ggo->glivrc_arg);
    g_free(ggo->info_arg);
    g_free(ggo->loop_arg);
    g_free(ggo->scale_down_arg);
    g_free(ggo->maximize_arg);
    g_free(ggo->make_fit_arg);
    g_free(ggo->mnemonics_arg);
    g_free(ggo->scrollbars_arg);
    g_free(ggo->mipmap_arg);
    g_free(ggo->recursive_arg);
    g_free(ggo->sort_arg);
    g_free(ggo->shuffle_arg);
    g_free(ggo->thumbnails_arg);
    g_free(ggo->dither_arg);
    g_free(ggo->resize_win_arg);
    g_free(ggo->slide_show_arg);
    g_free(ggo->zoom_pointer_arg);
    g_free(ggo->null_arg);
    g_free(ggo->one_image_arg);
    g_free(ggo->no_center_arg);
}

/* May exit :) */
gboolean get_on_off(const gchar * str)
{
    if (str == NULL)
        return TRUE;

    while (*str == '=' || *str == ' ')
        str++;

    if (g_str_equal(str, "on"))
        return TRUE;

    if (g_str_equal(str, "off"))
        return FALSE;

    ERROR_MSG(_("Command line flags should be on or off, not %s\n"), str);
    quit(1);
}

typedef struct {
    gboolean given;
    gboolean *flag;
    gchar *str;
    gboolean not;
} ggo_flag;

typedef struct {
    gboolean given;
    gint *opt_val;
    gint value;
    gint minimum;
} ggo_int;

static void fill_options(struct gengetopt_args_info *ggo)
{
    ggo_flag flags[] = {
/* *INDENT-OFF* */
{ ggo->full_screen_given,  &options->fullscreen,   ggo->full_screen_arg,  0 },
{ ggo->scale_down_given,   &options->scaledown,    ggo->scale_down_arg,   0 },
{ ggo->maximize_given,     &options->maximize,     ggo->maximize_arg,     0 },
{ ggo->zoom_pointer_given, &options->zoom_pointer, ggo->zoom_pointer_arg, 0 },
{ ggo->dither_given,       &options->dither,       ggo->dither_arg,       0 },
{ ggo->force_load_given,   &options->force,        ggo->force_load_arg,   0 },
{ ggo->mipmap_given,       &options->mipmap,       ggo->mipmap_arg,       0 },
{ ggo->build_menus_given,  &options->build_menus,  ggo->build_menus_arg,  1 },
{ ggo->mnemonics_given,    &options->mnemonics,    ggo->mnemonics_arg,    1 },
{ ggo->menu_given,         &options->menu_bar,     ggo->menu_arg,         1 },
{ ggo->info_given,         &options->status_bar,   ggo->info_arg,         1 },
{ ggo->scrollbars_given,   &options->scrollbars,   ggo->scrollbars_arg,   1 },
{ ggo->alpha_checks_given, &options->alpha_checks, ggo->alpha_checks_arg, 1 },
{ ggo->loop_given,         &options->loop,         ggo->loop_arg,         0 },
{ ggo->one_image_given,    &options->one_image,    ggo->one_image_arg,    0 },
{ ggo->thumbnails_given,   &options->thumbnails,   ggo->thumbnails_arg,   1 },
{ ggo->slide_show_given,   &options->start_show,   ggo->slide_show_arg,   0 },
{ ggo->resize_win_given,   &options->resize_win,   ggo->resize_win_arg,   1 },
{ ggo->make_fit_given,     &options->maximize,     ggo->make_fit_arg,     0 },
{ ggo->make_fit_given,     &options->scaledown,    ggo->make_fit_arg,     0 },
{ ggo->no_center_given,    &options->no_center,    ggo->no_center_arg,    0 },
{ FALSE,                   NULL,                   NULL,                    }
/* *INDENT-ON* */
    };

    ggo_int ints[] = {
/* *INDENT-OFF* */
{ ggo->delay_given,        &options->delay,        ggo->delay_arg,     0 },
{ ggo->history_given,      &options->history_size, ggo->history_arg,  -1 },
{ ggo->duration_given,     &options->duration,     ggo->duration_arg,  0 },
{ ggo->fps_given,          &options->fps,          ggo->fps_arg,      -1 },
{ FALSE,                   NULL,                   FALSE,              0 }
/* *INDENT-ON* */
    };

    ggo_flag *flag;
    ggo_int *opt_int;

    for (flag = flags; flag->flag != NULL; flag++)
        if (flag->given) {
            *flag->flag = get_on_off(flag->str);
            if (flag->not)
                *flag->flag ^= TRUE;
        }

    for (opt_int = ints; opt_int->opt_val != NULL; opt_int++)
        if (opt_int->given && opt_int->value >= opt_int->minimum)
            *opt_int->opt_val = opt_int->value;
}

#define FLAG_ON(flag) (flag##_given && get_on_off(flag##_arg))

static void init_args(gint argc, gchar ** argv)
{
    struct gengetopt_args_info ggo;
    options_struct *rc_file;

    /* Command line (some flags only). */

    if (cmdline_parser(argc, argv, &ggo) != 0)
        quit(1);

    if (FLAG_ON(ggo.sort) && FLAG_ON(ggo.shuffle)) {
        g_printerr(_("Cannot sort and shuffle at the same time\n"));
        quit(1);
    }

    if (FLAG_ON(ggo.client) && connect_server(argc, argv))
        /* Successfully reused a GLiv window. */
        quit(0);

    /* Configuration file. */
    rc_file = load_rc(!FLAG_ON(ggo.glivrc));
    options = g_memdup(rc_file, sizeof(options_struct));

    /* Command line (remaining options). */
    fill_options(&ggo);

    if (FLAG_ON(ggo.glivrc))
        options->save_quit = FALSE;

    rt = g_new(rt_struct, 1);

    rt->cursor_hidden = FALSE;
    rt->help = FALSE;
    rt->alpha_checks_changed = TRUE;
    rt->scr_width = gdk_screen_width();
    rt->scr_height = gdk_screen_height();

    if (FLAG_ON(ggo.null)) {
        read_null_filenames(FLAG_ON(ggo.recursive), FLAG_ON(ggo.shuffle),
                            FLAG_ON(ggo.sort));

        insert_after_current(ggo.inputs, ggo.inputs_num,
                             FLAG_ON(ggo.recursive),
                             FLAG_ON(ggo.shuffle), FLAG_ON(ggo.sort), TRUE);

    } else if (ggo.inputs_num > 0)
        /* There are filenames on the command line. */
        init_list(ggo.inputs, ggo.inputs_num,
                  FLAG_ON(ggo.recursive),
                  FLAG_ON(ggo.shuffle), FLAG_ON(ggo.sort));

    destroy_ggo_inputs(&ggo);
    destroy_ggo_flags(&ggo);
}

/*
 * We call cmdline_parser() a first time to quickly exit if --help or --version
 * was given.  We do the real work after the next call in init_args(), after
 * gtk_init() removed the gtk arguments.
 *
 * We temporarily close stderr because unknown and gtk arguments are handled
 * afterwards.
 */
static void check_quick_exit(gint argc, gchar ** argv)
{
    struct gengetopt_args_info ggo;
    gint fd;

    /* Silence stderr. */
    fd = dup(STDERR_FILENO);
    close(STDERR_FILENO);

    cmdline_parser(argc, argv, &ggo);

    /* Restore stderr. */
    dup2(fd, STDERR_FILENO);
    close(fd);

    destroy_ggo_inputs(&ggo);
    destroy_ggo_flags(&ggo);
}

G_GNUC_NORETURN void quit(gint code)
{
    if (options->save_quit)
        save_rc(options);

    exit(code);
}

gboolean gui_quit(void)
{
    GtkMessageDialog *dialog;
    gchar *msg;

    if (options->confirm_quit == FALSE)
        quit(0);

    msg = _("Do you really want to quit GLiv?");

    dialog = GTK_MESSAGE_DIALOG(gtk_message_dialog_new(get_current_window(),
                                                       GTK_DIALOG_MODAL,
                                                       GTK_MESSAGE_QUESTION,
                                                       GTK_BUTTONS_YES_NO,
                                                       "%s", msg));

    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_YES);

    set_hide_cursor_enabled(FALSE);
    if (run_modal_dialog(GTK_DIALOG(dialog)) == GTK_RESPONSE_YES)
        quit(0);

    gtk_widget_destroy(GTK_WIDGET(dialog));
    set_hide_cursor_enabled(TRUE);
    return TRUE;
}

gint main(gint argc, gchar ** argv)
{
    check_quick_exit(argc, argv);

    /* i18n */
#ifdef ENABLE_NLS
    gtk_set_locale();
#ifdef LOCALEDIR
    bindtextdomain(PACKAGE, LOCALEDIR);
#endif
    textdomain(PACKAGE);
    bind_textdomain_codeset(PACKAGE, "UTF-8");
#endif

    g_thread_init(NULL);
    gtk_init(&argc, &argv);
    gtk_gl_init(&argc, &argv);

    init_args(argc, argv);

    create_windows();
    start_server();

    gtk_main();

    return 0;
}
