/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <guichaz@yahoo.fr>
 */

/*******************
 * The Open dialog *
 *******************/

#include <glob.h>               /* glob_t, glob(), GLOB_NOSORT */
#include <gdk/gdkkeysyms.h>     /* GDK_Escape */

#include "gliv.h"
#include "open_dialog.h"
#include "mnemonics.h"
#include "messages.h"
#include "next_image.h"
#include "files_list.h"
#include "windows.h"

static gchar *saved_path = NULL;
static gboolean shuffle_flag = FALSE;
static gboolean dir_flag = TRUE;

static gboolean toggle_flag(GtkToggleButton * button, gboolean * flag)
{
    *flag = gtk_toggle_button_get_active(button);
    return FALSE;
}

/* Shuffle button or add dir button. */
static void add_button(GtkFileSelection * dialog, const gchar * label,
                       gboolean * value)
{
    GtkCheckButton *button;

    label = add_mnemonic(label);
    button = GTK_CHECK_BUTTON(gtk_check_button_new_with_mnemonic(label));

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), *value);

    g_signal_connect(button, "toggled", G_CALLBACK(toggle_flag), value);

    gtk_container_add(GTK_CONTAINER(dialog->action_area), GTK_WIDGET(button));
}

/* A GtkFileSelection with two buttons. */
static GtkFileSelection *decorated_dialog(void)
{
    GtkFileSelection *dialog;
    const gchar *label;

    label = _("GLiv: Select files or use wildcards");
    dialog = GTK_FILE_SELECTION(gtk_file_selection_new(label));

    if (saved_path != NULL)
        gtk_file_selection_set_filename(dialog, saved_path);

    gtk_file_selection_set_select_multiple(dialog, TRUE);

    add_button(dialog, _("Shuffle"), &shuffle_flag);
    add_button(dialog, _("All files in the directory"), &dir_flag);

    return dialog;
}

static void single_selection(GtkFileSelection * dialog)
{
    glob_t pglob;

    /* Use what is in the entry. */
    glob(gtk_file_selection_get_filename(dialog), GLOB_NOSORT, NULL, &pglob);

    if (pglob.gl_pathc == 1)
        /* No wildcard expansion. */
        open_file(pglob.gl_pathv[0], dir_flag, shuffle_flag);

    else {
        /* Add all expansions from glob(). */
        gint nb_inserted;

        nb_inserted = insert_after_current(pglob.gl_pathv, pglob.gl_pathc,
                                           FALSE, shuffle_flag, !shuffle_flag,
                                           TRUE);
        if (nb_inserted > 0)
            open_next_image(nb_inserted == 1);
    }

    globfree(&pglob);
}

static gboolean process_selection(GtkFileSelection * dialog)
{
    gchar **array;
    gint nb;

    gtk_widget_hide(GTK_WIDGET(dialog));
    g_free(saved_path);
    saved_path = g_strdup(gtk_file_selection_get_filename(dialog));

    array = gtk_file_selection_get_selections(dialog);

    /* Count selected files. */
    for (nb = 0; array[nb] != NULL; nb++);

    if (nb == 1)
        single_selection(dialog);

    else {
        gint nb_inserted;

        nb_inserted = insert_after_current(array, nb, FALSE,
                                           shuffle_flag, !shuffle_flag, TRUE);
        if (nb_inserted > 0)
            open_next_image(nb_inserted == 1);
    }

    /* array[0] is constant. */
    for (nb = 1; array[nb] != NULL; nb++)
        g_free(array[nb]);

    g_free(array);
    gtk_widget_destroy(GTK_WIDGET(dialog));
    return FALSE;
}

/* Leaves if ESC is pressed. */
static gboolean key_press_event(GtkWidget * widget, GdkEventKey * event)
{
    if (event->keyval == GDK_Escape) {
        gtk_widget_destroy(widget);
        return TRUE;
    }

    return FALSE;
}

static gboolean delete_open_dialog(GtkFileSelection ** dialog)
{
    if (dialog)
        *dialog = NULL;

    return FALSE;
}

/* Called by File->Open (or the 'o' key). */
gboolean menu_open(void)
{
    static GtkFileSelection *dialog = NULL;

    if (dialog)
        return TRUE;

    dialog = decorated_dialog();

    g_signal_connect_swapped(dialog->ok_button, "clicked",
                             G_CALLBACK(process_selection), dialog);

    g_signal_connect_swapped(dialog->cancel_button, "clicked",
                             G_CALLBACK(gtk_widget_destroy), dialog);

    g_signal_connect(dialog, "key-press-event", G_CALLBACK(key_press_event),
                     GINT_TO_POINTER(FALSE));

    g_signal_connect_swapped(dialog, "delete-event",
                             G_CALLBACK(delete_open_dialog), &dialog);

    g_signal_connect_swapped(dialog, "destroy",
                             G_CALLBACK(delete_open_dialog), &dialog);

    show_dialog(GTK_WIDGET(dialog), NULL);
    return TRUE;
}
