/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <guichaz@yahoo.fr>
 */

/*******************
 * The Open dialog *
 *******************/

#include "gliv.h"
#include "open_dialog.h"
#include "mnemonics.h"
#include "messages.h"
#include "next_image.h"
#include "files_list.h"
#include "windows.h"

static gchar *saved_path = NULL;
static gboolean shuffle_flag = FALSE;

static gboolean toggle_shuffle_flag(GtkToggleButton * button)
{
    shuffle_flag = gtk_toggle_button_get_active(button);
    return FALSE;
}

static GtkWidget *make_extra_widget(void)
{
    GtkCheckButton *button;
    const gchar *label;

    label = add_mnemonic(_("Shuffle"));
    button = GTK_CHECK_BUTTON(gtk_check_button_new_with_mnemonic(label));

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), shuffle_flag);

    g_signal_connect(button, "toggled", G_CALLBACK(toggle_shuffle_flag), NULL);

    return GTK_WIDGET(button);
}

/* A GtkFileChooser with one check box */
static GtkFileChooser *decorated_dialog(void)
{
    GtkFileChooser *chooser;
    const gchar *label;

    label = _("GLiv: Select files to open");
    chooser =
        GTK_FILE_CHOOSER(gtk_file_chooser_dialog_new
                         (label, get_current_window(),
                          GTK_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL,
                          GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN,
                          GTK_RESPONSE_ACCEPT, NULL));

    if (saved_path != NULL)
        gtk_file_chooser_set_current_folder(chooser, saved_path);

    gtk_file_chooser_set_select_multiple(chooser, TRUE);

    gtk_file_chooser_set_extra_widget(chooser, make_extra_widget());

    return chooser;
}

static void load_selection(GtkFileChooser * chooser)
{
    GSList *selection = gtk_file_chooser_get_filenames(chooser);
    gint nb = g_slist_length(selection);
    gchar **filenames = g_new(gchar *, nb + 1);
    GSList *list_ptr = selection;
    gchar **tab_ptr = filenames;
    gint nb_inserted;

    while (list_ptr != NULL) {
        *tab_ptr = list_ptr->data;
        tab_ptr++;
        list_ptr = list_ptr->next;
    }

    *tab_ptr = NULL;

    nb_inserted = insert_after_current(filenames, nb, FALSE,
                                       shuffle_flag, !shuffle_flag, TRUE);

    if (nb_inserted > 0)
        open_next_image(nb_inserted == 1);

    g_strfreev(filenames);
    g_slist_free(selection);
}

/* Called by File->Open (or the 'o' key). */
gboolean menu_open(void)
{
    GtkFileChooser *dialog;
    gint response;

    dialog = decorated_dialog();
    response = run_modal_dialog(GTK_DIALOG(dialog));

    if (response == GTK_RESPONSE_ACCEPT) {
        g_free(saved_path);
        saved_path = gtk_file_chooser_get_current_folder(dialog);
        load_selection(dialog);
    }

    gtk_widget_destroy(GTK_WIDGET(dialog));

    return FALSE;
}
