/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * buffer.c
 *
 * Buffer management code.
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-packages/nexus/source/nexus/buffer.c,v 1.62 2006/01/19 05:57:05 mlink Exp $";

#include "internal.h"

#define DEFAULT_BASE_SEGMENT_SIZE	4096
#define DEFAULT_DIRECT_SEGMENT_SIZE	128

/* TODO: This value should really be calculated */
#define FIXED_HEADER_SIZE		64
#define DEFAULT_RESERVED_HEADER_SIZE	(NEXUS_MAX_TRANSFORM_INFO_SIZE \
					 + NEXUS_MAX_LIBA_SIZE \
					 + FIXED_HEADER_SIZE)
#define RESERVED_HEADER_SIZE_INCREMENT	64
static unsigned long default_reserved_header_size;

#define MALLOC_ALIGNMENT		sizeof(double)

#ifndef BUILD_LITE
typedef struct _threaded_handler_startup_info_t
{
    nexus_context_t *			context;
    nexus_endpoint_t *			endpoint;
#ifdef BUILD_PROFILE
    int					node_id;
    int					context_id;
    int					message_length;
#endif	    
    nexus_buffer_t			buffer;
    nexus_handler_func_t		func;
} threaded_handler_startup_info_t;

static
void *
threaded_handler_startup(
    void *                                  arg);
    
static void                   threaded_handler_startup_wakeup(void *arg);
#endif /* BUILD_LITE */

static int			sizeof_base_segment;
static int			sizeof_direct_segment;
static unsigned long		default_base_segment_size;
static unsigned long		default_direct_segment_size;

static nexus_mutex_t		freelists_mutex;
static nexus_buffer_t		buffer_freelist_head;
static nexus_base_segment_t *	base_freelist_head;
static nexus_direct_segment_t *	direct_freelist_head;

#define LockFreelists() \
    nexus_mutex_lock(&freelists_mutex)
#define UnlockFreelists() \
    nexus_mutex_unlock(&freelists_mutex)

static nexus_buffer_t buffer_alloc();
static void buffer_free(nexus_buffer_t buffer);
static void buffer_dispatch_non_threaded(struct globus_nexus_buffer_s *buffer,
					 globus_nexus_endpoint_t *endpoint,
					 nexus_handler_func_t handler);
static nexus_base_segment_t *base_segment_malloc(unsigned long size);
static nexus_direct_segment_t *direct_segment_malloc(unsigned long size);
static void base_segment_alloc(nexus_buffer_t buffer,
			       unsigned long size);
static void base_segments_free(nexus_base_segment_t *base_segments);
static void direct_segment_alloc(nexus_buffer_t buffer,
				 unsigned long size);
static void direct_segments_free(nexus_direct_segment_t *direct_segments);
static void call_nexus_unknown_handler(nexus_endpoint_t *endpoint,
				       struct globus_nexus_buffer_s *buffer,
				       int handler_id);


		    
/*
 * _nx_buffer_new_process_params()
 */
int _nx_buffer_new_process_params(char *buf, int size)
{
    return (0);
} /* _nx_buffer_new_process_params() */


/*
 * _nx_buffer_init()
 */
void _nx_buffer_init(void)
{
    /*
     * TODO: Add command line arguments to
     * change default_*_segment_size
     */

    if(_nx_tcp_mim_msg_size < DEFAULT_BASE_SEGMENT_SIZE)
    {
        default_base_segment_size = DEFAULT_BASE_SEGMENT_SIZE;
    }
    else
    {
        default_base_segment_size = _nx_tcp_mim_msg_size;
    }

    default_direct_segment_size = DEFAULT_DIRECT_SEGMENT_SIZE;

    default_reserved_header_size = DEFAULT_RESERVED_HEADER_SIZE;
    
    sizeof_base_segment = sizeof(nexus_base_segment_t);
    if (sizeof_base_segment % MALLOC_ALIGNMENT != 0)
    {
	sizeof_base_segment += (MALLOC_ALIGNMENT
				- (sizeof_base_segment % MALLOC_ALIGNMENT));
    }
    sizeof_direct_segment = sizeof(nexus_direct_segment_t);
    if (sizeof_direct_segment % MALLOC_ALIGNMENT != 0)
    {
	sizeof_direct_segment += (MALLOC_ALIGNMENT
				- (sizeof_direct_segment % MALLOC_ALIGNMENT));
    }

    nexus_mutex_init(&freelists_mutex, (nexus_mutexattr_t *)NULL);
    buffer_freelist_head = NULL;
    base_freelist_head = NULL;
    direct_freelist_head = NULL;
    
} /* _nx_buffer_init() */


/*
 * _nx_buffer_shutdown()
 */
void _nx_buffer_shutdown(void)
{
    nexus_buffer_t buffer;
    nexus_base_segment_t *buf_segment;
    nexus_direct_segment_t *direct_seg;
    
    nexus_mutex_destroy(&freelists_mutex);

    for (buffer = buffer_freelist_head;
	 buffer;
	 buffer_freelist_head = buffer)
    {
        buffer = buffer_freelist_head->next;
	NexusFree(buffer_freelist_head);
    }

    for (buf_segment = base_freelist_head;
	 buf_segment;
	 base_freelist_head = buf_segment)
    {
        buf_segment = base_freelist_head->next;
	if (base_freelist_head->size > 0)
	{
	    NexusFree(base_freelist_head->storage);
	}
	NexusFree(base_freelist_head);
    }

    for (direct_seg = direct_freelist_head;
	 direct_seg;
	 direct_freelist_head = direct_seg)
    {
        direct_seg = direct_freelist_head->next;
	NexusFree(direct_freelist_head);
    }
} /* _nx_buffer_shutdown() */


/*
 * nexus_buffer_init()
 */
int nexus_buffer_init(nexus_buffer_t *buffer,
		      unsigned long buffer_size,
		      unsigned long num_direct_puts)
{
    LockFreelists();
    *buffer = buffer_alloc();
    base_segment_alloc(*buffer,
		       (buffer_size + default_reserved_header_size));
    if (num_direct_puts > 0)
    {
        direct_segment_alloc(*buffer,
			     num_direct_puts);
    }
    UnlockFreelists();

#ifdef BUILD_DEBUG
    (*buffer)->magic = NEXUS_BUFFER_MAGIC;
#endif    
    (*buffer)->reserved_header_size = default_reserved_header_size;
    (*buffer)->format = GLOBUS_DC_FORMAT_LOCAL;

    /*
     * Set the buffers saved_state so that nexus_buffer_destroy()
     * will actually free up the memory assocaiated with this
     * buffer, either before or after a nexus_send_rsr is called.
     */
    (*buffer)->saved_state = GLOBUS_NEXUS_BUFFER_SAVED_STATE_SENDING;

    (*buffer)->current_base_segment->current
	+= (*buffer)->reserved_header_size;

    return(0);
} /* nexus_buffer_init() */


/*
 * nexus_buffer_reset()
 */
int nexus_buffer_reset(nexus_buffer_t *buffer,
		       unsigned long buffer_size,
		       unsigned long num_direct_puts)
{
    if (   !(*buffer)->base_segments
	|| (*buffer)->base_segments->next
	|| (buffer_size + default_reserved_header_size
	    > (*buffer)->base_segments->size)
	|| (   (num_direct_puts > 0)
	    && (   !(*buffer)->direct_segments
		|| (*buffer)->direct_segments->next
		|| (num_direct_puts > (*buffer)->direct_segments->size) ) ) )
    {
	int rc;
	nexus_buffer_destroy(buffer);
	rc = nexus_buffer_init(buffer,
			       buffer_size,
			       num_direct_puts);
	return(rc);
    }

    if (   (num_direct_puts <= 0)
	&& (*buffer)->direct_segments)
    {
	LockFreelists();
	direct_segments_free((*buffer)->direct_segments);
	UnlockFreelists();
	(*buffer)->direct_segments = NULL;
	(*buffer)->current_direct_segment = NULL;
    }
    
#ifdef BUILD_DEBUG
    (*buffer)->magic = NEXUS_BUFFER_MAGIC;
#endif
    (*buffer)->proto = NULL;
    (*buffer)->next = NULL;
    (*buffer)->reserved_header_size = default_reserved_header_size;
    (*buffer)->format = GLOBUS_DC_FORMAT_LOCAL;
    (*buffer)->saved_state = GLOBUS_NEXUS_BUFFER_SAVED_STATE_SENDING;
    (*buffer)->n_direct = 0;
    (*buffer)->current_base_segment = (*buffer)->base_segments;
    (*buffer)->current_base_segment->current
	+= (*buffer)->reserved_header_size;
    return(0);
} /* nexus_buffer_reset() */


/*
 * nexus_buffer_destroy()
 */
int nexus_buffer_destroy(nexus_buffer_t *buffer)
{
    LockFreelists();
    switch ((*buffer)->saved_state)
    {
    case GLOBUS_NEXUS_BUFFER_SAVED_STATE_UNSAVED:
    case GLOBUS_NEXUS_BUFFER_SAVED_STATE_SAVED_IN_HANDLER:
	/*
	 * This is being called from a non-threaded handler.
	 * So just mark the buffer as freed, and let the buffer
	 * destruction happen after the non-threaded handler returns.
	 */
	(*buffer)->saved_state = GLOBUS_NEXUS_BUFFER_SAVED_STATE_FREED;
        break;
    case GLOBUS_NEXUS_BUFFER_SAVED_STATE_SAVED:
    case GLOBUS_NEXUS_BUFFER_SAVED_STATE_SENDING:
	/* This is being called from a thread, or from a sending side
	 * after the send has completed.  So free the buffer.  */
	direct_segments_free((*buffer)->direct_segments);
	base_segments_free((*buffer)->base_segments);
	buffer_free(*buffer);
	break;
    case GLOBUS_NEXUS_BUFFER_SAVED_STATE_FREED:
	/*
	 * This buffer is already freed, so do nothing.
	 */
	break;
    }
    UnlockFreelists();
    return(0);
} /* nexus_buffer_destroy() */


/*
 * nexus_check_buffer_size()
 *
 * Return:
 *	0 if there is enough room in the buffer (perhaps after resizing)
 *	-1 if malloc fails
 *	-2 if there is not enough room, but the increment is 0
 */
int nexus_check_buffer_size(nexus_buffer_t *buffer,
			    int size_needed,
			    int size_increment,
			    int num_direct_puts_needed,
			    int num_direct_puts_increment)
{
    /* Add a base segment if necessary */
    if (   (size_needed > 0)
	&& (   (!((*(buffer))->current_base_segment))
	    || (((*(buffer))->current_base_segment->current + size_needed)
		  > ((*(buffer))->current_base_segment->storage
		     + (*(buffer))->current_base_segment->size) ) ) )
    {
        int real_size;

        if (size_increment <= 0)
	{
	    return(-2);
	}

	for (real_size = 0;
	     real_size < size_needed;
	     real_size += size_increment)
	  /* do nothing */ ;
	
	LockFreelists();
        base_segment_alloc(*buffer,
			   real_size);
	UnlockFreelists();
    }
    
    /* Add a direct segment if necessary */
    if (   (num_direct_puts_needed > 0)
	&& (   (!((*(buffer))->current_direct_segment))
	    || (((*(buffer))->current_direct_segment->current
		 + num_direct_puts_needed)
		> ((*(buffer))->current_direct_segment->storage
		   + (*(buffer))->current_base_segment->size) ) ) )
    {
        int real_puts;

	if (num_direct_puts_increment <= 0)
	{
	    return(-2);
	}
	
	for (real_puts = 0;
	     real_puts < num_direct_puts_needed;
	     real_puts += num_direct_puts_increment)
	  /* do nothing */ ;
	
	LockFreelists();
        direct_segment_alloc(*buffer,
			     real_puts);
	UnlockFreelists();
    }

    return(0);
} /* nexus_check_buffer_size() */
    

/*
 *  nexus_send_rsr()
 *
 * Take 'in_buffer', coalesce it into a canonical form, add a header
 * to it, and then call the appropriate protocol module.
 *
 * The 'new_buffer' that is passed to the protocol module's send_rsr()
 * function has the following properties:
 *	new_buffer->iovec_formatted is set to NEXUS_TRUE if the contents
 *		of new_buffer->base_segments is a struct iovec
 *	new_buffer->base_segments
 *		Points to either:
 *		- A single segment with the entire base message.
 *		  That message is at:
 *			new_buffer->base_segments->current
 *		  with a size of:
 *			new_buffer->base_segments->size_used
 *		- A single segment containing a struct iovec.
 *		  That iovec is at:
 *			new_buffer->base_segments->current
 *		  with an iocnt of:
 *			new_buffer->base_segments->size_used
 *	new_buffer->current_base_segment = new_buffer->base_segments
 *	new_buffer->base_segments->current
 *		See above.
 *	new_buffer->base_segments->size_used
 *		See above.
 *	new_buffer->reserved_header_size
 *		Set to the number of bytes at the
 *		start of new_buffer->base_segments->storage,
 *		before the body of the message.  The header that is
 *		added to the front of the message body comes out of
 *		this space, though reserved_header_size continues
 *		to be the number of bytes reserved before the message
 *		body.
 *	new_buffer->direct_segments
 *		Points to a single segment which contains
 *		all of the custom direct puts, or NULL if there
 *		are no direct segments.
 *		If new_buffer->iovec_formatted is NEXUS_TRUE, then this
 *		field is always NULL.
 *	new_buffer->n_direct
 *		Set to the number of custom direct puts
 *		contained in new_buffer->direct_segments.
 *		If this value is 0, then send_rsr()
 *		need not wait for the data to be written to the network
 *		before returning.
 *	new_buffer->proto
 *		Set to the nexus_proto_t from the startpoint
 *		to which this buffer is being sent.
 *	new_buffer->using_barrier = NEXUS_FALSE
 *		If the protocol want nexus_send_rsr() to wait for
 *		completion of the sent, it should change this
 *		value to NEXUS_TRUE, initialize new_buffer->barrier,
 *		and signal the barrier when the send has completed.
 *		Otherwise, the protocol module just call
 *		nexus_buffer_destroy() on the buffer after the
 *		send has completed.
 *
 * Input values are:
 *	in_buffer:
 *		The buffer to be sent.
 *	startpoint:
 *		The startpoint to which this message is being sent.
 *	handler_id:
 *		The handler_id to which this message is	being sent.
 *	destroy_buffer:
 *		If this is NEXUS_TRUE, then 'buffer' will be destroyed
 *		immediately after this send.  In that case, new_buffer
 *		can simply be set to point to in_buffer, rather than
 *		allocating a whole new buffer.
 *	called_from_non_threaded_handler:
 *		Is this call being made from a non-threaded handler?
 *		If so, then this function will copy all data so that
 *		there are no direct componenents.  This allows the
 *		send_rsr() to complete immediately.
 *
 * The message layout is standardized for use by all protocol modules
 * as follows:
 *
 *   Header:
 *       protocol version number     1 byte u_int
 *       senders_data_format         1 byte u_int
 *                                     (<128 for format,
 *                                      >=128 for signaling)
 *       full_message_size           1 u_long
 *       liba_size                   1 byte u_int
 *       liba[0..liba_size]          byte array of size liba_size
 *   Transform Info:
 *       transform_info              (optional, size depends on transform)
 *   Message Info:
 *       flags                       1 byte
 *           has_direct_info            :1
 *           has_other_info             :1
 *           pad_size                   :3
 *       pad[0..pad_size]            (pad Header + Message Info size
 *                                    to NEXUS_ALIGN_MESSAGE_SIZE byte
 *                                    multiple)
 *       handler_id                  1 int
 *       if (has_direct_info)
 *           direct_info_offset      1 u_long
 *       endif
 *       if (has_other_info)
 *           other_info_offset       1 u_long
 *       endif
 *   Message Body:
 *       message_body
 *   Other Info:
 *       if (has_other_info)
 *           loop
 *               info_tag (DEBUG_TAG, PROFILE_TAG)
 *               info_size (not including tag and size header)
 *               info
 *               (examples:
 *                   if (info_tag==PROFILE_TAG)
 *                       info_size=2*sizeof(int)
 *                       info=
 *                           node_id
 *                           context_id
 *                   endif
 *                   if (info_tag==DEBUG_TAG)
 *                       info_size=???
 *                       info=
 *                           n_debug_tags
 *                       for (0..n_debug_tags)
 *                           size
 *                           datatype
 *                       endfor
 *                   endif
 *           endloop
 *       endif (has_other_info)
 *   Direct Info:
 *       if (has_direct_info)
 *           n_direct_components                1 u_int
 *           protocol_specific_info_size        1 u_int
 *           protocol_specific_info[0..protocol_specific_info_size]
 *           for (0..n_direct_components)
 *               component_size                 1 u_long
 *               component_datatype             1 byte u_int
 *               component_approach             1 byte u_int
 *               switch(component_approach)
 *                 case INLINE==0:
 *                   component_offset           1 u_long
 *                 case POINTER==1:
 *                   component_pointer          1 u_long
 *                 default:
 *                   protocol_specific_info     1 u_long
 *           endfor
 *       endif
 *
 * Comments about the message format:
 *   - Each *_offset value is a byte offset from the beginning of
 *     the message body.
 *   - A message contains five parts:
 *         header, message info, message body, other info, direct info.
 *     Only the header and message info are manditory
 *   - Everything except the message header may be
 *     transformed (i.e. encrypted, compressed, etc).
 *     Therefore, the header must contain enough information
 *     to allow a receiving context to transform the message, namely:
 *         liba: This is needed to allow the receiving context
 *           to find the endpoint to which this message is directed.
 *           The endpoint contains the transform operation that
 *           needs to be applied to the message, plus any state
 *           required to perform the transform (i.e. a key to decrypt).
 *         transform_info: If the transform operation requires
 *           information from the sending context, then this
 *           information is included in the message header in
 *           the transform_info. For example, an MD5 hash value
 *           may be included here.
 *   - The senders_data_format is defined such that values >=128
 *     are free for use by the protocol module.  So all normal data
 *     messages, as created by nexus_send_rsr(), will have a first
 *     byte with the buffer version number, and a second bye with
 *     a value <128.  If the protocol module wants to send other
 *     non-data messages, it can signal these other messages by
 *     using a senders_data_format of >=128.  This allows a protocol
 *     to avoid adding yet another byte to the message header to
 *     distinguish these cases.
 *   - On many machines it is desirable to keep the message body
 *     aligned on a word boundary. In order to do this, the size
 *     of the header plus message info is always padded out to
 *     an 8 byte boundary (NEXUS_ALIGN_MESSAGE_SIZE).
 *     Therefore, by aligning the whole
 *     message buffer, the message body will be guaranteed to be
 *     aligned. The size of padding needed to align the message
 *     body is contained in the flags.pad_size field.
 *   - All of the other information (debug tags, profile info, etc)
 *     is all kept at the end of the message, and is marked by a
 *     single bit in the message header. This allows the optimized
 *     version to deal with (i.e. ignore) these various other pieces
 *     of information with a single check to see if it needs to
 *     skip the other_info_offset field in the header. But it
 *     allows for an extensible set of additional information
 *     to be added to a message.
 *   - The direct information is protocol specific. For example,
 *     the MP protocol module will say how many subsequent messages
 *     there are and their sizes, whereas the shared memory protocol
 *     module may just keep a list of sizes and addresses that it
 *     can grab the data from.
 *   - In the fully optimized case (32-bit cpu, no profile, no debug,
 *     no direct components, no transform, no security), a message
 *     would have a 16 byte header followed by the message body:
 *         byte  0      protocol version number
 *         byte  1      senders_data_format
 *         bytes 2-5    full_message_size
 *         byte  6      liba_size
 *         bytes 7-10   liba
 *         byte  11     flags (has_direct_info=0, has_other_info=0, pad_size=0)
 *         bytes 12-15  handler_id
 *         bytes 16-N   message_body
 *
 * Notes:
 *   - Most machines have an iovec limit of 16
 */
int nexus_send_rsr(nexus_buffer_t *in_buffer,
		   nexus_startpoint_t *startpoint,
		   int handler_id,
		   nexus_bool_t destroy_buffer,
		   nexus_bool_t called_from_non_threaded_handler)
{
    struct globus_nexus_buffer_s *	buffer;
    struct globus_nexus_buffer_s *	new_buffer;
    nexus_base_segment_t *	base_segments;
    nexus_direct_segment_t *	direct_segments;
    unsigned long		total_direct_puts;
    nexus_byte_t *		body_start;
    unsigned long		body_size;
    unsigned long		total_size;
    unsigned long		size;
    nexus_byte_t *		b;
    unsigned long		header_size;
    unsigned long		message_info_size;
    unsigned long		total_header_size;
    unsigned long		pad_size;
    unsigned long		required_header_size;
    nexus_bool_t		freelists_locked;
    nexus_base_segment_t *	bseg;
    nexus_direct_segment_t *	dseg;
    int				transform_id;
    nexus_bool_t		transform_modifies_data;
    unsigned long		transform_info_size;
    unsigned long		transform_trailer_size;
    nexus_proto_t *		proto;
    int				rc;
    int				i;
    nexus_bool_t		has_other_info;
    unsigned long		other_info_offset;
    unsigned long		direct_info_offset;
    nexus_byte_t		tmp_byte;
    nexus_byte_t *		liba;

    NexusBufferMagicCheck(nexus_send_rsr, in_buffer);
    buffer = *in_buffer;
    
    nexus_debug_printf(2,
		       ("nexus_send_rsr(): invoked with buffer: %x\n",buffer));


    /*
     * Verify that the buffer is not being re-sent
     */
    if(buffer->saved_state != GLOBUS_NEXUS_BUFFER_SAVED_STATE_SENDING)
    {
	return GLOBUS_NEXUS_ERROR_UNIMPLEMENTED;
    }

    /*
     * Verify that the startpoint is reasonable
     */
    if (startpoint == NULL
	|| startpoint->mi_proto == NULL
	|| startpoint->mi_proto->proto == NULL)
    {
	if (destroy_buffer)
	{
	    nexus_buffer_destroy(&buffer);
	}

	return(-1);
    }

    /*
     * Stash all of contents of the buffer into base_segments and
     * direct_segments.  These will be used to construct new_buffer,
     * with the properties described in this functions header comment.
     */
    base_segments = buffer->base_segments;
    direct_segments = buffer->direct_segments;
    proto = startpoint->mi_proto->proto;

    /*
     * Figure out what to do with each of the direct components
     * of the message, based on its size and on the properties
     * of the protocol module (as supplied in proto).
     *
     * The proto structure supplies us with two
     * ranges:
     *   direct_custom_{min,max}_size:
     *       Range of sizes for which the protocol will do custom
     *       messaging.
     *   direct_pointer_{min,max}_size:
     *       Range of sizes for which a just a pointer to the
     *       direct component needs to be stored in the message.
     *       In other words, the receiving end can make use of
     *       such pointers to get the data directly from the
     *       sending buffer.
     * These values can be set to NEXUS_DC_MAX_U_LONG to effectively
     * mean infinity.
     * If the size of the direct component does not fall within either
     * of these ranges, then that direct component will be inlined into
     * the message.  That is, it will be copied into the base_segments
     * buffer.
     */
    total_direct_puts = 0;
    for (dseg = direct_segments;
	 dseg;
	 dseg = dseg->next)
    {
	/*
	 * TODO: Stare at this more.  I think we need to count
	 * how many custom vs non-custom puts there are.
	 */
        for (i = 0; i < dseg->size; i++)
	{
	    if (   (dseg->storage[i].size >= proto->direct_custom_min_size)
		&& (dseg->storage[i].size <= proto->direct_custom_max_size) )
		
	    {
	        dseg->storage[i].action
		    = NEXUS_DIRECT_INFO_ACTION_CUSTOM;
	    }
	    else
	    if (   (dseg->storage[i].size >= proto->direct_pointer_min_size)
		&& (dseg->storage[i].size <= proto->direct_pointer_max_size) )
		
	    {
	        dseg->storage[i].action
		    = NEXUS_DIRECT_INFO_ACTION_POINTER;
	    }
	    else
	    {
	        dseg->storage[i].action
		    = NEXUS_DIRECT_INFO_ACTION_INLINE;
	    }
	}
	total_direct_puts += dseg->size;
    }

    has_other_info = NEXUS_FALSE;
    other_info_offset = 0;

    if (destroy_buffer)
    {
	/* Can use buffer without copying it */
	new_buffer = buffer;
	new_buffer->base_segments = (nexus_base_segment_t *) NULL;
	new_buffer->direct_segments = (nexus_direct_segment_t *) NULL;
	freelists_locked = NEXUS_FALSE;
    }
    else
    {
	/* Must create new buffer */
	LockFreelists();
	freelists_locked = NEXUS_TRUE;
	new_buffer = buffer_alloc();
	new_buffer->saved_state = GLOBUS_NEXUS_BUFFER_SAVED_STATE_SENDING;
    }

    /*
     * Get the information about the transform.
     *   transform_modifies_data: True if the transform may modify the data.
     *   transform_info_size: Number of bytes that must be reserved
     *		in the message header, immediately before the transformed
     *		data.  The transform can put whatever it wants here,
     *		such as a MAC.
     *   transform_trailer_size: This is the number of unused bytes
     *		that the transform would like at the end of the message.
     *		This is purely advisory.  That is, the transform module
     *		must be able to deal with a buffer that does not have
     *		these unused bytes at the end.  However, if new
     *		space must be allocated when coalescing the buffer,
     *		then, we should go ahead and leave this many unused
     *		bytes at the end.
     */
    transform_id = startpoint->transform_id;
    if (transform_id != NEXUS_TRANSFORM_NONE)
    {
	nexus_transform_info(startpoint->transform_id,
			     &transform_modifies_data,
			     &transform_info_size,
			     &transform_trailer_size);
    }
    else
    {
	transform_modifies_data = NEXUS_FALSE;
	transform_info_size = 0;
	transform_trailer_size = 0;
    }

    /*
     * Calculate the sizes of the various message header components:
     *     header_size
     *   + transform_info_size
     *   + message_info_size (- pad_size)
     *   -----------------------------
     *   = total_header_size
     *   + pad_size (round total_header_size up to next
     *               NEXUS_ALIGN_MESSAGE_SIZE byte boundary)
     *   -----------------------------
     *   = total_header_size
     *   + proto->reserved_header_size
     *   -----------------------------
     *   = required_header_size
     */
    header_size
	= (nexus_dc_sizeof_byte(1) * 3	/* format, version, liba_size */
	   + nexus_dc_sizeof_u_long(1)	/* full_message_size */
	   + startpoint->liba_size);	/* liba */
    message_info_size
	= (nexus_dc_sizeof_byte(1)		/* flags */
	   + nexus_dc_sizeof_int(1)		/* handler_id */
	   + (total_direct_puts > 0 ? nexus_dc_sizeof_u_long(1) : 0)
	   + (has_other_info ? nexus_dc_sizeof_u_long(1) : 0) );
    total_header_size = (header_size
			 + transform_info_size
			 + message_info_size);
    pad_size = ((total_header_size % NEXUS_ALIGN_MESSAGE_SIZE)
		? (NEXUS_ALIGN_MESSAGE_SIZE
		   - (total_header_size % NEXUS_ALIGN_MESSAGE_SIZE))
		: (0));
    message_info_size += pad_size;
    total_header_size += pad_size;
    required_header_size = total_header_size + proto->reserved_header_size;
    
    if (total_direct_puts == 0)
    {
	/*
	 * There are no direct puts in this buffer.
	 * So just do the necessary work to consolidate the
	 * message body into a single base_segment.
	 */
	if (base_segments->next == (nexus_base_segment_t *) NULL)
	{
	    /*
	     * All of the base data is in the first segment.
	     * So figure out if we can use this base segment directly,
	     * or if it needs to be copied into a new buffer.
	     */
	    body_start = (base_segments->storage
			  + buffer->reserved_header_size);
	    body_size = base_segments->current - body_start;

	    if (   destroy_buffer
		&& (buffer->reserved_header_size >= required_header_size) )
	    {
		/*
		 * There is enough room for the header, and
		 * this is the last time this buffer will be used.
		 * So we can use base segment without copying it.
		 */
		new_buffer->iovec_formatted = NEXUS_FALSE;
		new_buffer->base_segments = base_segments;
		new_buffer->current_base_segment = base_segments;
		new_buffer->base_segments->current = body_start;
		new_buffer->base_segments->size_used = body_size;
		new_buffer->reserved_header_size
		    = buffer->reserved_header_size;
		new_buffer->direct_segments = (nexus_direct_segment_t *) NULL;
		new_buffer->n_direct = 0;
	    }
	    else
	    {
		/*
		 * There is not enough room for the header, or
		 * the buffer may be reused in another nexus_send_rsr().
		 * Therefore we must copy the base segment.
		 */

		/*
		 * Adjust the default_reserved_header_size for future sends
		 * to be big enough for this header, so as to hopefully
		 * avoid this copy in the future.
		 */
		while (required_header_size > default_reserved_header_size)
		{
		    default_reserved_header_size
			+= RESERVED_HEADER_SIZE_INCREMENT;
		}
		
		/*
		 * Allocate the required space.
		 * This sets:
		 *	new_buffer->base_segments
		 *	new_buffer->current_base_segment
		 */
		base_segment_alloc(new_buffer,
				   (body_size
				    + default_reserved_header_size
				    + transform_trailer_size));

		/* Setup new_buffer correctly */
		new_buffer->iovec_formatted = NEXUS_FALSE;
		new_buffer->reserved_header_size
		    = default_reserved_header_size;
		new_buffer->base_segments->current
		    = (new_buffer->base_segments->storage
		       + new_buffer->reserved_header_size);
		new_buffer->base_segments->size_used = body_size;
		new_buffer->direct_segments = (nexus_direct_segment_t *) NULL;
		new_buffer->n_direct = 0;

		/* Copy the data into the new base segment */
		memcpy(new_buffer->base_segments->current,
		       body_start,
		       body_size);
		
		/* Free the old base segment if necessary */
		if (destroy_buffer)
		{
		    if (!freelists_locked)
		    {
			LockFreelists();
			freelists_locked = NEXUS_TRUE;
		    }
		    base_segments_free(base_segments);
		}
	    }
	}
	else
	{
	    /*
	     * The base data is spread across multiple segments.
	     * So consolidate it to a single segment.
	     */

	    /* Calculate the total number of bytes in the base segments */
	    body_start = (base_segments->storage
			  + buffer->reserved_header_size);
	    body_size = base_segments->current - body_start;
	    total_size = body_size;
	    for (bseg = base_segments->next;
		 bseg;
		 bseg = bseg->next)
	    {
		total_size += (bseg->current - bseg->storage);
	    }

	    /*
	     * Allocate a new base segment to hold all of the data.
	     * This sets:
	     *	new_buffer->base_segments
	     *	new_buffer->current_base_segment
	     */
	    base_segment_alloc(new_buffer,
			       (total_size
				+ default_reserved_header_size
				+ transform_trailer_size));
	    
	    /* Setup new_buffer correctly */
	    new_buffer->iovec_formatted = NEXUS_FALSE;
	    new_buffer->reserved_header_size
		= default_reserved_header_size;
	    new_buffer->base_segments->current
		= (new_buffer->base_segments->storage
		   + new_buffer->reserved_header_size);
	    new_buffer->base_segments->size_used = total_size;
	    new_buffer->direct_segments = (nexus_direct_segment_t *) NULL;
	    new_buffer->n_direct = 0;

	    /* Copy the data into the new base segment */
	    b = new_buffer->base_segments->current;
	    memcpy(b, body_start, body_size);
	    b += body_size;
	    for (bseg = base_segments->next;
		 bseg;
		 bseg = bseg->next)
	    {
		size = (bseg->current - bseg->storage);
		memcpy(b, bseg->storage, size);
		b += size;
	    }

	    /* Free the old base segments if necessary */
	    if (destroy_buffer)
	    {
		if (!freelists_locked)
		{
		    LockFreelists();
		    freelists_locked = NEXUS_TRUE;
		}
		base_segments_free(base_segments);
	    }
	}
    }
    else
    {
	/*
	 * There are direct components in this buffer.
	 * TODO: Fillin this case.
	 */
    }
	
    if (freelists_locked)
    {
	UnlockFreelists();
    }

    /*
     * Add the message info portion of the header to the buffer.
     */
    new_buffer->base_segments->current -= message_info_size;
    new_buffer->base_segments->size_used += message_info_size;
    b = new_buffer->base_segments->current;
    tmp_byte = pad_size;
    if (total_direct_puts > 0)
    {
	tmp_byte |= 0x10;
    }
    if (has_other_info)
    {
	tmp_byte |= 0x08;
    }
    nexus_dc_put_byte(&b, &tmp_byte, 1); /* flags */
    b += pad_size;
    nexus_dc_put_int(&b, &handler_id, 1);
    if (total_direct_puts > 0)
    {
	nexus_dc_put_u_long(&b, &direct_info_offset, 1);
    }
    if (has_other_info)
    {
	nexus_dc_put_u_long(&b, &other_info_offset, 1);
    }

    /*
     * new_buffer looks like this:
     *
     * new_buffer->base_segments->storage
     * |
     * |                      new_buffer->base_segments->current
     * |                                     |
     * v                                     v
     * +--------+-----------+----------------+------+------+--------+
     * | unused | saved for | saved for      | msg  | user | unused |
     * |        | header    | transform info | info | data |        |
     * +--------+-----------+----------------+------+------+--------+
     *
     *                                       |<----v------>|
     *                                             |
     *                        new_buffer->base_segments->size_used
     *
     * |<---- new_buffer->reserved_header_size ---->|
     *
     * |<---------- new_buffer->base_segments->size --------------->|
     *
     *                                       |<-transform->|
     */

    /*
     * Transform the buffer, if necessary
     */
    if (startpoint->transform_id != NEXUS_TRANSFORM_NONE)
    {
	nexus_byte_t *	transform_info_start;
	nexus_bool_t	must_alloc_new_buffer;
	nexus_byte_t *	out_storage_start = (nexus_byte_t *) NULL;
	nexus_byte_t *	out_data_start;
	unsigned long	out_storage_size;
	unsigned long	out_data_size;

	transform_info_start
	    = new_buffer->base_segments->current - transform_info_size;
	must_alloc_new_buffer = ((new_buffer == buffer) && !destroy_buffer);

	nexus_buffer_transform(startpoint,
			       new_buffer->base_segments->storage,
			       new_buffer->base_segments->size,
			       new_buffer->base_segments->current,
			       &(new_buffer->base_segments->size_used),
			       must_alloc_new_buffer,
			       transform_info_start,
			       &out_storage_start,
			       &out_storage_size,
			       &out_data_start,
			       &out_data_size);
	
	if (out_storage_start)
	{
	    nexus_base_segment_t *new_base_segment;
	    
            /*
             * if (out_storage != NULL)
             *
             * out_storage_start
             * |
             * |                              out_data_start
             * |                                     |
             * v                                     v
	     * +--------+-----------+----------------+------+------+--------+
	     * | unused | saved for | saved for      | msg  | user | unused |
	     * |        | header    | transform info | info | data |        |
	     * +--------+-----------+----------------+------+------+--------+
             *
             * |<-- size preserved from new_buffer ->|<-----v----->|
             *                                        out_data_size
             *
             * |<------------------- out_storage_size --------------------->|
             */

	    /* Create a new base segment to hold the transformed data */
	    NexusMalloc(nexus_send_rsr(),
			new_base_segment,
			nexus_base_segment_t *,
			sizeof(nexus_base_segment_t));
	    new_base_segment->next = (nexus_base_segment_t *) NULL;
	    new_base_segment->size = out_storage_size;
	    new_base_segment->size_used = out_data_size;
	    new_base_segment->storage = out_storage_start;
	    new_base_segment->current = out_data_start;
	    new_base_segment->storage_is_inline = NEXUS_FALSE;

	    /* Copy the transform_info over to this new base segment */
	    memcpy((new_base_segment->current - transform_info_size),
		   transform_info_start,
		   transform_info_size);

	    /* Free the old base segment */
	    base_segments_free(new_buffer->base_segments);
	    new_buffer->base_segments = new_base_segment;
	}

	/* Adjust new_buffer to account for the transform info */
	new_buffer->base_segments->current -= transform_info_size;
	new_buffer->base_segments->size_used += transform_info_size;
    }
	
    /*
     * Add the header to the buffer.
     */
    new_buffer->base_segments->current -= header_size;
    new_buffer->base_segments->size_used += header_size;
    b = new_buffer->base_segments->current;
    *b++ = (nexus_byte_t) proto->version;
    *b++ = (nexus_byte_t) GLOBUS_DC_FORMAT_LOCAL;
    nexus_dc_put_u_long(&b, &(new_buffer->base_segments->size_used), 1);
    tmp_byte = startpoint->liba_size;
    nexus_dc_put_byte(&b, &tmp_byte, 1);
    liba = (startpoint->liba_is_inline
	    ? startpoint->liba.array
	    : startpoint->liba.pointer);
    nexus_dc_put_byte(&b, liba, startpoint->liba_size);
    new_buffer->proto = proto;

    /*
     * Clear the endpoint pointer so that attempting to deliver this message
     * using the local protocol results in the the endpoint being obtained
     * from the LIBA.
     */
    new_buffer->endpoint = NULL;
    
    new_buffer->using_barrier = NEXUS_FALSE;
    rc = proto->funcs->send_rsr(new_buffer);

    /*
     * TODO: The new thread model should obviate the need for this.
     */
    if (!called_from_non_threaded_handler)
    {
	nexus_poll();
    }

    /*
     * Wait for completion of the send on new_buffer.
     * It is assumed that if wait_on_buffer_barrier is set to
     * true by the protocol's send_rsr(), then that send_rsr()
     * also initialized the barrier.
     */
    if (new_buffer->using_barrier)
    {
        nexus_mutex_lock(&(new_buffer->barrier.mutex));
	while (new_buffer->barrier.count > 0)
	{
	    nexus_cond_wait(&(new_buffer->barrier.cond),
			    &(new_buffer->barrier.mutex));
	}
        nexus_mutex_unlock(&(new_buffer->barrier.mutex));
        nexus_mutex_destroy(&(new_buffer->barrier.mutex));
        nexus_cond_destroy(&(new_buffer->barrier.cond));
	
	/*
	 * Destroy the buffer.
	 * If new_buffer->using_barrier==NEXUS_FALSE, then
	 * it is the protocol module's responsibility to
	 * destroy the buffer when it is done with it.
	 */
	nexus_buffer_destroy(&new_buffer);
    }
    
    return(rc);

#ifdef DONT_INCLUDE
    /* else (total_direct_puts > 0) */
    
    unsigned long total_size;
    unsigned long total_inline_sizes;
    unsigned long total_custom_puts;
    nexus_bool_t copy_inlines;
    nexus_bool_t copy_pointer;
    nexus_bool_t copy_custom;
    unsigned long required_header_size;
    unsigned long direct_info_size;
    unsigned long required_body_size;
    int iovec_count;
    int i, j;
    unsigned long data_size;
    nexus_byte_t *start_data;
    nexus_byte_t *cur_location;
    unsigned long cur_custom_put;
    nexus_base_segment_t *base_segment;
    nexus_direct_segment_t *direct_segment;
    nexus_byte_t tmp_byte;
    nexus_byte_t *cur_header;
    unsigned long total_message_size;
    nexus_byte_t pad_size;
    nexus_byte_t flags;
    unsigned long direct_info_offset;
    nexus_byte_t *start_transform;
    unsigned long save_header_size;
    unsigned long out_size;
    unsigned long out_header_size;
    unsigned long out_transform_info_size;
    unsigned long out_data_size;
    unsigned long out_untransform_size;
    nexus_bool_t out_must_be_freed;

    required_header_size
	= (nexus_transform_header_size(startpoint->trasform_id)
	   + startpoint->liba_size
	   + nexus_dc_sizeof_byte(1) * 3
	   + nexus_dc_sizeof_u_long(1)
	   + nexus_dc_sizeof_int(1)
	   + (total_direct_puts>0?nexus_dc_sizeof_u_long(1):0)
	   + (has_other_info?nexus_dc_sizeof_u_long(1):0));
    if (required_header_size % 8)
    {
        pad_size = 8 - (required_header_size % 8);
	required_header_size += pad_size;
    }
    else
    {
        pad_size = 0;
    }

    if (total_direct_puts > 0)
    {
	if (   !called_from_non_threaded_handler
	    && can_use_writev
	    && startpoint->transform_id == NEXUS_TRANSFORM_NONE)
	{
	    copy_inlines = NEXUS_FALSE;
	}
	else
	{
	    copy_inlines = NEXUS_TRUE;
	}

	if (   called_from_non_threaded_handler
	    || nexus_transform_modifies_data(startpoint->transform_id))
	{
	    copy_pointer = NEXUS_TRUE;
	    copy_custom = NEXUS_TRUE;
	}
	else
	{
	    copy_pointer = NEXUS_FALSE;
	    copy_custom = NEXUS_FALSE;
	}

	/*
	 * direct_info contains:
	 *
	 * u_int:          n_direct_components, proto_info_size
	 * byte_stream:    proto_info
	 * for (0..n_direct_components)
	 *   u_long:       size, {location,pointer,custom_info}
	 *   byte:         datatype, approach
	 * endfor
	 */
	direct_info_size = (total_direct_puts * (nexus_dc_sizeof_u_long(1) * 2
						 + nexus_dc_sizeof_byte(1) * 2)
			    + nexus_dc_sizeof_u_int(1) * 2
			    + (*buffer)->proto->funcs->direct_info_size());
    }
    else
    {
	direct_info_size = 0;
    }
    
    required_body_size = (total_size
			  + direct_info_size
			  - (*buffer)->reserved_header_size);

    if (   called_from_non_threaded_handler
	|| !can_use_writev)
    {
        (*c_buffer)->use_writev = NEXUS_FALSE;
    }
    else
    {
        (*c_buffer)->use_writev = NEXUS_TRUE;
    }

    total_inline_sizes = 0;
    total_custom_puts = 0;
    iovec_count = 1;
    for (direct_segment = (*buffer)->direct_segments;
	 direct_segment;
	 direct_segment = direct_segment->next)
    {
        for (i = 0; i < direct_segment->size; i++)
	{
	    switch (direct_segment->storage[i].action)
	    {
	      case NEXUS_DIRECT_INFO_ACTION_INLINE:
		if (copy_inlines)
		{
		    total_inline_sizes += direct_segment->storage[i].size;
		}
		else
		{
		    iovec_count++;
		}
		break;
	      case NEXUS_DIRECT_INFO_ACTION_POINTER:
		if (copy_pointer)
		{
		    total_inline_sizes += direct_segment->storage[i].size;
		}
		break;
	      case NEXUS_DIRECT_INFO_ACTION_CUSTOM:
		if (copy_custom)
		{
		    total_inline_sizes += direct_segment->storage[i].size;
		}
		else
		{
		    iovec_count++;
		    total_custom_puts++;
		}
		break;
	      default: /* error */
		break;
	    }
	}
    }
    required_body_size += total_inline_sizes;

    data_size = required_header_size + required_body_size;
    if (   destroy_buffer
	&& required_header_size < (*buffer)->reserved_header_size
	&& (*buffer)->base_segments->size > required_body_size + (*buffer)->reserved_header_size)
    {
        /* use first segment as coalesce buffer */
        start_data = ((*buffer)->base_segments->storage
		      + (*buffer)->reserved_header_size
		      - required_header_size);
    }
    else
    {
        /* get new coalesce buffer */
        NexusMalloc(nexus_send_rsr(),
		    start_data,
		    nexus_byte_t *,
		    data_size);
	cur_location = start_data + required_header_size;
	for (base_segment = (*buffer)->base_segments;
	     base_segment;
	     base_segment = base_segment->next)
	{
	    memcpy(cur_location, base_segment->storage, base_segment->size);
	    cur_location += base_segment->size;
	}
    }
    /*
     assert(we have a contiguous buffer big enough to hold the base message)
     assert(there is room for the header)
     assert(all non-direct info is in buffer)
     */
    
    if (!(*c_buffer)->use_writev)
    {
	(*c_buffer)->u.data.data = start_data;
	(*c_buffer)->u.data.data_size = data_size;
	NexusMalloc(nexus_send_rsr(),
		    (*c_buffer)->u.data.custom_direct_puts,
		    nexus_direct_info_t *,
		    sizeof(nexus_direct_info_t) * total_custom_puts);
	(*c_buffer)->u.data.num_custom_direct_puts = total_custom_puts;
	
	cur_location = start_data + data_size;
	cur_custom_put = 0;
	direct_segment = (*buffer)->direct_segments;
	for (i = 0;
	     i < total_direct_puts;
	     direct_segment = direct_segment->next)
	{
	    for (j = 0; j < direct_segment->size; i++, j++)
	    {
		switch(direct_segment->storage[j].action)
		{
		  case NEXUS_DIRECT_INFO_ACTION_INLINE:
		    /* assert(copy_inlines == NEXUS_TRUE) */
		    memcpy(cur_location,
			    direct_segment->storage[j].data,
			    direct_segment->storage[j].size);
		    cur_location += direct_segment->storage[j].size;
		    break;
		  case NEXUS_DIRECT_INFO_ACTION_POINTER:
		    if (copy_pointer)
		    {
			memcpy(cur_location,
			       direct_segment->storage[j].data,
			       direct_segment->storage[j].size);
			direct_segment->storage[j].data = cur_location;
			cur_location += direct_segment->storage[j].size;
		    }
		    else
		    {
		    }
		    break;
		  case NEXUS_DIRECT_INFO_ACTION_CUSTOM:
		    if (copy_custom)
		    {
			memcpy(cur_location,
			       direct_segment->storage[j].data,
			       direct_segment->storage[j].size);
			cur_location += direct_segment->storage[j].size;
		    }
		    else
		    {
		        memcpy((*c_buffer)->u.data.custom_direct_puts[cur_custom_put++].data,
			       direct_segment->storage[j].data,
			       sizeof(nexus_direct_info_t));
		    }
		    break;
		  default: /* error */
		    break;
		}
	    }
	}
		      
	(*c_buffer)->u.data.num_custom_direct_puts = total_custom_puts;
    }
    else
    {
        /* allocate iovec for writev */
        (*c_buffer)->u.writev.iovec_count = iovec_count;
	NexusMalloc(nexus_send_rsr(),
		    (*c_buffer)->u.writev.iovec,
		    struct iovec *,
		    sizeof(struct iovec) * (*c_buffer)->u.writev.iovec_count);
	/* fill in iovec */
	(*c_buffer)->u.writev.iovec[0].iov_base = (caddr_t)start_data;
	(*c_buffer)->u.writev.iovec[0].iov_len = data_size;
	
	direct_segment = (*buffer)->direct_segments;
	for (i = 0;
	     i < (*c_buffer)->u.writev.iovec_count;
	     direct_segment = direct_segment->next)
	{
	    for (j = 0; j < direct_segment->size; j++)
	    {
	        switch(direct_segment->storage[j].action)
		{
		  case NEXUS_DIRECT_INFO_ACTION_INLINE:
		    /* assert (copy_inline == NEXUS_FALSE) */
		    (*c_buffer)->u.writev.iovec[i].iov_base =
		        direct_segment->storage[j].data;
		    (*c_buffer)->u.writev.iovec[i].iov_len =
		        direct_segment->storage[j].size;
		    i++;
		    break;
		  case NEXUS_DIRECT_INFO_ACTION_POINTER:
		    /* assert (copy_pointer == NEXUS_FALSE) */
		    break;
		  case NEXUS_DIRECT_INFO_ACTION_CUSTOM:
		    /* assert (copy_custom == NEXUS_FALSE) */
		    (*c_buffer)->u.writev.iovec[i].iov_base =
		        direct_segment->storage[j].data;
		    (*c_buffer)->u.writev.iovec[i].iov_len =
		        direct_segment->storage[j].size;
		    i++;
		    break;
		  default: /* error */
		    break;
		}
	    }
	}
    }
    /*
     add direct info to segment
     if should use writev
         fill in writev segment
     else
         copy inlines to segment
     */
#endif /* DONT_INCLUDE */

} /* nexus_send_rsr() */


/*
 * _nx_buffer_create_from_raw()
 *
 * This yields a buffer with the same properties as the one created
 * by nexus_send_rsr().
 *
 * raw_buffer points to the beginning of a malloced buffer, with
 * a size of raw_size.
 *
 * offset_to_message_start is the number of unused bytes at the start
 * of raw_buffer, before the start of the actual message.
 *
 * message_length is the length of the message in the buffer.
 * If message_length==0, this indicates that the message length
 * is not known.  In this case _nx_buffer_dispatch() will
 * grab the size from the message header.
 *
 */
int _nx_buffer_create_from_raw(nexus_byte_t *raw_buffer,
			       unsigned long raw_size,
			       unsigned long offset_to_message_start,
			       unsigned long message_length,
			       nexus_endpoint_t *endpoint,
			       struct globus_nexus_buffer_s **buffer)
{
    LockFreelists();
    *buffer = buffer_alloc();
    UnlockFreelists();
    NexusMalloc(_nx_buffer_create_from_raw(),
		(*buffer)->base_segments,
		nexus_base_segment_t *,
		sizeof(nexus_base_segment_t));
    (*buffer)->base_segments->next = (nexus_base_segment_t *) NULL;
    (*buffer)->base_segments->size = raw_size;
    (*buffer)->base_segments->size_used = message_length;
    (*buffer)->base_segments->storage = raw_buffer;
    (*buffer)->base_segments->current = raw_buffer + offset_to_message_start;
    (*buffer)->base_segments->storage_is_inline = NEXUS_FALSE;

    (*buffer)->current_base_segment = (*buffer)->base_segments;
    (*buffer)->reserved_header_size = offset_to_message_start;
    (*buffer)->direct_segments = (nexus_direct_segment_t *) NULL;
    (*buffer)->n_direct = 0;

    (*buffer)->endpoint = endpoint;

    return(0);
} /* _nx_buffer_create_from_raw() */

/*
 * _nx_buffer_dispatch()
 *
 * TODO: Add checks to make sure the buffer is big enough, has
 * valid data, etc.
 */
int _nx_buffer_dispatch(struct globus_nexus_buffer_s *buffer)
{
    nexus_byte_t *		message_start;
    nexus_byte_t		tmp_byte;
    unsigned long		message_size;
    nexus_bool_t		has_direct_info;
    nexus_bool_t		has_other_info;
    nexus_byte_t		pad_size;
    int				liba_size;
    nexus_byte_t		liba[NEXUS_MAX_LIBA_SIZE];
    nexus_endpoint_t *		endpoint;
    int				handler_id;
    unsigned long		direct_info_offset;
    unsigned long		other_info_offset;
    nexus_handler_func_t	handler;
    unsigned long		ul;
    int version;

    /*
     * Unpack the header
     */
    message_start = buffer->base_segments->current;
    
    /* Skip the protocol version */
    buffer->base_segments->current++;
    
    /* Format */
    buffer->format = (int) (*(buffer->base_segments->current));
    buffer->base_segments->current++;
    
    /* Message size */
    nexus_dc_get_u_long(&(buffer->base_segments->current),
			&message_size, 1, buffer->format);
    if (buffer->base_segments->size_used == 0)
    {
	buffer->base_segments->size_used = message_size;
    }
    else
    {
	NexusAssert2((message_size == buffer->base_segments->size_used), ("_nx_buffer_dispatch(): Buffer size (%li) does not match header size (%li)", buffer->base_segments->size_used, message_size));
    }
    
    /* Liba */
    nexus_dc_get_byte(&(buffer->base_segments->current),
		      &tmp_byte, 1, buffer->format);
    liba_size = (int) tmp_byte;
    NexusAssert2((liba_size <= NEXUS_MAX_LIBA_SIZE), ("_nx_buffer_dispatch(): Liba size (%i) is greater than NEXUS_MAX_LIBA_SIZE (%i)", liba_size, NEXUS_MAX_LIBA_SIZE));
    nexus_dc_get_byte(&(buffer->base_segments->current),
		      liba, liba_size, buffer->format);
    if (buffer->endpoint)
    {
	endpoint = buffer->endpoint;
    }
    else
    {
	if (globus_i_nexus_endpoint_table_lookup(liba, &endpoint)
	    != GLOBUS_SUCCESS)
	{
	    _nx_fault_detected(GLOBUS_NEXUS_ERROR_ENDPOINT_DESTROYED);
	    return(GLOBUS_FAILURE);
	}
    }
    buffer->base_segments->size_used
	-= (buffer->base_segments->current - message_start);
    
    /* Untransform the message */
    if (endpoint->transform_id != NEXUS_TRANSFORM_NONE)
    {
	nexus_bool_t	transform_modifies_data;
	unsigned long	transform_info_size;
	unsigned long	transform_trailer_size;
	nexus_byte_t *	transform_info_start;
	nexus_byte_t *	destination_start;
	unsigned long	destination_size;

	nexus_transform_info(endpoint->transform_id,
			     &transform_modifies_data,
			     &transform_info_size,
			     &transform_trailer_size);

	transform_info_start = buffer->base_segments->current;
	buffer->base_segments->current += transform_info_size;
	buffer->base_segments->size_used -= transform_info_size;

        /*
         * buffer->base_segments->storage
	 * |
	 * |             transform_info_start
         * |                         |
         * |                         |    buffer->base_segments->current
	 * |                         |           |
         * v                         v           v
         * +--------+----------------+-----------+------+------+--------+
         * | unused | format,        | transform | msg  | user | unused |
         * |        | libasize, liba | info      | info | data |        |
         * +--------+----------------+-----------+------+------+--------+
         *
         *                                       |<-----v----->|
	 *                                              |
	 *                               buffer->base_segments->size_used
         *
         * |<-------------- buffer->base_segments->size --------------->|
         *
         *                                       |<-untrnsfrm->|
         */

	destination_start = NULL;
	destination_size = 0;
	
        nexus_buffer_untransform(endpoint,
				 buffer->base_segments->current,
				 &(buffer->base_segments->size_used),
				 transform_info_start,
				 buffer->format,
				 &destination_start,
				 &destination_size);
	if(endpoint->transform_id == NEXUS_TRANSFORM_RTP) { /* transform_rtp */
		int len = 16; 	/*size_transform_info_except_rtphdr();*/
				/* liba(8) + sequence(8) = 16         */
		buffer->base_segments->rtphdr
			= (unsigned char *)transform_info_start + len;
	} else
		buffer->base_segments->rtphdr = NULL;

        if (destination_start)
        {
	    NexusAssert2((!buffer->base_segments->storage_is_inline),
			 ("_nx_buffer_dispatch(): Internal error: Expected non-inline storage to be created by _nx_buffer_create_from_raw()"));
	    NexusFree(buffer->base_segments->storage);
            buffer->base_segments->size = destination_size;
            buffer->base_segments->size_used = destination_size;
            buffer->base_segments->storage = destination_start;
            buffer->base_segments->current = destination_start;
            buffer->base_segments->storage_is_inline = NEXUS_FALSE;
        }
    }

    /*
     * Unpack the message info
     */
    message_start = buffer->base_segments->current;
    nexus_dc_get_byte(&(buffer->base_segments->current),
		      &tmp_byte, 1, buffer->format);
    has_direct_info = tmp_byte & 0x10;
    has_other_info = tmp_byte & 0x08;
    pad_size = tmp_byte & 0x07;
    buffer->base_segments->current += pad_size;
    nexus_dc_get_int(&(buffer->base_segments->current),
		     &handler_id, 1, buffer->format);
    if (has_direct_info)
    {
        nexus_dc_get_u_long(&(buffer->base_segments->current),
			    &direct_info_offset,
			    1,
			    buffer->format);
    }
    else
    {
	buffer->n_direct = 0;
    }
    if (has_other_info)
    {
        nexus_dc_get_u_long(&(buffer->base_segments->current),
			    &other_info_offset,
			    1,
			    buffer->format);
    }
    buffer->base_segments->size_used
	-= (buffer->base_segments->current - message_start);

    buffer->reserved_header_size
	= (buffer->base_segments->current - buffer->base_segments->storage);

    if (handler_id < 0)
    {

	/* This is a system handler */
	switch (handler_id)
	{
	case GLOBUS_NEXUS_STARTPOINT_COPY_HANDLER_ID:
	    /* TODO: Check endpoint to see if it support context creation */
	    buffer_dispatch_non_threaded(buffer,
					 endpoint,
					 _nx_startpoint_copy_handler);
	    break;
	default:
	    call_nexus_unknown_handler(endpoint, buffer, handler_id);
	    break;
	}
    }
    else if (   (handler_id >= endpoint->handler_table_size)
	     || !(endpoint->handler_table[handler_id].func) )
    {
	call_nexus_unknown_handler(endpoint, buffer, handler_id);
    }
    else
    {
        /* Call the handler */
        handler = endpoint->handler_table[handler_id].func;
#ifndef BUILD_LITE
        if (endpoint->handler_table[handler_id].type
	    == NEXUS_HANDLER_TYPE_THREADED)
	{
	    globus_reltime_t                    delay_time;
	    threaded_handler_startup_info_t *   info;

	    NexusMalloc(_nx_buffer_dispatch(),
			info,
			threaded_handler_startup_info_t *,
			sizeof(threaded_handler_startup_info_t));
	    info->context = endpoint->context;
	    info->endpoint = endpoint;
#ifdef BUILD_PROFILE
	    info->node_id = node_id;
	    info->context_id = context_id;
	    info->message_length = message_length;
#endif
	    nexus_buffer_save(&buffer);
	    buffer->saved_state = GLOBUS_NEXUS_BUFFER_SAVED_STATE_SAVED;
	    info->buffer = buffer;
	    info->func = handler;

	    globus_thread_create(GLOBUS_NULL,
				(globus_threadattr_t *) NULL,
				threaded_handler_startup,
				(void *) info);
	}
	else /* type == NEXUS_HANDLER_TYPE_NON_THREADED */
#endif
	{
	    buffer_dispatch_non_threaded(buffer,
					 endpoint,
					 handler);
	}
    }
    
    return(GLOBUS_SUCCESS);
} /* _nx_buffer_dispatch() */

/*
 * call_nexus_unknown_handler()
 *
 * TODO: Allow this to be a threaded handler
 */
static void call_nexus_unknown_handler(nexus_endpoint_t *endpoint,
				       struct globus_nexus_buffer_s *buffer,
				       int handler_id)
{
    nexus_unknown_handler_func_t NexusUnknownHandler;
    nexus_context_t *save_context;

    NexusUnknownHandler = endpoint->unknown_handler;

    nexus_debug_printf(1, ("call_nexus_unknown_handler(): Handler[%d] not found. %s\n", handler_id, (NexusUnknownHandler ? "Calling unknown handler" : "Ignoring (unknown handler not registered)")));

    if (NexusUnknownHandler)
    {
	_nx_context(&save_context);
	_nx_set_context(endpoint->context);
	(*NexusUnknownHandler)(endpoint, &buffer, handler_id);
	_nx_set_context(save_context);
    }

    LockFreelists();
    if (   (buffer->saved_state == GLOBUS_NEXUS_BUFFER_SAVED_STATE_FREED)
	   || (buffer->saved_state == GLOBUS_NEXUS_BUFFER_SAVED_STATE_UNSAVED))
    {
	base_segments_free(buffer->base_segments);
	buffer_free(buffer);
    }
    else if (buffer->saved_state
	     == GLOBUS_NEXUS_BUFFER_SAVED_STATE_SAVED_IN_HANDLER)
    {
	buffer->saved_state = GLOBUS_NEXUS_BUFFER_SAVED_STATE_SAVED;
    }
    UnlockFreelists();
} /* call_nexus_unknown_handler() */

static void
buffer_dispatch_non_threaded(struct globus_nexus_buffer_s *buffer,
			     globus_nexus_endpoint_t *endpoint,
			     nexus_handler_func_t handler)
{
    nexus_context_t *save_context;
    
    _nx_context(&save_context);
    _nx_set_context(endpoint->context);
    (*handler)(endpoint, &buffer, NEXUS_TRUE);
    _nx_set_context(save_context);
    
    LockFreelists();
    if (   (buffer->saved_state == GLOBUS_NEXUS_BUFFER_SAVED_STATE_FREED)
	   || (buffer->saved_state == GLOBUS_NEXUS_BUFFER_SAVED_STATE_UNSAVED))
    {
	base_segments_free(buffer->base_segments);
	buffer_free(buffer);
    }
    else if (buffer->saved_state
	     == GLOBUS_NEXUS_BUFFER_SAVED_STATE_SAVED_IN_HANDLER)
    {
	buffer->saved_state = GLOBUS_NEXUS_BUFFER_SAVED_STATE_SAVED;
    }
    UnlockFreelists();
}

/*
 * nexus_buffer_save()
 */
int nexus_buffer_save(nexus_buffer_t *buffer)
{
    LockFreelists();
    if ((*buffer)->saved_state == GLOBUS_NEXUS_BUFFER_SAVED_STATE_UNSAVED)
    {
	(*buffer)->saved_state = GLOBUS_NEXUS_BUFFER_SAVED_STATE_SAVED_IN_HANDLER;
	/* TODO: Save direct segments */
    }
    UnlockFreelists();
    return(0);
} /* nexus_buffer_save() */


/*
 * nexus_buffer_save_linearly()
 */
int nexus_buffer_save_linearly(nexus_buffer_t *buffer)
{
    LockFreelists();
    if ((*buffer)->saved_state == GLOBUS_NEXUS_BUFFER_SAVED_STATE_UNSAVED)
    {
	(*buffer)->saved_state = GLOBUS_NEXUS_BUFFER_SAVED_STATE_SAVED_IN_HANDLER;
	/* TODO: Save direct segments */
    }
    UnlockFreelists();
    return(0);
} /* nexus_buffer_save_linearly() */



#ifndef BUILD_LITE
static void
threaded_handler_startup_wakeup(
    void *                                    arg)
{

}

/*
 * threaded_handler_startup()
 *
 * Startup a threaded handler.  The info about the handler is in arg.
 */
static
void *
threaded_handler_startup(
    void *                                  arg)
{
    threaded_handler_startup_info_t *         info;
    nexus_context_t *                         context;
    nexus_endpoint_t *                        endpoint;
    nexus_buffer_t                            buffer;
    nexus_handler_func_t                      func;
#ifdef BUILD_PROFILE
    int                                       node_id;
    int                                       context_id;
    int                                       message_length;
#endif

    info = (threaded_handler_startup_info_t *) arg;
    context = info->context;
    endpoint = info->endpoint;
#ifdef BUILD_PROFILE
    node_id = info->node_id;
    context_id = inf->context_id;
    message_length = info->message_length;
#endif
    buffer = info->buffer;
    func = info->func;
    NexusFree(info);
    
    _nx_set_context(context);

    (*func)(endpoint, &buffer, NEXUS_FALSE);
    
    globus_callback_signal_poll();
return GLOBUS_NULL;
} /* threaded_handler_startup() */
#endif /* BUILD_LITE */


/*
 * buffer_alloc()
 */
static nexus_buffer_t buffer_alloc()
{
    nexus_buffer_t buffer;
    if (buffer_freelist_head)
    {
        buffer = buffer_freelist_head;
	buffer_freelist_head = buffer_freelist_head->next;
    }
    else
    {
        NexusMalloc(buffer_alloc(),
		    buffer,
		    nexus_buffer_t,
		    sizeof(struct globus_nexus_buffer_s));
    }
    buffer->proto = NULL;
    buffer->next = NULL;
    buffer->iovec_formatted = NEXUS_FALSE;
    buffer->saved_state = GLOBUS_NEXUS_BUFFER_SAVED_STATE_UNSAVED;
    buffer->base_segments = NULL;
    buffer->current_base_segment = NULL;
    buffer->direct_segments = NULL;
    buffer->current_direct_segment = NULL;
    buffer->n_direct = 0;
    return(buffer);
} /* buffer_alloc() */


/*
 * buffer_free()
 */
static void buffer_free(nexus_buffer_t buffer)
{
    buffer->next = buffer_freelist_head;
    buffer_freelist_head = buffer;
} /* buffer_free() */


/*
 * base_segment_malloc()
 */
static nexus_base_segment_t *base_segment_malloc(unsigned long size)
{
    nexus_base_segment_t *segment;
    nexus_byte_t *b;
    NexusMalloc(base_segment_malloc(),
		b,
		nexus_byte_t *,
		(sizeof_base_segment + size));
    segment = (nexus_base_segment_t *) b;
    segment->size = size;
    segment->storage = (b + sizeof_base_segment);
    segment->storage_is_inline = NEXUS_TRUE;
    return(segment);
} /* base_segment_malloc() */


/*
 * direct_segment_malloc()
 */
static nexus_direct_segment_t *direct_segment_malloc(unsigned long size)
{
    nexus_direct_segment_t *segment;
    nexus_byte_t *b;
    NexusMalloc(direct_segment_malloc(),
		b,
		nexus_byte_t *,
		(sizeof_direct_segment
		 + (size * sizeof(nexus_direct_info_t))));
    segment = (nexus_direct_segment_t *) b;
    segment->size = size;
    segment->storage = (nexus_direct_info_t *) (b + sizeof_direct_segment);
    return(segment);
} /* direct_segment_malloc() */


/*
 * base_segment_alloc()
 */
static void base_segment_alloc(nexus_buffer_t buffer,
			       unsigned long size)
{
    nexus_base_segment_t *tmp;
    if (   size <= default_base_segment_size
	&& base_freelist_head)
    {
	/* Get a default size base segment from the freelist */
        tmp = base_freelist_head;
	base_freelist_head = base_freelist_head->next;
    }
    else
    {
	tmp = base_segment_malloc(NEXUS_MAX(size, default_base_segment_size));
    }
    tmp->next = NULL;
    tmp->current = tmp->storage;
    if (buffer->base_segments)
    {
	buffer->current_base_segment->next = tmp;
    }
    else
    {
	buffer->base_segments = tmp;
    }
    buffer->current_base_segment = tmp;
} /* base_segment_alloc() */


/*
 * base_segments_free()
 */
static void base_segments_free(nexus_base_segment_t *base_segments)
{
    nexus_base_segment_t *segment, *tmp;
    for (segment = base_segments;
	 segment;
	 segment = tmp)
    {
	tmp = segment->next;
        if (segment->storage_is_inline)
	{
	    if (segment->size != default_base_segment_size)
	    {
		NexusFree(segment);
	    }
	    else
	    {
		segment->next = base_freelist_head;
		base_freelist_head = segment;
	    }
	}
	else
	{
	    NexusFree(segment->storage);
	    NexusFree(segment);
	}
    }
} /* base_segments_free() */


/*
 * direct_segment_alloc()
 */
static void direct_segment_alloc(nexus_buffer_t buffer,
				 unsigned long size)
{
    nexus_direct_segment_t *tmp;
    if (   size <= default_direct_segment_size
	&& direct_freelist_head)
    {
	/* Get a default size direct segment from the freelist */
        tmp = direct_freelist_head;
	direct_freelist_head = direct_freelist_head->next;
    }
    else
    {
	tmp = direct_segment_malloc(NEXUS_MAX(size,
					      default_direct_segment_size));
    }
    tmp->next = NULL;
    tmp->n_left = tmp->size;
    tmp->current = tmp->storage;
    if (buffer->current_direct_segment)
    {
	buffer->current_direct_segment->next = tmp;
    }
    else
    {
	buffer->direct_segments = tmp;
    }
    buffer->current_direct_segment = tmp;
} /* direct_segment_alloc() */


/*
 * direct_segments_free()
 */
static void direct_segments_free(nexus_direct_segment_t *direct_segments)
{
    nexus_direct_segment_t *segment, *tmp;
    for (segment = direct_segments;
	 segment;
	 segment = tmp)
    {
	tmp = segment->next;
        if (segment->size != default_direct_segment_size)
	{
	    NexusFree(segment);
	}
	else
	{
	    segment->next = direct_freelist_head;
	    direct_freelist_head = segment;
	}
    }
} /* direct_segments_free() */
