/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Nexus
 * Authors:     Steven Tuecke and Robert Olson
 *              Argonne National Laboratory
 *
 * pr_local.c		- local protocol module
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-packages/nexus/source/nexus/pr_local.c,v 1.62 2006/01/19 05:57:06 mlink Exp $";

#include "internal.h"


/*
 * GLOBUS_L_NEXUS_LOCAL_PROTOCOL_VERSION
 *
 * The version of this protocol module's mi_proto.  If you change
 * the contents of this module's mi_proto, bump this version number.
 */
#define GLOBUS_L_NEXUS_LOCAL_PROTOCOL_VERSION (0+GLOBUS_I_NEXUS_BUFFER_VERSION)

#define GLOBUS_L_LOCAL_PROTO_COUNT 1

/*
 * Some forward typedef declarations...
 */
typedef struct _local_buffer_t	local_buffer_t;
typedef struct _local_proto_t	local_proto_t;


#ifdef BUILD_DEBUG
/*
static char tmpbuf1[10240];
static char tmpbuf2[10240];
*/
#endif

#ifdef BUILD_LITE
#define local_enter()
#define local_exit()
#else  /* BUILD_LITE */
static nexus_mutex_t	local_mutex;
#define local_enter()	nexus_mutex_lock(&local_mutex);
#define local_exit()	nexus_mutex_unlock(&local_mutex);
#endif /* BUILD_LITE */

static nexus_bool_t		handle_in_progress;
static struct globus_nexus_buffer_s *	handle_q_head;
static struct globus_nexus_buffer_s *	handle_q_tail;

/*
 * local_proto_t
 *
 * This is an overload of nexus_proto_t.  It adds the
 * local specific information to that structure.
 */
struct _local_proto_t
{
    nexus_proto_type_t		type;	/* NEXUS_PROTO_TYPE_LOCAL */
    nexus_proto_funcs_t *	funcs;
    int				version;
    unsigned long		direct_custom_min_size;
    unsigned long		direct_custom_max_size;
    unsigned long		direct_pointer_min_size;
    unsigned long		direct_pointer_max_size;
    nexus_bool_t		can_use_iovec;
    unsigned long		reserved_header_size;
};


/*
 * Some useful queue macros
 */
#define Enqueue(Qhead, Qtail, Item) \
{ \
    if (Qhead) \
    { \
	(Qtail)->next = (Item); \
	(Qtail) = (Item); \
    } \
    else \
    { \
	(Qhead) = (Qtail) = (Item); \
    } \
}

#define Dequeue(Qhead, Qtail, Item) \
{ \
    (Item) = (Qhead); \
    (Qhead) = (Qhead)->next; \
}

#define QueueNotEmpty(Qhead)	(Qhead)


/*
 * Various forward declarations of procedures
 */
static nexus_proto_type_t	local_proto_type(void);
static void		local_init(nexus_bool_t * add_to_my_mi_proto);
static void		local_shutdown(void);
static int		local_send_rsr(struct globus_nexus_buffer_s *buffer);

static globus_bool_t
local_send_rsr_outstanding(globus_nexus_proto_t *nproto);

static nexus_bool_t	local_construct_from_mi_proto(nexus_proto_t **proto,
						    nexus_mi_proto_t *mi_proto,
						    nexus_byte_t *array,
						    int size);
static int              local_direct_info_size(void);
static int              local_direct_get(nexus_byte_t *dest,
					 size_t n_bytes,
					 int action,
					 unsigned long info);

static globus_bool_t    local_startpoint_proto_match(
					globus_nexus_mi_proto_t *  mi_proto0,
					int                        offset0,
				        globus_byte_t *            subarray0,
				        int                        sub_length0,
					globus_nexus_mi_proto_t *  mi_proto1,
					int                        offset1,
				        globus_byte_t *            subarray1,
				        int                        sub_length1);

static int              local_count_proto();

void
globus_l_nexus_local_dispatch_callback(
    void *                              user_args);

static nexus_proto_funcs_t local_proto_funcs =
{
    local_proto_type,
    local_init,
    local_shutdown,
    NULL /* local_increment_reference_count */,
    NULL /* local_decrement_reference_count */,
    NULL /* local_get_my_mi_proto */,
    local_construct_from_mi_proto,
    NULL /* local_destroy_my_mi_proto */,
    NULL /* local_test_proto */,
    local_send_rsr,
    local_send_rsr_outstanding,
    local_direct_info_size,
    local_direct_get,
    local_startpoint_proto_match,
    local_count_proto,
};


/*
 * _nx_pr_local_info()
 *
 * Return the nexus_proto_funcs_t function table for this protocol module.
 *
 * This procedure is used for bootstrapping the protocol module.
 * The higher level Nexus code needs to call this routine to
 * retrieve the functions it needs to use this protocol module.
 */
void *_nx_pr_local_info(void)
{
    return((void *) (&local_proto_funcs));
} /* _nx_pr_local_info() */


/*
 * local_proto_type()
 *
 * Return the nexus_proto_type_t for this protocol module.
 */
static nexus_proto_type_t local_proto_type(void)
{
    return (NEXUS_PROTO_TYPE_LOCAL);
} /* local_proto_type() */


/*
 * local_init()
 *
 * Initialize the LOCAL protocol.
 */
static void local_init(nexus_bool_t * add_to_my_mi_proto)
{
    handle_in_progress = NEXUS_FALSE;
#ifndef BUILD_LITE
    nexus_mutex_init(&local_mutex, (nexus_mutexattr_t *) NULL);
#endif
    
    *add_to_my_mi_proto = NEXUS_TRUE;
    return ;
} /* local_init() */


/*
 * local_shutdown()
 *
 * This routine is called during normal shutdown of a process.
 */
static void local_shutdown(void)
{
} /* local_shutdown() */

/*
 * local_send_rsr()
 *
 * Generate a local remote service request for 'buffer'.
 */
static int local_send_rsr(struct globus_nexus_buffer_s *buffer)
{

    nexus_debug_printf(2,
		       ("local_send_rsr(): invoked with buffer: %x\n",buffer));

    /*
     * This buffer should arrive as NEXUS_BUFFER_SAVED_STATE_SENDING.
     * This converts it to UNSAVED for dispatch to free after handler
     * completes.
     */
    buffer->saved_state = GLOBUS_NEXUS_BUFFER_SAVED_STATE_UNSAVED;
    
    local_enter();

    Enqueue(handle_q_head, handle_q_tail, buffer);
    if (!handle_in_progress)
    { 
        globus_reltime_t             delay;

        GlobusTimeReltimeSet(delay, 0, 0);

	handle_in_progress  = NEXUS_TRUE;
	while (QueueNotEmpty(handle_q_head))
	{
	    Dequeue(handle_q_head, handle_q_tail, buffer);
	    local_exit();
	 /*   _nx_buffer_dispatch(buffer);*/
            globus_callback_register_oneshot(
                GLOBUS_NULL,
                &delay,
                globus_l_nexus_local_dispatch_callback,
                (void *)buffer);
                   
	    local_enter();
	}
	handle_in_progress = NEXUS_FALSE;
    }
    local_exit();
    return(0);

} /* local_send_rsr() */


void
globus_l_nexus_local_dispatch_callback(
    void *                              user_args)
{
    struct globus_nexus_buffer_s *      buffer;

    buffer = (struct globus_nexus_buffer_s *)user_args;

    _nx_buffer_dispatch(buffer);
    
    globus_callback_signal_poll();
}


/*
 * local_send_rsr_outstanding()
 *
 * Return true if there are any sends outstanding for this proto,
 * otherwise false.
 */
static globus_bool_t
local_send_rsr_outstanding(globus_nexus_proto_t *nproto)
{
    globus_bool_t rc = GLOBUS_FALSE;
    local_enter();
    if (handle_in_progress)
    {
	rc = GLOBUS_TRUE;
    }
    local_exit();
    return(rc);
} /* local_send_rsr_outstanding() */


/*
 * local_construct_from_mi_proto()
 *
 * Return my proto.  This is called only during initialization by
 * pr_iface.c:_nx_proto_init(), so that it can cache this proto.
 */
static nexus_bool_t local_construct_from_mi_proto(nexus_proto_t **proto,
						  nexus_mi_proto_t *mi_proto,
						  nexus_byte_t *array,
						  int size)
{
    local_proto_t *local_proto;
    
    NexusMalloc(local_construct_from_mi_proto(),
		local_proto,
		local_proto_t *,
		sizeof(local_proto_t) );
    local_proto->type = NEXUS_PROTO_TYPE_LOCAL;
    local_proto->funcs = &local_proto_funcs;
    local_proto->version = GLOBUS_L_NEXUS_LOCAL_PROTOCOL_VERSION;
    local_proto->direct_custom_min_size = NEXUS_DC_MAX_U_LONG;
    local_proto->direct_custom_max_size = NEXUS_DC_MAX_U_LONG;
    local_proto->direct_pointer_min_size = 0;
    local_proto->direct_pointer_max_size = NEXUS_DC_MAX_U_LONG;
    local_proto->can_use_iovec = NEXUS_FALSE;
    local_proto->reserved_header_size = 0;
    *proto = (nexus_proto_t *) local_proto;
    
    return (NEXUS_TRUE);
} /* local_construct_from_mi_proto() */


/*
 * local_direct_info_size()
 */
static int local_direct_info_size(void)
{
    return 0;
}


/*
 * local_direct_get()
 */
static int local_direct_get(nexus_byte_t *dest,
			    size_t n_bytes,
			    int action,
			    unsigned long info)
{
    memcpy(dest, (void *)info, n_bytes);
    return(0);
} /* local_direct_get() */

/* added by bresnaha */
globus_bool_t              local_startpoint_proto_match(
				  globus_nexus_mi_proto_t *  mi_proto0,
				  int                        offset0,
				  globus_byte_t *            subarray0,
				  int                        sub_length0,
				  globus_nexus_mi_proto_t *  mi_proto1,
				  int                        offset1,
				  globus_byte_t *            subarray1,
				  int                        sub_length1)
{
    return GLOBUS_TRUE;
}

static int
local_count_proto()
{
    return GLOBUS_L_LOCAL_PROTO_COUNT;
}
