/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * transform_iface.c
 *
 * Data transform module interface routines
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-packages/nexus/source/nexus/transform_iface.c,v 1.19 2006/01/19 05:57:06 mlink Exp $";

#include "internal.h"

/*
 * nexus_transform_new_process_params()
 *
 * Call the new_process_params() function for each protocol
 *
 * Each of those functions may add stuff to 'buf', returning the number
 * of characters that they added.
 *
 * Return: The total number of characters added to 'buf'.
 */
int nexus_transform_new_process_params(char *buf, int size)
{
    return (0);
} /* nexus_transform_new_process_params() */


/*
 * nexus_transform_init()
 */
int nexus_transform_init(nexus_module_list_t module_list[])
{
    int i;

    /* Initialize the transform table */
    for (i = 0; i < NEXUS_TRANSFORM_TABLE_SIZE; i++)
    {
	_nx_transform_table[i].funcs = (nexus_transform_funcs_t *) NULL;
	_nx_transform_table[i].name = (char *) NULL;
	_nx_transform_table[i].modifies_data = NEXUS_FALSE;
	_nx_transform_table[i].transform_info_size = 0;
	_nx_transform_table[i].trailer_size = 0;
    }

    /*
     * Scan the module_list looking for transform modules.
     * For each of these, get the function table,
     * and add that module to the _nx_transform_table
     */
    for (i = 0; module_list[i].family_name != (char *) NULL; i++)
    {
	if (strcmp(module_list[i].family_name, "transform") == 0)
	{
	    nexus_transform_add_module(module_list[i].module_name,
				       module_list[i].info_func);
	}
    }

    return(0);
} /* nexus_transform_init() */


/*
 * nexus_transform_add_module()
 */
void nexus_transform_add_module(char *module_name,
				void *(*info_func)(void))
{
    nexus_transform_funcs_t *funcs;
    int id;

    funcs = (nexus_transform_funcs_t *)(*info_func)();
    id = (*funcs->transform_id)();
    _nx_transform_table[id].funcs = funcs;
    _nx_transform_table[id].name = _nx_copy_string(module_name);
    (*funcs->init)(&(_nx_transform_table[id].modifies_data),
		   &(_nx_transform_table[id].transform_info_size),
		   &(_nx_transform_table[id].trailer_size));
} /* nexus_transform_add_module() */


/*
 * nexus_transform_shutdown()
 */
void nexus_transform_shutdown(void)
{
    int i;
    for (i = 0; i < NEXUS_TRANSFORM_TABLE_SIZE; i++)
    {
	if (   _nx_transform_table[i].funcs
	    && _nx_transform_table[i].funcs->shutdown)
	{
	    (*_nx_transform_table[i].funcs->shutdown)();
	}
    }
} /* nexus_transform_shutdown() */


/*
 * nexus_transformattr_init()
 */
void nexus_transformattr_init(int id,
			      void *info,
			      nexus_transformattr_t **attr)
{
    if (   id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->transformattr_init)
    {
	(*_nx_transform_table[id].funcs->transformattr_init)(info,
							     attr);
    }
    else
    {
	*attr = (nexus_transformattr_t *) NULL;
    }
} /* nexus_transformattr_init() */


/*
 * nexus_transformattr_destroy()
 */
void nexus_transformattr_destroy(int id,
				 nexus_transformattr_t *attr)
{
    if (   id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->transformattr_destroy)
    {
	(*_nx_transform_table[id].funcs->transformattr_destroy)(attr);
    }
} /* nexus_transformattr_destroy() */


/*
 * nexus_transformattr_get_info()
 *
 * This basically reverses nexus_transformattr_init().  It takes
 * a nexus_transformattr_t*, and returns an info structure
 * in *info.
 *
 * *info must be created with nexus_malloc(), or set to NULL
 * if there is no info structure for this attribute.
 */
void nexus_transformattr_get_info(int id,
				  nexus_transformattr_t *attr,
				  void **info)
{
    if (   id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->transformattr_get_info)
    {
	(*_nx_transform_table[id].funcs->transformattr_get_info)(attr,
								 info);
    }
} /* nexus_transformattr_get_info() */


/*
 * nexus_transformstate_init_on_endpoint()
 *
 * This is called by nexus_endpoint_init().
 * As input it takes the nexus_transformattr_t* from the endpointattr,
 * and returns a nexus_transformstate_t* for the endpoint.
 */
void nexus_transformstate_init_on_endpoint(int id,
					   nexus_transformattr_t *attr,
					   nexus_transformstate_t **ep_state)
{
    if (   id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->init_endpoint_state)
    {
	(*_nx_transform_table[id].funcs->init_endpoint_state)(attr,
							      ep_state);
    }
    else
    {
	*ep_state = (nexus_transformstate_t *) NULL;
    }
} /* nexus_transformstate_init_on_endpoint() */


/*
 * nexus_transformstate_destroy_on_endpoint()
 */
void nexus_transformstate_destroy_on_endpoint(int id,
					      nexus_transformstate_t *ep_state)
{
    if (   id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->destroy_endpoint_state)
    {
	(*_nx_transform_table[id].funcs->destroy_endpoint_state)(ep_state);
    }
} /* nexus_transformstate_destroy_on_endpoint() */


/*
 * nexus_transformstate_update_endpoint_with_startpoint_destroy()
 *
 * Use the sp_state to find and remove the its corresponding
 * state in ep_state.
 */
void nexus_transformstate_update_endpoint_with_startpoint_destroy(
				int id,
				nexus_transformstate_t *ep_state,
				nexus_transformstate_t *sp_state)
{
    if (   id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->update_endpoint_state)
    {
	(*_nx_transform_table[id].funcs->update_endpoint_state)(ep_state,
								sp_state);
    }
} /* nexus_transformstate_update_endpoint_with_startpoint_destroy() */


/*
 * nexus_transformstate_init_on_startpoint()
 *
 * This is called by nexus_startpoint_bind(), taking the endpoint's
 * nexus_transformstate_t* as input.
 * It returns:
 *   - A nexus_transformstate_t* for the startpoint.
 *   - A unique id that identifies this startpoint within
 *     the endpoint state.  This should be 0 if its not needed.
 *   - A boolean specifying whether nexus_startpoint_copy() can
 *     be done locally (NEXUS_TRUE), or must be done as a round
 *     trip to the context with the endpoint to which this
 *     startpoint is bound.
 */
void nexus_transformstate_init_on_startpoint(int id,
					     nexus_transformstate_t *ep_state,
					     nexus_transformstate_t **sp_state,
					     nexus_bool_t *copy_sp_locally,
					     nexus_bool_t *destroy_sp_locally)
{
    if (   id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->init_startpoint_state)
    {
	(*_nx_transform_table[id].funcs->init_startpoint_state)(
	    						ep_state,
							sp_state,
							copy_sp_locally,
							destroy_sp_locally);
    }
    else
    {
	*sp_state = (nexus_transformstate_t *) NULL;
	*copy_sp_locally = NEXUS_TRUE;
	*destroy_sp_locally = NEXUS_TRUE;
    }
} /* nexus_transformstate_init_on_startpoint() */


/*
 * nexus_transformstate_copy()
 *
 * If a startpoint can be copied locally (i.e. it doesn't require a
 * roundtrip back to the endpoint to copy), then nexus_startpoint_copy()
 * calls this routine to copy its nexus_transformstate_t*.
 */
void nexus_transformstate_copy(int id,
			       nexus_transformstate_t *sp_state,
			       nexus_transformstate_t **sp_state_copy)
{
    if (   id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->copy_startpoint_state)
    {
	(*_nx_transform_table[id].funcs->copy_startpoint_state)(sp_state,
								sp_state_copy);
    }
    else
    {
	*sp_state_copy = (nexus_transformstate_t *) NULL;
    }
} /* nexus_transformstate_copy() */


/*
 * nexus_transformstate_destroy_on_startpoint()
 */
void nexus_transformstate_destroy_on_startpoint(int id,
					      nexus_transformstate_t *sp_state)
{
    if (   id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->destroy_startpoint_state)
    {
	(*_nx_transform_table[id].funcs->destroy_startpoint_state)(sp_state);
    }
} /* nexus_transformstate_destroy_on_startpoint() */


/*
 * nexus_transformstate_sizeof()
 */
int nexus_transformstate_sizeof(int id,
				nexus_transformstate_t *sp_state)
{
    if (   id > NEXUS_TRANSFORM_NONE && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->sizeof_state)
    {
	return(*_nx_transform_table[id].funcs->sizeof_state)(sp_state);
    }
    return(0);
} /* nexus_transformstate_sizeof() */


/*
 * nexus_transformstate_put()
 *
 * Use nexus_dc_put_*() routines to put the transform state.
 */
void nexus_transformstate_put(int id,
			      nexus_byte_t **buffer,
			      nexus_transformstate_t *sp_state)
{
    if (   id > NEXUS_TRANSFORM_NONE && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->put_state)
    {
	(*_nx_transform_table[id].funcs->put_state)(buffer, sp_state);
    }
} /* nexus_transformstate_put() */


/*
 * nexus_transformstate_get()
 *
 * Use nexus_dc_get_*() routines to put the transform state.
 */
void nexus_transformstate_get(int id,
			      nexus_byte_t **buffer,
			      int format,
			      nexus_transformstate_t **sp_state)
{
    if (   id > NEXUS_TRANSFORM_NONE && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->get_state)
    {
	(*_nx_transform_table[id].funcs->get_state)(buffer,
						   format,
						   sp_state);
    }
} /* nexus_transformstate_get() */


/*
 * nexus_buffer_transform()
 *
 * The input variables look like this:
 *
 * storage_start
 * |
 * |           data_start
 * |                |
 * v                v
 * +----------------+-----------------------+----------------+
 * |  unused space  |       user data       |  unused space  |
 * +----------------+-----------------------+----------------+
 *
 *                  |<----- data_size ----->|
 *
 * |<--------------------- storage_size -------------------->|
 *
 * That is, storage_start and storage_size pointer to a byte array.
 * A subset of the storage array is the data array, which is described
 * by data_start and data_size.
 *
 * must_alloc_new_buffer is set to NEXUS_TRUE, the transform
 * module must allocate a new buffer for the transformed user data,
 * or NEXUS_FALSE if the user data can be modified inplace.  (However,
 * the transform module has the option of allocating a new buffer,
 * even if must_alloc_new_buffer==NEXUS_FALSE.)
 *
 * transform_info_start points to a byte array which is at least
 * as large as the transform_info_size value returned
 * by nexus_transform_info().
 *
 * This function should call down to the appropriate transform
 * module, as dictated by the startpoint, to transform the user data.
 * Any transform info should be filled into the byte array pointed
 * to by transform_info_start.
 *
 * If the transform module needs to create a new array for the
 * transformed data (either because must_alloc_new_buffer==NEXUS_TRUE,
 * or because it needs a larger array than is provided by storage_*),
 * then it should allocate and return that array in the out_* variables.
 * In doing so, it must preserve the amount of unused space at the
 * front of the message (data_start - storage_start).
 *
 * The amount of unused space at the end of the buffer
 * is NOT guaranteed to be at least as large as the trailer_size
 * value returned by nexus_transform_info().  That argument is
 * meant only as a hint.  If there is not enough unused space at the
 * end of the storage to perform the transform, then the transform
 * must create a new out_* buffer.
 *
 * If the data can be transformed in place, but required some of the
 * unused space at the end of the buffer to be used, then *data_size
 * can be modified, so long as it stays within the storage_* array.
 */
int nexus_buffer_transform(nexus_startpoint_t *sp,
                           nexus_byte_t *storage_start,
                           unsigned long storage_size,
                           nexus_byte_t *data_start,
                           unsigned long *data_size,
                           nexus_bool_t must_alloc_new_buffer,
                           nexus_byte_t *transform_info_start,
                           nexus_byte_t **out_storage_start,
                           unsigned long *out_storage_size,
                           nexus_byte_t **out_data_start,
                           unsigned long *out_data_size)
{
    int id;

    id = sp->transform_id;
    if (   id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
        && _nx_transform_table[id].funcs
        && _nx_transform_table[id].funcs->transform)
    {
        return(*_nx_transform_table[id].funcs->transform)(
                                        sp->transform_state,
                                        storage_start,
					storage_size,
					data_start,
					data_size,
                                        must_alloc_new_buffer,
                                        transform_info_start,
                                        out_storage_start,
                                        out_storage_size,
                                        out_data_start,
                                        out_data_size);
    }
    return(-1);
} /* nexus_buffer_transform() */


/*
 * nexus_buffer_untransform()
 *
 * A byte array containing transformed data is pointed to by
 * data_start, and is data_size bytes long.
 *
 * transform_info_start points to the transform info associated
 * with this transformed byte array.
 *
 * if (destination_start == NULL)
 *     The data should be untransformed inplace, if possible.
 *       The *data_size can be shrunk if appropriate.
 *     If it cannot, then an array should be allocated to hold
 *       the untransformed data, and that array's start and
 *       size should be returned in *destionation_start
 *       and *destination_size.
 * else
 *     The data should be untransformed into the
 *       array pointed to by *destination_start, with a size
 *       of *destination_size.  If the untransformed data
 *       is not exactly *destination_size bytes long,
 *       then return a non-0 error code, and do not bother
 *       doing the untransform.
 */
int nexus_buffer_untransform(nexus_endpoint_t *ep,
			     nexus_byte_t *data_start,
			     unsigned long *data_size,
			     nexus_byte_t *transform_info_start,
			     int format,
			     nexus_byte_t **destination_start,
			     unsigned long *destination_size)
{
    int id;

    id = ep->transform_id;

    if (   id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
        && _nx_transform_table[id].funcs
        && _nx_transform_table[id].funcs->untransform)
    {
        return(*_nx_transform_table[id].funcs->untransform)(
                                                ep->transform_state,
						data_start,
						data_size,
						transform_info_start,
						format,
						destination_start,
						destination_size);
    }
    return(-1);
} /* nexus_buffer_untransform() */


/*
 * nexus_transform_info()
 *
 * modifies_data:	Set to NEXUS_TRUE if the transform modifies
 *			the user data that is passed to
 *                      nexus_buffer_transform()
 *			and nexus_buffer_untransform().
 * transform_info_size	Set to the number of bytes that are needed
 *			to hold the transform information for this
 *			transform module.
 * trailer_size:	Set to number of bytes the transform may need to
 *			add to the end of the message.
 */
void nexus_transform_info(int id,
			  nexus_bool_t *modifies_data,
			  unsigned long *transform_info_size,
			  unsigned long *trailer_size)
{
    if (id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE)
    {
	*modifies_data = _nx_transform_table[id].modifies_data;
	*transform_info_size = _nx_transform_table[id].transform_info_size;
	*trailer_size = _nx_transform_table[id].trailer_size;
    }
    else
    {
	*modifies_data = NEXUS_FALSE;
	*transform_info_size = 0;
	*trailer_size = 0;
    }
} /* nexus_transform_info() */


void nexus_startpoint_info(int id,
			   nexus_transformstate_t *sp_state,
			   void *info)
{
    if (id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->transformstate_get_info)
    {
	(*_nx_transform_table[id].funcs->transformstate_get_info)(
				sp_state, info, 0); /* set flag to 0 */
    }
}


void nexus_endpoint_info(int id,
			 nexus_transformstate_t *ep_state,
			 void *info)
{
     if (id >= 0 && id < NEXUS_TRANSFORM_TABLE_SIZE
	&& _nx_transform_table[id].funcs
	&& _nx_transform_table[id].funcs->transformstate_get_info)
     {
	(*_nx_transform_table[id].funcs->transformstate_get_info)(
			ep_state, info, 1); /* set flag to 1 */
     }
}
