/*
 *    filter.c  --  FIR filter
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "cmplx.h"
#include "filter.h"

#undef	DEBUG

#ifdef DEBUG
#include <stdio.h>
#endif

/* ---------------------------------------------------------------------- */

/*
 * This gets used when not optimising
 */
#ifndef	__OPTIMIZE__
float mac(const float *a, const float *b, unsigned int len)
{ 
        float sum = 0;
        unsigned int i;

        for (i = 0; i < len; i++)
                sum += (*a++) * (*b++);
        return sum;
}
#endif

/* ---------------------------------------------------------------------- */

/*
 * Sinc done properly.
 */
static inline double sinc(double x)
{
	if (fabs(x) < 1e-10)
		return 1.0;
	else
		return sin(M_PI * x) / (M_PI * x);
}

/*
 * Don't ask...
 */
static inline double cosc(double x)
{
	if (fabs(x) < 1e-10)
		return 0.0;
	else
		return (1.0 - cos(M_PI * x)) / (M_PI * x);
}

/*
 * Hamming window function.
 */
static inline double hamming(double x)
{
	return 0.54 - 0.46 * cos(2 * M_PI * x);
}

/* ---------------------------------------------------------------------- */

/*
 * Create a band pass FIR filter with 6 dB corner frequencies
 * of 'f1' and 'f2'. (0 <= f1 < f2 <= 0.5)
 */
static float *mk_filter(int len, int hilbert, float f1, float f2)
{
	float *fir;
	float t, h, x;
	int i;

	if ((fir = malloc(len * sizeof(float))) == NULL)
		return NULL;


	for (i = 0; i < len; i++) {
		t = i - (len - 1.0) / 2.0;
		h = i * (1.0 / (len - 1.0));

		if (!hilbert) {
			x = (2 * f2 * sinc(2 * f2 * t) -
			     2 * f1 * sinc(2 * f1 * t)) * hamming(h);
		} else {
			x = (2 * f2 * cosc(2 * f2 * t) -
			     2 * f1 * cosc(2 * f1 * t)) * hamming(h);
			/*
			 * The actual filter code assumes the impulse response
			 * is in time reversed order. This will be anti-
			 * symmetric so the minus sign handles that for us.
			 */
			x = -x;
		}

		fir[i] = x;
	}

	return fir;
}

struct filter *filter_init(int len, int dec, float *itaps, float *qtaps)
{
	struct filter *f;

	if ((f = calloc(1, sizeof(struct filter))) == NULL)
		return NULL;

	f->length = len;
	f->decimateratio = dec;

	if (itaps) {
		if ((f->ifilter = malloc(len * sizeof(float))) == NULL) {
			filter_free(f);
			return NULL;
		}
		memcpy(f->ifilter, itaps, len * sizeof(float));
	}

	if (qtaps) {
		if ((f->qfilter = malloc(len * sizeof(float))) == NULL) {
			filter_free(f);
			return NULL;
		}
		memcpy(f->qfilter, qtaps, len * sizeof(float));
	}

	f->pointer = len;
	f->counter = 0;

	return f;
}

struct filter *filter_init_lowpass(int len, float freq)
{
	struct filter *f;
	float *i, *q;

	if ((i = mk_filter(len, 0, 0.0, freq)) == NULL)
		return NULL;

	if ((q = mk_filter(len, 0, 0.0, freq)) == NULL)
		return NULL;

	f = filter_init(len, 1, i, q);

	free(i);
	free(q);

	return f;
}

struct filter *filter_init_bandpass(int len, float f1, float f2)
{
	struct filter *f;
	float *i, *q;

	if ((i = mk_filter(len, 0, f1, f2)) == NULL)
		return NULL;

	if ((q = mk_filter(len, 0, f1, f2)) == NULL)
		return NULL;

	f = filter_init(len, 1, i, q);

	free(i);
	free(q);

	return f;
}

struct filter *filter_init_hilbert(int len)
{
	struct filter *f;
	float *i, *q;

	if ((i = mk_filter(len, 0, 0.05, 0.45)) == NULL)
		return NULL;

	if ((q = mk_filter(len, 1, 0.05, 0.45)) == NULL) {
		free(i);
		return NULL;
	}

	f = filter_init(len, 1, i, q);

	free(i);
	free(q);

	return f;
}

void filter_free(struct filter *f)
{
	if (f) {
		free(f->ifilter);
		free(f->qfilter);
		free(f);
	}
}

/* ---------------------------------------------------------------------- */

int filter_run(struct filter *f, complex in, complex *out)
{
	float *iptr = f->ibuffer + f->pointer;
	float *qptr = f->qbuffer + f->pointer;

	*iptr = c_re(in);
	*qptr = c_im(in);

	out->re = mac(iptr - f->length, f->ifilter, f->length);
	out->im = mac(qptr - f->length, f->qfilter, f->length);

	f->pointer++;
	if (f->pointer == BufferLen) {
		iptr = f->ibuffer + BufferLen - f->length;
		qptr = f->qbuffer + BufferLen - f->length;
		memcpy(f->ibuffer, iptr, f->length * sizeof(float));
		memcpy(f->qbuffer, qptr, f->length * sizeof(float));
		f->pointer = f->length;
	}

	f->counter++;
	if (f->counter == f->decimateratio) {
		f->counter = 0;
		return 1;
	}

	return 0;
}

int filter_I_run(struct filter *f, float in, float *out)
{
	float *iptr = f->ibuffer + f->pointer;

	*iptr = in;

	*out = mac(iptr - f->length, f->ifilter, f->length);

	f->pointer++;
	if (f->pointer == BufferLen) {
		iptr = f->ibuffer + BufferLen - f->length;
		memcpy(f->ibuffer, iptr, f->length * sizeof(float));
		f->pointer = f->length;
	}

	f->counter++;
	if (f->counter == f->decimateratio) {
		f->counter = 0;
		return 1;
	}

	return 0;
}

int filter_Q_run(struct filter *f, float in, float *out)
{
	float *qptr = f->ibuffer + f->pointer;

	*qptr = in;

	*out = mac(qptr - f->length, f->qfilter, f->length);

	f->pointer++;
	if (f->pointer == BufferLen) {
		qptr = f->qbuffer + BufferLen - f->length;
		memcpy(f->qbuffer, qptr, f->length * sizeof(float));
		f->pointer = f->length;
	}

	f->counter++;
	if (f->counter == f->decimateratio) {
		f->counter = 0;
		return 1;
	}

	return 0;
}

/* ---------------------------------------------------------------------- */

void filter_dump(struct filter *f)
{
	int i;

	fprintf(stderr, "# len = %d\n", f->length);

	for (i = 0; i < f->length; i++) {
		if (f->ifilter)
			fprintf(stderr, "% .10f  ", f->ifilter[i]);
		else
			fprintf(stderr, "             ");

		if (f->qfilter)
			fprintf(stderr, "% .10f\n", f->qfilter[i]);
		else
			fprintf(stderr, "\n");
	}
}

/* ---------------------------------------------------------------------- */
