/*
 *    snd.c  --  Sound card routines.
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>

#include <sys/soundcard.h>
#include <sys/ioctl.h>

#include "snd.h"
#include "misc.h"
#include "cwirc.h"

#undef SND_DEBUG

/* ---------------------------------------------------------------------- */

#define	SOUND_BUF_LEN	16384
#define	SOUND_VOL	0.5

static gchar sounddev[256] =	"";
static gint eightbit =		FALSE;
static gint stereo =		FALSE;
static gint fulldup =		FALSE;

static gchar config_dev[256] =	"";
static gint config_eb =		FALSE;
static gint config_st =		FALSE;
static gint config_fd =		FALSE;

static gint testmode = SOUND_TESTMODE_NONE;

static gint sound_fd = -1;

static gint16 snd_16_buffer[2 * SOUND_BUF_LEN];
static guint8 snd_8_buffer[2 * SOUND_BUF_LEN];

static gchar snd_error_str[1024];

/* ---------------------------------------------------------------------- */

void sound_set_conf(const gchar *dev,
		    gboolean eb, gboolean st,
		    gboolean fd)
{
	g_strlcpy(config_dev, dev, sizeof(config_dev));
	config_eb = eb;
	config_st = st;
	config_fd = fd;
}

void sound_set_testmode(gint mode)
{
	testmode = mode;
}

/* ---------------------------------------------------------------------- */

static void snderr(const gchar *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsnprintf(snd_error_str, sizeof(snd_error_str), fmt, args);
	snd_error_str[sizeof(snd_error_str) - 1] = 0;
	va_end(args);

#ifdef SND_DEBUG
	fprintf(stderr, "%s\n", snd_error_str);
#endif
}

#ifdef SND_DEBUG
static void dprintf(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vfprintf(stderr, fmt, args);
	va_end(args);
}
#endif

/* ---------------------------------------------------------------------- */

static gint opensnd(gint direction, gint samplerate)
{
#ifdef SND_DEBUG
	audio_buf_info info;
	gchar *str;
#endif
	guint sndparam, wanted;
	gint fd;

#ifdef SND_DEBUG
	switch (direction) {
	case O_RDONLY:
		str = "reading";
		break;
	case O_WRONLY:
		str = "writing";
		break;
	case O_RDWR:
		str = "read/write";
		break;
	default:
		str = "???";
		break;
	}
	dprintf("Opening %s for %s .. ", sounddev, str);
#endif

	/* non-blocking open */
	if ((fd = open(sounddev, direction | O_NONBLOCK)) < 0) {
		snderr("open: %s: %m", sounddev);
		return -1;
	}

	/* make it block again - (SNDCTL_DSP_NONBLOCK ???) */
	if (fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) & ~O_NONBLOCK) < 0) {
		snderr("ioctl: SNDCTL_DSP_NONBLOCK: %m");
		goto error;
	}

#ifdef SND_DEBUG
	if (eightbit)
		str = "8 bit unsigned";
	else
		str = "16 bit signed, native byteorder";

	dprintf("ok\nSetting sample format (%s) .. ", str);
#endif

	if (eightbit)
		wanted = AFMT_U8;	/* 8 bit unsigned */
	else
		wanted = AFMT_S16_NE;	/* 16 bit signed, native byteorder */

	sndparam = wanted;
	if (ioctl(fd, SNDCTL_DSP_SETFMT, &sndparam) < 0) {
		snderr("ioctl: SNDCTL_DSP_SETFMT: %m");
		goto error;
	}
	if (sndparam != wanted) {
		snderr("Requested sample format not supported");
		goto error;
	}

#ifdef SND_DEBUG
	dprintf("ok\nSetting %s audio .. ", stereo ? "stereo" : "mono");
#endif

	if (stereo)
		wanted = 1;		/* stereo */
	else
		wanted = 0;		/* mono */

	sndparam = wanted;
	if (ioctl(fd, SNDCTL_DSP_STEREO, &sndparam) < 0) {
		snderr("ioctl: SNDCTL_DSP_STEREO: %m");
		goto error;
	}
	if (sndparam != wanted) {
		snderr("Cannot set %s audio", stereo ? "stereo" : "mono");
		goto error;
	}

#ifdef SND_DEBUG
	dprintf("ok\nSetting samplerate to %u .. ", samplerate);
#endif

	sndparam = samplerate;
	if (ioctl(fd, SNDCTL_DSP_SPEED, &sndparam) < 0) {
		snderr("ioctl: SNDCTL_DSP_SPEED: %m");
		goto error;
	}
	if (sndparam != samplerate) {
		snderr("Warning: Sampling rate is %u, requested %u .. ",
			sndparam,
			samplerate);
	}

#ifdef SND_DEBUG
	dprintf("ok\n");
#endif

	/* Request a buffer size of 512 samples */
	if (eightbit)
		sndparam = 0x00000009;
	else
		sndparam = 0x0000000A;

	if (stereo)
		sndparam += 1;

	/* Unlimited amount of buffers for RX, four for TX */
	if (direction == O_RDONLY)
		sndparam |= 0x7FFF0000;
	else
		sndparam |= 0x00040000;

#ifdef SND_DEBUG
	dprintf("Setting fragment size (param = 0x%08X) .. ", sndparam);
#endif

	if (ioctl(fd, SNDCTL_DSP_SETFRAGMENT, &sndparam) < 0) {
		snderr("ioctl: SNDCTL_DSP_SETFRAGMENT: %m");
		goto error;
	}

#ifdef SND_DEBUG
	dprintf("ok\n");

	if (direction == O_RDONLY) {
		if (ioctl(fd, SNDCTL_DSP_GETISPACE, &info) < 0) {
			dprintf("ioctl: SNDCTL_DSP_GETISPACE: %m");
		}
	} else {
		if (ioctl(fd, SNDCTL_DSP_GETOSPACE, &info) < 0) {
			dprintf("ioctl: SNDCTL_DSP_GETOSPACE: %m");
		}
	}

	dprintf("Audio buffer size: %d bytes, number of buffers: %d\n",
		info.fragsize, info.fragstotal);
#endif

#ifdef SND_DEBUG
	dprintf("-- \n");
#endif

	return fd;

error:
	close(fd);
	return -1;
}

/* ---------------------------------------------------------------------- */

gint sound_open_for_write(gint srate)
{
	/* copy current config */
	g_strlcpy(sounddev, config_dev, sizeof(sounddev));
	eightbit = config_eb;
	stereo = config_st;
	fulldup = config_fd;

	if (cwirc_extension_mode) {
		cwirc_sound_set_srate(srate);
		return 0;
	}

	/* use stdout for sound output */
	if (testmode) {
		sound_fd = 1;
		return 1;
	}

	if (fulldup == TRUE) {
		if (sound_fd < 0)
			sound_fd = opensnd(O_RDWR, srate);

		return sound_fd;
	}

	sound_fd = opensnd(O_WRONLY, srate);

	return sound_fd;
}

gint sound_open_for_read(gint srate)
{
	/* copy current config */
	g_strlcpy(sounddev, config_dev, sizeof(sounddev));
	eightbit = config_eb;
	stereo = config_st;
	fulldup = config_fd;

	if (cwirc_extension_mode) {
		cwirc_sound_set_srate(srate);
		return 0;
	}

	/* use stdin for sound input */
	if (testmode) {
		sound_fd = 0;
		return 0;
	}

	if (fulldup == TRUE) {
		if (sound_fd < 0)
			sound_fd = opensnd(O_RDWR, srate);

		return sound_fd;
	}

	sound_fd = opensnd(O_RDONLY, srate);

	return sound_fd;
}

void sound_close(void)
{
	if (cwirc_extension_mode)
		return;

	if (testmode == TRUE) {
		sound_fd = -1;
		return;
	}

	if (fulldup == TRUE)
		return;

	/* never close stdin/out/err */
	if (sound_fd > 2) {
		if (ioctl(sound_fd, SNDCTL_DSP_SYNC, 0) < 0)
			snderr("ioctl: SNDCTL_DSP_SYNC: %m");
		close(sound_fd);
		sound_fd = -1;
	}
}

char *sound_error(void)
{
	return snd_error_str;
}

/* ---------------------------------------------------------------------- */

gint sound_write(gfloat *buf, gint count)
{
	void *p;
	gint i, j;

	if (cwirc_extension_mode)
		return(cwirc_sound_write(buf, count));

	if (testmode == SOUND_TESTMODE_RX)
		return count;

	if (sound_fd < 0) {
		snderr("sound_write: fd < 0");
		return -1;
	}

	if (count > SOUND_BUF_LEN) {
		snderr("sound_write: count > SOUND_BUF_LEN");
		return -1;
	}

	if (eightbit) {
		for (i = j = 0; i < count; i++) {
			snd_8_buffer[j++] = (buf[i] * 127.0 * SOUND_VOL) + 128.0;
			if (stereo)
				snd_8_buffer[j++] = 128;
		}

		count *= sizeof(u_int8_t);
		p = snd_8_buffer;
	} else {
		for (i = j = 0; i < count; i++) {
			snd_16_buffer[j++] = buf[i] * 32767.0 * SOUND_VOL;
			if (stereo)
				snd_16_buffer[j++] = 0;
		}

		count *= sizeof(int16_t);
		p = snd_16_buffer;
	}

	if (stereo)
		count *= 2;

	if ((i = write(sound_fd, p, count)) < 0)
		snderr("sound_write: write: %m");

	return i;
}

gint sound_read(gfloat *buf, gint count)
{
	struct timespec ts;
	gint len, i, j;

	if (cwirc_extension_mode)
		return(cwirc_sound_read(buf, count));

	if (testmode == SOUND_TESTMODE_TX) {
		ts.tv_sec = 0;
		ts.tv_nsec = 125000 * count;	/* assume 8000 sps */
		nanosleep(&ts, NULL);
		return 0;
	}

	if (sound_fd < 0) {
		snderr("sound_read: fd < 0");
		return -1;
	}

	if (count > SOUND_BUF_LEN) {
		snderr("sound_read: count > SOUND_BUF_LEN");
		return -1;
	}

	if (stereo)
		count *= 2;

	if (eightbit) {
		count *= sizeof(u_int8_t);

		if ((len = read(sound_fd, snd_8_buffer, count)) < 0)
			goto error;

		len /= sizeof(u_int8_t);

		if (stereo)
			len /= 2;

		for (i = j = 0; i < len; i++) {
			buf[i] = (snd_8_buffer[j++] - 128) / 128.0;
			if (stereo)
				j++;
		}
	} else {
		count *= sizeof(int16_t);

		if ((len = read(sound_fd, snd_16_buffer, count)) < 0)
			goto error;

		len /= sizeof(int16_t);

		if (stereo)
			len /= 2;

		for (i = j = 0; i < len; i++) {
			buf[i] = snd_16_buffer[j++] / 32768.0;
			if (stereo)
				j++;
		}
	}

	return len;

error:
	if (errno != EAGAIN)
		snderr("sound_read: read: %m");
	return len;
}

/* ---------------------------------------------------------------------- */

