/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "gtkgmotriggers.h"

/* free all strigns, and the trigger itself */
void gtk_gmo_trigger_free(GtkGmoTrigger *t) {
    if(!t) return;
    if(debug) printf("TRIGGER: freeing: %s\n", t->name);
    
    g_free(t->name);
    g_free(t->pattern);
    
    g_list_foreach(t->hoover_actions, (GFunc)gtk_gmo_trigger_hoover_free, NULL);
    g_list_free(t->hoover_actions);

    g_free(t->send_text_text);
    g_free(t->run_script_function);
    g_free(t->play_file_name);

    g_list_free(t->changed_functions);
    g_list_free(t->changed_data);
    g_list_free(t->deleted_functions);
    g_list_free(t->deleted_data);

    g_free(t);
    
    if(debug) printf("\tfree()ed.\n");
}

/* Create a default trigger */
GtkGmoTrigger *gtk_gmo_trigger_new() {
    GtkGmoTrigger *t = g_malloc(sizeof(GtkGmoTrigger));

    /* General */
    t->name = g_strdup("(New Trigger)");
    t->disabled = FALSE;
    
    /* Matching */
    t->type = GTK_GMO_MATCH_START;
    t->pattern = g_strdup("");
    t->case_sensitive = FALSE;
    t->chance = 100;
    t->more_matches = TRUE;
    
    /* Normal color */
    t->n_color_fg = FALSE;
    t->n_fgcolor = FALSE;
    t->n_fgcolors[0] = t->n_fgcolors[1] = t->n_fgcolors[2] = 0;

    t->n_color_bg = FALSE;
    t->n_bgcolor = FALSE;
    t->n_bgcolors[0] = t->n_bgcolors[1] = t->n_bgcolors[2] = 0;

    t->n_color_bar = FALSE;
    t->n_barcolor = FALSE;
    t->n_barcolors[0] = t->n_barcolors[1] = t->n_barcolors[2] = 0;

    t->n_bold = FALSE;
    t->n_underline = FALSE;
    
    /* Hoover color */
    t->h_color_fg = FALSE;
    t->h_fgcolor = FALSE;
    t->h_fgcolors[0] = t->h_fgcolors[1] = t->h_fgcolors[2] = 0;

    t->h_color_bg = FALSE;
    t->h_bgcolor = FALSE;
    t->h_bgcolors[0] = t->h_bgcolors[1] = t->h_bgcolors[2] = 0;
    
    t->h_bold = FALSE;
    t->h_underline = FALSE;

    /* Hoover actions */
    t->hoover_actions = NULL;

    /* Trigger actions */
    t->send_text = FALSE;
    t->send_text_text = g_strdup("");
    t->run_script = FALSE;
    t->run_script_function = g_strdup("");

    /* Sound */
    t->beep = FALSE;
    t->play_file = FALSE;
    t->play_file_name = g_strdup("");

    /* Event handling */
    t->changed_functions = NULL;
    t->changed_data = NULL;
    t->deleted_functions = NULL;
    t->deleted_data = NULL;
    
    /* Misc */
    t->refcount = 1;
    
    if(debug) printf("TRIGGER: new trigger created\n");
    return t;
}

/* copies an existing one */
GtkGmoTrigger *gtk_gmo_trigger_copy(const GtkGmoTrigger *trigger) {
    GList *l, *func, *data;
    GtkGmoTrigger *t = g_malloc(sizeof(GtkGmoTrigger));

    /* General */
    t->name = g_strdup(trigger->name);
    t->disabled = trigger->disabled;

    /* Matching */
    t->type = trigger->type;
    t->pattern = g_strdup(trigger->pattern);
    t->case_sensitive = trigger->case_sensitive;
    t->chance = trigger->chance;
    t->more_matches = trigger->more_matches;

    /* Normal Color */
    t->n_color_fg = trigger->n_color_fg;
    t->n_fgcolor = trigger->n_fgcolor;
    t->n_fgcolors[0] = trigger->n_fgcolors[0];
    t->n_fgcolors[1] = trigger->n_fgcolors[1];
    t->n_fgcolors[2] = trigger->n_fgcolors[2];
    
    t->n_color_bg = trigger->n_color_bg;
    t->n_bgcolor = trigger->n_bgcolor;
    t->n_bgcolors[0] = trigger->n_bgcolors[0];
    t->n_bgcolors[1] = trigger->n_bgcolors[1];
    t->n_bgcolors[2] = trigger->n_bgcolors[2];
    
    t->n_color_bar = trigger->n_color_bar;
    t->n_barcolor = trigger->n_barcolor;
    t->n_barcolors[0] = trigger->n_barcolors[0];
    t->n_barcolors[1] = trigger->n_barcolors[1];
    t->n_barcolors[2] = trigger->n_barcolors[2];
    
    t->n_bold = trigger->n_bold;
    t->n_underline = trigger->n_underline;

    /* Hoover color */
    t->h_color_fg = trigger->h_color_fg;
    t->h_fgcolor = trigger->h_fgcolor;
    t->h_fgcolors[0] = trigger->h_fgcolors[0];
    t->h_fgcolors[1] = trigger->h_fgcolors[1];
    t->h_fgcolors[2] = trigger->h_fgcolors[2];
    
    t->h_color_bg = trigger->h_color_bg;
    t->h_bgcolor = trigger->h_bgcolor;
    t->h_bgcolors[0] = trigger->h_bgcolors[0];
    t->h_bgcolors[1] = trigger->h_bgcolors[1];
    t->h_bgcolors[2] = trigger->h_bgcolors[2];
    
    t->h_bold = trigger->h_bold;
    t->h_underline = trigger->h_underline;

    /* Hoover action */
    t->hoover_actions = NULL;
    for(l = trigger->hoover_actions; l; l = g_list_next(l)) {
	t->hoover_actions = 
	    g_list_append(t->hoover_actions,
			  gtk_gmo_trigger_hoover_copy((GtkGmoTriggerHoover*)l->data));
    }
	
    /* Trigger actions */
    t->send_text = trigger->send_text;
    t->send_text_text = g_strdup(trigger->send_text_text);
    t->run_script = trigger->run_script;
    t->run_script_function = g_strdup(trigger->run_script_function);

    /* Sound */
    t->beep = trigger->beep;
    t->play_file = trigger->play_file;
    t->play_file_name = g_strdup(trigger->play_file_name);
    
    /* Event Handling */
    t->changed_functions = NULL;
    t->changed_data = NULL;
    
    func = trigger->changed_data;
    data = trigger->changed_data;
    while(func && data) {
	gtk_gmo_trigger_reg_change(t, func->data, data->data);
	func = g_list_next(func);
	data = g_list_next(data);
    }
    
    t->deleted_functions = NULL;
    t->deleted_data = NULL;
    func = trigger->deleted_data;
    data = trigger->deleted_data;
    while(func && data) {
	gtk_gmo_trigger_reg_delete(t, func->data, data->data);
	func = g_list_next(func);
	data = g_list_next(data);
    }
    
    t->refcount = 1;
    if(debug) printf("TRIGGER: trigger \"%s\" duplicated\n", t->name);
    return t;
}

const char *get_str(FILE *file) {
    static char line[4096];
    int len;
    line[0] = '\0';
    if(!fgets(line, 4095, file))
	return NULL;
    len = strlen(line);
    if(len && line[len-1] == '\n')
	line[len-1] = '\0';
    return line;
}
int get_int(FILE *file) {
    const char *s = get_str(file);
    return s ? atoi(s) : 0;
}

void put_int(FILE *file, int i) {
    fprintf(file, "%d\n", i);
}
void put_str(FILE *file, const char *s) {
    fprintf(file, "%s\n", s);
}

   

GtkGmoTrigger *gtk_gmo_trigger_load(FILE *f) {
    int i, count;
    GtkGmoTrigger *t;
    const char *name = get_str(f);
    if(!name)
	return NULL;

    t = g_malloc(sizeof(GtkGmoTrigger));

    /* General */
    t->name = g_strdup(name);
    t->disabled = get_int(f);
    t->type = get_int(f);
    t->pattern = g_strdup(get_str(f));
    t->case_sensitive = get_int(f);
    t->chance = get_int(f);
    t->more_matches = get_int(f);

    /* normal color */
    t->n_color_fg = get_int(f);
    t->n_fgcolor = get_int(f);
    t->n_fgcolors[0] = get_int(f);
    t->n_fgcolors[1] = get_int(f);
    t->n_fgcolors[2] = get_int(f);
    
    t->n_color_bg = get_int(f);
    t->n_bgcolor = get_int(f);
    t->n_bgcolors[0] = get_int(f);
    t->n_bgcolors[1] = get_int(f);
    t->n_bgcolors[2] = get_int(f);
    
    t->n_color_bar = get_int(f);
    t->n_barcolor = get_int(f);
    t->n_barcolors[0] = get_int(f);
    t->n_barcolors[1] = get_int(f);
    t->n_barcolors[2] = get_int(f);
    
    t->n_bold = get_int(f);
    t->n_underline = get_int(f);

    /* Hoover color */
    t->h_color_fg = get_int(f);
    t->h_fgcolor = get_int(f);
    t->h_fgcolors[0] = get_int(f);
    t->h_fgcolors[1] = get_int(f);
    t->h_fgcolors[2] = get_int(f);
    
    t->h_color_bg = get_int(f);
    t->h_bgcolor = get_int(f);
    t->h_bgcolors[0] = get_int(f);
    t->h_bgcolors[1] = get_int(f);
    t->h_bgcolors[2] = get_int(f);
    
    t->h_bold = get_int(f);
    t->h_underline = get_int(f);

    /* Hoover action */
    t->hoover_actions = NULL;
    count = get_int(f);
    for(i = 0; i < count; i++) {
	t->hoover_actions =
	    g_list_append(t->hoover_actions,
			  gtk_gmo_trigger_hoover_load(f));
    }

    /* Trigger action */
    t->send_text = get_int(f);
    t->send_text_text = g_strdup(get_str(f));
    t->run_script = get_int(f);
    t->run_script_function = g_strdup(get_str(f));

    /* Sound */
    t->beep = get_int(f);
    t->play_file = get_int(f);
    t->play_file_name = g_strdup(get_str(f));

    /* Event handling */
    t->changed_functions = NULL;
    t->changed_data = NULL;
    t->deleted_functions = NULL;
    t->deleted_data = NULL;

    /* Misc */
    t->refcount = 1;
    if(debug) printf("TRIGGER: trigger \"%s\" loaded from FILE\n", t->name);
    return t;
}

void gtk_gmo_trigger_save(const GtkGmoTrigger *t, FILE *f) {
    /* General */
    put_str(f, t->name);
    put_int(f, t->disabled);
    put_int(f, t->type);
    put_str(f, t->pattern);
    put_int(f, t->case_sensitive);
    put_int(f, t->chance);
    put_int(f, t->more_matches);

    /* normal color */
    put_int(f, t->n_color_fg);
    put_int(f, t->n_fgcolor);
    put_int(f, t->n_fgcolors[0]);
    put_int(f, t->n_fgcolors[1]);
    put_int(f, t->n_fgcolors[2]);
    
    put_int(f, t->n_color_bg);
    put_int(f, t->n_bgcolor);
    put_int(f, t->n_bgcolors[0]);
    put_int(f, t->n_bgcolors[1]);
    put_int(f, t->n_bgcolors[2]);
    
    put_int(f, t->n_color_bar);
    put_int(f, t->n_barcolor);
    put_int(f, t->n_barcolors[0]);
    put_int(f, t->n_barcolors[1]);
    put_int(f, t->n_barcolors[2]);
    
    put_int(f, t->n_bold);
    put_int(f, t->n_underline);

    /* Hoover color */
    put_int(f, t->h_color_fg);
    put_int(f, t->h_fgcolor);
    put_int(f, t->h_fgcolors[0]);
    put_int(f, t->h_fgcolors[1]);
    put_int(f, t->h_fgcolors[2]);
   
    put_int(f, t->h_color_bg);
    put_int(f, t->h_bgcolor);
    put_int(f, t->h_bgcolors[0]);
    put_int(f, t->h_bgcolors[1]);
    put_int(f, t->h_bgcolors[2]);
    
    put_int(f, t->h_bold);
    put_int(f, t->h_underline);

    /* Hoover action */
    put_int(f, g_list_length(t->hoover_actions));
    g_list_foreach(t->hoover_actions,
		   (GFunc) gtk_gmo_trigger_hoover_save, f);

    /* Trigger action */
    put_int(f, t->send_text);
    put_str(f, t->send_text_text);
    put_int(f, t->run_script);
    put_str(f, t->run_script_function);

    /* Sound */
    put_int(f, t->beep);
    put_int(f, t->play_file);
    put_str(f, t->play_file_name);

    if(debug) printf("TRIGGER: trigger \"%s\" save to FILE\n", t->name);
}



GtkGmoTriggerHoover *gtk_gmo_trigger_hoover_new() {
    GtkGmoTriggerHoover *h = g_malloc(sizeof(GtkGmoTriggerHoover));
    h->button = 0;
    h->type = 0;
    h->action = 0;
    h->action_data = g_strdup("");
    return h;
}
GtkGmoTriggerHoover *gtk_gmo_trigger_hoover_copy(GtkGmoTriggerHoover *hoover) {
    GtkGmoTriggerHoover *h = g_malloc(sizeof(GtkGmoTriggerHoover));
    h->button = hoover->button;
    h->type = hoover->type;
    h->action = hoover->action;
    h->action_data = g_strdup(hoover->action_data);
    return h;
}
GtkGmoTriggerHoover *gtk_gmo_trigger_hoover_load(FILE *f) {
    GtkGmoTriggerHoover *h = g_malloc(sizeof(GtkGmoTriggerHoover));
    h->button = get_int(f);
    h->type = get_int(f);
    h->action = get_int(f);
    h->action_data = g_strdup(get_str(f));
    return h;
}
void gtk_gmo_trigger_hoover_save(GtkGmoTriggerHoover *h, FILE *f) {
    put_int(f, h->button);
    put_int(f, h->type);
    put_int(f, h->action);
    put_str(f, h->action_data);
}
void gtk_gmo_trigger_hoover_free(GtkGmoTriggerHoover *h) {
    g_free(h->action_data);
    g_free(h);
}


/* claim a refcount */
void gtk_gmo_trigger_ref(GtkGmoTrigger *trigger) {
    trigger->refcount++;
    if(debug) printf("TRIGGER: refed (refcount now is %d\n", trigger->refcount);
}

void gtk_gmo_trigger_unref(GtkGmoTrigger *trigger) {
    trigger->refcount--;
    if(debug) printf("TRIGGER: unrefed (refcount now is %d\n", trigger->refcount);
    if(trigger->refcount < 1) {
	if(debug) printf("\tfree()ing\n");
	gtk_gmo_trigger_free(trigger);
    }
}

/* tell the other they should changed their reference */
void gtk_gmo_trigger_emit_change(GtkGmoTrigger *from,
				 GtkGmoTrigger *to) {
    GList *func;
    GList *data;
    func = from->changed_functions;
    data = from->changed_data;
    while(func) {
	((GtkGmoTriggerChanged) func->data)(from, to, data);
	func = g_list_next(func);
	data = g_list_next(data);
    }
}

/* inform the others to unref */
void gtk_gmo_trigger_emit_delete(GtkGmoTrigger *trigger) {
    GList *func;
    GList *data;
    func = trigger->deleted_functions;
    data = trigger->deleted_data;
    while(func) {
	((GtkGmoTriggerDeleted) func->data)(trigger, data);
	func = g_list_next(func);
	data = g_list_next(data);
    }
}

/* (un)registrate change/delete events. */
void gtk_gmo_trigger_reg_change(GtkGmoTrigger *trigger,
				GtkGmoTriggerChanged *function,
				void *data) {
    trigger->changed_functions =
	g_list_append(trigger->changed_functions, function);
    trigger->changed_data =
	g_list_append(trigger->changed_data, data);
}

void gtk_gmo_trigger_unreg_change(GtkGmoTrigger *trigger,
				  GtkGmoTriggerChanged *function,
				  void *data) {
    trigger->changed_functions =
	g_list_remove(trigger->changed_functions, function);
    trigger->changed_data =
	g_list_remove(trigger->changed_data, data);
}

void gtk_gmo_trigger_reg_delete  (GtkGmoTrigger *trigger,
				  GtkGmoTriggerDeleted *function,
				  void *data) {
    trigger->deleted_functions =
	g_list_append(trigger->deleted_functions, function);
    trigger->deleted_data =
	g_list_append(trigger->deleted_data, data);
}
void gtk_gmo_trigger_unreg_delete(GtkGmoTrigger *trigger,
				  GtkGmoTriggerDeleted *function,
				  void *data) {
    trigger->deleted_functions =
	g_list_remove(trigger->deleted_functions, function);
    trigger->deleted_data =
	g_list_remove(trigger->deleted_data, data);
}
