/*
 *  Copyright 2004-2006 Michael Terry
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "MultPuzzle.h"
#include "config.h"
#include "gettext.h"
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <ctype.h>
#include <string.h>

bool MultPuzzle::seeded = false;

/* Declare the letters we are going to be using for translators */
const char *MultPuzzle::LETTERS[10] = {
		/* Translators: Replace with an appropriate variable symbol */
		N_("A"),
		/* Translators: Replace with an appropriate variable symbol */
		N_("B"),
		/* Translators: Replace with an appropriate variable symbol */
		N_("C"),
		/* Translators: Replace with an appropriate variable symbol */
		N_("D"),
		/* Translators: Replace with an appropriate variable symbol */
		N_("E"),
		/* Translators: Replace with an appropriate variable symbol */
		N_("F"),
		/* Translators: Replace with an appropriate variable symbol */
		N_("G"),
		/* Translators: Replace with an appropriate variable symbol */
		N_("H"),
		/* Translators: Replace with an appropriate variable symbol */
		N_("I"),
		/* Translators: Replace with an appropriate variable symbol */
		N_("J")};

MultPuzzle::MultPuzzle (int numXDigits, int numYDigits)
	: x (NULL), y (NULL), z (NULL), numAddends (0), addends (NULL),
	totalGuesses (0), wrongGuesses (0)
{
	int valX, valY, valZ;
	
	/**
	 * First, seed the random number generator if not yet done.
	 */
	if (!seeded)
	{
		srand (time (0));
		seeded = true;
	}
	
	/**
	 * Now, randomize the symbols '0'...'9'.
	 */
	shuffle ();
	
	/**
	 * Get value for x and y.
	 */
	valX = numWithNDigits (numXDigits);
	do {
	valY = numWithNDigits (numYDigits);
	} while (valY % 10 == 0);
	valZ = valX * valY;
	
	x = numToString (valX);
	y = numToString (valY);
	z = numToString (valZ);
	expandNumString (&z, strlen (x) + strlen (y));
	
	for (int i = 0; i < 10; i++)
	{
		needed[i] = false;
	}
	noteNeeded (x);
	noteNeeded (y);
	noteNeeded (z);
	
	/**
	 * Calculate addends.
	 */
	numAddends = strlen (y);
	addends = (char **) malloc (sizeof (char *) * numAddends);
	
	for (int i = 0; i < numAddends; i++)
	{
		int addend = valX * (y[numAddends - i - 1] - 48);
		
		addends[i] = numToString (addend);
		
		expandNumString (&addends[i], strlen (x) + 1);
		
		noteNeeded (addends[i]);
		
		interpret (addends[i]);
	}
	
	interpret (x);
	interpret (y);
	interpret (z);
	
	for (int i = 0; i < 10; i++)
	{
		unknown [i] = true;
		
		for (int j = 0; j < 10; j++)
		{
			have_guessed[j][i] = false;
		}
	}
}


MultPuzzle::~MultPuzzle (void)
{
	free (x);
	free (y);
	free (z);
	
	for (int i = 0; i < numAddends; i++)
	{
		free (addends[i]);
	}
	
	free (addends);
}


int MultPuzzle::guess (char digit, char letter)
{
	int valDigit;
	int rv;
	
	if (!isdigit (digit))
		return INVALID_INPUT;
	
	if (letter < STARTING_LETTER ||
		letter > STARTING_LETTER + 9)
		return INVALID_INPUT;
	
	valDigit = digit - 48;
	if (symbols[valDigit] == letter)
	{
		solve (valDigit);
		
		totalGuesses ++;
		have_guessed[letter - STARTING_LETTER][valDigit] = true;
		
		rv = CORRECT;
	}
	else if (symbols[valDigit] == digit)
	{
		rv =  ALREADY_KNOWN;
	}
	else
	{
		totalGuesses ++;
		wrongGuesses ++;
		have_guessed[letter - STARTING_LETTER][valDigit] = true;
		
		rv =  WRONG;
	}
	
	signal_guessed ().emit (digit, letter, rv);
	
	return rv;
}

int MultPuzzle::getTotalGuesses (void) const
{
	return totalGuesses;
}

int MultPuzzle::getWrongGuesses (void) const
{
	return wrongGuesses;
}

int MultPuzzle::getCorrectGuesses (void) const
{
	return totalGuesses - wrongGuesses;
}

/* returned string must be free'd */
char *MultPuzzle::getMultiplicand (void) const
{
	return strdup (x);
}

/* returned string must be free'd */
char *MultPuzzle::getMultiplier (void) const
{
	return strdup (y);
}

/* returned string must be free'd */
char *MultPuzzle::getAnswer (void) const
{
	return strdup (z);
}

int MultPuzzle::getNumAddends (void) const
{
	return numAddends;
}

/* returned string must be free'd */
char *MultPuzzle::getAddend (int n) const
{
	if (n < 0 || n > numAddends - 1)
		return NULL;
	
	return strdup (addends[n]);
}


bool MultPuzzle::isDone (void) const
{
	for (int i = 0; i < 10; i++)
	{
		if (needed[i])
			return false;
	}
	
	return true;
}

// returned array must be free'd
bool *MultPuzzle::getUnknownDigits (void) const
{
	bool *rv = (bool *) malloc (sizeof (bool) * 10);
	
	memcpy (rv, unknown, sizeof (bool) * 10);
	
	return rv;
}

// returned array must be free'd
bool *MultPuzzle::getLetterGuesses (char letter) const
{
	bool *rv;
	
	if (letter < STARTING_LETTER ||
		letter > STARTING_LETTER + 9)
		return NULL;
	
	rv = (bool *) malloc (sizeof (bool) * 10);
	
	memcpy (rv, have_guessed[letter - STARTING_LETTER], sizeof (bool) * 10);
	
	return rv;
}


sigc::signal <void, char, char, int> MultPuzzle::signal_guessed (void) const
{
	return m_signal_guessed;
}

sigc::signal <void> MultPuzzle::signal_changed (void) const
{
	return m_signal_changed;
}

void MultPuzzle::solve (void)
{
	for (int i = 0; i < 10; i++)
	{
		solve (i);
	}
}

void MultPuzzle::solve (int digit)
{	
	if (digit < 0 || digit > 9)
		return;
	
	if (!needed[digit])
		return;
	
	replace (x, digit);
	replace (y, digit);
	replace (z, digit);
	
	for (int i = 0; i < numAddends; i++)
	{
		replace (addends[i], digit);
	}
	
	symbols[digit] = digit + 48;
	needed[digit] = false;
	unknown[digit] = false;
	
	signal_changed ().emit ();
}


/* pads a string with 0's in the front if it is less than the minimum */
void MultPuzzle::expandNumString (char **n, int minimum)
{
	int slen = strlen (*n);
	int difference = minimum - slen;
	
	if (difference <= 0)
		return;
	
	*n = (char *) realloc (*n, slen + 1 + difference);
	
	memmove (*n + difference, *n, slen);
	
	for (int i = 0; i < difference; i++)
	{
		(*n)[i] = '0';
	}
	
	(*n)[slen + difference] = '\0';
}


/* n is  a string full of digit characters */
void MultPuzzle::noteNeeded (char *n)
{
	int size;
	
	size = strlen (n);
	
	for (int i = 0; i < size; i++)
	{
		int digit;
		
		if (!isdigit (n[i]))
			continue;
		
		digit = n[i] - 48;
		
		needed[digit] = true;
	}
}


/* replaces all instances of whatever character represents the number 'digit' with
 the character for 'digit' */
char *MultPuzzle::replace (char *n, int digit)
{
	char *place;
	
	if (digit < 0 || digit > 9 || !n)
		return NULL;
	
	while ((place = strchr (n, symbols[digit])))
	{
		place[0] = digit + 48;
	}
	
	return n;
}


/* n is a string of numbers */
/* after return, it will consist of each unguessed number replaced
with the appropriate letter */
char *MultPuzzle::interpret (char *n)
{
	int size = strlen (n);
	
	for (int i = 0; i < size; i++)
	{
		if (!isdigit (n[i]))
			continue;
		
		n[i] = symbols[n[i] - 48];
	}
	
	return n;
}


/**
 * Returned string must be free'd.
 */
char *MultPuzzle::numToString (int n)
{
	char *dest = (char *) malloc (13);
	
	sprintf (dest, "%i", n);
	
	return dest;
}


int MultPuzzle::numWithNDigits (int n)
{
	int low, high;
	
	if (n <= 1)
		low = 0;
	else
		low = (int) pow (10, n - 1);
	
	high = (int) pow (10, n) - 1;
	
	return (rand () % (high - low + 1)) + low;
}


void MultPuzzle::shuffle (void)
{
	/**
	 * Here, we need to pick a spot from 0...9, and place the first digit there.
	 * Then, we pick a spot k from 0...8, and place the next digit in the kth free spot.
	 * Repeat.
	 */
	
	for (int i = 0; i < 10; i++)
	{
		symbols[i] = '#';
	}
	
	for (int i = 0; i < 10; i++)
	{
		int spot = rand () % (10 - i);
		int j;
		
		for (j = 0; j < 10; j++)
		{
			if (symbols[j] == '#')
			{
				if (spot)
					spot --;
				else
					break;
			}
		}
		
		symbols[j] = STARTING_LETTER + i;
	}
}
