/**
 * GMyth Library
 *
 * @file_local gmyth/gmyth_file_local.c
 * 
 * @brief <p> GMythFileLocal deals with the file_local streaming media remote/local
 * transfering to the MythTV frontend.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Rosfran Lins Borges <rosfran.borges@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gmyth_file_local.h"
#include "gmyth_recorder.h"
#include "gmyth_util.h"
#include "gmyth_socket.h"
#include "gmyth_stringlist.h"
#include "gmyth_debug.h"
#include "gmyth_uri.h"
#include "gmyth_marshal.h"

#include <unistd.h>
#include <glib.h>

#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <errno.h>
#include <stdlib.h>
#include <assert.h>

#define GMYTH_FILE_LOCAL_GET_PRIVATE(obj) \
	(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GMYTH_FILE_LOCAL_TYPE, GMythFileLocalPrivate))

struct _GMythFileLocalPrivate {

    gboolean        disposed;

    GMutex         *mutex;

    gint            fd;

    GIOChannel     *file_io;

};

static void     gmyth_file_local_class_init(GMythFileLocalClass * klass);
static void     gmyth_file_local_init(GMythFileLocal * object);
static void     gmyth_file_local_dispose(GObject * object);
static void     gmyth_file_local_finalize(GObject * object);

static gboolean _control_acquire_context(GMythFileLocal * file_local,
                                         gboolean do_wait);

static gboolean _control_release_context(GMythFileLocal * file_local);

G_DEFINE_TYPE(GMythFileLocal, gmyth_file_local, GMYTH_FILE_TYPE)
    static void     gmyth_file_local_class_init(GMythFileLocalClass *
                                                klass)
{
    GObjectClass   *gobject_class;
    GMythFileLocalClass *gtransfer_class;

    gobject_class = (GObjectClass *) klass;
    gtransfer_class = (GMythFileLocalClass *) gobject_class;

    gobject_class->dispose = gmyth_file_local_dispose;
    gobject_class->finalize = gmyth_file_local_finalize;

    g_type_class_add_private(gobject_class, sizeof(GMythFileLocalPrivate));

}

static void
gmyth_file_local_init(GMythFileLocal * file_local)
{
    GMythFileLocalPrivate *priv;

    g_return_if_fail(file_local != NULL);

    priv = GMYTH_FILE_LOCAL_GET_PRIVATE(file_local);

    priv->mutex = g_mutex_new();
}

static void
gmyth_file_local_dispose(GObject * object)
{
    GMythFileLocalPrivate *priv;
    GMythFileLocal *file_local = GMYTH_FILE_LOCAL(object);

    g_return_if_fail(file_local != NULL);

    priv = GMYTH_FILE_LOCAL_GET_PRIVATE(file_local);

    if (priv->disposed) {
        /*
         * If dispose did already run, return. 
         */
        return;
    }

    /*
     * Make sure dispose does not run twice. 
     */
    priv->disposed = TRUE;

    if (priv->mutex != NULL) {
        g_mutex_free(priv->mutex);
        priv->mutex = NULL;
    }

    if (priv->file_io != NULL) {
        g_io_channel_unref(priv->file_io);
        priv->file_io = NULL;
    }

    G_OBJECT_CLASS(gmyth_file_local_parent_class)->dispose(object);
}

static void
gmyth_file_local_finalize(GObject * object)
{
    g_signal_handlers_destroy(object);

    G_OBJECT_CLASS(gmyth_file_local_parent_class)->finalize(object);
}

/** 
 * Creates a new instance of GMythFileLocal.
 * 
 * @param backend_info The BackendInfo instance, with all the MythTV network 
 * 										 configuration data.
 * 
 * @return a new instance of the File Transfer. 
 */
GMythFileLocal *
gmyth_file_local_new(GMythBackendInfo * backend_info)
{
    GMythFileLocal *file_local =
        GMYTH_FILE_LOCAL(g_object_new(GMYTH_FILE_LOCAL_TYPE, NULL));

    g_object_set(GMYTH_FILE(file_local), "backend-info", &backend_info,
                 NULL);

    return file_local;
}

/** 
 * Creates a new instance of GMythFileLocal.
 * 
 * @param uri_str The URI poiting to the MythTV backend server.
 * 
 * @return a new instance of the File Transfer.
 */
GMythFileLocal *
gmyth_file_local_new_with_uri(const gchar * uri_str)
{
    GMythFileLocal *file_local =
        GMYTH_FILE_LOCAL(g_object_new(GMYTH_FILE_LOCAL_TYPE, NULL));
    GMythURI       *uri = gmyth_uri_new_with_value(uri_str);

    gmyth_debug("GMythURI path segment = %s", gmyth_uri_get_path(uri));

    g_object_set(GMYTH_FILE(file_local),
                 "backend-info", gmyth_backend_info_new_with_uri(uri_str),
                 "filename", g_strdup(gmyth_uri_get_path(uri)), NULL);

    g_object_unref(uri);

    return file_local;
}

gchar          *
gmyth_file_local_get_file_name(GMythFileLocal * file_local)
{
    return gmyth_file_get_file_name(GMYTH_FILE(file_local));
}

void
gmyth_file_local_set_file_name(GMythFileLocal * file_local,
                               const gchar * filename)
{
    gmyth_file_set_file_name(GMYTH_FILE(file_local), filename);
}

/** 
 * Open a File in order to get a local file.
 * 
 * @param file_local The actual File Transfer instance. 
 * 
 * @return <code>true</code>, if the connection opening had been done successfully. 
 */
gboolean
gmyth_file_local_open(GMythFileLocal * file_local)
{
    gboolean        ret = TRUE;
    GMythFileLocalPrivate *priv;
    gchar          *file_name_uri = NULL;

    g_return_val_if_fail(file_local != NULL, FALSE);

    priv = GMYTH_FILE_LOCAL_GET_PRIVATE(file_local);
    file_name_uri = gmyth_file_local_get_file_name(file_local);

    if (file_name_uri != NULL) {
        priv->file_io =
            g_io_channel_new_file(g_strdup(file_name_uri), "r+", NULL);
        g_free(file_name_uri);
    }

    if (priv->file_io < 0)
        ret = FALSE;

    return ret;
}

/** 
 * Closes a remote File Transfer connection.
 * 
 * @param file_local The actual File Transfer instance. 
 */
void
gmyth_file_local_close(GMythFileLocal * file_local)
{
    g_return_if_fail(file_local != NULL);
}

/** 
 * Acquire access to a local file read/write pointer.
 * 
 * @param transfer The actual File Local instance.
 * @param do_wait Waits or not on a GCond, when trying to read from the remote socket.
 * 
 * @return <code>true</code>, if the acquire had been got. 
 */
static          gboolean
_control_acquire_context(GMythFileLocal * file_local, gboolean do_wait)
{
    gboolean        ret = TRUE;
    GMythFileLocalPrivate *priv;

    g_return_val_if_fail(file_local != NULL, FALSE);
    priv = GMYTH_FILE_LOCAL_GET_PRIVATE(file_local);

    g_mutex_lock(priv->mutex);
    return ret;
}

/** 
 * Release access to a local file read/write pointer.
 * 
 * @param transfer The actual File Transfer instance.
 * 
 * @return <code>true</code>, if the local file read/write permissions had been releaseds. 
 */
static          gboolean
_control_release_context(GMythFileLocal * file_local)
{
    gboolean        ret = TRUE;
    GMythFileLocalPrivate *priv;

    g_return_val_if_fail(file_local != NULL, FALSE);
    priv = GMYTH_FILE_LOCAL_GET_PRIVATE(file_local);

    g_mutex_unlock(priv->mutex);

    return ret;
}

/** 
 * Reads a block from a remote file.
 * 
 * @param transfer The actual File Transfer instance.
 * @param data A GByteArray instance, where all the binary data representing 
 *                       the remote file will be stored.
 * @param size The block size, in bytes, to be requested from a remote file.
 * @param read_unlimited Tells the backend to read indefinitely (LiveTV), or only 
 *                                           gets the actual size
 * 
 * @return The actual block size (in bytes) returned by REQUEST_BLOCK message,
 *              or the error code. 
 */
GMythFileReadResult
gmyth_file_local_read(GMythFileLocal * file_local, GByteArray * data,
                      gint size, gboolean read_unlimited)
{
    gsize           bytes_read = 0;
    gint64          total_read = 0;
    GMythFileReadResult retval = GMYTH_FILE_READ_OK;
    GMythFileLocalPrivate *priv;

    GError         *error = NULL;

    GIOCondition    io_cond;
    GIOStatus       io_status = G_IO_STATUS_NORMAL;

    g_return_val_if_fail(file_local != NULL, FALSE);
    g_return_val_if_fail(data != NULL, GMYTH_FILE_READ_ERROR);

    priv = GMYTH_FILE_LOCAL_GET_PRIVATE(file_local);

    io_status = g_io_channel_set_encoding(priv->file_io, NULL, &error);
    if (io_status == G_IO_STATUS_NORMAL)
        gmyth_debug("Setting encoding to binary file data stream.\n");

    io_cond = g_io_channel_get_buffer_condition(priv->file_io);

    _control_acquire_context(file_local, TRUE);

    if (size > 0) {
        gchar          *data_buffer = g_new0(gchar, size);

        io_status = g_io_channel_read_chars(priv->file_io,
                                            data_buffer, (gsize) size,
                                            &bytes_read, &error);

        if (io_status != G_IO_STATUS_NORMAL) {
            gmyth_debug("Error on io_channel");
            g_free(data_buffer);
            retval = GMYTH_FILE_READ_ERROR;
            goto error;
        }

        /*
         * append new data to the increasing byte array 
         */
        data =
            g_byte_array_append(data, (const guint8 *) data_buffer,
                                bytes_read);
        total_read += bytes_read;

        if (!read_unlimited
            && (gmyth_file_local_get_filesize(file_local) > 0)
            && (gmyth_file_local_get_offset(file_local) ==
                gmyth_file_local_get_filesize(file_local))) {
            retval = GMYTH_FILE_READ_EOF;
            goto error;
        }

        g_free(data_buffer);
    } else {
        retval = GMYTH_FILE_READ_ERROR;
    }

  error:
    _control_release_context(file_local);

    if (error != NULL) {
        gmyth_debug("Cleaning-up ERROR: [msg = %s, code = %d]\n",
                    error->message, error->code);
        g_error_free(error);
    }

    if (total_read > 0)
        gmyth_file_local_set_offset(file_local,
                                    (gmyth_file_local_get_offset
                                     (file_local) + total_read));

    return retval;
}

gint64
gmyth_file_local_seek(GMythFileLocal * file_local, gint64 pos,
                      GSeekType whence)
{
    GMythFileLocalPrivate *priv;

    GError         *error;

    GIOStatus       io_status = G_IO_STATUS_NORMAL;

    g_return_val_if_fail(file_local != NULL, -1);

    priv = GMYTH_FILE_LOCAL_GET_PRIVATE(file_local);

    io_status =
        g_io_channel_seek_position(priv->file_io, pos, whence, &error);

    if (io_status == G_IO_STATUS_ERROR)
        pos = -1;

    return pos;

}

/** 
 * Gets the actual file_local size of the binary content.
 * 
 * @param file_local The actual File Transfer instance.
 * 
 * @return The actual file_local size in bytes. 
 */
guint64
gmyth_file_local_get_filesize(GMythFileLocal * file_local)
{
    g_return_val_if_fail(file_local != NULL, 0);

    return gmyth_file_get_filesize(GMYTH_FILE(file_local));
}

/** 
 * Sets the actual file_local size.
 * 
 * @param file_local The actual File Transfer instance.
 * @param filesize The actual File Transfer size, in bytes.
 */
void
gmyth_file_local_set_filesize(GMythFileLocal * file_local,
                              guint64 filesize)
{
    g_return_if_fail(file_local != NULL);

    gmyth_file_set_filesize(GMYTH_FILE(file_local), filesize);
}

/** 
 * Gets the actual file offset of the binary content.
 * 
 * @param file_local The actual File Transfer instance.
 * 
 * @return The actual file offset in bytes.
 */
gint64
gmyth_file_local_get_offset(GMythFileLocal * file_local)
{
    g_return_val_if_fail(file_local != NULL, 0);

    return gmyth_file_get_offset(GMYTH_FILE(file_local));
}

/** 
 * Sets the actual file offset.
 * 
 * @param file_local The actual File Local instance.
 * @param offset The actual File Local offset, in bytes.
 */
void
gmyth_file_local_set_offset(GMythFileLocal * file_local, gint64 offset)
{
    g_return_if_fail(file_local != NULL);

    gmyth_file_set_offset(GMYTH_FILE(file_local), offset);
}
