/**
 * GMyth Library
 *
 * @file gmyth/gmyth_file_transfer.c
 * 
 * @brief <p> GMythFileTransfer deals with the file streaming media remote/local
 * transfering to the MythTV frontend.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Rosfran Lins Borges <rosfran.borges@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * GStreamer MythTV plug-in properties:
 * - location (backend server hostname/URL) [ex.: myth://192.168.1.73:28722/1000_1092091.nuv]
 * - path (qurl - remote file to be opened)
 * - port number *   
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gmyth_file_transfer.h"
#include "gmyth_recorder.h"
#include "gmyth_util.h"
#include "gmyth_socket.h"
#include "gmyth_stringlist.h"
#include "gmyth_debug.h"
#include "gmyth_uri.h"
#include "gmyth_marshal.h"

#include <unistd.h>
#include <glib.h>

#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <errno.h>
#include <stdlib.h>
#include <assert.h>

#define GMYTHTV_QUERY_HEADER		"QUERY_FILETRANSFER "


#define GMYTH_FILE_TRANSFER_GET_PRIVATE(obj) \
	(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GMYTH_FILE_TRANSFER_TYPE, GMythFileTransferPrivate))

enum myth_sock_types {
    GMYTH_PLAYBACK_TYPE = 0,
    GMYTH_MONITOR_TYPE,
    GMYTH_FILETRANSFER_TYPE,
    GMYTH_RINGBUFFER_TYPE
};

struct _GMythFileTransferPrivate {
    GMythRecorder  *recorder;

    gboolean        do_next_program_chain;
    gboolean        disposed;
    gboolean        livetv_wait;

    /*
     * MythTV version number 
     */
    gint            mythtv_version;

    /*
     * socket descriptors 
     */
    GMythSocket    *control_sock;
    GMythSocket    *sock;
    GMutex         *mutex;
    gint            file_id;
};

static void     gmyth_file_transfer_class_init(GMythFileTransferClass *
                                               klass);
static void     gmyth_file_transfer_init(GMythFileTransfer * object);
static void     gmyth_file_transfer_dispose(GObject * object);
static void     gmyth_file_transfer_finalize(GObject * object);
static void     _file_transfer_program_info_changed(GMythFileTransfer *
                                                    transfer,
                                                    gint msg_code,
                                                    gpointer
                                                    livetv_recorder);
static gboolean _connect_to_backend(GMythFileTransfer * transfer);
static gboolean _control_acquire_context(GMythFileTransfer * transfer,
                                         gboolean do_wait);
static gboolean _control_release_context(GMythFileTransfer * transfer);

G_DEFINE_TYPE(GMythFileTransfer, gmyth_file_transfer, GMYTH_FILE_TYPE)
    static void     gmyth_file_transfer_class_init(GMythFileTransferClass *
                                                   klass)
{
    GObjectClass   *gobject_class;
    GMythFileTransferClass *gtransfer_class;

    gobject_class = (GObjectClass *) klass;
    gtransfer_class = (GMythFileTransferClass *) gobject_class;

    gobject_class->dispose = gmyth_file_transfer_dispose;
    gobject_class->finalize = gmyth_file_transfer_finalize;

    g_type_class_add_private(gobject_class,
                             sizeof(GMythFileTransferPrivate));

    gtransfer_class->program_info_changed_handler =
        _file_transfer_program_info_changed;

    gtransfer_class->program_info_changed_handler_signal_id =
        g_signal_new("program-info-changed",
                     G_TYPE_FROM_CLASS(gtransfer_class),
                     G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE |
                     G_SIGNAL_NO_HOOKS, 0, NULL, NULL,
                     gmyth_marshal_VOID__INT_POINTER, G_TYPE_NONE, 2,
                     G_TYPE_INT, G_TYPE_POINTER);

}

static void
gmyth_file_transfer_init(GMythFileTransfer * transfer)
{
    g_return_if_fail(transfer != NULL);

    transfer->priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);
    transfer->priv->mutex = g_mutex_new();

    g_signal_connect(G_OBJECT(transfer), "program-info-changed",
                     (GCallback) (GMYTH_FILE_TRANSFER_GET_CLASS
                         (transfer)->program_info_changed_handler),
                         NULL);
}

static void
gmyth_file_transfer_dispose(GObject * object)
{
    GMythFileTransferPrivate *priv;
    GMythFileTransfer *transfer = GMYTH_FILE_TRANSFER(object);

    g_return_if_fail(transfer != NULL);

    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);

    if (priv->disposed) {
        /*
         * If dispose did already run, return. 
         */
        return;
    }

    /*
     * Make sure dispose does not run twice. 
     */
    priv->disposed = TRUE;

    if (priv->mutex != NULL) {
        g_mutex_free(priv->mutex);
        priv->mutex = NULL;
    }

    if (priv->control_sock != NULL) {
        g_object_unref(priv->control_sock);
        priv->control_sock = NULL;
    }

    if (priv->sock != NULL) {
        g_object_unref(priv->sock);
        priv->sock = NULL;
    }

    if (priv->recorder != NULL) {
        g_object_unref(priv->recorder);
        priv->recorder = NULL;
    }

    G_OBJECT_CLASS(gmyth_file_transfer_parent_class)->dispose(object);
}

static void
gmyth_file_transfer_finalize(GObject * object)
{
    g_signal_handlers_destroy(object);

    G_OBJECT_CLASS(gmyth_file_transfer_parent_class)->finalize(object);
}

/** 
 * Creates a new instance of GMythFileTransfer.
 * 
 * @param backend_info The BackendInfo instance, with all the MythTV network 
 * 										 configuration data.
 * 
 * @return a new instance of the File Transfer. 
 */
GMythFileTransfer *
gmyth_file_transfer_new(GMythBackendInfo * backend_info)
{
    GMythFileTransfer *transfer = g_object_new(GMYTH_FILE_TRANSFER_TYPE,
                                               "backend-info",
                                               backend_info,
                                               NULL);

    // GValue val = {0,}; 
    // backend_info = g_object_ref( backend_info );
    gmyth_debug("Creating FileTransfer BackendInfo hostname = %s",
                gmyth_backend_info_get_hostname(backend_info));
    // GMythBackendInfo *backend_info = gmyth_backend_info_new_with_uri
    // (uri_str);
    // g_value_init (&val, G_TYPE_OBJECT);
    // g_value_set_object (&val, backend_info); 
    // g_object_set (G_OBJECT (transfer), "backend-info", &val, NULL);

    return transfer;
}

gchar          *
gmyth_file_transfer_get_file_name(GMythFileTransfer * transfer)
{
    gchar          *filename;

    g_object_get(G_OBJECT(transfer), "filename", &filename, NULL);

    return filename;
}

/** 
 * Creates a new instance of GMythFileTransfer.
 * 
 * @param uri_str The URI poiting to the MythTV backend server.
 * 
 * @return a new instance of the File Transfer. 
 */
GMythFileTransfer *
gmyth_file_transfer_new_with_uri(const gchar * uri_str)
{
    GMythFileTransfer *transfer =
        GMYTH_FILE_TRANSFER(g_object_new(GMYTH_FILE_TRANSFER_TYPE, NULL));
    gmyth_debug("URI str = %s", uri_str);
    // GMythBackendInfo *backend_info = gmyth_backend_info_new_with_uri
    // (uri_str);
    GValue          val = { 0, };
    g_value_init(&val, G_TYPE_OBJECT);
    g_value_set_object(&val, gmyth_backend_info_new_with_uri(uri_str));
    g_object_set(G_OBJECT(transfer), "backend-info", &val, NULL);

    return transfer;
}

/** 
 * Open a File Transfer connection in order to get a remote file.
 * 
 * @param transfer The actual File Transfer instance. 
 * @param filename The file name of the remote file to be transferred to the client.
 * 
 * @return <code>true</code>, if the connection opening had been done successfully. 
 */
gboolean
gmyth_file_transfer_open(GMythFileTransfer * transfer,
                         const gchar * filename)
{
    gboolean        ret = TRUE;
    GMythFileTransferPrivate *priv;

    g_return_val_if_fail(transfer != NULL, FALSE);
    g_return_val_if_fail(filename != NULL && strlen(filename) > 0, FALSE);

    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);

    gmyth_debug("Opening the FileTransfer... (%s)", filename);

    g_object_set(GMYTH_FILE(transfer), "filename", filename, NULL);

    /*
     * configure the control socket 
     */
    if (TRUE /* priv->control_sock == NULL */ ) {
        if (!_connect_to_backend(transfer)) {
            gmyth_debug("Connection to backend failed (Control Socket).");
            ret = FALSE;
        }

        if (priv->do_next_program_chain) {
            priv->do_next_program_chain = FALSE;    // fixme
            gmyth_debug
                ("New file available before the current file was opened");
            GMythProgramInfo *prog_info =
                gmyth_recorder_get_current_program_info(priv->recorder);

            if (prog_info != NULL && prog_info->pathname != NULL
                && strlen(prog_info->pathname->str) > 0
                && g_ascii_strcasecmp(prog_info->pathname->str,
                                      gmyth_file_get_file_name(GMYTH_FILE
                                                               (transfer)))
                != 0)
                ret =
                    gmyth_file_transfer_open(transfer,
                                             g_strrstr(prog_info->
                                                       pathname->str,
                                                       "/"));

            if (prog_info != NULL)
                g_object_unref(prog_info);

            if (!ret)
                gmyth_debug("Cannot change to the next program info!");
            else
                gmyth_debug("OK!!! MOVED to the next program info [%s]!",
                            gmyth_file_get_file_name(GMYTH_FILE
                                                     (transfer)));

        } else {
            gmyth_debug
                ("None new file found. We continue with the same file opened before");
        }

    } else {
        gmyth_debug("Remote transfer control socket already created.");
    }

    gmyth_debug("Got file with size = %lld.\n",
                gmyth_file_get_filesize(GMYTH_FILE(transfer)));

    return ret;
}

/** 
 * Connect a File Transfer binary client socket to a remote file.
 * 
 * @param transfer The actual File Transfer instance. 
 * 
 * @return <code>true</code>, if the connection had been configured successfully. 
 */
static          gboolean
_connect_to_backend(GMythFileTransfer * transfer)
{
    GString        *base_str = NULL;
    GString        *hostname = NULL;
    GMythStringList *strlist = NULL;
    gboolean        ret = TRUE;
    GMythFileTransferPrivate *priv;
    GMythBackendInfo *backend_info;

    g_return_val_if_fail(transfer != NULL, FALSE);

    g_object_get(GMYTH_FILE(transfer), "backend-info", &backend_info,
                 NULL);

    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);
    _control_acquire_context(transfer, TRUE);

    /*
     * Creates the control socket 
     */

    if (priv->control_sock != NULL) {
        g_object_unref(priv->control_sock);
        priv->control_sock = NULL;
    }

    base_str = g_string_new("");

    priv->control_sock = gmyth_socket_new();
    // Connects the socket, send Mythtv ANN command and verify Mythtv
    // protocol version 
    if (!gmyth_socket_connect_to_backend(priv->control_sock,
                                         backend_info->hostname,
                                         backend_info->port, TRUE)) {

        _control_release_context(transfer);
        g_object_unref(priv->control_sock);
        priv->control_sock = NULL;
        return FALSE;
    }

    /*
     * Creates the data socket 
     */
    if (priv->sock != NULL) {
        g_object_unref(priv->sock);
        priv->sock = NULL;
    }

    priv->sock = gmyth_socket_new();
    gmyth_socket_connect(priv->sock, backend_info->hostname,
                         backend_info->port);
    gmyth_debug("Connecting file transfer... (%s, %d)",
                backend_info->hostname, backend_info->port);

    strlist = gmyth_string_list_new();
    hostname = gmyth_socket_get_local_hostname();
    gmyth_debug("[%s] MythTV version (from backend) = %d.\n", __FUNCTION__,
                priv->control_sock->mythtv_version);
    if (priv->control_sock->mythtv_version > 26)
        g_string_printf(base_str, "ANN FileTransfer %s 1 -1",
                        hostname->str);
    else
        g_string_printf(base_str, "ANN FileTransfer %s", hostname->str);

    gmyth_string_list_append_string(strlist, base_str);
    gmyth_string_list_append_char_array(strlist,
                                        gmyth_file_get_file_name(GMYTH_FILE
                                                                 (transfer)));

    gmyth_socket_write_stringlist(priv->sock, strlist);

    /*
     * MONITOR Handler - DVB TV Chain update messages!!! 
     */

    gmyth_socket_read_stringlist(priv->sock, strlist);

    /*
     * file identification used in future file transfer requests to
     * backend 
     */
    priv->file_id = gmyth_string_list_get_int(strlist, 1);

    /*
     * Myth URI stream file size - decoded using two 8-bytes sequences (64 
     * bits/long long types) 
     */
    gmyth_file_set_filesize(GMYTH_FILE(transfer),
                            gmyth_string_list_get_int64(strlist, 2));

    gmyth_debug("***** Received: recordernum = %d, filesize = %"
                G_GUINT64_FORMAT "\n", priv->file_id,
                gmyth_file_get_filesize(GMYTH_FILE(transfer)));

    if (gmyth_file_get_filesize(GMYTH_FILE(transfer)) < 0) {
        gmyth_debug
            ("Got filesize equals to %llu is lesser than 0 [invalid stream file]\n",
             gmyth_file_get_filesize(GMYTH_FILE(transfer)));
        g_object_unref(priv->sock);
        priv->sock = NULL;
        ret = FALSE;
    }

    _control_release_context(transfer);

    if (strlist != NULL)
        g_object_unref(strlist);

    if (base_str != NULL)
        g_string_free(base_str, TRUE);

    if (hostname != NULL)
        g_string_free(hostname, TRUE);

    return ret;
}

/** 
 * Receives a GObject signal coming from a LiveTV instance, all the time a 
 * program info changes.
 * 
 * @param transfer The actual File Transfer instance. 
 * @param msg_code The MythTV backend message status code.
 * @param live_tv A pointer to the LiveTV instance. * 
 */
void
gmyth_file_transfer_emit_program_info_changed_signal(GMythFileTransfer *
                                                     transfer,
                                                     gint msg_code,
                                                     gpointer
                                                     live_tv_recorder)
{
    gmyth_debug("Calling signal handler... [FILE_TRANSFER]");

    g_signal_emit(transfer, GMYTH_FILE_TRANSFER_GET_CLASS(transfer)->program_info_changed_handler_signal_id, 0, /* details 
                                                                                                                 */
                  msg_code, live_tv_recorder);

}

/** 
 * Checks if the actual File Transfer connection is open.
 * 
 * @param transfer The actual File Transfer instance. 
 * 
 * @return <code>true</code>, if the File Transfer connection is opened. 
 */
gboolean
gmyth_file_transfer_is_open(GMythFileTransfer * transfer)
{
    GMythStringList *strlist;
    GMythFileTransferPrivate *priv;
    GString        *query;

    g_return_val_if_fail(transfer != NULL, FALSE);

    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);
    g_return_val_if_fail(priv->control_sock != NULL, FALSE);
    g_return_val_if_fail(priv->sock != NULL, FALSE);

    _control_acquire_context(transfer, TRUE);

    strlist = gmyth_string_list_new();
    query = g_string_new(GMYTHTV_QUERY_HEADER);
    g_string_append_printf(query, "%d", priv->file_id);

    gmyth_string_list_append_string(strlist, query);
    gmyth_string_list_append_char_array(strlist, "IS_OPEN");

    gmyth_socket_write_stringlist(priv->control_sock, strlist);
    gmyth_socket_read_stringlist(priv->control_sock, strlist);

    _control_release_context(transfer);

    g_string_free(query, TRUE);
    g_object_unref(strlist);

    return (strlist != NULL && gmyth_string_list_get_int(strlist, 0) == 1);
}

/** 
 * Closes a remote File Transfer connection.
 * 
 * @param transfer The actual File Transfer instance. 
 */
void
gmyth_file_transfer_close(GMythFileTransfer * transfer)
{
    GMythStringList *strlist;
    GMythFileTransferPrivate *priv;
    GString        *query;

    g_return_if_fail(transfer != NULL);

    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);

    if (priv->control_sock == NULL)
        return;

    _control_acquire_context(transfer, TRUE);

    strlist = gmyth_string_list_new();
    query = g_string_new(GMYTHTV_QUERY_HEADER);
    g_string_append_printf(query, "%d", priv->file_id);

    gmyth_string_list_append_string(strlist, query);
    gmyth_string_list_append_char_array(strlist, "DONE");

    if (gmyth_socket_sendreceive_stringlist(priv->control_sock, strlist) <=
        0) {
        // fixme: time out???
        gmyth_debug("Remote file timeout.\n");
    }

    g_string_free(query, TRUE);
    g_object_unref(strlist);

    if (priv->sock) {
        g_object_unref(priv->sock);
        priv->sock = NULL;
    }

    if (priv->control_sock) {
        g_object_unref(priv->control_sock);
        priv->control_sock = NULL;
    }

    _control_release_context(transfer);
}

/** 
 * Do a seek operation (moves the read/write file pointer) on a remote file.
 * 
 * @param transfer The actual File Transfer instance. 
 * @param pos The position to be moved in the remote file.
 * @param whence Tells to what direction seek movement should be done.
 * 
 * @return The actual position on the remote file (after seek has been done). 
 */
gint64
gmyth_file_transfer_seek(GMythFileTransfer * transfer, guint64 pos,
                         gint whence)
{
    GMythStringList *strlist = gmyth_string_list_new();
    GMythFileTransferPrivate *priv;
    GString        *query;

    g_return_val_if_fail(transfer != NULL, FALSE);
    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);

    g_return_val_if_fail(priv->sock != NULL, -1);
    g_return_val_if_fail(priv->control_sock != NULL, -1);

    strlist = gmyth_string_list_new();
    query = g_string_new(GMYTHTV_QUERY_HEADER);
    g_string_append_printf(query, "%d", priv->file_id);

    /*
     * myth_control_acquire_context( transfer, TRUE ); 
     */

    gmyth_string_list_append_string(strlist, query);
    gmyth_string_list_append_char_array(strlist, "SEEK");
    gmyth_string_list_append_uint64(strlist, pos);

    gmyth_string_list_append_int(strlist, whence);

    if (pos > 0)
        gmyth_string_list_append_uint64(strlist, pos);
    else
        gmyth_string_list_append_uint64(strlist,
                                        gmyth_file_get_offset(GMYTH_FILE
                                                              (transfer)));

    gmyth_socket_sendreceive_stringlist(priv->control_sock, strlist);

    gint64          retval = gmyth_string_list_get_int64(strlist, 0);

    gmyth_file_set_offset(GMYTH_FILE(transfer), retval);
    gmyth_debug("Got reading position pointer from the streaming = %lld\n",
                retval);

    g_object_unref(strlist);
    g_string_free(query, TRUE);

    /*
     * myth_control_release_context( transfer ); 
     */

    return retval;
}

/** 
 * Acquire access to a remote file socket read/write pointer.
 * 
 * @param transfer The actual File Transfer instance.
 * @param do_wait Waits or not on a GCond, when trying to read from the remote socket.
 * 
 * @return <code>true</code>, if the acquire had been got. 
 */
static          gboolean
_control_acquire_context(GMythFileTransfer * transfer, gboolean do_wait)
{
    gboolean        ret = TRUE;
    GMythFileTransferPrivate *priv;

    g_return_val_if_fail(transfer != NULL, FALSE);
    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);

    g_mutex_lock(priv->mutex);
    return ret;
}

/** 
 * Release access to a remote file socket read/write pointer.
 * 
 * @param transfer The actual File Transfer instance.
 * 
 * @return <code>true</code>, if the socket read/write permissions had been releaseds. 
 */
static          gboolean
_control_release_context(GMythFileTransfer * transfer)
{
    gboolean        ret = TRUE;
    GMythFileTransferPrivate *priv;

    g_return_val_if_fail(transfer != NULL, FALSE);
    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);

    g_mutex_unlock(priv->mutex);

    return ret;
}

/** 
 * Reads a block from a remote file.
 * 
 * @param transfer The actual File Transfer instance.
 * @param data A GByteArray instance, where all the binary data representing 
 * 						 the remote file will be stored.
 * @param size The block size, in bytes, to be requested from a remote file.
 * @param read_unlimited Tells the backend to read indefinitely (LiveTV), or only 
 * 			 gets the actual size
 * 
 * @return The actual block size (in bytes) returned by REQUEST_BLOCK message,
 * 				or the error code. 
 */
GMythFileReadResult
gmyth_file_transfer_read(GMythFileTransfer * transfer, GByteArray * data,
                         gint size, gboolean read_unlimited)
{
    gint            bytes_sent = 0;
    gsize           bytes_read = 0;
    gint64          total_read = 0;
    GMythFileReadResult retval = GMYTH_FILE_READ_OK;
    GMythFileTransferPrivate *priv;

    GError         *error = NULL;

    GIOChannel     *io_channel;
    GIOChannel     *io_channel_control;

    GIOCondition    io_cond;
    GIOCondition    io_cond_control;
    GIOStatus       io_status = G_IO_STATUS_NORMAL;
    GIOStatus       io_status_control = G_IO_STATUS_NORMAL;

    GMythStringList *strlist;
    GMythStringList *ret_strlist = NULL;
    gboolean        ret = TRUE;
    GString        *query;

    g_return_val_if_fail(transfer != NULL, FALSE);
    g_return_val_if_fail(data != NULL, GMYTH_FILE_READ_ERROR);

    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);

    strlist = gmyth_string_list_new();

    io_channel = priv->sock->sd_io_ch;
    io_channel_control = priv->control_sock->sd_io_ch;

    io_status = g_io_channel_set_encoding(io_channel, NULL, &error);
    if (io_status == G_IO_STATUS_NORMAL)
        gmyth_debug("[%s] Setting encoding to binary data socket).\n",
                    __FUNCTION__);

    io_cond = g_io_channel_get_buffer_condition(io_channel);

    io_cond_control = g_io_channel_get_buffer_condition(io_channel);
    if (priv->sock == NULL || (io_status == G_IO_STATUS_ERROR)) {
        g_printerr
            ("gmyth_file_transfer_read(): Called with no raw socket.\n");
        return GMYTH_FILE_READ_ERROR;
    }

    if (priv->control_sock == NULL
        || (io_status_control == G_IO_STATUS_ERROR)) {
        g_printerr
            ("gmyth_file_transfer_read(): Called with no control socket.\n");
        return GMYTH_FILE_READ_ERROR;
    }

    query = g_string_new(GMYTHTV_QUERY_HEADER);
    g_string_append_printf(query, "%d", priv->file_id);
    gmyth_debug("[%s] Transfer_query = %s\n", __FUNCTION__, query->str);

    _control_acquire_context(transfer, TRUE);
    // Do Read
    gmyth_string_list_append_char_array(strlist, query->str);
    gmyth_string_list_append_char_array(strlist, "REQUEST_BLOCK");
    gmyth_string_list_append_int(strlist, size - total_read);

    guint           iter_count = 3;

    do {
        bytes_sent = 0;

        // Request the block to the backend
        gmyth_socket_write_stringlist(priv->control_sock, strlist);

        if (ret_strlist != NULL)
            g_object_unref(ret_strlist);

        ret_strlist = gmyth_string_list_new();
        // Receives the backand answer 
        gmyth_socket_read_stringlist(priv->control_sock, ret_strlist);

        if (ret_strlist != NULL
            && gmyth_string_list_length(ret_strlist) > 0) {
            bytes_sent = gmyth_string_list_get_int(ret_strlist, 0); // -1
            // on
            // backend 
            // error
            gmyth_debug("[%s] got SENT buffer message = %d\n",
                        __FUNCTION__, bytes_sent);
        }

        if (read_unlimited && (bytes_sent == 0)) {
            g_usleep(300);
        }

        --iter_count;

    }
    while (read_unlimited && (bytes_sent == 0) && iter_count > 0);

    if (bytes_sent > 0) {
        gchar          *data_buffer = g_new0(gchar, bytes_sent);

        io_status = g_io_channel_read_chars(io_channel,
                                            data_buffer,
                                            (gsize) bytes_sent,
                                            &bytes_read, &error);

        if (io_status != G_IO_STATUS_NORMAL) {
            gmyth_debug("Error on io_channel");
            g_free(data_buffer);
            g_object_unref(strlist);
            retval = GMYTH_FILE_READ_ERROR;
            goto error;
        }

        /*
         * append new data to the increasing byte array 
         */
        data =
            g_byte_array_append(data, (const guint8 *) data_buffer,
                                bytes_read);
        gmyth_file_set_offset(GMYTH_FILE(transfer),
                              gmyth_file_get_offset(GMYTH_FILE(transfer)) +
                              bytes_read);

        if (!read_unlimited
            && (gmyth_file_get_filesize(GMYTH_FILE(transfer)) > 0)
            && (gmyth_file_get_offset(GMYTH_FILE(transfer)) ==
                gmyth_file_get_filesize(GMYTH_FILE(transfer)))) {
            retval = GMYTH_FILE_READ_EOF;
            goto error;
        }

        g_free(data_buffer);
    } else {
        retval = GMYTH_FILE_READ_ERROR;
    }

    if (strlist != NULL) {
        g_object_unref(strlist);
        strlist = NULL;
    }

    if (ret_strlist != NULL) {
        g_object_unref(ret_strlist);
        ret_strlist = NULL;
    }

    if (read_unlimited && (bytes_sent == 0)) {
        gmyth_debug("Trying to move to the next program chain...");
        if (priv->recorder != NULL && priv->do_next_program_chain) {
            priv->do_next_program_chain = FALSE;
            retval = GMYTH_FILE_READ_NEXT_PROG_CHAIN;
            GMythProgramInfo *prog_info =
                gmyth_recorder_get_current_program_info(priv->recorder);

            gmyth_debug
                ("Comparing if the current prog. info = %s [strlen == %d] is equals to "
                 " %s [strlen == %d]...", prog_info->pathname->str,
                 strlen(prog_info->pathname->str),
                 gmyth_file_get_file_name(GMYTH_FILE(transfer)),
                 strlen(gmyth_file_get_file_name(GMYTH_FILE(transfer))));

            if (prog_info != NULL && prog_info->pathname != NULL
                && strlen(prog_info->pathname->str) > 0
                && (NULL ==
                    g_strstr_len(prog_info->pathname->str,
                                 strlen(prog_info->pathname->str),
                                 gmyth_file_get_file_name(GMYTH_FILE
                                                          (transfer))))) {
                /*
                 * releasing context got at this function starting... 
                 */
                _control_release_context(transfer);
                ret =
                    gmyth_file_transfer_open(transfer,
                                             g_strrstr(prog_info->
                                                       pathname->str,
                                                       "/"));
                _control_acquire_context(transfer, TRUE);
                /*
                 * acquiring context released at this function stopping... 
                 */

                if (prog_info != NULL)
                    g_object_unref(prog_info);

                if (!ret)
                    gmyth_debug("Cannot change to the next program info!");
                else
                    gmyth_debug
                        ("OK!!! MOVED to the next program info [%s]!",
                         gmyth_file_get_file_name(GMYTH_FILE(transfer)));
            }

        }

    }
    /*
     * if 
     */
  error:

    _control_release_context(transfer);
    g_string_free(query, TRUE);

    if (error != NULL) {
        gmyth_debug("Cleaning-up ERROR: %s [msg = %s, code = %d]\n",
                    __FUNCTION__, error->message, error->code);
        g_error_free(error);
    }

    if (total_read > 0)
        gmyth_file_set_offset(GMYTH_FILE(transfer),
                              gmyth_file_get_offset(GMYTH_FILE
                                                    (transfer)) +
                              total_read);

    return retval;
}

static void
_file_transfer_program_info_changed(GMythFileTransfer * transfer,
                                    gint msg_code,
                                    gpointer livetv_recorder)
{
    GMythRecorder  *recorder;
    GMythFileTransferPrivate *priv;

    g_return_if_fail(transfer != NULL);

    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);

    recorder = GMYTH_RECORDER(livetv_recorder);
    gmyth_debug
        ("Program info changed! ( file transfer orig. = %p, ptr. = [%s] )",
         transfer, livetv_recorder != NULL ? "[NOT NULL]" : "[NULL]");

    if (NULL != recorder) {
        gmyth_debug
            ("YES, the requested program info movement on the LiveTV transfer is authentical!");
    }

    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);
    g_object_ref(recorder);
    priv->recorder = recorder;
    priv->do_next_program_chain = TRUE;
}

/** 
 * Sets the timeout flag of a file being transfered.
 * 
 * @param transfer The actual File Transfer instance.
 * @param fast If this is <code>true</code>, sets the remote timeout to be as fast
 * 						as possible.
 * 
 * @return <code>true</code>, if the acquire had been got. 
 */
gboolean
gmyth_file_transfer_settimeout(GMythFileTransfer * transfer, gboolean fast)
{
    GMythFileTransferPrivate *priv;
    GMythStringList *strlist = NULL;

    g_return_val_if_fail(transfer != NULL, FALSE);

    priv = GMYTH_FILE_TRANSFER_GET_PRIVATE(transfer);

    g_return_val_if_fail(priv->sock != NULL, FALSE);
    g_return_val_if_fail(priv->control_sock != NULL, FALSE);

    _control_acquire_context(transfer, TRUE);

    strlist = gmyth_string_list_new();
    gmyth_string_list_append_char_array(strlist, GMYTHTV_QUERY_HEADER);
    gmyth_string_list_append_char_array(strlist, "SET_TIMEOUT");
    gmyth_string_list_append_int(strlist, fast);

    gint            strlist_len =
        gmyth_socket_sendreceive_stringlist(priv->control_sock,
                                            strlist);

    if (strlist_len > 0)
        gmyth_debug("Yes, timeout was changed: %s.",
                    gmyth_string_list_get_char_array(strlist, 0));
    else
        gmyth_debug("Timeout cannot be changed!");

    _control_release_context(transfer);

    gmyth_debug("%s setting timeout flag of this file transfer = %s\n",
                strlist_len > 0 ? "Yes," : "NOT",
                fast ? "FAST" : "NOT FAST");

    g_object_unref(strlist);

    return TRUE;
}

/** 
 * Gets the actual file size of the binary content.
 * 
 * @param transfer The actual File Transfer instance.
 * 
 * @return The actual file size in bytes. 
 */
guint64
gmyth_file_transfer_get_filesize(GMythFileTransfer * transfer)
{
    guint64         filesize;

    g_return_val_if_fail(transfer != NULL, 0);

    g_object_get(GMYTH_FILE(transfer), "file-size", &filesize, NULL);

    return filesize;
}
