------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2003 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-connections.adb,v 1.8 2003/11/26 10:56:12 vagul Exp $

with Ada.Exceptions;
with Ada.Strings.Fixed;

with OCI.Environments;
with OCI.Thread;

package body OCI.Thick.Connections is

   use Lib;
   use type SWord;
   use type Ub4;
   use type OCIHandle;

   -----------
   -- Break --
   -----------

   procedure Break (It : Connection) is
      Rc : SWord := OCIBreak (Handle (It), Thread.Error);
   begin
      Check_Error (Rc);
   end Break;

   ------------
   -- Commit --
   ------------

   procedure Commit (Connect : in Connection) is
      RC : SWord := OCITransCommit (OCISvcCtx (Connect.Handle), Thread.Error);
   begin
      Check_Error (RC);
   end Commit;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Object : in out Connection) is
      Rc : SWord;
      H : OCIHandle := Object.Handle;
   begin
      if H /= Empty_Handle then
         if Object.Sessn /= OCISession (Empty_Handle) then
            Rc := OCISessionEnd (Svchp => OCISvcCtx (H),
                                 errhp => Thread.Error,
                                 usrhp => Object.Sessn);

            Free (OCIHandle (Object.Sessn), OCI_HTYPE_SESSION);
            Free (H, OCI_HTYPE_SVCCTX);
         else
            Rc := OCILogoff (Svchp => OCISvcCtx (H), Errhp => Thread.Error);
            --  ??? Check_Error(Rc); -- 8i bug
         end if;
      end if;
   end Destroy;

   -----------------
   -- Is_Blocking --
   -----------------

   function Is_Blocking (Connect : in Connection) return Boolean is
      function Get_Attrib is new Get_Attr_G (Ub1);
      use type Ub1;
   begin
      return Get_Attrib
               (Connect.Handle, OCI_HTYPE_SVCCTX, OCI_ATTR_NONBLOCKING_MODE)
                = 0;
   end Is_Blocking;

   --------------------
   -- Get_Connection --
   --------------------

   function Get_Connection
     (Context : in OCIExtProcContext)
      return Connection
   is
      Rc : SWord;
      H  : aliased OCISvcCtx;
      E  : aliased OCIEnv;
      Er : aliased OCIError;
      tr : OCI.Environments.Thread_Environment;
      Result : Connection;
   begin

      Rc := OCIExtProcGetEnv
        (octxp => Context,
         Envhp => E'Access,
         Errhp => Er'Access,
         Svchp => H'Access);

      tr := (RF.Controlled_Reference with Handle => E);
      --  Thread.Set_Environment (tr);

      Thread.Set_Error (E, Er);

      Check_Error (Rc);

      Result.Handle := OCIHandle (H);
      Result.Environment := Thread.Synch.Environment;

      return Result;
   end Get_Connection;

   -----------
   -- Reset --
   -----------

   procedure Reset (It : in Connection) is
      Rc : SWord := OCIReset (Handle (It), Thread.Error);
   begin
      Check_Error (Rc);
   end Reset;

   -----------
   -- Logon --
   -----------

   function Logon
     (Server_Name : in String;
      User        : in String;
      Password    : in String)
      return      Connection is
   begin
      return Logon (DB       => Attach (Server_Name),
                    User     => User,
                    Password => Password);
   end Logon;

   function Logon
     (DB       : Server;
      User     : String;
      Password : String)
      return   Connection
   is
      Result : Connection;
   begin
      Result.Environment := Thread.Synch.Environment;
      Result.DB := DB;
      Result.Handle := Alloc_Handle (Thread.Environment, OCI_HTYPE_SVCCTX);
      Set_Attr (Result.Handle, OCI_HTYPE_SVCCTX, Handle (DB), OCI_ATTR_SERVER);
      Result.Sessn := OCISession
                        (Alloc_Handle
                           (Thread.Environment, OCI_HTYPE_SESSION));

      Set_Attr (OCIHandle (Result.Sessn),
                OCI_HTYPE_SESSION,
                User,
                OCI_ATTR_USERNAME);
      Set_Attr (OCIHandle (Result.Sessn),
                OCI_HTYPE_SESSION,
                Password,
                OCI_ATTR_PASSWORD);
      Set_Attr (Result.Handle,
                OCI_HTYPE_SVCCTX,
                OCIHandle (Result.Sessn),
                OCI_ATTR_SESSION);
      Check_Error
        (OCISessionBegin
           (OCISvcCtx (Result.Handle), Thread.Error, Result.Sessn));
      return Result;
   end Logon;

   function Logon (Connect : String) return Connection is
      D1, D2 : Natural;
      use Ada.Strings.Fixed, Ada.Strings;
   begin
      D1 := Index (Connect, "/");
      D2 := Index (Connect, "@", Backward);

      if D1 = 0 then
         Ada.Exceptions.Raise_Exception
           (Constraint_Error'Identity,
            "Wrong connect string format """ & Connect & '"');
      elsif D2 = 0 then
         return Logon
           (Server_Name => "",
            User        => Connect (Connect'First .. D1 - 1),
            Password    => Connect (D1 + 1 .. Connect'Last));
      else
         return Logon
           (Server_Name => Connect (D2 + 1 .. Connect'Last),
            User        => Connect (Connect'First .. D1 - 1),
            Password    => Connect (D1 + 1 .. D2 - 1));
      end if;
   end Logon;

   ------------
   -- Logoff --
   ------------

   procedure Logoff (Connect : in out Connection) is
      Result : Connection;
   begin
      Connect := Result;
   end Logoff;

   --------------
   -- Rollback --
   --------------

   procedure Rollback (Connect : in Connection) is
      RC : SWord := OCITransRollback
                      (OCISvcCtx (Connect.Handle), Thread.Error);
   begin
      Check_Error (RC);
   end Rollback;

   --------------------
   -- Server_Version --
   --------------------

   function Server_Version (Connect : in Connection) return String is
      Buff : aliased Text := (0 .. 511 => c.nul);
      Rc   : SWord := OCIServerVersion
              (hndlp    => Connect.Handle,
               errhp    => Thread.Error,
               bufp     => CStr.To_Chars_Ptr (Buff'Unchecked_Access),
               bufsz    => buff'Length - 1,
               hndltype => OCI_HTYPE_SVCCTX);
   begin
      Check_Error (RC);
      return C.To_Ada (Buff);
   end Server_Version;

   ------------------
   -- Set_Blocking --
   ------------------

   procedure Set_Blocking (Connect : in out Connection; Mode : in Boolean) is
   begin
      Set_Attr
        (Connect.Handle,
         OCI_HTYPE_SVCCTX,
         Boolean'Pos (not Mode),
         OCI_ATTR_NONBLOCKING_MODE);
   end Set_Blocking;

end OCI.Thick.Connections;
