------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2003 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-string_var.adb,v 1.8 2003/12/10 17:05:22 vagul Exp $

with
   OCI.Lib,
   OCI.Thread,
   OCI.Thick.Parameters;

with OCI.Environments;

package body OCI.Thick.String_Var is

   use Lib, Parameters;
   use type OCIHandle;

   procedure Correct_Size
     (Stmt     : in     Statement;
      Value    : in out Variable;
      Position : in     Positive);

   function Allocated_Size (Value : in Variable) return Natural;

   --------------------
   -- Allocated_Size --
   --------------------

   function Allocated_Size (Value : in Variable) return Natural is
      alsize : aliased Ub4 := 0;
      Str : OCIString := OCIString (Get_Value_Internal (Value));
      rc : SWord := OCIStringAllocSize
            (env       => Thread.Environment,
             vs        => Str,
             allocsize => alsize'Access);

   begin
      Check_Error (rc);
      return Natural (alsize);
   end Allocated_Size;

   ------------------
   -- Correct_Size --
   ------------------

   procedure Correct_Size
     (Stmt     : in     Statement;
      Value    : in out Variable;
      Position : in     Positive)
   is
      Param : Parameter := Column (Stmt, Position);
      NSize : Natural;
      TCode : constant SQL_Type := Get_SQL_Type (Param);
   begin
      if TCode = SQL_Varchar2
        or TCode = SQL_Char
        or TCode = SQL_Varchar
      then
         NSize := Data_Size (Param) + 1;
      else
         NSize := 64;  -- ot fanaria
      end if;

      if Get_Value_Internal (Value) = Empty_Handle
        or else Allocated_Size (Value) < NSize
      then
         Resize (Value, NSize);
      end if;
   end Correct_Size;

   ------------
   -- Define --
   ------------

   procedure Define
     (Stmt     : in     Statement;
      Value    : in out Variable;
      Position : in     Positive) is
   begin
      if Described (Stmt) then
         Correct_Size (Stmt, Value, Position);
      end if;

      Define (Stmt, OCIHandle_Var.Var_Type (Value), Position);
   end Define;

   --------------
   -- Finalize --
   --------------

   procedure Finalize  (Object : in out Variable) is
   begin
      if Get_Value_Internal (Object) /= Empty_Handle then
         Resize (Object, 0);
      end if;
      Finalize (Var_Type (Object));
   end Finalize;

   ---------------
   -- Get_Value --
   ---------------

   function Get_Value (Var : in Variable) return String is
      Env : OCIEnv := Thread.Environment;
      Str : OCIString := OCIString (OCIHandle_Var.Get_Value (Var_Type (Var)));
   begin
      return C.To_Ada (CStr.Value (OCIStringPtr (Env, Str),
                                   C.size_t (OCIStringSize (Env, Str))),
                       False);
   end Get_Value;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (Object : in out Variable) is
   begin
      Initialize (Var_Type (Object));
      Set_Value (Object, Empty_Handle);
   end Initialize;

   ------------
   -- Resize --
   ------------

   procedure Resize (Var : in out Variable; Length : in Natural) is
      Value : aliased OCIString := OCIString (Get_Value_Internal (Var));
      rc : SWord;
      use type Ub4;
   begin
      rc := OCIStringResize
        (env      => Thread.Environment,
         err      => Thread.Error,
         new_size => Ub4 (Length),
         str      => Value'Access);
      Check_Error (rc);

      Set_Value (Var, OCIHandle (Value));
   end Resize;

   ---------------
   -- Set_Value --
   ---------------

   procedure Set_Value (Var : in out Variable; Value : in String) is
      str : aliased OCIString := OCIString (Get_Value_Internal (Var));
      rc  : SWord := OCIStringAssignText
         (env     => Thread.Environment,
          err     => Thread.Error,
          rhs     => C.To_C (Value),
          rhs_len => Value'Length,
          lhs     => str'Access);
   begin
      Check_Error (rc);
      Set_Value (Var, OCIHandle (str));
   end Set_Value;

begin
   OCI.Environments.Set_Create_Mode_Flag (OCI_OBJECT);
end OCI.Thick.String_Var;
