------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2004 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-lobs.adb,v 1.10 2004/06/28 04:57:05 vagul Exp $

with OCI.Thread;
with System;
with Ada.Unchecked_Deallocation;

package body OCI.Thick.Lobs is

   use Ada.Strings.Unbounded;

   type4alloc : array (Lob_Type) of Ub4
     := (File  => OCI_DTYPE_FILE,
         Bin   => OCI_DTYPE_LOB,
         NChar => OCI_DTYPE_LOB,
         Char  => OCI_DTYPE_LOB);

   type4create_temp : array (Lob_Type) of Ub1
     := (File  => 0,
         Bin   => OCI_TEMP_BLOB,
         NChar => OCI_TEMP_NCLOB,
         Char  => OCI_TEMP_CLOB);

   procedure Read
     (Loc    : in     Lob;
      Amount : in out Count;
      Offset : in     Positive_Count;
      Buffer : in     System.Address);

   procedure Write
     (Loc    : in out Lob;
      Offset : in     Positive_Count;
      Buffer : in     System.Address;
      Length : in     Ub4);

   ------------------
   -- Create_Empty --
   ------------------

   function Create_Empty
     (Connect : in Connection;
      LType   : in Lob_Type := Char)
      return Lob
   is
      use type System.Address;
      Object : Lob;
   begin
      Object.Handle
        := Alloc_Descriptor (Thread.Environment, type4alloc (LType));

      Object.LType   := LType;
      Object.Locp    := new OCIHandle'(Object.Handle);
      Object.Ind     := new Sb2'(Null_Indicator);
      Object.Connect := Connect;

      return Object;
   end Create_Empty;

   ----------------------
   -- Create_Temporary --
   ----------------------

   function Create_Temporary
     (Connect : in Connection;
      LType   : in Lob_Type := Char)
      return Lob
   is
      Loc : Lob := Create_Empty (Connect, LType);
   begin
      Create_Temporary (Loc);
      return Loc;
   end Create_Temporary;

   procedure Create_Temporary
     (Loc   : in out Lob;
      Cache : in     Boolean := True)
   is
      Rc : SWord;
      To_OCI_Boolean : constant array (Boolean) of C.int :=
         (True  => Lib.TRUE,
          False => Lib.FALSE);
   begin
      Rc := OCILobCreateTemporary (svchp  => OCISvcCtx (Handle (Loc.Connect)),
                                   errhp   => Thread.Error,
                                   locp    => OCILobLocator (Loc.Handle),
                                   csid    => OCI_DEFAULT,
                                   csfrm   => SQLCS_IMPLICIT,
                                   lobtype => type4create_temp (Loc.LType),
                                   cache    => To_OCI_Boolean (Cache),
                                   duration => OCI_DURATION_SESSION);
      Check_Error (Rc);

      Loc.Ind.all := Not_Null_Indicator;
   end Create_Temporary;

   procedure Create_Temporary
       (Stream  : in out Lob_Stream;
        Connect : in     Connection;
        LType   : Lob_Type := Char) is
   begin
      Stream.Loc := Create_Empty (Connect, LType);
      Create_Temporary (Stream.Loc);
   end Create_Temporary;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Object : in out Lob) is
      Rc : SWord;
      use type System.Address;

      procedure Free is
         new Ada.Unchecked_Deallocation (OCIHandle, AC.Object_Pointer);

      procedure Free is new Ada.Unchecked_Deallocation (Sb2, A_Sb2);
   begin
      if Object.Handle /= Empty_Handle then

         Rc := OCIDescriptorFree
            (descp => Object.Handle,
             dtype => type4alloc (Object.LType));
         Check_Error (Rc);
      end if;

      Free (Object.Locp);
      Free (Object.Ind);
   end Destroy;

   -----------------
   -- End_Of_File --
   -----------------

   function End_Of_File (Stream : in Lob_Stream) return Boolean is
   begin
      return Stream.Position > Length (Stream.Loc);
   end End_Of_File;

   --------------------
   -- Free_Temporary --
   --------------------

   procedure Free_Temporary (Loc : in out Lob) is
      Rc : SWord := OCILobFreeTemporary
                      (svchp => OCISvcCtx (Handle (Loc.Connect)),
                       errhp => Thread.Error,
                       locp  => OCILobLocator (Loc.Handle));
   begin
      Check_Error (Rc);
   end Free_Temporary;

   procedure Free_Temporary (Stream : in out Lob_Stream) is
   begin
      Free_Temporary (Stream.Loc);
   end Free_Temporary;

   ----------------------
   -- Get_Bind_Address --
   ----------------------

   function Get_Bind_Address (Loc : Lob) return DVoid is
      use type System.Address;
   begin
      pragma Assert (Loc.Handle = OCIHandle (Loc.Locp.all));
      return AC.To_Address (Loc.Locp);
   end Get_Bind_Address;

   ------------------------
   -- Get_Bind_Indicator --
   ------------------------

   function Get_Bind_Indicator (Loc : Lob) return A_Sb2 is
   begin
      return Loc.Ind;
   end Get_Bind_Indicator;

   ------------------
   -- Get_Lob_Type --
   ------------------

   function Get_Lob_Type (Loc : Lob) return Lob_Type is
   begin
      return Loc.LType;
   end Get_Lob_Type;

   -------------
   -- Is_Init --
   -------------

   function Is_Init (Value : in Lob) return Boolean is
      Result : aliased C.int;
      Rc     : SWord;
      use type DVoid;
   begin
      if Value.Handle = Empty_Handle then
         return False;
      end if;

      Rc := OCILobLocatorIsInit
               (envhp          => Thread.Environment,
                errhp          => Thread.Error,
                locp           => OCILobLocator (Value.Handle),
                is_initialized => Result'Unchecked_Access);

      Check_Error (Rc);

      return Boolean'Val (Result);
   end Is_Init;

   -------------
   -- Is_Open --
   -------------

   function Is_Open (Value : in Lob) return Boolean is
      Result : aliased C.int;
      Rc : SWord;
      use type DVoid;
   begin
      if Value.Handle = Empty_Handle then
         return false;
      end if;

      Rc := OCILobIsOpen (svchp => OCISvcCtx (Handle (Value.Connect)),
                          errhp => Thread.Error,
                          locp  => OCILobLocator (Value.Handle),
                          flag  => Result'Unchecked_Access);

      Check_Error (Rc);
      return Boolean'Val (Result);
   end Is_Open;

   ------------------
   -- Is_Temporary --
   ------------------

   function Is_Temporary (Value : in Lob) return Boolean is
      Result : aliased C.int;

      Rc : SWord := OCILobIsTemporary
                      (Envhp        => Thread.Environment,
                       Errhp        => Thread.Error,
                       Locp         => OCILobLocator (Value.Handle),
                       Is_Temporary => Result'Unchecked_Access);
   begin
      Check_Error (Rc);
      return Boolean'Val (Result);
   end Is_Temporary;

   ------------
   -- Length --
   ------------

   function Length (Loc : in Lob) return Count is
      Result : aliased Ub4;
      Rc : SWord := OCILobGetLength
                      (svchp => OCISvcCtx (Handle (Loc.Connect)),
                       errhp => Thread.Error,
                       locp  => OCILobLocator (Loc.Handle),
                       lenp  => Result'Unchecked_Access);
   begin
      Check_Error (Rc);

      return Count (Result);
   end Length;

   ----------
   -- Read --
   ----------

   procedure Read
     (Loc    : in     Lob;
      Amount : in out Count;
      Offset : in     Positive_Count;
      Buffer : in     System.Address)
   is
      Amt : aliased Ub4 := Ub4 (Amount);
      use System;
      Rc : SWord := OCILobRead
                       (svchp  => OCISvcCtx (Handle (Loc.Connect)),
                        errhp  => Thread.Error,
                        locp   => OCILobLocator (Loc.Handle),
                        amtp   => Amt'Unchecked_Access,
                        offset => Ub4 (Offset),
                        bufp   => Buffer,
                        bufl   => Ub4 (Amount),
                        ctxp   => Empty_Handle,
                        cbfp   => Empty_Handle,
                        csid   => 0);
   begin
      pragma Assert (Loc.Handle = OCIHandle (Loc.Locp.all));
      Check_Error (Rc);
      Amount := Count (Amt);
   end Read;

   procedure Read
     (Loc    :  in     Lob;
      Offset :  in     Positive_Count;
      Buffer :     out Raw;
      Last   :     out Raw_Offset)
   is
      Amt : Count := Buffer'Length;
   begin
      Read (Loc, Amt, Offset, Buffer'Address);
      Last := Buffer'Last - Raw_Offset (Buffer'Length - Amt);
   end Read;

   procedure Read
     (Loc    : in     Lob;
      Offset : in     Positive_Count;
      Buffer :    out String;
      Last   :    out Integer)
   is
      Amt : Count := Buffer'Length;
   begin
      Read (Loc, Amt, Offset, Buffer'Address);
      Last := Buffer'Last - Integer (Buffer'Length - Amt);
   end Read;

   procedure Read
     (Stream : in out Lob_Stream;
      Item   : out Stream_Element_Array;
      Last   : out Stream_Element_Offset) is
   begin
      Read
        (Stream.Loc,
         offset => Stream.Position,
         Buffer => Item,
         Last   => Last);
      Stream.Position := Stream.Position
        + Count (Last - Item'First + 1);
   end Read;

   -----------
   -- Reset --
   -----------

   procedure Reset (Stream : in out Lob_Stream) is
   begin
      Stream.Position := Positive_Count'First;
   end Reset;

   ------------
   -- Stream --
   ------------

   procedure Stream (Stream : in out Lob_Stream'Class; Loc : in Lob) is
   begin
      Stream.Loc := Loc;
      Stream.Position   := Positive_Count'First;
   end Stream;

   ---------------
   -- To_String --
   ---------------

   function To_String (Loc : in Lob) return String is
      Len : Count := Length (Loc);
      Result : C.char_array (0 .. C.size_t (Len - 1));
   begin
      Read (Loc, Len, 1, Result'Address);
      return C.To_Ada (Result, False);
   end To_String;

   function To_String (Loc : in Lob_Stream) return String is
   begin
      return To_String (Loc.Loc);
   end To_String;

   -------------------------
   -- To_Unbounded_String --
   -------------------------

   function To_Unbounded_String (Loc : in Lob) return Unbounded_String is
      Position : Count := 1;
      Buffer   : String (1 .. 16#4000#);
      Result   : Unbounded_String;
      Last     : Natural;
   begin
      loop
         Read (Loc    => Loc,
               Offset => Position,
               Buffer => Buffer,
               Last   => Last);

         Append (Result, Buffer (1 .. Last));

         exit when Last < Buffer'Last;

         Position := Position + Count (Last);
      end loop;

      return Result;
   end To_Unbounded_String;

   -----------
   -- Write --
   -----------

   procedure Write
     (Loc    : in out Lob;
      Offset : in     Positive_Count;
      Buffer : in     System.Address;
      Length : in     Ub4)
   is
      Amt : aliased Ub4 := Length;
      Rc : SWord := OCILobWrite
                      (svchp => OCISvcCtx (Handle (Loc.Connect)),
                       errhp => Thread.Error,
                       locp  => OCILobLocator (Loc.Handle),
                       amtp  => Amt'Unchecked_Access,
                       offset => Ub4 (Offset),
                       bufp   => Buffer,
                       buflen => Length,
                       piece => OCI_ONE_PIECE,
                       ctxp  => Empty_Handle,
                       cbfp  => Empty_Handle,
                       csid  => 0);
   begin
      Check_Error (Rc);
   end Write;

   procedure Write
     (Loc    : in out Lob;
      Offset : in     Positive_Count;
      Buffer : in     Raw) is
   begin
      Write (Loc, Offset, Buffer'Address, Buffer'Length);
   end Write;

   procedure Write
     (Loc    : in out Lob;
      Offset : in     Positive_Count;
      Buffer : in     String) is
   begin
      Write (Loc, Offset, Buffer'Address, Buffer'Length);
   end Write;

   -----------
   -- Write --
   -----------

   procedure Write
     (Stream : in out Lob_Stream;
      Item   : in     Stream_Element_Array) is
   begin
      Write
        (Loc    => Stream.Loc,
         Buffer => Item,
         Offset => Stream.Position);

      Stream.Position := Stream.Position + Item'Length;
   end Write;

end OCI.Thick.Lobs;
