------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-gen_ocinumber.adb,v 1.24 2008/05/05 09:10:58 vagul Exp $

with Ada.Decimal;
with Interfaces.C;
with OCI.Thread;

package body OCI.Thick.Gen_OCINumber is

   ---------------------
   -- Binary_Function --
   ---------------------

   function Binary_Function (Left, Right : OCINumber) return OCINumber is
      Result : aliased OCINumber;
   begin
      Check_Error (OCINumberOper (Thread.Error, Left, Right, Result'Access));

      return Result;
   end Binary_Function;

   ----------------------
   -- Compare_Function --
   ----------------------

   function Compare_Function (Left, Right : OCINumber) return Boolean is
      Result : aliased SWord;
   begin
      Check_Error (OCINumberCmp
                     (Err     => Thread.Error,
                      Number1 => Left,
                      Number2 => Right,
                      Result  => Result'Access));
      return Compare (Result, 0);
   end Compare_Function;

   -----------------------
   -- Decimal_To_Number --
   -----------------------

   function Decimal_To_Number (From : Decimal_Type) return OCINumber is
      --  float conversion variant.
      --  function To_Number is new Float_To_Number (C.long_double);

      type LLI is range System.Min_Int .. System.Max_Int;
      type DLLI is delta 1.0 digits System.Max_Digits;

      function To_Number is new Int_To_Number (LLI);

      function Shift is new Integer_Function (Integer, OCINumberShift);

      procedure Divide is new Ada.Decimal.Divide
        (Dividend_Type  => Decimal_Type,
         Divisor_Type   => Decimal_Type,
         Quotient_Type  => DLLI,
         Remainder_Type => DLLI);

      Quotient  : DLLI;
      Remainder : DLLI;

   begin
      Divide
        (Dividend  => From,
         Divisor   => Decimal_Type'Delta,
         Quotient  => Quotient,
         Remainder => Remainder);

      if Remainder /= 0.0 then
         raise Program_Error;
      end if;

      return Shift (To_Number (LLI (Quotient)),  -Decimal_Type'Aft);

      --  float conversion variant not support Max_Digits.
      --  return To_Number (C.long_double (From));
   end Decimal_To_Number;

   ---------------------
   -- Float_To_Number --
   ---------------------

   function Float_To_Number (From : in Float_Type) return OCINumber is
      use type UWord;
      Result : aliased OCINumber;

      procedure Double_Convert;

      procedure Double_Convert is
         F1 : constant Long_Float := Long_Float (From);
         F2 : constant Long_Float := Long_Float (From - Float_Type (F1));
         P1 : aliased OCINumber;
         P2 : aliased OCINumber;
      begin
         Check_Error (OCINumberFromReal
                        (Thread.Error,
                         F1'Address,
                         F1'Size / System.Storage_Unit,
                         P1'Access));
         Check_Error (OCINumberFromReal
                        (Thread.Error,
                         F2'Address,
                         F2'Size / System.Storage_Unit,
                         P2'Access));
         Check_Error (OCINumberAdd (Thread.Error, P1, P2, Result'Access));
      end Double_Convert;

   begin
      if Float_Type'Size <= Long_Float'Size then
         --  Oracle could not correct make convertion if Float size more than
         --  sizeof double.

         Check_Error (OCINumberFromReal
                        (Thread.Error,
                         From'Address,
                         From'Size / System.Storage_Unit,
                         Result'Access));
      else
         Double_Convert;
      end if;

      return Result;
   end Float_To_Number;

   ----------------------------
   -- Float_To_Number_Better --
   ----------------------------

   function Float_To_Number_Better (From : Float_Type) return OCINumber is
      type Unsigned_Type is mod System.Max_Binary_Modulus;

      function To_Number is new Unsigned_To_Number (Unsigned_Type);

      N : aliased OCINumber
        := To_Number (Unsigned_Type
                        (abs Float_Type'Fraction (From)
                         * Float_Type (Float_Type'Machine_Radix)
                           ** Float_Type'Machine_Mantissa));
      P : Integer
        := Float_Type'Exponent (From) - Float_Type'Machine_Mantissa;

      E : OCINumber;

      function "/" is new Binary_Function (OCINumberDiv);
      pragma Inline ("/");
      function "*" is new Binary_Function (OCINumberMul);
      pragma Inline ("*");
      function Prec  is new Integer_Function (Integer, OCINumberPrec);

   begin
      if From < 0.0 then
         Check_Error (OCINumberNeg (Thread.Error, N, N'Access));
      end if;

      while abs P >= Unsigned_Type'Size loop
         E := To_Number (2 ** (Unsigned_Type'Size - 1));

         if P >= 0 then
            N := N * E;
            P := P - Unsigned_Type'Size + 1;
         else
            N := N / E;
            P := P + Unsigned_Type'Size - 1;
         end if;
      end loop;

      E := To_Number (Float_Type'Machine_Radix ** (abs P));

      --  3 is just experimental constant for each Float would be converted
      --  to OCINumber and back without modification.

      if P >= 0 then
         return Prec (N * E, Float_Type'Digits + 3);
      else
         return Prec (N / E, Float_Type'Digits + 3);
      end if;
   end Float_To_Number_Better;

   -------------------
   -- Int_To_Number --
   -------------------

   function Int_To_Number (From : in Integer_Type) return OCINumber is
      use type UWord;
      Result : aliased OCINumber;
   begin
      Check_Error (OCINumberFromInt
                     (Thread.Error,
                      Inum        => From'Address,
                      Inum_Length => From'Size / System.Storage_Unit,
                      Inum_S_Flag => OCI_NUMBER_SIGNED,
                      Number      => Result'Access));
      return Result;
   end Int_To_Number;

   ----------------------
   -- Integer_Function --
   ----------------------

   function Integer_Function
     (Left : OCINumber; Right : Integer_Type) return OCINumber
   is
      Result : aliased OCINumber;
   begin
      Check_Error
        (OCINumberOper (Thread.Error, Left, SWord (Right), Result'Access));

      return Result;
   end Integer_Function;

   --------------------
   -- Num_To_Decimal --
   --------------------

   function Num_To_Decimal (Numb : OCINumber) return Decimal_Type is
      --  float variant.
      --  function To_Float is new Num_To_Float (C.long_double);

      type LLI is range System.Min_Int .. System.Max_Int;
      type DLLI is delta 1.0 digits System.Max_Digits;

      function To_Integer is new Num_To_Integer (LLI);
      function Shift is new Integer_Function (Integer, OCINumberShift);
      function Round is new Integer_Function (Integer, OCINumberRound);

      procedure Divide is new Ada.Decimal.Divide
        (Dividend_Type  => DLLI,
         Divisor_Type   => DLLI,
         Quotient_Type  => Decimal_Type,
         Remainder_Type => Decimal_Type);

      L : constant DLLI :=
        DLLI (To_Integer
                (Shift (Round (Numb, Decimal_Type'Aft), Decimal_Type'Aft)));

      Quotient  : Decimal_Type;
      Remainder : Decimal_Type;
   begin
      Divide
        (Dividend  => L,
         Divisor   => DLLI (10.0**Decimal_Type'Aft),
         Quotient  => Quotient,
         Remainder => Remainder);

      if Remainder /= 0.0 then
         raise Program_Error;
      end if;

      return Quotient;

      --  float variant.
      --  return Decimal_Type (To_Float (Numb));
   end Num_To_Decimal;

   ------------------
   -- Num_To_Float --
   ------------------

   function Num_To_Float (Numb : in OCINumber) return Float_Type is
      use type UWord;
      Result : aliased Float_Type;

      procedure Double_Convert;

      procedure Double_Convert is
         N1 : aliased OCINumber;
         N2 : aliased OCINumber;
         P1 : aliased Long_Float;
         P2 : aliased Long_Float;
      begin
         Check_Error (OCINumberToReal
                        (Thread.Error,
                         Numb,
                         P1'Size / System.Storage_Unit,
                         P1'Address));
         Check_Error (OCINumberFromReal
                        (Thread.Error,
                         P1'Address,
                         P1'Size / System.Storage_Unit,
                         N1'Access));
         Check_Error (OCINumberSub (Thread.Error, Numb, N1, N2'Access));
         Check_Error (OCINumberToReal
                        (Thread.Error,
                         N2,
                         P2'Size / System.Storage_Unit,
                         P2'Address));
         Result := Float_Type (P1) + Float_Type (P2);
      end Double_Convert;

   begin
      if Float_Type'Size <= Long_Float'Size then
         --  Oracle could not correct make convertion if Float size more than
         --  sizeof double.

         Check_Error (OCINumberToReal
                        (Thread.Error,
                         Numb,
                         Result'Size / System.Storage_Unit,
                         Result'Address));
      else
         Double_Convert;
      end if;

      return Result;
   end Num_To_Float;

   -------------------------
   -- Num_To_Float_Better --
   -------------------------

   function Num_To_Float_Better (Numb : in OCINumber) return Float_Type is
      use type SWord;

      type Max_Float is digits System.Max_Digits;

      type Unsigned_Type is mod System.Max_Binary_Modulus;
      function To_Number  is new Unsigned_To_Number (Unsigned_Type);
      function To_Integer is new Num_To_Integer (Integer);
      function To_Unsigned is new Num_To_Unsigned (Unsigned_Type);

      function "/" is new Binary_Function (OCINumberDiv);
      pragma Inline ("/");
      function "*" is new Binary_Function (OCINumberMul);
      pragma Inline ("*");
      function "+" is new Binary_Function (OCINumberAdd);
      pragma Inline ("+");
      function Log is new Binary_Function (OCINumberLog);
      function "abs"  is new Unary_Function (OCINumberAbs);
      pragma Inline ("abs");
      function Ceil is new Unary_Function (OCINumberCeil);
      function Round is new Integer_Function (Integer, OCINumberRound);
      function Power is new Integer_Function (Integer, OCINumberIntPower);
      function "=" is new Compare_Function (Interfaces.C."=");

      NMax : constant OCINumber
        := To_Number (Unsigned_Type'Last) + To_Number (1);
      N2   : constant OCINumber := To_Number (2);
      ANum : constant OCINumber := abs Numb;

      P  : Integer;
      NR : OCINumber;
      FR : Max_Float;

      Sign : aliased SWord;
      Zero : aliased C_Boolean;

      --  N * 2 ** X < UMax;
      --  N * 2 ** X > UMax / 2;

      --  2 ** -X > N / UMax;
      --  2 ** -X < N / UMax / 2;

      --  -X > L2 (N / UMax)
      --  -X < L2 (N / UMax / 2)
   begin
      Check_Error (OCINumberSign (Thread.Error, Numb, Sign'Access));

      if Sign = 0 then
         --  Strange, but sometimes OCINumberIsZero show non zero when Sign
         --  show zero.

         Check_Error (OCINumberIsZero (Thread.Error, Numb, Zero'Access));

         if Zero /= OCI.Lib.FALSE then
            return 0.0;
         end if;
      end if;

      P := To_Integer (Ceil (Log (N2, ANum))) - Unsigned_Type'Size;

      if P > 0 then
         NR := Round (ANum / Power (N2, P), 0);
      else
         declare
            VP   : Integer := P;
            Skip : constant := 400;
         begin
            NR := ANum;

            while VP < -Skip loop
               NR := NR * Power (N2, Skip);
               VP := VP + Skip;
            end loop;

            NR := Round (NR * Power (N2, -VP), 0);
         end;
      end if;

      if NR = NMax then
         FR := 2.0 ** Unsigned_Type'Size;
      else
         FR := Max_Float (To_Unsigned (NR));
      end if;

      if P < 0 then
         FR := FR / 2.0 ** (-P);
      else
         FR := FR * 2.0 ** P;
      end if;

      if Sign < 0 then
         return Float_Type (-FR);
      end if;

      return Float_Type (FR);

   end Num_To_Float_Better;

   --------------------
   -- Num_To_Integer --
   --------------------

   function Num_To_Integer (Numb : in OCINumber) return Integer_Type is
      use type UWord;
      Result : aliased Integer_Type;
   begin
      Check_Error (OCINumberToInt
                     (Thread.Error,
                      Numb,
                      Rsl_Length => Result'Size / System.Storage_Unit,
                      Rsl_Flag   => OCI_NUMBER_SIGNED,
                      Rsl        => Result'Address));
      return Result;
   end Num_To_Integer;

   ---------------------
   -- Num_To_Unsigned --
   ---------------------

   function Num_To_Unsigned (Numb : OCINumber) return Unsigned_Type is
      use type UWord;
      Result : aliased Unsigned_Type;
   begin
      Check_Error (OCINumberToInt
                     (Thread.Error,
                      Numb,
                      Rsl_Length => Result'Size / System.Storage_Unit,
                      Rsl_Flag   => OCI_NUMBER_UNSIGNED,
                      Rsl        => Result'Address));
      return Result;
   end Num_To_Unsigned;

   --------------------
   -- Unary_Function --
   --------------------

   function Unary_Function (Item : OCINumber) return OCINumber is
      Result : aliased OCINumber;
   begin
      Check_Error (OCINumberOper (Thread.Error, Item, Result'Access));

      return Result;
   end Unary_Function;

   ------------------------
   -- Unsigned_To_Number --
   ------------------------

   function Unsigned_To_Number (From : in Unsigned_Type) return OCINumber is
      use type UWord;
      Result : aliased OCINumber;
   begin
      Check_Error (OCINumberFromInt
                     (Thread.Error,
                      Inum        => From'Address,
                      Inum_Length => From'Size / System.Storage_Unit,
                      Inum_S_Flag => OCI_NUMBER_UNSIGNED,
                      Number      => Result'Access));
      return Result;
   end Unsigned_To_Number;

end OCI.Thick.Gen_OCINumber;
