-----------------------------------------------------------------------
--                               G P S                               --
--                                                                   --
--                     Copyright (C) 2003                            --
--                            ACT-Europe                             --
--                                                                   --
-- GPS is free  software;  you can redistribute it and/or modify  it --
-- under the terms of the GNU General Public License as published by --
-- the Free Software Foundation; either version 2 of the License, or --
-- (at your option) any later version.                               --
--                                                                   --
-- This program is  distributed in the hope that it will be  useful, --
-- but  WITHOUT ANY WARRANTY;  without even the  implied warranty of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details. You should have received --
-- a copy of the GNU General Public License along with this program; --
-- if not,  write to the  Free Software Foundation, Inc.,  59 Temple --
-- Place - Suite 330, Boston, MA 02111-1307, USA.                    --
-----------------------------------------------------------------------

with Ada.Unchecked_Conversion;
with Ada.Unchecked_Deallocation;
with GNAT.OS_Lib;          use GNAT.OS_Lib;
with Glib.Object;          use Glib.Object;
with Glide_Intl;           use Glide_Intl;
with Gtk.Dialog;           use Gtk.Dialog;
with Gtk.Label;            use Gtk.Label;
with Gtk.Size_Group;       use Gtk.Size_Group;
with Gtk.Box;              use Gtk.Box;
with Gtk.Widget;           use Gtk.Widget;
with Gtk.Enums;            use Gtk.Enums;
with Gtk.GEntry;           use Gtk.GEntry;
with Gtk.Stock;            use Gtk.Stock;
with Gtkada.Dialogs;       use Gtkada.Dialogs;
with Gtkada.MDI;           use Gtkada.MDI;
with Glide_Kernel.Custom;  use Glide_Kernel.Custom;
with Glide_Kernel.Modules; use Glide_Kernel.Modules;
with Glide_Kernel.Project; use Glide_Kernel.Project;
with Glide_Main_Window;    use Glide_Main_Window;
with Glide_Kernel.Task_Manager; use Glide_Kernel.Task_Manager;
with Src_Info.Queries;     use Src_Info, Src_Info.Queries;
with String_Hash;
with System;               use System;
with String_Utils;         use String_Utils;
with Basic_Types;
with Projects;             use Projects;
with Projects.Registry;    use Projects.Registry;
with Projects.Editor;      use Projects.Editor;
with Types;                use Types;
with Traces;               use Traces;
with VFS;                  use VFS;

package body Glide_Kernel.Scripts is

   Me : constant Debug_Handle := Create ("Glide_Kernel.Scripts");

   procedure Unchecked_Free is new Ada.Unchecked_Deallocation
     (Class_Instance_Record'Class, Class_Instance);

   type Scripting_Language_Data;
   type Scripting_Language_List is access Scripting_Language_Data;
   type Scripting_Language_Data is record
      Script : Scripting_Language;
      Next   : Scripting_Language_List;
   end record;

   procedure Free (Class : in out Class_Type);
   package Classes_Hash is new String_Hash (Class_Type, Free, No_Class);
   use Classes_Hash.String_Hash_Table;

   type Scripting_Data_Record is new Kernel_Scripting_Data_Record with record
      Scripting_Languages  : Scripting_Language_List;
      Classes              : Classes_Hash.String_Hash_Table.HTable;
      Entity_Class         : Class_Type := No_Class;
      File_Class           : Class_Type := No_Class;
      Project_Class        : Class_Type := No_Class;
      File_Context_Class   : Class_Type := No_Class;
      File_Location_Class  : Class_Type := No_Class;
      Entity_Context_Class : Class_Type := No_Class;
   end record;
   type Scripting_Data is access all Scripting_Data_Record'Class;

   type Entity_Information_Access is access Entity_Information;
   function Convert is new Ada.Unchecked_Conversion
     (System.Address, Entity_Information_Access);
   procedure Unchecked_Free is new Ada.Unchecked_Deallocation
     (Entity_Information, Entity_Information_Access);
   procedure On_Destroy_Entity (Value : System.Address);
   pragma Convention (C, On_Destroy_Entity);

   type File_Info_Access is access all File_Info;
   function Convert is new Ada.Unchecked_Conversion
     (System.Address, File_Info_Access);
   procedure Unchecked_Free is new Ada.Unchecked_Deallocation
     (File_Info, File_Info_Access);
   procedure On_Destroy_File (Value : System.Address);
   pragma Convention (C, On_Destroy_File);

   type File_Location_Info_Access is access all File_Location_Info;
   function Convert is new Ada.Unchecked_Conversion
     (System.Address, File_Location_Info_Access);
   procedure Unchecked_Free is new Ada.Unchecked_Deallocation
     (File_Location_Info, File_Location_Info_Access);
   procedure On_Destroy_File_Location (Value : System.Address);
   pragma Convention (C, On_Destroy_File_Location);

   function Convert is new Ada.Unchecked_Conversion
     (System.Address, Selection_Context_Access);
   procedure On_Destroy_Context (Value : System.Address);
   pragma Convention (C, On_Destroy_Context);

   procedure Default_Command_Handler
     (Data : in out Callback_Data'Class; Command : String);
   --  Handler for the default commands

   procedure Create_Entity_Command_Handler
     (Data : in out Callback_Data'Class; Command : String);
   --  Handler for the "Entity" command

   procedure Create_File_Command_Handler
     (Data : in out Callback_Data'Class; Command : String);
   --  Handler for the "File" command

   procedure Create_Project_Command_Handler
     (Data : in out Callback_Data'Class; Command : String);
   --  Handler for the "Project" command

   procedure Create_Location_Command_Handler
     (Data : in out Callback_Data'Class; Command : String);
   --  Handler for the "Location" command

   procedure Context_Command_Handler
     (Data : in out Callback_Data'Class; Command : String);
   --  Handler for all context-related commands

   procedure Entity_Context_Command_Handler
     (Data : in out Callback_Data'Class; Command : String);
   --  Handler for all entity_context-related commands

   procedure Set_Data
     (Instance : access Class_Instance_Record'Class; File : File_Info);
   procedure Set_Data
     (Instance : access Class_Instance_Record'Class;
      Project  : Projects.Project_Type);
   procedure Set_Data
     (Instance : access Class_Instance_Record'Class;
      Location : File_Location_Info);
   procedure Set_Data
     (Instance : access Class_Instance_Record'Class;
      Context  : Selection_Context_Access);
   --  Set the data for an instance

   procedure Free (File : in out File_Info);
   --  Free the contents of File_Info

   Name_Cst       : aliased constant String := "name";
   Filename_Cst   : aliased constant String := "filename";
   File_Cst       : aliased constant String := "file";
   Line_Cst       : aliased constant String := "line";
   Col_Cst        : aliased constant String := "column";
   Shared_Lib_Cst : aliased constant String := "shared_lib";
   Module_Cst     : aliased constant String := "module";
   Msg_Cst        : aliased constant String := "msg";
   Param1_Cst     : aliased constant String := "param1";
   Xml_Cst        : aliased constant String := "xml";
   Attribute_Cst  : aliased constant String := "attribute";
   Package_Cst    : aliased constant String := "package";
   Index_Cst      : aliased constant String := "index";
   Tool_Cst       : aliased constant String := "tool";
   Force_Cst      : aliased constant String := "force";
   Action_Cst     : aliased constant String := "action";
   Prefix_Cst     : aliased constant String := "prefix";
   Project_Cmd_Parameters : constant Cst_Argument_List :=
     (1 => Name_Cst'Access);
   Insmod_Cmd_Parameters  : constant Cst_Argument_List :=
     (1 => Shared_Lib_Cst'Access, 2 => Module_Cst'Access);
   Entity_Cmd_Parameters   : constant Cst_Argument_List :=
     (Name_Cst'Access, File_Cst'Access, Line_Cst'Access, Col_Cst'Access);
   File_Cmd_Parameters     : constant Cst_Argument_List :=
     (1 => Name_Cst'Access);
   Dialog_Cmd_Parameters   : constant Cst_Argument_List :=
     (1 => Msg_Cst'Access);
   Open_Cmd_Parameters     : constant Cst_Argument_List :=
     (1 => Filename_Cst'Access);
   Location_Cmd_Parameters : constant Cst_Argument_List :=
     (1 => Filename_Cst'Access,
      2 => Line_Cst'Access,
      3 => Col_Cst'Access);
   Input_Dialog_Cmd_Parameters : constant Cst_Argument_List :=
     (1 => Msg_Cst'Access,
      2 => Param1_Cst'Access);
   Xml_Custom_Parameters : constant Cst_Argument_List :=
     (1 => Xml_Cst'Access);
   Get_Attributes_Parameters : constant Cst_Argument_List :=
     (1 => Attribute_Cst'Access,
      2 => Package_Cst'Access,
      3 => Index_Cst'Access);
   Tool_Parameters : constant Cst_Argument_List := (1 => Tool_Cst'Access);
   Save_Windows_Parameters : constant Cst_Argument_List :=
     (1 => Force_Cst'Access);
   Exec_Action_Parameters : constant Cst_Argument_List :=
     (1 => Action_Cst'Access);
   Scenar_Var_Parameters : constant Cst_Argument_List :=
     (1 => Prefix_Cst'Access);
   Exit_Cmd_Parameters : constant Cst_Argument_List :=
     (1 => Force_Cst'Access);

   ----------
   -- Free --
   ----------

   procedure Free (Class : in out Class_Type) is
   begin
      Free (Class.Name);
   end Free;

   ---------------------------------
   -- Register_Scripting_Language --
   ---------------------------------

   procedure Register_Scripting_Language
     (Kernel : access Glide_Kernel.Kernel_Handle_Record'Class;
      Script : access Scripting_Language_Record'Class) is
   begin
      Scripting_Data (Kernel.Scripts).Scripting_Languages :=
        new Scripting_Language_Data'
          (Script => Scripting_Language (Script),
           Next   => Scripting_Data (Kernel.Scripts).Scripting_Languages);
   end Register_Scripting_Language;

   -------------------------------
   -- Lookup_Scripting_Language --
   -------------------------------

   function Lookup_Scripting_Language
     (Kernel : access Glide_Kernel.Kernel_Handle_Record'Class;
      Name   : String) return Scripting_Language
   is
      Tmp : Scripting_Language_List :=
        Scripting_Data (Kernel.Scripts).Scripting_Languages;
   begin
      while Tmp /= null loop
         if Case_Insensitive_Equal (Get_Name (Tmp.Script), Name) then
            return Tmp.Script;
         end if;

         Tmp := Tmp.Next;
      end loop;

      return null;
   end Lookup_Scripting_Language;

   ----------------------
   -- Register_Command --
   ----------------------

   procedure Register_Command
     (Kernel        : access Glide_Kernel.Kernel_Handle_Record'Class;
      Command       : String;
      Params        : String  := "";
      Return_Value  : String  := "";
      Description   : String;
      Minimum_Args  : Natural := 0;
      Maximum_Args  : Natural := 0;
      Handler       : Module_Command_Function;
      Class         : Class_Type := No_Class;
      Static_Method : Boolean := False)
   is
      Tmp : Scripting_Language_List :=
        Scripting_Data (Kernel.Scripts).Scripting_Languages;
   begin
      Assert (Me,
              Command /= Constructor_Method or else Class /= No_Class,
              "Constructors can only be specified for classes");
      Assert (Me, not Static_Method or else Class /= No_Class,
              "Static method can only be created for classes");
      Assert (Me,
              Params = "" or else Params (Params'First) = '(',
              "Invalid usage string for "
              & Command & ": must start with '('");

      while Tmp /= null loop
         Register_Command
           (Tmp.Script, Command, Params, Return_Value, Description,
            Minimum_Args, Maximum_Args, Handler, Class, Static_Method);
         Tmp := Tmp.Next;
      end loop;
   end Register_Command;

   ---------------
   -- New_Class --
   ---------------

   function New_Class
     (Kernel      : access Glide_Kernel.Kernel_Handle_Record'Class;
      Name        : String;
      Description : String := "";
      Base        : Class_Type := No_Class) return Class_Type
   is
      Tmp   : Scripting_Language_List :=
        Scripting_Data (Kernel.Scripts).Scripting_Languages;
      Class : Class_Type;

   begin
      Class := Get (Scripting_Data (Kernel.Scripts).Classes, Name);

      if Class = No_Class then
         while Tmp /= null loop
            Register_Class (Tmp.Script, Name, Description, Base);
            Tmp := Tmp.Next;
         end loop;

         Class := Class_Type'(Name => new String'(Name));
         Set (Scripting_Data (Kernel.Scripts).Classes, Name, Class);
      end if;

      return Class;
   end New_Class;

   --------------
   -- Get_Name --
   --------------

   function Get_Name (Class : Class_Type) return String is
   begin
      if Class.Name = null then
         return "";
      else
         return Class.Name.all;
      end if;
   end Get_Name;

   ----------
   -- Free --
   ----------

   procedure Free (Instance : access Class_Instance_Record'Class) is
      Ins : Class_Instance := Class_Instance (Instance);
   begin
      Primitive_Free (Instance.all);
      Unchecked_Free (Ins);
   end Free;

   -----------------------
   -- On_Destroy_Entity --
   -----------------------

   procedure On_Destroy_Entity (Value : System.Address) is
      Ent : Entity_Information_Access := Convert (Value);
   begin
      Destroy (Ent.all);
      Unchecked_Free (Ent);
   end On_Destroy_Entity;

   ----------
   -- Free --
   ----------

   procedure Free (File : in out File_Info) is
      pragma Unreferenced (File);
   begin
      null;
   end Free;

   ---------------------
   -- On_Destroy_File --
   ---------------------

   procedure On_Destroy_File (Value : System.Address) is
      File : File_Info_Access := Convert (Value);
   begin
      Free (File.all);
      Unchecked_Free (File);
   end On_Destroy_File;

   --------------
   -- Set_Data --
   --------------

   procedure Set_Data
     (Instance : access Class_Instance_Record'Class;
      Entity   : Entity_Information)
   is
      Ent    : constant Entity_Information_Access :=
        new Entity_Information'(Copy (Entity));
      Script : constant Scripting_Language := Get_Script (Instance);

   begin
      if not Is_Subclass
        (Script, Get_Class (Instance), Get_Entity_Class (Get_Kernel (Script)))
      then
         raise Invalid_Data;
      end if;

      Set_Data
        (Instance,
         Value      => Ent.all'Address,
         On_Destroy => On_Destroy_Entity'Access);
   end Set_Data;

   ------------------------------
   -- On_Destroy_File_Location --
   ------------------------------

   procedure On_Destroy_File_Location (Value : System.Address) is
      File : File_Location_Info_Access := Convert (Value);
   begin
      Free (File.File);
      Unchecked_Free (File);
   end On_Destroy_File_Location;

   --------------
   -- Set_Data --
   --------------

   procedure Set_Data
     (Instance : access Class_Instance_Record'Class;
      Location : File_Location_Info)
   is
      Loc    : constant File_Location_Info_Access :=
        new File_Location_Info'(Location);
      Script : constant Scripting_Language := Get_Script (Instance);

   begin
      if not Is_Subclass
        (Script, Get_Class (Instance),
         Get_File_Location_Class (Get_Kernel (Script)))
      then
         raise Invalid_Data;
      end if;

      Set_Data
        (Instance,
         Value      => Loc.all'Address,
         On_Destroy => On_Destroy_File_Location'Access);
   end Set_Data;

   --------------
   -- Get_Data --
   --------------

   function Get_Data
     (Instance : access Class_Instance_Record'Class)
      return Entity_Information
   is
      Script : constant Scripting_Language := Get_Script (Instance);
      Ent    : Entity_Information_Access;
   begin
      if not Is_Subclass
        (Script, Get_Class (Instance), Get_Entity_Class (Get_Kernel (Script)))
      then
         raise Invalid_Data;
      end if;

      Ent := Convert (Get_Data (Instance));
      return Ent.all;
   end Get_Data;

   --------------
   -- Get_Data --
   --------------

   function Get_Data (Instance : access Class_Instance_Record'Class)
      return File_Location_Info
   is
      Script : constant Scripting_Language := Get_Script (Instance);
      Loc    : File_Location_Info_Access;
   begin
      if not Is_Subclass
        (Script, Get_Class (Instance),
         Get_File_Location_Class (Get_Kernel (Script)))
      then
         raise Invalid_Data;
      end if;

      Loc := Convert (Get_Data (Instance));
      return Loc.all;
   end Get_Data;

   --------------
   -- Set_Data --
   --------------

   procedure Set_Data
     (Instance : access Class_Instance_Record'Class;
      File     : File_Info)
   is
      Ent    : File_Info_Access;
      Script : constant Scripting_Language := Get_Script (Instance);
      Kernel : constant Kernel_Handle := Get_Kernel (Script);

   begin
      if not Is_Subclass
        (Script, Get_Class (Instance), Get_File_Class (Kernel))
      then
         raise Invalid_Data;
      end if;

      Ent      := new File_Info;
      Ent.File := Get_File (File);

      Set_Data
        (Instance,
         Value      => Ent.all'Address,
         On_Destroy => On_Destroy_File'Access);
   end Set_Data;

   --------------
   -- Get_Data --
   --------------

   function Get_Data (Instance : access Class_Instance_Record'Class)
      return File_Info
   is
      Ent    : File_Info_Access;
      Script : constant Scripting_Language := Get_Script (Instance);
   begin
      if not Is_Subclass
        (Script, Get_Class (Instance), Get_File_Class (Get_Kernel (Script)))
      then
         Trace (Me, "Expected FileClass, got "
                & Get_Name (Get_Class (Instance)));
         raise Invalid_Data;
      end if;

      Ent := Convert (Get_Data (Instance));
      return Ent.all;
   end Get_Data;

   --------------
   -- Set_Data --
   --------------

   procedure Set_Data
     (Instance : access Class_Instance_Record'Class;
      Project  : Project_Type)
   is
      Script : constant Scripting_Language := Get_Script (Instance);
   begin
      if not Is_Subclass
        (Script, Get_Class (Instance), Get_Project_Class (Get_Kernel (Script)))
      then
         raise Invalid_Data;
      end if;

      Set_Data (Instance, Value => Integer (Name_Id'(Project_Name (Project))));
   end Set_Data;

   --------------
   -- Get_Data --
   --------------

   function Get_Data
     (Instance : access Class_Instance_Record'Class)
      return Project_Type
   is
      Script : constant Scripting_Language := Get_Script (Instance);
   begin
      if not Is_Subclass
        (Script, Get_Class (Instance), Get_Project_Class (Get_Kernel (Script)))
      then
         raise Invalid_Data;
      end if;

      return Get_Project_From_Name
        (Project_Registry (Get_Registry (Get_Kernel (Script))),
         Name_Id (Integer'(Get_Data (Instance))));
   end Get_Data;

   -----------------------------
   -- Default_Command_Handler --
   -----------------------------

   procedure Default_Command_Handler
     (Data    : in out Callback_Data'Class;
      Command : String)
   is
      Kernel : constant Kernel_Handle := Get_Kernel (Data);
   begin
      if Command = "insmod" then
         Name_Parameters (Data, Insmod_Cmd_Parameters);

         declare
            Shared  : constant String := Nth_Arg (Data, 1);
            Module  : constant String := Nth_Arg (Data, 2);
            Success : Boolean;
         begin
            Dynamic_Register_Module (Kernel, Shared, Module, Success);

            if Success then
               Set_Return_Value (Data, -"Module successfully loaded.");
            else
               Set_Return_Value (Data, -"Couldn't load module.");
            end if;
         end;

      elsif Command = "lsmod" then
         declare
            use type Module_List.List_Node;
            Current : Module_List.List_Node;
            List    : constant Module_List.List := List_Of_Modules (Kernel);

         begin
            Current := Module_List.First (List);

            Set_Return_Value_As_List (Data);

            while Current /= Module_List.Null_Node loop
               Set_Return_Value
                 (Data,
                  Module_Name (Module_List.Data (Current)));
               Current := Module_List.Next (Current);
            end loop;
         end;

      elsif Command = "exit" then
         Name_Parameters (Data, Exit_Cmd_Parameters);
         Quit (Glide_Window (Get_Main_Window (Kernel)),
               Force => Nth_Arg (Data, 1, False));

      elsif Command = "set_busy" then
         Push_State (Kernel, Processing);

      elsif Command = "unset_busy" then
         Pop_State (Kernel);

      elsif Command = "parse_xml" then
         Name_Parameters (Data, Xml_Custom_Parameters);
         Glide_Kernel.Custom.Add_Customization_String
           (Kernel, Nth_Arg (Data, 1));

      elsif Command = "dialog" then
         Name_Parameters (Data, Dialog_Cmd_Parameters);

         declare
            Result : Message_Dialog_Buttons;
            pragma Unreferenced (Result);
         begin
            Result := Message_Dialog
              (Msg     => Nth_Arg (Data, 1),
               Buttons => Button_OK,
               Justification => Justify_Left,
               Parent  => Get_Main_Window (Kernel));
         end;

      elsif Command = "yes_no_dialog" then
         Name_Parameters (Data, Dialog_Cmd_Parameters);
         Set_Return_Value
           (Data, Message_Dialog
            (Msg           => Nth_Arg (Data, 1),
             Buttons       => Button_Yes + Button_No,
             Justification => Justify_Left,
             Dialog_Type   => Confirmation,
             Parent        => Get_Main_Window (Kernel)) = Button_Yes);

      elsif Command = "save_all" then
         Name_Parameters (Data, Save_Windows_Parameters);

         if not Save_MDI_Children
           (Kernel, No_Children, Nth_Arg (Data, 1, False))
         then
            Set_Error_Msg (Data, -"Cancelled by user");
         end if;

      elsif Command = "execute_action" then
         Name_Parameters (Data, Exec_Action_Parameters);

         declare
            Action : constant Action_Record := Lookup_Action
              (Kernel, Nth_Arg (Data, 1));
            Context : constant Selection_Context_Access :=
              Get_Current_Context (Kernel);
         begin
            if Action = No_Action then
               Set_Error_Msg (Data, -"No such registered action");

            elsif not Filter_Matches (Action.Filter, Context, Kernel) then
               Set_Error_Msg (Data, -"Invalid context for the action");

            else
               --  Have a small delay, since custom actions would launch
               --  external commands in background
               Launch_Background_Command
                 (Kernel, Action.Command, Destroy_On_Exit => False,
                  Active => False, Queue_Id => "");
            end if;
         end;

      elsif Command = "scenario_variables" then
         declare
            Vars : constant Scenario_Variable_Array :=
              Scenario_Variables (Kernel);
         begin
            for V in Vars'Range loop
               Set_Return_Value (Data, Value_Of (Vars (V)));
               Set_Return_Value_Key
                 (Data, External_Reference_Of (Vars (V)));
            end loop;
         end;

      elsif Command = "scenario_variables_cmd_line" then
         Name_Parameters (Data, Scenar_Var_Parameters);
         declare
            Prefix : constant String := Nth_Arg (Data, 1, "");
         begin
            Set_Return_Value
              (Data, Scenario_Variables_Cmd_Line (Kernel, Prefix));
         end;

      elsif Command = "input_dialog" then
         declare
            Dialog : Gtk_Dialog;
            Label  : Gtk_Label;
            Group  : Gtk_Size_Group;
            Hbox   : Gtk_Hbox;
            Button : Gtk_Widget;

            type Ent_Array
               is array (2 .. Number_Of_Arguments (Data)) of Gtk_Entry;
            Ent : Ent_Array;

         begin
            Name_Parameters (Data, Input_Dialog_Cmd_Parameters);
            Gtk_New (Dialog,
                     Title  => Nth_Arg (Data, 1),
                     Parent => Get_Main_Window (Kernel),
                     Flags  => Modal);

            Gtk_New (Label, Nth_Arg (Data, 1));
            Set_Alignment (Label, 0.0, 0.5);
            Pack_Start (Get_Vbox (Dialog), Label, Expand => False);

            Gtk_New (Group);

            for Num in Ent'Range loop
               Gtk_New_Hbox (Hbox, Homogeneous => False);
               Pack_Start (Get_Vbox (Dialog), Hbox);

               Gtk_New (Label, Nth_Arg (Data, Num) & ':');
               Set_Alignment (Label, 0.0, 0.5);
               Add_Widget (Group, Label);
               Pack_Start (Hbox, Label, Expand => False);

               Gtk_New (Ent (Num));
               Set_Activates_Default (Ent (Num),  True);
               Pack_Start (Hbox, Ent (Num));
            end loop;

            Button := Add_Button (Dialog, Stock_Ok, Gtk_Response_OK);
            Grab_Default (Button);
            Button := Add_Button (Dialog, Stock_Cancel, Gtk_Response_Cancel);

            Show_All (Dialog);

            Set_Return_Value_As_List (Data);

            if Run (Dialog) = Gtk_Response_OK then
               for Num in Ent'Range loop
                  Set_Return_Value (Data, Get_Text (Ent (Num)));
               end loop;
            end if;

            Destroy (Dialog);
         end;

      end if;
   end Default_Command_Handler;

   -------------------------------------
   -- Create_Location_Command_Handler --
   -------------------------------------

   procedure Create_Location_Command_Handler
     (Data : in out Callback_Data'Class; Command : String)
   is
      Kernel   : constant Kernel_Handle := Get_Kernel (Data);
      Instance : constant Class_Instance :=
        Nth_Arg (Data, 1, Get_File_Location_Class (Kernel));
      Location : File_Location_Info;

   begin
      if Command = Constructor_Method then
         Name_Parameters (Data, Location_Cmd_Parameters);

         declare
            File : constant Class_Instance  :=
              Nth_Arg (Data, 2, Get_File_Class (Kernel));
            L    : constant Integer := Nth_Arg (Data, 3);
            C    : constant Integer := Nth_Arg (Data, 4);

         begin
            Ref (File);
            Set_Data (Instance, File_Location_Info'(File, L, C));
         end;

      elsif Command = "line" then
         Location := Get_Data (Instance);
         Set_Return_Value (Data, Get_Line (Location));

      elsif Command = "column" then
         Location := Get_Data (Instance);
         Set_Return_Value (Data, Get_Column (Location));

      elsif Command = "file" then
         Location := Get_Data (Instance);
         Set_Return_Value (Data, Get_File (Location));
      end if;
   end Create_Location_Command_Handler;

   -----------------------------------
   -- Create_Entity_Command_Handler --
   -----------------------------------

   procedure Create_Entity_Command_Handler
     (Data : in out Callback_Data'Class; Command : String)
   is
      Kernel   : constant Kernel_Handle := Get_Kernel (Data);
      Entity   : Entity_Information;
      Instance : constant Class_Instance :=
        Nth_Arg (Data, 1, Get_Entity_Class (Kernel));

   begin
      if Command = Constructor_Method then
         Name_Parameters (Data, Entity_Cmd_Parameters);

         declare
            Name     : constant String  := Nth_Arg (Data, 2);
            File     : constant Class_Instance  :=
              Nth_Arg (Data, 3, Get_File_Class (Kernel),
                       Default    => null,
                       Allow_Null => True);
            L        : constant Integer := Nth_Arg (Data, 4, Default => 1);
            C        : constant Integer := Nth_Arg (Data, 5, Default => 1);
            Status   : Find_Decl_Or_Body_Query_Status;
            Lib_Info : LI_File_Ptr;
            F        : File_Info;

         begin
            if File = null then
               Entity := Create_Predefined_Entity
                 (Name, (Unresolved_Entity, False, True, False));
               Set_Data (Instance, Entity);
               Destroy (Entity);
               return;
            end if;

            F := Get_Data (File);
            Lib_Info := Locate_From_Source_And_Complete (Kernel, Get_File (F));

            if Lib_Info = No_LI_File then
               Set_Error_Msg
                 (Data, -"Xref information not found for: """
                  & Full_Name (Get_File (F)).all & '"');
               return;
            end if;

            Find_Declaration_Or_Overloaded
              (Kernel      => Kernel,
               Lib_Info    => Lib_Info,
               File_Name   => Get_File (F),
               Entity_Name => Name,
               Line        => L,
               Column      => C,
               Entity      => Entity,
               Status      => Status);

            if Status /= Success and then Status /= Fuzzy_Match then
               Set_Error_Msg (Data, -"Entity not found");
               Destroy (Entity);
            else
               Set_Data (Instance, Entity);
               Destroy (Entity);
            end if;
         end;

      elsif Command = "name" then
         Entity := Get_Data (Instance);
         Set_Return_Value (Data, Get_Name (Entity));

      elsif Command = "decl_file" then
         Entity := Get_Data (Instance);

         if not Is_Predefined_Entity (Entity) then
            Set_Return_Value
              (Data,
               Create_File
                 (Get_Script (Data), Get_Declaration_File_Of (Entity)));
         end if;

      elsif Command = "decl_line" then
         Entity := Get_Data (Instance);
         Set_Return_Value (Data, Get_Declaration_Line_Of (Entity));

      elsif Command = "decl_column" then
         Entity := Get_Data (Instance);
         Set_Return_Value (Data, Get_Declaration_Column_Of (Entity));

      elsif Command = "body" then
         declare
            Status : Find_Decl_Or_Body_Query_Status;
            Lib_Info : LI_File_Ptr;
            Location : File_Location;
         begin
            Entity := Get_Data (Instance);
            Lib_Info := Locate_From_Source_And_Complete
              (Kernel, Get_Declaration_File_Of (Entity));
            Find_Next_Body
              (Kernel,
               Lib_Info    => Lib_Info,
               File_Name   => Get_Declaration_File_Of (Entity),
               Entity_Name => Get_Name (Entity),
               Line        => Get_Declaration_Line_Of (Entity),
               Column      => Get_Declaration_Column_Of (Entity),
               Location    => Location,
               Status      => Status);

            if Status = Success then
               Set_Return_Value
                 (Data, Create_File_Location
                    (Get_Script (Data),
                     File   => Create_File
                       (Get_Script (Data), Get_File (Location)),
                     Line   => Get_Line (Location),
                     Column => Get_Column (Location)));

            else
               Set_Error_Msg (Data, -"Body not found for the entity");
            end if;
         end;
      end if;
   end Create_Entity_Command_Handler;

   ---------------------------------
   -- Create_File_Command_Handler --
   ---------------------------------

   procedure Create_File_Command_Handler
     (Data : in out Callback_Data'Class; Command : String)
   is
      Kernel   : constant Kernel_Handle := Get_Kernel (Data);
      Instance : constant Class_Instance :=
        Nth_Arg (Data, 1, Get_File_Class (Kernel));
      Info     : File_Info;
   begin
      if Command = Constructor_Method then
         Name_Parameters (Data, File_Cmd_Parameters);
         Info := (File => Create (Nth_Arg (Data, 2), Kernel));
         Set_Data (Instance, Info);
         Free (Info);

      elsif Command = "name" then
         Info := Get_Data (Instance);
         Set_Return_Value (Data, Full_Name (Info.File).all);

      elsif Command = "project" then
         Info := Get_Data (Instance);
         Set_Return_Value
           (Data, Create_Project
            (Get_Script (Data),
             Get_Project_From_File
             (Registry          => Project_Registry (Get_Registry (Kernel)),
              Source_Filename   => Info.File,
              Root_If_Not_Found => True)));

      elsif Command = "other_file" then
         Info := Get_Data (Instance);
         Set_Return_Value
           (Data, Create_File
            (Get_Script (Data), Other_File_Name (Kernel, Info.File)));
      end if;
   end Create_File_Command_Handler;

   ------------------------------------
   -- Create_Project_Command_Handler --
   ------------------------------------

   procedure Create_Project_Command_Handler
     (Data : in out Callback_Data'Class; Command : String)
   is
      Kernel : constant Kernel_Handle := Get_Kernel (Data);

      procedure Set_Return_Attribute
        (Project          : Project_Type;
         Attr, Pkg, Index : String;
         As_List          : Boolean);
      --  Store in Data the value of a specific attribute

      --------------------------
      -- Set_Return_Attribute --
      --------------------------

      procedure Set_Return_Attribute
        (Project          : Project_Type;
         Attr, Pkg, Index : String;
         As_List          : Boolean)
      is
         List : Argument_List := Get_Attribute_Value
           (Project, Build (Pkg, Attr), Index);
      begin
         if As_List then
            Set_Return_Value_As_List (Data);
         end if;

         --  If the attribute was a string in fact
         if List'Length = 0 then
            Set_Return_Value
              (Data, Get_Attribute_Value
                 (Project, Build (Pkg, Attr), "", Index));

         else
            if As_List then
               for L in List'Range loop
                  Set_Return_Value (Data, List (L).all);
               end loop;
            else
               Set_Return_Value
                 (Data, Argument_List_To_String (List, True));
            end if;
         end if;

         Basic_Types.Free (List);
      end Set_Return_Attribute;

      Instance : Class_Instance;
      Project  : Project_Type;

   begin
      if Command = "load" then
         Name_Parameters (Data, Open_Cmd_Parameters);
         Load_Project (Kernel, Nth_Arg (Data, 1));
         Set_Return_Value
           (Data, Create_Project (Get_Script (Data), Get_Project (Kernel)));

      elsif Command = "recompute" then
         Recompute_View (Get_Kernel (Data));

      elsif Command = "root" then
         Set_Return_Value
           (Data, Create_Project (Get_Script (Data), Get_Project (Kernel)));

      else
         Instance := Nth_Arg (Data, 1, Get_Project_Class (Kernel));

         if Command = Constructor_Method then
            Name_Parameters (Data, Project_Cmd_Parameters);
            Project  := Get_Project_From_Name
              (Project_Registry (Get_Registry (Kernel)),
               Get_String (Nth_Arg (Data, 2)));

            if Project = No_Project then
               Set_Error_Msg (Data, -"No such project: " & Nth_Arg (Data, 2));
            else
               Set_Data (Instance, Project);
            end if;

         elsif Command = "name" then
            Project := Get_Data (Instance);
            Set_Return_Value (Data, Project_Name (Project));

         elsif Command = "file" then
            Project := Get_Data (Instance);
            Set_Return_Value
              (Data,
               Create_File
                 (Get_Script (Data), Create (Project_Path (Project))));

         elsif Command = "ancestor_deps" then
            declare
               Iter : Imported_Project_Iterator;
               P    : Project_Type;
            begin
               Project := Get_Data (Instance);
               Set_Return_Value_As_List (Data);
               Iter := Find_All_Projects_Importing
                 (Get_Project (Kernel), Project, Include_Self => True);

               loop
                  P := Current (Iter);
                  exit when P = No_Project;
                  Set_Return_Value
                    (Data, Create_Project (Get_Script (Data), P));
                  Next (Iter);
               end loop;
            end;

         elsif Command = "get_attribute_as_list"
           or else Command = "get_attribute_as_string"
         then
            Name_Parameters (Data, Get_Attributes_Parameters);
            Set_Return_Attribute
              (Project => Get_Data (Instance),
               Attr => Nth_Arg (Data, 2),
               Pkg  => Nth_Arg (Data, 3, ""),
               Index => Nth_Arg (Data, 4, ""),
               As_List => Command = "get_attribute_as_list");

         elsif Command = "get_tool_switches_as_list"
           or else Command = "get_tool_switches_as_string"
         then
            Name_Parameters (Data, Tool_Parameters);
            declare
               Tool  : constant String := Nth_Arg (Data, 2);
               Props : constant Tool_Properties_Record :=
                 Get_Tool_Properties (Kernel, Tool);

            begin
               if Props = No_Tool then
                  Set_Error_Msg (Data, -"No such tool: " & Tool);
               else
                  Set_Return_Attribute
                    (Project => Get_Data (Instance),
                     Attr    => Props.Project_Attribute.all,
                     Pkg     => Props.Project_Package.all,
                     Index   => Props.Project_Index.all,
                     As_List => Command = "get_tool_switches_as_list");
               end if;
            end;
         end if;
      end if;
   end Create_Project_Command_Handler;

   ------------------------------------
   -- Entity_Context_Command_Handler --
   ------------------------------------

   procedure Entity_Context_Command_Handler
     (Data : in out Callback_Data'Class; Command : String)
   is
      Kernel   : constant Kernel_Handle := Get_Kernel (Data);
      Instance : constant Class_Instance := Nth_Arg
        (Data, 1, Get_Entity_Context_Class (Kernel));
      Entity   : constant Entity_Selection_Context_Access :=
        Entity_Selection_Context_Access'(Get_Data (Instance));
      L, C     : Integer := -1;

   begin
      if Command = "location" then
         if Has_Line_Information (Entity) then
            L := Line_Information (Entity);
         end if;

         if Has_Column_Information (Entity) then
            C := Column_Information (Entity);
         end if;

         if Has_File_Information (Entity) then
            Set_Return_Value
              (Data,
               Create_File_Location
                 (Get_Script (Data),
                  (Create_File (Get_Script (Data), File_Information (Entity))),
                  L,
                  C));
         else
            Set_Error_Msg
              (Data, -"No file information stored in the context");
         end if;

      elsif Command = "entity" then
         Set_Return_Value
           (Data, Create_Entity (Get_Script (Data), Get_Entity (Entity)));
      end if;
   end Entity_Context_Command_Handler;

   -----------------------------
   -- Context_Command_Handler --
   -----------------------------

   procedure Context_Command_Handler
     (Data : in out Callback_Data'Class; Command : String)
   is
      Kernel   : constant Kernel_Handle := Get_Kernel (Data);
      Instance : Class_Instance;
      File     : File_Selection_Context_Access;
      Context  : Selection_Context_Access;
      L, C     : Integer := -1;

   begin
      if Command = Constructor_Method then
         Set_Error_Msg (Data, -"Cannot create an instance of this class");

      elsif Command = "file" then
         Instance := Nth_Arg (Data, 1, Get_File_Context_Class (Kernel));
         File := File_Selection_Context_Access'(Get_Data (Instance));
         if Has_File_Information (File) then
            Set_Return_Value
              (Data, Create_File (Get_Script (Data), File_Information (File)));
         else
            Set_Error_Msg (Data, -"No file information stored in the context");
         end if;

      elsif Command = "project" then
         Instance := Nth_Arg (Data, 1, Get_File_Context_Class (Kernel));
         File := File_Selection_Context_Access'(Get_Data (Instance));
         if Has_Project_Information (File) then
            Set_Return_Value
              (Data,
               Create_Project (Get_Script (Data), Project_Information (File)));
         elsif Has_File_Information (File) then
         --  Since the editor doesn't provide the project, we emulate it
            --  here
            Set_Return_Value
              (Data,
               Create_Project
                 (Get_Script (Data),
                  Get_Project_From_File
                    (Project_Registry (Get_Registry (Kernel)),
                     File_Information (File),
                     Root_If_Not_Found => False)));
         else
            Set_Error_Msg (Data, -"No project stored in the context");
         end if;

      elsif Command = "directory" then
         Instance := Nth_Arg (Data, 1, Get_File_Context_Class (Kernel));
         File := File_Selection_Context_Access'(Get_Data (Instance));
         if Has_Directory_Information (File) then
            Set_Return_Value (Data, Directory_Information (File));
         else
            Set_Error_Msg (Data, -"No directory stored in the context");
         end if;

      elsif Command = "location" then
         Instance := Nth_Arg
           (Data, 1, Get_File_Context_Class (Kernel));
         File := File_Selection_Context_Access'(Get_Data (Instance));

         if Has_Line_Information (File) then
            L := Line_Information (File);
         end if;

         if Has_Column_Information (File) then
            C := Column_Information (File);
         end if;

         if Has_File_Information (File) then
            Set_Return_Value
              (Data,
               Create_File_Location
                 (Get_Script (Data),
                  (Create_File (Get_Script (Data), File_Information (File))),
                  L,
                  C));
         else
            Set_Error_Msg
              (Data, -"No file information stored in the context");
         end if;

      elsif Command = "current_context" then
         Context := Get_Current_Context (Kernel);
         if Context = null then
            Set_Error_Msg (Data, -"There is no current context");

         elsif Context.all in Entity_Selection_Context'Class then
            Set_Return_Value
             (Data, Create_Entity_Context
               (Get_Script (Data), Entity_Selection_Context_Access (Context)));

         elsif Context.all in File_Selection_Context'Class then
            Set_Return_Value
             (Data, Create_File_Context
              (Get_Script (Data), File_Selection_Context_Access (Context)));

         else
            Set_Error_Msg (Data, -"Unknown current context");
         end if;
      end if;
   end Context_Command_Handler;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize
     (Kernel : access Glide_Kernel.Kernel_Handle_Record'Class) is
   begin
      Kernel.Scripts := new Scripting_Data_Record;
   end Initialize;

   --------------
   -- Finalize --
   --------------

   procedure Finalize
     (Kernel : access Glide_Kernel.Kernel_Handle_Record'Class)
   is
      procedure Unchecked_Free is new Ada.Unchecked_Deallocation
        (Scripting_Language_Data, Scripting_Language_List);

      List : Scripting_Language_List :=
        Scripting_Data (Kernel.Scripts).Scripting_Languages;
      Tmp  : Scripting_Language_List;

   begin
      while List /= null loop
         Tmp := List.Next;
         Destroy (List.Script);
         Unchecked_Free (List);

         List := Tmp;
      end loop;

      --  Various classes instances stored in the kernel are freed when this
      --  table is freed.
      Classes_Hash.String_Hash_Table.Reset
        (Scripting_Data (Kernel.Scripts).Classes);
   end Finalize;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Script : access Scripting_Language_Record) is
      pragma Unreferenced (Script);
   begin
      null;
   end Destroy;

   --------------------------------------
   -- Register_Default_Script_Commands --
   --------------------------------------

   procedure Register_Default_Script_Commands
     (Kernel : access Glide_Kernel.Kernel_Handle_Record'Class) is
   begin
      Register_Command
        (Kernel,
         Command      => "insmod",
         Params       => Parameter_Names_To_Usage (Insmod_Cmd_Parameters),
         Description  => -"Dynamically register from shared-lib a new module.",
         Minimum_Args => 2,
         Maximum_Args => 2,
         Handler      => Default_Command_Handler'Access);

      Register_Command
        (Kernel,
         Command      => "lsmod",
         Return_Value => "list of modules",
         Description  => -"List modules currently loaded.",
         Minimum_Args => 0,
         Maximum_Args => 0,
         Handler      => Default_Command_Handler'Access);

      Register_Command
        (Kernel,
         Command      => "exit",
         Params       => Parameter_Names_To_Usage (Exit_Cmd_Parameters, 1),
         Description  =>
           -("Exit GPS. If there are unsaved changes, a dialog is first"
             & " displayed to ask whether these should be saved. If the"
             & " user cancels the operation through the dialog, GPS will not"
             & " exit. If force is true, then no dialog is open, and nothing"
             & " is saved"),
         Minimum_Args => Exit_Cmd_Parameters'Length - 1,
         Maximum_Args => Exit_Cmd_Parameters'Length,
         Handler      => Default_Command_Handler'Access);

      Register_Command
        (Kernel,
         Command      => "dialog",
         Params       => Parameter_Names_To_Usage (Dialog_Cmd_Parameters),
         Description  =>
           -("Display a modal dialog to report information to a user. This"
             & " blocks the interpreter until the dialog is closed."),
         Minimum_Args => 1,
         Maximum_Args => 1,
         Handler      => Default_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "yes_no_dialog",
         Params       => Parameter_Names_To_Usage (Dialog_Cmd_Parameters),
         Return_Value => "boolean",
         Description  =>
           -("Display a modal dialog to ask a question to the user. This"
             & " blocks the interpreter until the dialog is closed. The"
             & " dialog has two buttons Yes and No, and the selected button"
             & " is returned to the caller"),
         Minimum_Args => 1,
         Maximum_Args => 1,
         Handler      => Default_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "input_dialog",
         Params       =>
           Parameter_Names_To_Usage (Input_Dialog_Cmd_Parameters),
         Return_Value => "list",
         Description  =>
           -("Display a modal dialog and request some input from the user."
             & " The message is displayed at the top, and one input field"
             & " is displayed for each remaining argument. The return value"
             & " is the value that the user has input for each of these"
             & " parameters." & ASCII.LF
             & "An empty list is returned if the user presses Cancel"),
         Minimum_Args => 2,
         Maximum_Args => 100,
         Handler      => Default_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "save_all",
         Params       => Parameter_Names_To_Usage (Save_Windows_Parameters),
         Description  =>
           -("Save all currently unsaved windows. This includes open editors,"
             & " the project, and any other window that has registered some"
             & " save callbacks." & ASCII.LF
             & "If the force parameter is false, then a confirmation dialog"
             & " is displayed so that the user can select which windows"
             & " to save."),
         Minimum_Args => 0,
         Maximum_Args => 1,
         Handler      => Default_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "execute_action",
         Params       => Parameter_Names_To_Usage (Exec_Action_Parameters),
         Description  =>
           -("Execute a GPS action. These are the same actions to which key"
             & " bindings can be assigned through the keymanager. They are"
             & " either exported by GPS itself, or created through XML"
             & " customization files. The action is not executed if the"
             & " current context is not appropriate for the action. This"
             & " function is asynchronous, ie will return immediately before"
             & " the action is completed."),
         Minimum_Args => 1,
         Maximum_Args => 1,
         Handler      => Default_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "parse_xml",
         Params       => Parameter_Names_To_Usage (Xml_Custom_Parameters),
         Description  =>
           -("Load an XML customization string. This string should contain"
             & " one or more toplevel tags similar to what is normally found"
             & " in custom files, such as <key>, <alias>, <action>,.."
             & ASCII.LF
             & "Optionally you can also pass the full contents of an XML file,"
             & " starting from the <?xml?> header"),
         Minimum_Args => 1,
         Maximum_Args => 1,
         Handler      => Default_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "scenario_variables",
         Return_Value => "hash",
         Description  =>
           -("Return the list of scenario variables for the current project "
             & " hierarchy, and their current value"),
         Minimum_Args => 0,
         Maximum_Args => 0,
         Handler      => Default_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "scenario_variables_cmd_line",
         Params       => Parameter_Names_To_Usage (Scenar_Var_Parameters, 1),
         Return_Value => "string",
         Description  =>
           -("Return a concatenation of VARIABLE=VALUE, each preceded by the"
             & " given prefix. This string will generally be used when calling"
             & " external tool, for instance make or GNAT"),
         Minimum_Args => 0,
         Maximum_Args => 1,
         Handler      => Default_Command_Handler'Access);

      Register_Command
        (Kernel,
         Command      => "set_busy",
         Params       => "()",
         Description  =>
           -("Activate the ""busy"" indicator in GPS."),
         Minimum_Args => 0,
         Maximum_Args => 0,
         Handler      => Default_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "unset_busy",
         Params       => "()",
         Description  =>
           -("Deactivate the ""busy"" indicator in GPS."),
         Minimum_Args => 0,
         Maximum_Args => 0,
         Handler      => Default_Command_Handler'Access);

      Register_Command
        (Kernel,
         Command      => Constructor_Method,
         Params       => Parameter_Names_To_Usage (File_Cmd_Parameters),
         Return_Value => "file",
         Description  => -"Create a new file, from its name.",
         Minimum_Args => 1,
         Maximum_Args => 1,
         Class        => Get_File_Class (Kernel),
         Handler      => Create_File_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "name",
         Return_Value => "string",
         Description  => -"Return the name of the file",
         Class        => Get_File_Class (Kernel),
         Handler      => Create_File_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "other_file",
         Return_Value => "file",
         Description  =>
           -("Return the name of the other file semantically associated with"
             & " this one. In Ada this is the spec or body of the same package"
             & " depending on the type of this file."),
         Class        => Get_File_Class (Kernel),
         Handler      => Create_File_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "project",
         Return_Value => "project",
         Description  =>
           -("Return the project to which file belongs. If file is not one"
             & " of the souces of the project, the root project is returned"),
         Class        => Get_File_Class (Kernel),
         Handler      => Create_File_Command_Handler'Access);

      Register_Command
        (Kernel,
         Command      => Constructor_Method,
         Params       => Parameter_Names_To_Usage (Entity_Cmd_Parameters, 2),
         Return_Value => "entity",
         Description  =>
           -("Create a new entity, from any of its references. File must be"
             & " an instance of the File method, or omitted for predefined"
             & " entities of the language"),
         Minimum_Args => 1,
         Maximum_Args => 4,
         Class        => Get_Entity_Class (Kernel),
         Handler      => Create_Entity_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "name",
         Return_Value => "string",
         Description  => -"Return the name of the entity",
         Class        => Get_Entity_Class (Kernel),
         Handler      => Create_Entity_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "decl_file",
         Return_Value => "GPS.File",
         Description  =>
           -("Return the file in which the entity is declared. This file's"
             & " name is empty for predefined entities"),
         Class        => Get_Entity_Class (Kernel),
         Handler      => Create_Entity_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "decl_line",
         Return_Value => "integer",
         Description  => -("Return the line in decl_file() at which the"
                           & " entity is defined"),
         Class        => Get_Entity_Class (Kernel),
         Handler      => Create_Entity_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "decl_column",
         Return_Value => "integer",
         Description  => -("Return the column in decl_file() at which the"
                           & " entity is defined"),
         Class        => Get_Entity_Class (Kernel),
         Handler      => Create_Entity_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "body",
         Return_Value => "FileLocation",
         Description  =>
           -("Return the location of the body for this entity. This is the"
             & " place where the subprogram is implemented, or where the full"
             & " definition of a type is visible. For types which do not have"
             & " the notion of body, this returns the location of the"
             & " declaration"),
         Class        => Get_Entity_Class (Kernel),
         Handler      => Create_Entity_Command_Handler'Access);

      Register_Command
        (Kernel,
         Command      => Constructor_Method,
         Params       => Parameter_Names_To_Usage (Location_Cmd_Parameters),
         Return_Value => "location",
         Description  => -"Create a new file location, from its position.",
         Minimum_Args => 3,
         Maximum_Args => 3,
         Class        => Get_File_Location_Class (Kernel),
         Handler      => Create_Location_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command       => "line",
         Return_Value  => "integer",
         Description   => -"Return the line of the location",
         Class         => Get_File_Location_Class (Kernel),
         Handler       => Create_Location_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command       => "column",
         Return_Value  => "integer",
         Description   => -"Return the column of the location",
         Class         => Get_File_Location_Class (Kernel),
         Handler       => Create_Location_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "file",
         Return_Value => "File",
         Description  => -"Return the file of the location",
         Class        => Get_File_Location_Class (Kernel),
         Handler      => Create_Location_Command_Handler'Access);

      Register_Command
        (Kernel,
         Command      => Constructor_Method,
         Params       => Parameter_Names_To_Usage (Project_Cmd_Parameters),
         Return_Value => "project",
         Description  =>
           -("Create a project handle, from its name. The project must have"
             & " been loaded already."),
         Minimum_Args => 1,
         Maximum_Args => 1,
         Class        => Get_Project_Class (Kernel),
         Handler      => Create_Project_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command       => "root",
         Return_Value  => "project",
         Description   =>
           -("Return the root project of the currently loaded hierarchy"),
         Class         => Get_Project_Class (Kernel),
         Static_Method => True,
         Handler       => Create_Project_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command       => "recompute",
         Description   =>
            -("Recompute the contents of a project, including the list of"
              & " source files that are automatically loaded from the"
              & " source directories"),
         Class         => Get_Project_Class (Kernel),
         Static_Method => True,
         Handler       => Create_Project_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command       => "load",
         Params        => Parameter_Names_To_Usage (Open_Cmd_Parameters),
         Return_Value  => "project",
         Description   =>
         -("Load a new project, which replaces the current root project, and"
           & " return a handle to it. All imported projects are also"
           & " loaded at the same time. If the project is not found, a"
           & " default project is loaded"),
         Minimum_Args  => 1,
         Maximum_Args  => 1,
         Class         => Get_Project_Class (Kernel),
         Static_Method => True,
         Handler       => Create_Project_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "name",
         Return_Value => "string",
         Description  => -"Return the name of the project",
         Class        => Get_Project_Class (Kernel),
         Handler      => Create_Project_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "file",
         Return_Value => "File",
         Description  => -"Return a handle to the project file itself",
         Class        => Get_Project_Class (Kernel),
         Handler      => Create_Project_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "ancestor_deps",
         Return_Value => "list",
         Description  =>
           -("Return the list of projects that might contain sources that"
             & " depend on the project's sources. When doing extensive"
             & " searches it isn't worth checking other projects. Project"
             & " itself is included in the list."),
         Class        => Get_Project_Class (Kernel),
         Handler      => Create_Project_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "get_attribute_as_string",
         Params       =>
           Parameter_Names_To_Usage (Get_Attributes_Parameters, 2),
         Return_Value => "string",
         Description  =>
           -("Fetch the value of the attribute in the project." & ASCII.LF
             & "If the package is not specified, the attribute at the"
             & " toplevel of the project is queried." & ASCII.LF
             & "The index only needs to be specified if it applies to that"
             & " attribute." & ASCII.LF
             & "If the attribute value is stored as a list, the result string"
             & " is a concatenation of all the elements of the list."
             & " This function always returns the value of the attribute in"
             & " the currently selected scenario."),
         Minimum_Args => 2,
         Maximum_Args => 4,
         Class        => Get_Project_Class (Kernel),
         Handler      => Create_Project_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "get_attribute_as_list",
         Params       =>
           Parameter_Names_To_Usage (Get_Attributes_Parameters, 2),
         Return_Value => "list",
         Description  =>
           -("Fetch the value of the attribute in the project." & ASCII.LF
             & "If the package is not specified, the attribute at the"
             & " toplevel of the project is queried." & ASCII.LF
             & "The index only needs to be specified if it applies to that"
             & " attribute." & ASCII.LF
             & "If the attribute value is stored as a simple string, a list"
             & " with a single element is returned."
             & " This function always returns the value of the attribute in"
             & " the currently selected scenario."),
         Minimum_Args => 1,
         Maximum_Args => 3,
         Class        => Get_Project_Class (Kernel),
         Handler      => Create_Project_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "get_tool_switches_as_list",
         Params       => Parameter_Names_To_Usage (Tool_Parameters),
         Return_Value => "list",
         Description  =>
           -("Same as get_attribute_as_list, but specialized for the switches"
             & " of a specific tool"),
         Minimum_Args => 1,
         Maximum_Args => 1,
         Class        => Get_Project_Class (Kernel),
         Handler      => Create_Project_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "get_tool_switches_as_string",
         Params       => Parameter_Names_To_Usage (Tool_Parameters),
         Return_Value => "list",
         Description  =>
           -("Same as get_attribute_as_string, but specialized for the"
             & " switches of a specific tool"),
         Minimum_Args => 1,
         Maximum_Args => 1,
         Class        => Get_Project_Class (Kernel),
         Handler      => Create_Project_Command_Handler'Access);

      Register_Command
        (Kernel,
         Command      => Constructor_Method,
         Return_Value => "FileContext",
         Description  => -"Prevents creation of FileContext instances",
         Class        => Get_File_Context_Class (Kernel),
         Handler      => Context_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "file",
         Return_Value => "File",
         Description  => -"Return the name of the file in the context",
         Class        => Get_File_Context_Class (Kernel),
         Handler      => Context_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "project",
         Return_Value => "Project",
         Description  =>
           -("Return the project in the context, or the root project if none"
             & " was specified in the context"),
         Class        => Get_File_Context_Class (Kernel),
         Handler      => Context_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "directory",
         Return_Value => "string",
         Description  => -("Return the current directory in the context"),
         Class        => Get_File_Context_Class (Kernel),
         Handler      => Context_Command_Handler'Access);

      Register_Command
        (Kernel,
         Command      => Constructor_Method,
         Return_Value => "EntityContext",
         Description  => -"Prevents creation of EntityContext instances",
         Class        => Get_Entity_Context_Class (Kernel),
         Handler      => Context_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "entity",
         Return_Value => "Entity",
         Description  => -"Return the entity stored in the context",
         Class        => Get_Entity_Context_Class (Kernel),
         Handler      => Entity_Context_Command_Handler'Access);
      Register_Command
        (Kernel,
         Command      => "location",
         Return_Value => "FileLocation",
         Description  => -"Return the file location stored in the context",
         Class        => Get_Entity_Context_Class (Kernel),
         Handler      => Entity_Context_Command_Handler'Access);

      Register_Command
        (Kernel,
         Command      => "current_context",
         Return_Value => "FileContext",
         Description  =>
           -("Returns the current context in GPS. This is the currently"
             & " selected file, line, column, project,... depending on"
             & " what window is currently active"),
         Handler      => Context_Command_Handler'Access);
   end Register_Default_Script_Commands;

   ----------------------
   -- Get_Entity_Class --
   ----------------------

   function Get_Entity_Class
     (Kernel : access Glide_Kernel.Kernel_Handle_Record'Class)
      return Class_Type is
   begin
      if Scripting_Data (Kernel.Scripts).Entity_Class = No_Class then
         Scripting_Data (Kernel.Scripts).Entity_Class := New_Class
           (Kernel,
            "Entity", "Represents an entity from the source, based on the"
            & " location of its declaration");
      end if;

      return Scripting_Data (Kernel.Scripts).Entity_Class;
   end Get_Entity_Class;

   --------------------
   -- Get_File_Class --
   --------------------

   function Get_File_Class
     (Kernel : access Glide_Kernel.Kernel_Handle_Record'Class)
      return Class_Type is
   begin
      if Scripting_Data (Kernel.Scripts).File_Class = No_Class then
         Scripting_Data (Kernel.Scripts).File_Class := New_Class
           (Kernel,
            "File", "Represents a source file of your application");
      end if;

      return Scripting_Data (Kernel.Scripts).File_Class;
   end Get_File_Class;

   -----------------------
   -- Get_Project_Class --
   -----------------------

   function Get_Project_Class
     (Kernel : access Glide_Kernel.Kernel_Handle_Record'Class)
      return Class_Type is
   begin
      if Scripting_Data (Kernel.Scripts).Project_Class = No_Class then
         Scripting_Data (Kernel.Scripts).Project_Class := New_Class
           (Kernel, "Project", "Represents a project file");
      end if;

      return Scripting_Data (Kernel.Scripts).Project_Class;
   end Get_Project_Class;

   -----------------------------
   -- Get_File_Location_Class --
   -----------------------------

   function Get_File_Location_Class
     (Kernel : access Glide_Kernel.Kernel_Handle_Record'Class)
      return Class_Type is
   begin
      if Scripting_Data (Kernel.Scripts).File_Location_Class = No_Class then
         Scripting_Data (Kernel.Scripts).File_Location_Class := New_Class
           (Kernel, "FileLocation", "Represents a location in a file");
      end if;

      return Scripting_Data (Kernel.Scripts).File_Location_Class;
   end Get_File_Location_Class;

   -------------------------------
   -- Execute_Command_With_Args --
   -------------------------------

   function Execute_Command_With_Args
     (Script             : access Scripting_Language_Record;
      Command            : String;
      Args               : GNAT.OS_Lib.Argument_List) return String
   is
      pragma Unreferenced (Script, Command, Args);
   begin
      raise Program_Error;
      return "";
   end Execute_Command_With_Args;

   -------------------------------
   -- Execute_GPS_Shell_Command --
   -------------------------------

   function Execute_GPS_Shell_Command
     (Kernel  : access Glide_Kernel.Kernel_Handle_Record'Class;
      Command : String) return String
   is
      Errors : aliased Boolean;
   begin
      return Execute_Command
        (Lookup_Scripting_Language (Kernel, GPS_Shell_Name),
         Command, null, True, True, Errors'Unchecked_Access);
   end Execute_GPS_Shell_Command;

   -------------------------------
   -- Execute_GPS_Shell_Command --
   -------------------------------

   procedure Execute_GPS_Shell_Command
     (Kernel  : access Glide_Kernel.Kernel_Handle_Record'Class;
      Command : String;
      Args    : GNAT.OS_Lib.Argument_List)
   is
      Output : constant String := Execute_Command_With_Args
        (Lookup_Scripting_Language (Kernel, GPS_Shell_Name),
         Command, Args);
      pragma Unreferenced (Output);
   begin
      null;
   end Execute_GPS_Shell_Command;

   -------------------------------
   -- Execute_GPS_Shell_Command --
   -------------------------------

   function Execute_GPS_Shell_Command
     (Kernel  : access Glide_Kernel.Kernel_Handle_Record'Class;
      Command : String;
      Args    : GNAT.OS_Lib.Argument_List) return String is
   begin
      return Execute_Command_With_Args
        (Lookup_Scripting_Language (Kernel, GPS_Shell_Name),
         Command, Args);
   end Execute_GPS_Shell_Command;

   -------------------------------
   -- Execute_GPS_Shell_Command --
   -------------------------------

   procedure Execute_GPS_Shell_Command
     (Kernel  : access Glide_Kernel.Kernel_Handle_Record'Class;
      Command : String)
   is
      Errors : aliased Boolean;
      Str : constant String := Execute_Command
        (Lookup_Scripting_Language (Kernel, GPS_Shell_Name),
         Command, null, True, True, Errors'Unchecked_Access);
      pragma Unreferenced (Str);
   begin
      null;
   end Execute_GPS_Shell_Command;

   ---------------------
   -- Execute_Command --
   ---------------------

   function Execute_Command
     (Script             : access Scripting_Language_Record;
      Command            : String;
      Console            : Interactive_Consoles.Interactive_Console := null;
      Hide_Output        : Boolean := False;
      Show_Command       : Boolean := True;
      Errors             : access Boolean) return String is
   begin
      Execute_Command
        (Scripting_Language (Script),
         Command, Console, Hide_Output, Show_Command, Errors.all);
      return "";
   end Execute_Command;

   --------------
   -- Get_File --
   --------------

   function Get_File (File : File_Info) return Virtual_File is
   begin
      return File.File;
   end Get_File;

   -------------
   -- Nth_Arg --
   -------------

   function Nth_Arg
     (Data : Callback_Data; N : Positive; Default : String)
      return String is
   begin
      return Nth_Arg (Callback_Data'Class (Data), N);
   exception
      when No_Such_Parameter =>
         return Default;
   end Nth_Arg;

   -------------
   -- Nth_Arg --
   -------------

   function Nth_Arg
     (Data : Callback_Data; N : Positive; Default : Integer)
      return Integer is
   begin
      return Nth_Arg (Callback_Data'Class (Data), N);
   exception
      when No_Such_Parameter =>
         return Default;
   end Nth_Arg;

   -------------
   -- Nth_Arg --
   -------------

   function Nth_Arg
     (Data : Callback_Data; N : Positive; Default : Boolean)
      return Boolean is
   begin
      return Nth_Arg (Callback_Data'Class (Data), N);
   exception
      when No_Such_Parameter =>
         return Default;
   end Nth_Arg;

   -------------
   -- Nth_Arg --
   -------------

   function Nth_Arg
     (Data    : Callback_Data;
      N       : Positive;
      Default : System.Address)
      return System.Address is
   begin
      return Nth_Arg (Callback_Data'Class (Data), N);
   exception
      when No_Such_Parameter =>
         return Default;
   end Nth_Arg;

   -------------
   -- Nth_Arg --
   -------------

   function Nth_Arg
     (Data    : Callback_Data;
      N       : Positive;
      Class   : Class_Type;
      Default : Class_Instance;
      Allow_Null : Boolean := False)
      return Class_Instance is
   begin
      return Nth_Arg (Callback_Data'Class (Data), N, Class, Allow_Null);
   exception
      when No_Such_Parameter =>
         return Default;
   end Nth_Arg;

   ------------------------------
   -- Parameter_Names_To_Usage --
   ------------------------------

   function Parameter_Names_To_Usage
     (Parameters            : Cst_Argument_List;
      Optional_Params_Count : Natural := 0) return String
   is
      Length : Natural := 0;
   begin
      for P in Parameters'Range loop
         Length := Length + Parameters (P)'Length + 2;
      end loop;

      Length := Length + Optional_Params_Count * 2;

      declare
         Usage : String (1 .. Length);
         Index : Natural := Usage'First + 1;
      begin
         Usage (Usage'First) := '(';

         for P in Parameters'Range loop
            if Parameters'Last - P < Optional_Params_Count then
               Usage (Index) := '[';
               Index := Index + 1;
            end if;

            Usage (Index .. Index + Parameters (P)'Length - 1) :=
              Parameters (P).all;
            Index := Index + Parameters (P)'Length;

            if Parameters'Last - P < Optional_Params_Count then
               Usage (Index) := ']';
               Index := Index + 1;
            end if;

            if P /= Parameters'Last then
               Usage (Index .. Index + 1) := ", ";
               Index := Index + 2;
            end if;
         end loop;

         Usage (Index .. Usage'Last) := ")";
         return Usage;
      end;
   end Parameter_Names_To_Usage;

   ----------------
   -- Get_Kernel --
   ----------------

   function Get_Kernel (Data : Callback_Data)
      return Glide_Kernel.Kernel_Handle is
   begin
      return Get_Kernel (Get_Script (Callback_Data'Class (Data)));
   end Get_Kernel;

   -------------------
   -- Create_Entity --
   -------------------

   function Create_Entity
     (Script : access Scripting_Language_Record'Class;
      Entity : Src_Info.Queries.Entity_Information) return Class_Instance
   is
      Instance : constant Class_Instance := New_Instance
        (Script, Get_Entity_Class (Get_Kernel (Script)));
   begin
      Set_Data (Instance, Entity);
      return Instance;
   end Create_Entity;

   -----------------
   -- Create_File --
   -----------------

   function Create_File
     (Script : access Scripting_Language_Record'Class;
      File   : Virtual_File) return Class_Instance
   is
      Instance : constant Class_Instance := New_Instance
        (Script, Get_File_Class (Get_Kernel (Script)));
      Info     : File_Info := (File => File);

   begin
      Set_Data (Instance, Info);
      Free (Info);
      return Instance;
   end Create_File;

   --------------------
   -- Create_Project --
   --------------------

   function Create_Project
     (Script  : access Scripting_Language_Record'Class;
      Project : Project_Type) return Class_Instance
   is
      Instance : Class_Instance := null;
   begin
      if Project /= No_Project then
         Instance := New_Instance
           (Script, Get_Project_Class (Get_Kernel (Script)));
         Set_Data (Instance, Project);
      end if;
      return Instance;
   end Create_Project;

   --------------------------
   -- Create_File_Location --
   --------------------------

   function Create_File_Location
     (Script : access Scripting_Language_Record'Class;
      File   : Class_Instance;
      Line   : Natural;
      Column : Natural) return Class_Instance
   is
      Instance : constant Class_Instance := New_Instance
        (Script, Get_File_Location_Class (Get_Kernel (Script)));
      Info     : constant File_Location_Info := (File, Line, Column);

   begin
      Set_Data (Instance, Info);
      return Instance;
   end Create_File_Location;

   --------------
   -- Get_File --
   --------------

   function Get_File (Location : File_Location_Info) return Class_Instance is
   begin
      return Location.File;
   end Get_File;

   --------------
   -- Get_Line --
   --------------

   function Get_Line (Location : File_Location_Info) return Integer is
   begin
      return Location.Line;
   end Get_Line;

   ----------------
   -- Get_Column --
   ----------------

   function Get_Column (Location : File_Location_Info) return Integer is
   begin
      return Location.Column;
   end Get_Column;

   ----------------------------
   -- Get_File_Context_Class --
   ----------------------------

   function Get_File_Context_Class
     (Kernel : access Glide_Kernel.Kernel_Handle_Record'Class)
      return Class_Type is
   begin
      if Scripting_Data (Kernel.Scripts).File_Context_Class = No_Class then
         Scripting_Data (Kernel.Scripts).File_Context_Class := New_Class
           (Kernel,
            "FileContext",
            "Represents an context that contains file information");
      end if;

      return Scripting_Data (Kernel.Scripts).File_Context_Class;
   end Get_File_Context_Class;

   ------------------------------
   -- Get_Entity_Context_Class --
   ------------------------------

   function Get_Entity_Context_Class
     (Kernel : access Glide_Kernel.Kernel_Handle_Record'Class)
      return Class_Type is
   begin
      if Scripting_Data (Kernel.Scripts).Entity_Context_Class = No_Class then
         Scripting_Data (Kernel.Scripts).Entity_Context_Class := New_Class
           (Kernel,
            "EntityContext",
            "Represents an context that contains entity information",
            Base => Get_File_Context_Class (Kernel));
      end if;

      return Scripting_Data (Kernel.Scripts).Entity_Context_Class;
   end Get_Entity_Context_Class;

   ------------------------
   -- On_Destroy_Context --
   ------------------------

   procedure On_Destroy_Context (Value : System.Address) is
      C : Selection_Context_Access := Convert (Value);
   begin
      Unref (C);
   end On_Destroy_Context;

   --------------
   -- Set_Data --
   --------------

   procedure Set_Data
     (Instance : access Class_Instance_Record'Class;
      Context  : Selection_Context_Access)
   is
      Script : constant Scripting_Language := Get_Script (Instance);
   begin
      if not Is_Subclass
        (Script,
         Get_Class (Instance),
         Get_File_Context_Class (Get_Kernel (Script)))
      then
         raise Invalid_Data;
      end if;

      Ref (Context);
      Set_Data
        (Instance,
         Value      => Context.all'Address,
         On_Destroy => On_Destroy_Context'Access);
   end Set_Data;

   --------------
   -- Get_Data --
   --------------

   function Get_Data
     (Instance : access Class_Instance_Record'Class)
      return Glide_Kernel.Modules.File_Selection_Context_Access
   is
      Script : constant Scripting_Language := Get_Script (Instance);
   begin
      if not Is_Subclass
        (Script,
         Get_Class (Instance),
         Get_File_Context_Class (Get_Kernel (Script)))
      then
         raise Invalid_Data;
      end if;

      return File_Selection_Context_Access
        (Selection_Context_Access'(Convert (Get_Data (Instance))));
   end Get_Data;

   --------------
   -- Get_Data --
   --------------

   function Get_Data (Instance : access Class_Instance_Record'Class)
      return Glide_Kernel.Modules.Entity_Selection_Context_Access
   is
      Script : constant Scripting_Language := Get_Script (Instance);
   begin
      if not Is_Subclass
        (Script,
         Get_Class (Instance),
         Get_Entity_Context_Class (Get_Kernel (Script)))
      then
         raise Invalid_Data;
      end if;

      return Entity_Selection_Context_Access
        (Selection_Context_Access'(Convert (Get_Data (Instance))));
   end Get_Data;

   -------------------------
   -- Create_File_Context --
   -------------------------

   function Create_File_Context
     (Script  : access Scripting_Language_Record'Class;
      Context : Glide_Kernel.Modules.File_Selection_Context_Access)
      return Class_Instance
   is
      Instance : constant Class_Instance := New_Instance
        (Script, Get_File_Context_Class (Get_Kernel (Script)));
   begin
      Set_Data (Instance, Selection_Context_Access (Context));
      return Instance;
   end Create_File_Context;

   ---------------------------
   -- Create_Entity_Context --
   ---------------------------

   function Create_Entity_Context
     (Script  : access Scripting_Language_Record'Class;
      Context : Glide_Kernel.Modules.Entity_Selection_Context_Access)
      return Class_Instance
   is
      Instance : constant Class_Instance := New_Instance
        (Script, Get_Entity_Context_Class (Get_Kernel (Script)));
   begin
      Set_Data (Instance, Selection_Context_Access (Context));
      return Instance;
   end Create_Entity_Context;

   ---------
   -- Ref --
   ---------

   procedure Ref (Instance : access Class_Instance_Record) is
      pragma Unreferenced (Instance);
   begin
      null;
   end Ref;

end Glide_Kernel.Scripts;
