#!/bin/sh

fail=0
verbose=0

. CONFIG

if [ x$VALGRIND = xyes ]; then
   valgrind="valgrind --tool=memcheck --leak-check=full --num-callers=30 --show-reachable=no --leak-resolution=high --suppressions=valgrind.supp -q --suppressions=valgrind-python.supp --gen-suppressions=no"
   verbose=1
else
   valgrind=""
fi

GNATLIB_LIBRARY_TYPE=${GNATLIB_LIBRARY_TYPE:-relocatable}

# In case gnatlib was not installed but we still want to run the tests, make sure
# we at least are able to find the project files
if [ "$OS" = "Windows_NT" ]; then
   DIRSEP=";"
   EXE=".exe"
else
   DIRSEP=":"
   EXE=""
fi
ADA_PROJECT_PATH=${ADA_PROJECT_PATH}${DIRSEP}`pwd`/../src
export ADA_PROJECT_PATH

gnatmake -q -P test_traces -XGNATLIB_KIND=${GNATLIB_LIBRARY_TYPE}
gnatmake -q -P test_scripts -XGNATLIB_KIND=${GNATLIB_LIBRARY_TYPE} 2>&1 | grep -v "tmpnam" | grep -v tempnam

## Check we do not embed python.
## These commands should not output anything

nm traces1${EXE} | grep Py

case `uname` in
  Linux|SunOS)
    ldd traces1${EXE} | grep libpython
    ;;
esac

## Running a test

expect() {
  cmd="$1"
  expected="$2"
  remove_cr="$3"

  if test $verbose = 1; then
     echo "====== $cmd $expected"
  fi

  if [ "$remove_cr" = "1" ]; then
     $valgrind $cmd 2>&1 | tr -d '\r' > out.$$
  else
     $valgrind $cmd 2>&1 > out.$$
  fi

  if [ "$expected" = "" ] ; then
     if test -s out.$$ ; then
        if test $verbose != 1; then
           echo "======= $cmd (expecting no output)"
        fi
        cat out.$$
        print_status 1
        fail=1
     fi

  elif diff $expected out.$$ >/dev/null ; then
     :
  else
     if test $verbose != 1; then
        echo "======= $cmd $expected"
     fi
     ls -l $expected out.$$
     diff $expected out.$$
     print_status 1
     fail=1
  fi
  rm -f out.$$
}

diff_file() {
  expected="$1"
  output="$2"
  msg="$3"
  if diff "$expected" "$output" >/dev/null; then
     :
  else
     echo "======= $msg"
     diff "$expected" "$output"
     fail=1
  fi
}



## GNAT.Traces tests

print_header "GNAT.Traces"
expect "./traces1${EXE}" "traces1.out" 1
expect "./traces2"${EXE} "traces2.out" 1
tr -d '\r' < outfile.out > outfile.out.$$
tr -d '\r' < outfile > outfile.$$
diff outfile.out.$$ outfile.$$
rm -f outfile outfile.out.$$ outfile.$$
print_status 0

## GNAT.Scripts tests
print_header "GNAT.Scripts"
expect "./testapi${EXE}" "testapi.out" 1
expect "./testsuite${EXE} python scripts1.py"  "scripts1.out" 1
expect "./testsuite${EXE} shell  scripts2.gsh" "scripts2.out" 1
expect "./testsuite${EXE} python scripts3.py"  "scripts3.out" 1
print_status 0

## GNAT.Mmap tests
print_header "GNAT.Mmap"
./test_mmap${EXE} test_mmap${EXE} full mmap
diff mmap.out test_mmap${EXE}
#expect "./test_mmap${EXE} test_mmap${EXE} full mmap"    "test_mmap${EXE}" 0

./test_mmap${EXE} test_mmap${EXE} partial mmap
diff mmap.out test_mmap${EXE}
#expect "./test_mmap${EXE} test_mmap${EXE} partial mmap" "test_mmap${EXE}" 0

./test_mmap${EXE} test_mmap${EXE} full read
diff mmap.out test_mmap${EXE}
#expect "./test_mmap${EXE} test_mmap${EXE} full read"    "test_mmap${EXE}" 0

./test_mmap${EXE} test_mmap${EXE} partial read
diff mmap.out test_mmap${EXE}
#expect "./test_mmap${EXE} test_mmap${EXE} partial read" "test_mmap${EXE}" 0

## Reading empty files
expect "./test_mmap${EXE} empty.txt full mmap"    "empty.txt" 0
expect "./test_mmap${EXE} empty.txt full read"    "empty.txt" 0

cp mmap1.in mmap.in
expect "./test_mmap_write${EXE} read"   "" 0
diff_file mmap1.out mmap.in "./test_mmap_write read"

cp mmap1.in mmap.in
expect "./test_mmap_write${EXE} mmap"   "" 0
diff_file mmap1.out mmap.in "./test_mmap_write mmap"

print_status 0

## Templates
print_header "GNAT.Templates"
expect "./templates${EXE}" "empty.txt" 0
print_status 0

## Email
(cd email; ./run.sh)

rm -f mmap.in mmap.out

if test $fail = 0; then
   exit 0
else
   exit 1
fi
