/******************************************************************************\
 gnofin/exp-latex.c   $Revision: 1.6 $
 Copyright (C) 1999 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <string.h>
#include <errno.h>
#include "dialogs.h"
#include "data-if.h"
#include "record-stringizer.h"
#include "latex-export.h"
#include "ui-record-list.h"
#include <gnome.h>


static gboolean write_a4_format(GtkWindow *parent, const gchar *filename, const Bankbook *book);
static gboolean write_us_letter_format(GtkWindow *parent, const gchar *filename, const Bankbook *book);
static void paper_format_option_menu_select(GtkWidget *item, int format);
static void encoding_format_option_menu_select(GtkWidget *item, int format);

enum
{
  PAPER_FORMAT_A4,
  PAPER_FORMAT_US_LETTER
};

struct latex_formats
{
  guint id;
  gchar *name;
  gboolean (*fct)(GtkWindow *parent, const gchar *filename, const Bankbook *book);
};

struct latex_formats paper_formats[] = 
{
  { PAPER_FORMAT_A4,             N_("A4")       ,write_a4_format},
  { PAPER_FORMAT_US_LETTER,      N_("US letter"),write_us_letter_format},
  { 0, NULL }
};

/* Encoding Format (latin1, ...) */

enum
{
  ENCODING_FORMAT_LATIN1,
  ENCODING_FORMAT_7BITS
};

struct latex_formats encoding_formats[] = 
{
  { ENCODING_FORMAT_LATIN1,             N_("latin1")       },
  { ENCODING_FORMAT_7BITS,              N_("7 bits")       },
  { 0, NULL }
};

static struct _latex_context
{
  gint paper_format;
  gint encoding_format;
  gint font_size;
  GtkWidget *paper_format_option_menu;
  GtkWidget *encoding_format_option_menu;
} latex_context;


/******************************************************************************
 * Signals
 */
static void paper_format_option_menu_select(GtkWidget *item, int format)
{
  latex_context.paper_format=format;
}

static void encoding_format_option_menu_select(GtkWidget *item, int format)
{
  latex_context.encoding_format=format;
}

static void font_size_select(GtkWidget *item, int size)
{
  latex_context.font_size=size;
}

/******************************************************************************
 * UI Functions
 */
static gboolean latex_ask_options(GtkWindow *win)
{
  GnomeDialog *dialog;
  GtkWidget *frame, *label, *menuitem, *option_menu;
  GtkWidget *hbox, *vbox;
  GtkMenu *menu;
  struct latex_formats *fmt;
  int i;

  trace ("");

  dialog = GNOME_DIALOG (gnome_dialog_new (_("Latex Export Parameters"),
	GNOME_STOCK_BUTTON_OK, NULL));

  frame = gtk_frame_new (_("Latex Export Parameters"));
  gtk_box_pack_start (GTK_BOX (dialog->vbox), frame, TRUE, TRUE, 0);

  vbox = gtk_vbox_new (FALSE, 10);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  /* 1st line: choose paper format */
  hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

  label=gtk_label_new(_("Paper format "));
  gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

  latex_context.paper_format_option_menu=gtk_option_menu_new();
  menu=(GtkMenu *)gtk_menu_new();
  latex_context.paper_format=paper_formats[0].id;
  for (fmt=paper_formats; fmt->name!=NULL; fmt++)
   {
     menuitem=gtk_menu_item_new_with_label(fmt->name);
     gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
	 GTK_SIGNAL_FUNC(paper_format_option_menu_select),
	 GINT_TO_POINTER(fmt->id));
     gtk_widget_show(menuitem);
     gtk_menu_append(menu, menuitem);
   }
  gtk_option_menu_set_menu(GTK_OPTION_MENU(latex_context.paper_format_option_menu),GTK_WIDGET(menu));
  gtk_box_pack_start(GTK_BOX(hbox),latex_context.paper_format_option_menu,TRUE,TRUE,0);

  /* 2e line: Encoding */
  hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

  label=gtk_label_new(_("Encoding "));
  gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

  latex_context.encoding_format_option_menu=gtk_option_menu_new();
  menu=(GtkMenu *)gtk_menu_new();
  latex_context.encoding_format=encoding_formats[0].id;
  for (fmt=encoding_formats; fmt->name!=NULL; fmt++)
   {
     menuitem=gtk_menu_item_new_with_label(fmt->name);
     gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
	 GTK_SIGNAL_FUNC(encoding_format_option_menu_select),
	 GINT_TO_POINTER(fmt->id));
     gtk_widget_show(menuitem);
     gtk_menu_append (menu, menuitem);
   }
  gtk_option_menu_set_menu(GTK_OPTION_MENU(latex_context.encoding_format_option_menu),GTK_WIDGET(menu));
  gtk_box_pack_start(GTK_BOX(hbox),latex_context.encoding_format_option_menu,TRUE,TRUE,0);

  /* 3e line: Size of the font */
  hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

  label=gtk_label_new(_("Font Size: "));
  gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);

  option_menu=gtk_option_menu_new();
  menu=(GtkMenu *)gtk_menu_new();
  latex_context.font_size=7;
  for (i=7;i<20;i++)
   {
     char tempstring[5];
     sprintf(tempstring,"%d",i);
     menuitem=gtk_menu_item_new_with_label(tempstring);
     gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
	 GTK_SIGNAL_FUNC(font_size_select),
	 GINT_TO_POINTER(i));
     gtk_widget_show(menuitem);
     gtk_menu_append (menu, menuitem);
   }
  gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu),GTK_WIDGET(menu));
  gtk_box_pack_start(GTK_BOX(hbox),option_menu,TRUE,TRUE,0);

  /* View the dialog box */
  gtk_widget_show_all (frame);

  gnome_dialog_set_parent (dialog, win);
  gnome_dialog_run (dialog);

  /* Get preference */

  gnome_dialog_close (dialog);

  return TRUE;
}





static gchar
map_align_to_char (guint align)
{
  switch (align)
  {
  case GTK_JUSTIFY_LEFT:
    return 'l';
  case GTK_JUSTIFY_CENTER:
    return 'c';
  case GTK_JUSTIFY_RIGHT:
    return 'r';
  }
  return 'l';
}

static void
write_text (FILE *file, const gchar *text)
{
  gchar reqesc[] = "&#${}_%\"";	/* characters requiring escape prefix '\' in latex 
  			    	 * probably there are other characters that I have
			    	 * forgotten, so these will have to do for now */
  gchar mathch[] = "><";	/* these characters must be set in math mode */
  const gchar *c;

  trace ("");

  /* latex filter: necessary to translate, for example, "&" to "\&" */

  for (c=text; *c; ++c)
  {
    if (strchr (reqesc, *c) != NULL)
      fprintf (file, "\\%c", *c);
    else if (strchr (mathch, *c) != NULL)
      fprintf (file, "$%c$", *c);
    else
      fprintf (file, "%c", *c);
  }
}

gboolean
latex_export (GtkWindow *parent, const gchar *filename, const Bankbook *book)
{
  FILE *file;
  UI_RecordListColumnInfo *ci;
  guint ncol;

  trace ("");

  if (latex_ask_options(parent)==FALSE)
    return FALSE;

  file = fopen (filename, "wt");
  if (file == NULL)
  {
    dialog_error (parent, _("Unable to create file: %s"), strerror (errno));
    return FALSE;
  }

  ncol = ui_record_list_get_column_info (&ci);

  switch (latex_context.paper_format)
   {
     case PAPER_FORMAT_A4:
       return write_a4_format(parent,filename,book);
     case PAPER_FORMAT_US_LETTER:
       return write_us_letter_format(parent,filename,book);
    default:
      dialog_error (parent, _("Unable to find Paper format"));
      return FALSE;
   }
}


static gboolean write_a4_format(GtkWindow *parent, const gchar *filename, const Bankbook *book)
{
  FILE *file;
  const GList *ac, *rc;
  UI_RecordListColumnInfo *ci;
  guint ncol;

  trace ("");

  file = fopen (filename, "wt");
  if (file == NULL)
  {
    dialog_error (parent, _("Unable to create file: %s"), strerror (errno));
    return FALSE;
  }

  ncol = ui_record_list_get_column_info (&ci);

  fprintf (file, "\\documentclass[a4,%dpt]{article}\n\n"
  		 "\\setlength{\\textheight}{6in}\n"
		 "\\setlength{\\textwidth}{9in}\n"
		 "\\setlength{\\oddsidemargin}{0in}\n"
		 "\\setlength{\\parindent}{0in}\n"
		 "\\setlength{\\parsep}{0.2in}\n"
		 "\\setlength{\\headheight}{0in}\n"
		 "\\setlength{\\headsep}{0in}\n"
		 "\\setlength{\\topmargin}{0in}\n"
		 "\\setlength{\\topskip}{0in}\n"
		 "\\setlength{\\parskip}{1.2ex}\n\n"
                 "\\begin{document}\n\n"
		 "{\\small\\sf ",latex_context.font_size);
  write_text (file, g_basename (filename));
  fprintf (file, "}\\hrule\\vspace*{.1in}\n"
		 "{\\small\\sf\n");

  for (ac=if_bankbook_get_accounts (book); ac; ac=ac->next)
  {
    Account *account = LIST_DEREF (Account, ac);
    guint i;

    fprintf (file, "{\\bf ");
    write_text (file, if_account_get_name (account));
    fprintf (file, "}\n\n" 
    		   "\\begin{tabular}{|");

    for (i=0; i<ncol; ++i)
      fprintf (file, "%c|", map_align_to_char (ci[i].align));

    fprintf (file, "}\n\\hline\n");

    for (i=0; i<ncol; ++i)
    {
      write_text (file, ci[i].label);
      
      if (i == (ncol - 1))
	fprintf (file, " \\\\\n");
      else
	fprintf (file, " & ");
    }
    fprintf (file, "\\hline\\hline\n");

    for (rc=if_account_get_records (account); rc; rc=rc->next)
    {
      RecordInfo rec = {0};
      Record *record = LIST_DEREF (Record, rc);
      gchar *text;

      if_record_get_info (record, 0, &rec);

      for (i=0; i<ncol; ++i)
      {
        text = stringize_record_field (NULL, 0, ci[i].field, &rec);
	write_text (file, text);

	if (i == (ncol - 1))
	  fprintf (file, " \\\\\n");
	else
	  fprintf (file, " & ");

	g_free (text);
      }
    }
    fprintf (file, "\\hline\n\\end{tabular}\n\n");
  }
 
  fprintf (file, "}\\end{document}\n");
  fclose (file);
  return TRUE;
}


static gboolean write_us_letter_format(GtkWindow *parent, const gchar *filename, const Bankbook *book)
{
  FILE *file;
  const GList *ac, *rc;
  UI_RecordListColumnInfo *ci;
  guint ncol;

  trace ("");

  file = fopen (filename, "wt");
  if (file == NULL)
  {
    dialog_error (parent, _("Unable to create file: %s"), strerror (errno));
    return FALSE;
  }

  ncol = ui_record_list_get_column_info (&ci);

  /* at some point, it would be cool to surface a lot of the latex
   * style settings.. especially the font style.. and of course the 
   * paper type.. for now i am just going to go with US Letter */

  fprintf (file, "\\documentclass[letterpaper,%dpt]{article}\n\n"
  		 "\\setlength{\\textheight}{6in}\n"
		 "\\setlength{\\textwidth}{9in}\n"
		 "\\setlength{\\oddsidemargin}{0in}\n"
		 "\\setlength{\\parindent}{0in}\n"
		 "\\setlength{\\parsep}{0.2in}\n"
		 "\\setlength{\\headheight}{0in}\n"
		 "\\setlength{\\headsep}{0in}\n"
		 "\\setlength{\\topmargin}{0in}\n"
		 "\\setlength{\\topskip}{0in}\n"
		 "\\setlength{\\parskip}{1.2ex}\n\n"
                 "\\begin{document}\n\n"
		 "{\\small\\sf ",latex_context.font_size);
  write_text (file, g_basename (filename));
  fprintf (file, "}\\hrule\\vspace*{.1in}\n"
		 "{\\small\\sf\n");

  for (ac=if_bankbook_get_accounts (book); ac; ac=ac->next)
  {
    Account *account = LIST_DEREF (Account, ac);
    guint i;

    fprintf (file, "{\\bf ");
    write_text (file, if_account_get_name (account));
    fprintf (file, "}\n\n" 
    		   "\\begin{tabular}{|");

    for (i=0; i<ncol; ++i)
      fprintf (file, "%c|", map_align_to_char (ci[i].align));

    fprintf (file, "}\n\\hline\n");

    for (i=0; i<ncol; ++i)
    {
      write_text (file, ci[i].label);
      
      if (i == (ncol - 1))
	fprintf (file, " \\\\\n");
      else
	fprintf (file, " & ");
    }
    fprintf (file, "\\hline\\hline\n");

    for (rc=if_account_get_records (account); rc; rc=rc->next)
    {
      RecordInfo rec = {0};
      Record *record = LIST_DEREF (Record, rc);
      gchar *text;

      if_record_get_info (record, 0, &rec);

      for (i=0; i<ncol; ++i)
      {
        text = stringize_record_field (NULL, 0, ci[i].field, &rec);
	write_text (file, text);

	if (i == (ncol - 1))
	  fprintf (file, " \\\\\n");
	else
	  fprintf (file, " & ");

	g_free (text);
      }
    }
    fprintf (file, "\\hline\n\\end{tabular}\n\n");
  }
 
  fprintf (file, "}\\end{document}\n");
  fclose (file);
  return TRUE;
}


