/******************************************************************************\
 gnofin/data-events.c   $Revision: 1.13 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <gtk/gtksignal.h>
#include "data-events.h"
#include "data-if.h"
#include "record.h"

typedef struct {
  guint np, nm, nc, nl;
} CacheLengths;

static void
remember_cache_lengths (const Bankbook *book, CacheLengths *lengths)
{
  lengths->np = book->payee_cache.length;
  lengths->nm = book->memo_cache.length;
  lengths->nc = book->category_cache.length;
  lengths->nl = book->link_cache.length;
}

static void
signal_if_cache_lengths_changed (Bankbook *book, CacheLengths *lengths)
{
  if (lengths->np != book->payee_cache.length)
    gtk_signal_emit_by_name (GTK_OBJECT (book), "payee_list_invalidated");
  if (lengths->nm != book->memo_cache.length)
    gtk_signal_emit_by_name (GTK_OBJECT (book), "memo_list_invalidated");
  if (lengths->nc != book->category_cache.length)
    gtk_signal_emit_by_name (GTK_OBJECT (book), "category_list_invalidated");
//  if (lengths->nl != book->link_cache.length)
    gtk_signal_emit_by_name (GTK_OBJECT (book), "linked_account_list_invalidated");
}

/******************************************************************************
 * History event handlers
 */

/* Account insert/delete */

static void
on_undo_account_insert (Bankbook *book, EventAccountInsert *event)
{
  CacheLengths cl;
  guint index;

  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  remember_cache_lengths (book, &cl);

  index = account_index (event->account);
  account_detach (event->account, TRUE);

  gtk_signal_emit_by_name (GTK_OBJECT (book), "account_deleted", index);
  gtk_signal_emit_by_name (GTK_OBJECT (book), "balance_changed", event->account);

  signal_if_cache_lengths_changed (book, &cl);
}

static void
on_redo_account_insert (Bankbook *book, EventAccountInsert *event)
{
  CacheLengths cl;

  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  remember_cache_lengths (book, &cl);

  account_attach (event->account, book, TRUE);

  gtk_signal_emit_by_name (GTK_OBJECT (book), "account_inserted",
  			   account_index (event->account));
  gtk_signal_emit_by_name (GTK_OBJECT (book), "balance_changed", event->account);

  signal_if_cache_lengths_changed (book, &cl);
}

static void
on_free_account_insert (Bankbook *book, EventAccountInsert *event)
{
  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  account_unref (event->account);
  g_free (event);
}


/* Account update info */

static void
on_undo_account_set_info (Bankbook *book, EventAccountSetInfo *event)
{
  AccountInfo info = {0};
  guint index;

  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  index = account_index (event->account);
  account_detach (event->account, FALSE);

  /* Get account info, but duplicate allocated fields since
   * they will be free'd when we go to set the info */
  account_get_info (event->account, event->mask, &info);
  account_info_copy (&info, &info, event->mask);

  account_set_info (event->account, event->mask, &event->info);
  memcpy (&event->info, &info, sizeof(info));

  account_attach (event->account, book, FALSE);

  gtk_signal_emit_by_name (GTK_OBJECT (book), "account_info_changed",
  			   event->account, event->mask, index);
}

static void
on_free_account_set_info (Bankbook *book, EventAccountSetInfo *event)
{
  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  account_unref (event->account);
  account_info_clear (&event->info, event->mask);
  g_free (event);
}


/* Record insert/delete */

static void
on_undo_record_insert (Bankbook *book, EventRecordInsert *event)
{
  Account *linked_acc = NULL;
  guint index, linked_rec_index = 0;
  CacheLengths cl;

  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  remember_cache_lengths (book, &cl);

  index = record_index (event->record);

  /* The record may be linked, in which case we also have to update
   * the linked account */
  linked_acc = record_linked_acc (event->record);
  if (linked_acc)
    linked_rec_index = record_index (record_linked_rec (event->record));

  record_detach (event->record, TRUE);
  account_recompute_bal (event->account, index);

  gtk_signal_emit_by_name (GTK_OBJECT (book), "record_deleted",
  			   event->account, index);
  gtk_signal_emit_by_name (GTK_OBJECT (book), "balance_changed",
  			   event->account);

  if (linked_acc)
  {
    account_recompute_bal (linked_acc, linked_rec_index);
    
    gtk_signal_emit_by_name (GTK_OBJECT (book), "record_deleted",
			     linked_acc, linked_rec_index);
    gtk_signal_emit_by_name (GTK_OBJECT (book), "balance_changed",
			     linked_acc);
  }
  
  signal_if_cache_lengths_changed (book, &cl);
}

static void
on_redo_record_insert (Bankbook *book, EventRecordInsert *event)
{
  Account *linked_acc;
  guint index;
  CacheLengths cl;

  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  remember_cache_lengths (book, &cl);
  record_attach (event->record, event->account);

  index = record_index (event->record);
  account_recompute_bal (event->account, index);

  gtk_signal_emit_by_name (GTK_OBJECT (book), "record_inserted", event->account, index);
  gtk_signal_emit_by_name (GTK_OBJECT (book), "balance_changed", event->account);

  /* The record may be linked, in which case we also have to update
   * the linked account */
  linked_acc = record_linked_acc (event->record);
  if (linked_acc)
  {
    Record *linked_rec = record_linked_rec (event->record);

    index = record_index (linked_rec);
    account_recompute_bal (linked_acc, index);
    
    gtk_signal_emit_by_name (GTK_OBJECT (book), "record_inserted", linked_acc, index);
    gtk_signal_emit_by_name (GTK_OBJECT (book), "balance_changed", linked_acc);
  }

  signal_if_cache_lengths_changed (book, &cl);
}

static void
on_free_record_insert (Bankbook *book, EventRecordInsert *event)
{
  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  record_unref (event->record);
  account_unref (event->account);
  g_free (event);
}


/* Record update info */

static void
on_undo_record_set_info (Bankbook *book, EventRecordSetInfo *event)
{
  CacheLengths cl;
  RecordInfo info = {0};
  Account *linked_acc;
  Account *linked_acc_old;
  Account *account = event->record->parent;
  guint i1, i2, index = 0;
  guint li1=0, li2=0;
  gboolean breaking_link = FALSE;

  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  remember_cache_lengths (book, &cl);

  i1 = record_index (event->record);

  if (event->breaks_link && !event->record->link_broken)
  {
    record_break_link (event->record);
    breaking_link = TRUE;
  }

  linked_acc_old = record_linked_acc (event->record);
  if (linked_acc_old)
    li1 = record_index (record_linked_rec (event->record));

  record_detach (event->record, TRUE);

  /* It is important to copy the info that is returned by record_get_info
   * because the payee, category, and memo fields might not exist after we call 
   * record_set_info.  Likewise we have to free the info that was previously
   * stored after we call record_set_info to avoid a memory leak. */

  record_get_info (event->record, event->mask, &info);
  record_info_copy (&info, &info, event->mask);

  record_set_info (event->record, event->mask, &event->info);
  record_info_clear (&event->info, event->mask);

  memcpy (&event->info, &info, sizeof (info));

  record_attach (event->record, account);

  i2 = record_index (event->record);

  linked_acc = record_linked_acc (event->record);
  if (linked_acc)
    li2 = record_index (record_linked_rec (event->record));

  index = MIN (i1, i2);
  account_recompute_bal (account, index);
  gtk_signal_emit_by_name (GTK_OBJECT (book), "record_info_changed",
  			   event->record, event->mask, i1);
  gtk_signal_emit_by_name (GTK_OBJECT (book), "balance_changed", account);

  /* Update linked account(s)
   */
  if (linked_acc && (linked_acc == linked_acc_old))
  {
    Record *linked_rec = record_linked_rec (event->record);
    index = MIN (li1, li2);
    account_recompute_bal (linked_acc, index);
    gtk_signal_emit_by_name (GTK_OBJECT (book), "record_info_changed",
			     linked_rec, event->mask, index);
    gtk_signal_emit_by_name (GTK_OBJECT (book), "balance_changed",
			     linked_acc);
  }
  else
  {
    if (linked_acc_old)
    {
      account_recompute_bal (linked_acc_old, li1);
      gtk_signal_emit_by_name (GTK_OBJECT (book), "record_deleted",
			       linked_acc_old, li1);
      gtk_signal_emit_by_name (GTK_OBJECT (book), "balance_changed",
			       linked_acc_old);
    }
    if (linked_acc)
    {
      account_recompute_bal (linked_acc, li2);
      gtk_signal_emit_by_name (GTK_OBJECT (book), "record_inserted",
			       linked_acc, li2);
      gtk_signal_emit_by_name (GTK_OBJECT (book), "balance_changed",
			       linked_acc);
    }
  }

  if (!breaking_link && event->breaks_link && event->record->link_broken)
    record_unbreak_link (event->record);

  signal_if_cache_lengths_changed (book, &cl);
}

static void
on_free_record_set_info (Bankbook *book, EventRecordSetInfo *event)
{
  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  record_unref (event->record);
  record_info_clear (&event->info, event->mask);
  g_free (event);
}


/* Record type insert/delete */

static void
on_undo_record_type_insert (Bankbook *book, EventRecordTypeInsert *event)
{
  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  record_type_detach (event->type);
  gtk_signal_emit_by_name (GTK_OBJECT (book), "type_list_invalidated");
}

static void
on_redo_record_type_insert (Bankbook *book, EventRecordTypeInsert *event)
{
  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  record_type_attach (event->type, book);
  gtk_signal_emit_by_name (GTK_OBJECT (book), "type_list_invalidated");
}

static void
on_free_record_type_insert (Bankbook *book, EventRecordTypeInsert *event)
{
  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  record_type_unref (event->type);
  g_free (event);
}


/* Record type update info */

static void
on_undo_record_type_set_info (Bankbook *book, EventRecordTypeSetInfo *event)
{
  RecordTypeInfo info = {0};
  gboolean resort;
  GList *it;
  
  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  /* We only have to re-sort the type list if the name changes */
  resort = (event->mask & RECORD_TYPE_FIELD_NAME);

  if (resort)
    record_type_detach (event->type);

  record_type_get_info (event->type, event->mask, &info);
  record_type_info_copy (&info, &info, event->mask);  /* eliminate referenced fields */

  record_type_set_info (event->type, event->mask, &event->info);
  record_type_info_clear (&event->info, event->mask);

  memcpy (&event->info, &info, sizeof (info));
  
  if (resort)
    record_type_attach (event->type, book);
  
  gtk_signal_emit_by_name (GTK_OBJECT (book), "type_list_invalidated");

  /* We have to rebuild all of the record lists when a property of a
   * record type changes.  We also have to recompute the balance as
   * iff the sign property change. 
   *
   * NOTE: right now the linked property can't be changed for a record
   *       type that is in use.  if this ever changes, we would need
   *       to also recompute the balances for the affected accounts.
   */
  for (it=book->accounts; it; it=it->next)
  {
    Account *acct = LIST_DEREF (Account, it);

    if (event->mask & RECORD_TYPE_FIELD_SIGN)
      account_recompute_bal (acct, 0);
    gtk_signal_emit_by_name (GTK_OBJECT (book), "record_list_invalidated", acct);
  }
}

static void
on_free_record_type_set_info (Bankbook *book, EventRecordTypeSetInfo *event)
{
  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event);

  record_type_unref (event->type);
  record_type_info_clear (&event->info, event->mask);
  g_free (event);
}


/******************************************************************************
 * Copy functions
 */

static EventAccountInsert *
on_copy_account_insert (EventAccountInsert *event)
{
  EventAccountInsert *copy;

  trace ("");
  g_return_val_if_fail (event, NULL);

  copy = g_new (EventAccountInsert, 1);
  copy->account = event->account;

  account_ref (copy->account);

  return copy;
}

static EventAccountSetInfo *
on_copy_account_set_info (EventAccountSetInfo *event)
{
  EventAccountSetInfo *copy;

  trace ("");
  g_return_val_if_fail (event, NULL);

  copy = g_new (EventAccountSetInfo, 1);
  copy->account = event->account;
  copy->mask = account_info_copy (&copy->info, &event->info, event->mask);

  account_ref (copy->account);

  return copy;
}

static EventRecordInsert *
on_copy_record_insert (EventRecordInsert *event)
{
  EventRecordInsert *copy;

  trace ("");
  g_return_val_if_fail (event, NULL);

  copy = g_new (EventRecordInsert, 1);
  copy->account = event->account;
  copy->record = event->record;

  account_ref (copy->account);
  record_ref (copy->record);

  return copy;
}

static EventRecordSetInfo *
on_copy_record_set_info (EventRecordSetInfo *event)
{
  EventRecordSetInfo *copy;

  trace ("");
  g_return_val_if_fail (event, NULL);

  copy = g_new (EventRecordSetInfo, 1);
  copy->record = event->record;
  copy->mask = record_info_copy (&copy->info, &event->info, event->mask);
  copy->breaks_link = event->breaks_link;

  record_ref (copy->record);

  return copy;
}

static EventRecordTypeInsert *
on_copy_record_type_insert (EventRecordTypeInsert *event)
{
  EventRecordTypeInsert *copy;

  trace ("");
  g_return_val_if_fail (event, NULL);

  copy = g_new (EventRecordTypeInsert, 1);
  copy->type = event->type;

  record_type_ref (copy->type);

  return copy;
}

static EventRecordTypeSetInfo *
on_copy_record_type_set_info (EventRecordTypeSetInfo *event)
{
  EventRecordTypeSetInfo *copy;

  trace ("");
  g_return_val_if_fail (event, NULL);

  copy = g_new (EventRecordTypeSetInfo, 1);
  copy->type = event->type;
  copy->mask = record_type_info_copy (&copy->info, &event->info, event->mask);

  record_type_ref (copy->type);

  return copy;
}


/******************************************************************************
 * Dump functions
 */

static void
on_dump_account_insert (gpointer d, EventAccountInsert *event)
{
  g_print ("    I - %p {ACCOUNT_INSERT,%p}\n", event, event->account);
}

static void
on_dump_account_delete (gpointer d, EventAccountDelete *event)
{
  g_print ("    I - %p {ACCOUNT_DELETE,%p}\n", event, event->account);
}

static void
on_dump_account_set_info (gpointer d, EventAccountSetInfo *event)
{
  g_print ("    I - %p {ACCOUNT_SET_INFO,%p}\n", event, event->account);
}

static void
on_dump_record_insert (gpointer d, EventRecordInsert *event)
{
  g_print ("    I - %p {RECORD_INSERT,%p,%p}\n", event, event->account, event->record);
}

static void
on_dump_record_delete (gpointer d, EventRecordDelete *event)
{
  g_print ("    I - %p {RECORD_DELETE,%p,%p}\n", event, event->account, event->record);
}

static void
on_dump_record_set_info (gpointer d, EventRecordSetInfo *event)
{
  g_print ("    I - %p {RECORD_SET_INFO,%p}\n", event, event->record);
}

static void
on_dump_record_type_insert (gpointer d, EventRecordTypeInsert *event)
{
  g_print ("    I - %p {RECORD_TYPE_INSERT,%p}\n", event, event->type);
}

static void
on_dump_record_type_delete (gpointer d, EventRecordTypeDelete *event)
{
  g_print ("    I - %p {RECORD_TYPE_DELETE,%p}\n", event, event->type);
}

static void
on_dump_record_type_set_info (gpointer d, EventRecordTypeSetInfo *event)
{
  g_print ("    I - %p {RECORD_TYPE_SET_INFO,%p}\n", event, event->type);
}


/******************************************************************************
 * Array of history signals, indexed by EventType
 */

static HistorySignals event_signals[] =
{
  { (HistoryFunc) on_undo_account_insert,
    (HistoryFunc) on_redo_account_insert,
    (HistoryFunc) on_free_account_insert,
    (HistoryFunc) on_dump_account_insert },
  
  { (HistoryFunc) on_redo_account_insert,
    (HistoryFunc) on_undo_account_insert,
    (HistoryFunc) on_free_account_insert,
    (HistoryFunc) on_dump_account_delete },
  
  { (HistoryFunc) on_undo_account_set_info,
    (HistoryFunc) on_undo_account_set_info,
    (HistoryFunc) on_free_account_set_info,
    (HistoryFunc) on_dump_account_set_info },

  { (HistoryFunc) on_undo_record_insert,
    (HistoryFunc) on_redo_record_insert,
    (HistoryFunc) on_free_record_insert,
    (HistoryFunc) on_dump_record_insert },

  { (HistoryFunc) on_redo_record_insert,
    (HistoryFunc) on_undo_record_insert,
    (HistoryFunc) on_free_record_insert,
    (HistoryFunc) on_dump_record_delete },
   
  { (HistoryFunc) on_undo_record_set_info,
    (HistoryFunc) on_undo_record_set_info,
    (HistoryFunc) on_free_record_set_info,
    (HistoryFunc) on_dump_record_set_info },

  { (HistoryFunc) on_undo_record_type_insert,
    (HistoryFunc) on_redo_record_type_insert,
    (HistoryFunc) on_free_record_type_insert,
    (HistoryFunc) on_dump_record_type_insert },
  
  { (HistoryFunc) on_redo_record_type_insert,
    (HistoryFunc) on_undo_record_type_insert,
    (HistoryFunc) on_free_record_type_insert,
    (HistoryFunc) on_dump_record_type_delete },

  { (HistoryFunc) on_undo_record_type_set_info,
    (HistoryFunc) on_undo_record_type_set_info,
    (HistoryFunc) on_free_record_type_set_info,
    (HistoryFunc) on_dump_record_type_set_info },
};


/******************************************************************************
 * Array of event copy functions, indexed by EventType
 */

typedef gpointer (* EventCopyFunc) (gpointer);

static EventCopyFunc event_copy_funcs[] =
{
  (EventCopyFunc) on_copy_account_insert,
  (EventCopyFunc) on_copy_account_insert,
  (EventCopyFunc) on_copy_account_set_info,
  (EventCopyFunc) on_copy_record_insert,
  (EventCopyFunc) on_copy_record_insert,
  (EventCopyFunc) on_copy_record_set_info,
  (EventCopyFunc) on_copy_record_type_insert,
  (EventCopyFunc) on_copy_record_type_insert,
  (EventCopyFunc) on_copy_record_type_set_info,
};


/******************************************************************************
 * Interface
 */

void
realize_data_event (Bankbook *book, EventType type, gpointer event_data)
{
  HistorySignals *signals;

  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (event_data);

  signals = &event_signals[type];

  if (book->remember_events)
  {
    EventCopyFunc copy;

    /* We copy the event data instead of just passing along the given
     * pointer, because the memory associated with event_data may not
     * persist after this function returns. */
    copy = event_copy_funcs[type];
    event_data = copy (event_data);
    history_remember (&book->history, signals, event_data);
  }

  /* Finally we call the redo handler to "make the event happen" */
  signals->redo_item (book, event_data);

  if (book->remember_events && (book->history.batch_mode == 0))
  {
    book->dirty++;
  
    gtk_signal_emit_by_name (GTK_OBJECT (book), "history_changed");
    gtk_signal_emit_by_name (GTK_OBJECT (book), "dirty_flag_changed");
  }
}

// vim: ts=8 sw=2
