/******************************************************************************\
 gnofin/dialog-edit-record-types.c   $Revision: 1.13 $
 Copyright (C) 2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <gtk/gtk.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-uidefs.h>
#include <libgnome/gnome-config.h>
#include "dialogs.h"
#include "config-saver.h"
#include "data-if.h"
#include "gnofin-defaults.h"
#include <string.h>


typedef struct
{
  GtkCList  *clist;

  GtkWidget *ok_button;

  GtkWidget *insert_button;
  GtkWidget *remove_button;

  GtkWidget *name_entry;
  GtkWidget *desc_entry;
  GtkWidget *numbered_button;
  GtkWidget *linked_button;
  GtkWidget *sign_menu;

  GSList *removed_list;

  guint mod_count;
  guint selection;
  guint selection_changed : 1;

} DialogInfo;


static gchar *clist_titles[] =
{
  N_("Name"),
  N_("Description"),
  N_("Numbered"),
  N_("Linked"),
  N_("Sign"),
  N_("Usage"),
};
#define NUMCOLS  (sizeof_array (clist_titles))

static const struct
{
  guint id;
  const gchar *name;
}
signs[] =
{
  { RECORD_TYPE_SIGN_ANY, N_("Unrestricted")   },
  { RECORD_TYPE_SIGN_POS, N_("Positive") },
  { RECORD_TYPE_SIGN_NEG, N_("Negative") }
};


/******************************************************************************
 * Configuration
 */

typedef struct {
  gint *col_widths;
} Config;

static gint clist_default_col_widths[] = {
  40, 130, 60, 40, 80, 60
};

#define CAT  "Dialog_EditRecordTypes"
#define KEY  "/" PACKAGE "/" CAT "/"

static void
load_config (Config *config)
{
  gchar path[256];
  guint i;

  trace ("");
  g_return_if_fail (config);

  config->col_widths = g_new0 (gint, NUMCOLS);

  for (i=0; i < NUMCOLS; ++i)
  {
    g_snprintf (path, sizeof path, KEY "col_widths[%d]=%d", i,
    		clist_default_col_widths[i]);
    config->col_widths[i] = gnome_config_get_int (path);
  }
}

static void
save_config (Config *config)
{
  gchar path[256];
  guint i;

  trace ("");
  g_return_if_fail (config);

  for (i=0; i < NUMCOLS; ++i)
  {
    g_snprintf (path, sizeof path, KEY "col_widths[%d]", i);
    gnome_config_set_int (path, config->col_widths[i]);
  }
}

static Config *
get_config (void)
{
  static Config config = {0};
  static gboolean init = FALSE;

  if (!init)
  {
    init = TRUE;
    load_config (&config);
    config_saver_register (CAT, (ConfigSaveFunc) save_config, &config);
  }
  return &config;
}


/******************************************************************************
 * Helper functions
 */

static const gchar *
stringize_sign (guint id)
{
  guint i;

  trace ("");

  for (i=0; i < sizeof_array (signs); ++i)
  {
    if (signs[i].id == id)
      return _(signs[i].name);
  }
  return "";
}

static guint
lookup_sign (const gchar *name)
{
  guint i;

  trace ("");

  for (i=0; i < sizeof_array (signs); ++i)
  {
    if (strcmp (name, _(signs[i].name)) == 0)
      return i;
  }
  return (guint) -1;
}

static GSList *
gather_results (DialogInfo *info)
{
  RecordTypeContext *c;
  GSList *list = NULL, *names = NULL;
  guint i;

  trace ("");
  g_return_val_if_fail (info, NULL);

  for (i=0; i < info->clist->rows; ++i)
  {
    const gchar *name;

    c = (RecordTypeContext *) gtk_clist_get_row_data (info->clist, i);

    /* If the name of the record type was changed, it will have a name
     * entry, otherwise we get the name from the record type object. */
    if (c->info.name)
      name = c->info.name;
    else
      name = if_record_type_get_name (c->type);

    /* Verify that record type name is not going to be defined twice */
    if (g_slist_find_custom (names, (gpointer) name, (GCompareFunc) strcmp) != NULL)
    {
      dialog_error (GTK_WINDOW (gtk_widget_get_toplevel (info->ok_button)),
      		  _("One or more transaction types have the same name.  You must\n"
		    "correct the problem before your changes can be accepted."));
      g_slist_free (list);
      list = NULL;
      goto end;
    }

    names = g_slist_prepend (names, (gpointer) name);
    list = g_slist_prepend (list, c);
  }
  if (list)
    list = g_slist_reverse (list);
  /* Append list of removed record type contexts */
  list = g_slist_concat (list, info->removed_list);
end:
  g_slist_free (names);
  return list;
}


/******************************************************************************
 * Signal handlers
 */

static void
on_select_row (GtkCList *clist, gint row, gint col, GdkEvent *e, DialogInfo *info)
{
  gchar *text;
  guint i;

  trace ("");

  info->selection = row;
  info->selection_changed = 1;

  gtk_clist_get_text (clist, row, 0, &text);
  gtk_entry_set_text (GTK_ENTRY (info->name_entry), text);

  gtk_clist_get_text (clist, row, 1, &text);
  gtk_entry_set_text (GTK_ENTRY (info->desc_entry), text);

  gtk_clist_get_text (clist, row, 2, &text);
  i = (!strncmp(_("x"),text,1));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (info->numbered_button), i);

  gtk_clist_get_text (clist, row, 3, &text);
  i = (!strncmp(_("x"),text,1));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (info->linked_button), i);

  gtk_clist_get_text (clist, row, 4, &text);
  i = lookup_sign (text);
  gtk_option_menu_set_history (GTK_OPTION_MENU (info->sign_menu), i);

  gtk_clist_get_text (clist, row, 5, &text);
  i = (text[0] == '0');
  gtk_widget_set_sensitive (info->linked_button, i != 0);
  gtk_widget_set_sensitive (info->remove_button, i != 0);

  info->selection_changed = 0;
}

static void
on_unselect_row (GtkCList *clist, guint row, guint col, GdkEvent *e, DialogInfo *info)
{
  trace ("");
  info->selection = (guint) -1;
}

static void
on_insert_clicked (GtkButton *button, DialogInfo *info)
{
  RecordTypeContext *c;
  gchar *text[NUMCOLS];

  trace ("");

  c = g_new0 (RecordTypeContext, 1);
  c->info.name = g_strdup (_("(UNK)"));
  c->info.description = g_strdup (_("(Unknown)"));

  text[0] = _("(UNK)");
  text[1] = _("(Unknown)");
  text[2] = "";
  text[3] = "";
  text[4] = (gchar *) stringize_sign (RECORD_TYPE_SIGN_ANY);
  text[5] = "0";

  gtk_clist_append (info->clist, text);
  gtk_clist_set_row_data (info->clist, info->clist->rows - 1, c);
  gtk_clist_select_row (info->clist, info->clist->rows - 1, 0);
  gtk_clist_moveto (info->clist, info->clist->rows - 1, 0, 1.0, 0.0);

  info->mod_count++;
  gtk_widget_set_sensitive (info->ok_button, info->mod_count);
}

static void
on_remove_clicked (GtkButton *button, DialogInfo *info)
{
  RecordTypeContext *c;
  guint i;

  trace ("");

  i = info->selection;
  c = (RecordTypeContext *) gtk_clist_get_row_data (info->clist, i);

  /* We don't need to hold onto any changes made to this record type */
  if (c->mask)
    if_record_type_info_clear (&c->info, c->mask);

  /* If the record type context is not associated with an existing
   * record type than all we have to do is free the context structure.
   * Otherwise, we have to flag the record type as needing to be removed. 
   */
  if (c->type)
  {
    /* Flag this record type for removal */
    c->remove = 1;
    info->removed_list = g_slist_prepend (info->removed_list, c);
  }
  else
    g_free (c);
  gtk_clist_remove (info->clist, i);

  info->mod_count++;
  gtk_widget_set_sensitive (info->ok_button, info->mod_count);
}

static void
on_save_as_defaults_clicked (GtkButton *button, DialogInfo *info)
{
  GSList *types = NULL;

  trace ("");

  /* Generate record type list */
  {
    RecordTypeContext *c;
    RecordTypeInfo *typ;
    guint i;
    for (i=0; i < info->clist->rows; ++i)
    {
      c = (RecordTypeContext *) gtk_clist_get_row_data (info->clist, i);
      if (c->remove == 0)
      {
	typ = g_new0 (RecordTypeInfo, 1);
	if (c->type)
	{
	  if_record_type_get_info (c->type, 0, typ);
	  if_record_type_info_copy (typ, typ, 0); /* want to realloc the strings */
	}
	if (c->mask)
	  if_record_type_info_copy (typ, &c->info, c->mask);
	types = g_slist_prepend (types, typ);
      }
    }
    types = g_slist_reverse (types);
  }

  set_default_record_types (types);
}

static void
on_name_changed (GtkEntry *entry, DialogInfo *info)
{
  trace ("");
  if (!info->selection_changed)
  {
    RecordTypeContext *c = gtk_clist_get_row_data (info->clist, info->selection);
    RecordTypeInfo typ = {0};
    gchar *text = gtk_entry_get_text (entry);

    if (c->type)
      if_record_type_get_info (c->type, RECORD_TYPE_FIELD_NAME, &typ);

    if (c->mask & RECORD_TYPE_FIELD_NAME)
    {
      g_free (c->info.name);
      c->mask &= ~RECORD_TYPE_FIELD_NAME;

      info->mod_count--;
    }

    if (c->type && (strcmp (typ.name, text) == 0))
      c->info.name = typ.name;
    else
    {
      c->info.name = g_strdup (text);
      c->mask |= RECORD_TYPE_FIELD_NAME;

      info->mod_count++;
    }

    gtk_clist_set_text (info->clist, info->selection, 0, c->info.name);
    gtk_widget_set_sensitive (info->ok_button, info->mod_count);
  }
}

static void
on_desc_changed (GtkEntry *entry, DialogInfo *info)
{
  trace ("");
  if (!info->selection_changed)
  {
    RecordTypeContext *c = gtk_clist_get_row_data (info->clist, info->selection);
    RecordTypeInfo typ = {0};
    gchar *text = gtk_entry_get_text (entry);

    if (c->type)
      if_record_type_get_info (c->type, RECORD_TYPE_FIELD_DESCRIPTION, &typ);

    if (c->mask & RECORD_TYPE_FIELD_DESCRIPTION)
    {
      g_free (c->info.description);
      c->mask &= ~RECORD_TYPE_FIELD_DESCRIPTION;

      info->mod_count--;
    }

    if (c->type && (strcmp (typ.description, text) == 0))
      c->info.description = typ.description;
    else
    {
      c->info.description = g_strdup (text);
      c->mask |= RECORD_TYPE_FIELD_DESCRIPTION;

      info->mod_count++;
    }

    gtk_clist_set_text (info->clist, info->selection, 1, c->info.description);
    gtk_widget_set_sensitive (info->ok_button, info->mod_count);
  }
}

static void
on_numbered_toggled (GtkToggleButton *button, DialogInfo *info)
{
  trace ("");
  if (!info->selection_changed)
  {
    RecordTypeContext *c = gtk_clist_get_row_data (info->clist, info->selection);
    RecordTypeInfo typ = {0};
    gboolean value = gtk_toggle_button_get_active (button);

    if (c->type)
      if_record_type_get_info (c->type, RECORD_TYPE_FIELD_NUMBERED, &typ);

    if (c->mask & RECORD_TYPE_FIELD_NUMBERED)
    {
      c->mask &= ~RECORD_TYPE_FIELD_NUMBERED;
      info->mod_count--;
    }

    c->info.numbered = value;

    if ((!c->type) || (typ.numbered != value))
    {
      c->mask |= RECORD_TYPE_FIELD_NUMBERED;
      info->mod_count++;
    }

    gtk_clist_set_text (info->clist, info->selection, 2, value ? "x" : "");
    gtk_widget_set_sensitive (info->ok_button, info->mod_count);
  }
}

static void
on_linked_toggled (GtkToggleButton *button, DialogInfo *info)
{
  trace ("");
  if (!info->selection_changed)
  {
    RecordTypeContext *c = gtk_clist_get_row_data (info->clist, info->selection);
    RecordTypeInfo typ = {0};
    gboolean value = gtk_toggle_button_get_active (button);

    if (c->type)
      if_record_type_get_info (c->type, RECORD_TYPE_FIELD_LINKED, &typ);

    if (c->mask & RECORD_TYPE_FIELD_LINKED)
    {
      c->mask &= ~RECORD_TYPE_FIELD_LINKED;
      info->mod_count--;
    }

    c->info.linked = value;

    if ((!c->type) || (typ.linked != value))
    {
      c->mask |= RECORD_TYPE_FIELD_LINKED;
      info->mod_count++;
    }

    gtk_clist_set_text (info->clist, info->selection, 3, value ? "x" : "");
    gtk_widget_set_sensitive (info->ok_button, info->mod_count);
  }
}

static void
on_sign_activated (GtkMenuItem *item, DialogInfo *info)
{
  trace ("");
  if (!info->selection_changed)
  {
    RecordTypeContext *c = gtk_clist_get_row_data (info->clist, info->selection);
    RecordTypeInfo typ = {0};
    gchar *text;
    GtkWidget *label;
    guint sign;

    label = GTK_BIN (info->sign_menu)->child;
    gtk_label_get (GTK_LABEL (label), &text);

    /* Update the context structure */
    if (c->type)
      if_record_type_get_info (c->type, RECORD_TYPE_FIELD_SIGN, &typ);

    if (c->mask & RECORD_TYPE_FIELD_SIGN)
    {
      c->mask &= ~RECORD_TYPE_FIELD_SIGN;
      info->mod_count--;
    }
    
    sign = lookup_sign (text);
    c->info.sign = sign;

    if ((!c->type) || (typ.sign != sign))
    {
      c->mask |= RECORD_TYPE_FIELD_SIGN;
      info->mod_count++;
    }

    gtk_clist_set_text (info->clist, info->selection, 4, text);
    gtk_widget_set_sensitive (info->ok_button, info->mod_count);
  }
}


/******************************************************************************
 * Initialization
 */

static GtkWidget *
make_dialog (DialogInfo *info)
{
  Config *config = get_config ();
  GtkWidget *top;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *table;
  GtkWidget *scrolled_win;
  GtkWidget *clist;
  GtkWidget *button;
  GtkWidget *frame;
  GtkWidget *entry;
  GtkWidget *label;
  GtkWidget *optmenu;
  GtkWidget *pixmap;
  GtkTooltips *tips;
  gint i;

  tips = gtk_tooltips_new ();

  frame = gtk_frame_new (_("Define the current transaction types"));
  top = frame;

  vbox = gtk_vbox_new (FALSE, 10);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), GNOME_PAD);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  /* Create the list
   */
  hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

  scrolled_win = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_win),
                                  GTK_POLICY_AUTOMATIC,
                                  GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize (scrolled_win, -1, 120);
  gtk_box_pack_start (GTK_BOX (hbox), scrolled_win, TRUE, TRUE, 0);

# ifdef ENABLE_NLS
  for (i=0; i < NUMCOLS; ++i)
    clist_titles[i] = _(clist_titles[i]);
# endif

  clist = gtk_clist_new_with_titles (sizeof_array (clist_titles), clist_titles);
  gtk_clist_set_selection_mode (GTK_CLIST (clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_reorderable (GTK_CLIST (clist), FALSE);
  gtk_clist_column_titles_passive (GTK_CLIST (clist));
  gtk_container_add (GTK_CONTAINER (scrolled_win), clist);
  
  for (i=0; i < NUMCOLS; ++i)
    gtk_clist_set_column_width (GTK_CLIST (clist), i, config->col_widths[i]);

  gtk_signal_connect (GTK_OBJECT (clist), "select_row",
                      GTK_SIGNAL_FUNC (on_select_row), info);
  gtk_signal_connect (GTK_OBJECT (clist), "unselect_row",
                      GTK_SIGNAL_FUNC (on_unselect_row), info);
  
  info->clist = GTK_CLIST (clist);

  /* Create the button panel
   */
  hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  pixmap = gnome_stock_pixmap_widget (NULL, GNOME_STOCK_PIXMAP_ADD);
  button = gnome_pixmap_button (pixmap, _("Insert"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (on_insert_clicked), info);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  info->insert_button = button;

  gtk_tooltips_set_tip (tips, button,
  		      _("Insert a new transaction type"), NULL);

  pixmap = gnome_stock_pixmap_widget (NULL, GNOME_STOCK_PIXMAP_REMOVE);
  button = gnome_pixmap_button (pixmap, _("Remove"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (on_remove_clicked), info);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  info->remove_button = button;

  gtk_tooltips_set_tip (tips, button,
  		      _("Remove selected transaction type (provided usage count is zero)"), NULL);

  pixmap = gnome_stock_pixmap_widget (NULL, GNOME_STOCK_MENU_SAVE_AS);
  button = gnome_pixmap_button (pixmap, _("Save as defaults"));
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (on_save_as_defaults_clicked), info);
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);

  gtk_tooltips_set_tip (tips, button,
                      _("Gnofin data files that you create in the future will "
		        "use the transaction types you have defined here instead "
			"of the default ones"), NULL);

  /* Create the parameters editor table
   */
  frame = gtk_frame_new (_("Parameters"));
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (2, 5, TRUE);
  gtk_table_set_row_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
  gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD_SMALL);
  gtk_container_set_border_width (GTK_CONTAINER (table), GNOME_PAD_SMALL);
  gtk_container_add (GTK_CONTAINER (frame), table);

  entry = gtk_entry_new ();
  gtk_widget_set_usize (entry, 90, -1);
  gtk_table_attach_defaults (GTK_TABLE (table), entry, 0, 1, 0, 1);
  info->name_entry = entry;

  gtk_signal_connect (GTK_OBJECT (entry), "changed",
  		      GTK_SIGNAL_FUNC (on_name_changed), info);

  gtk_tooltips_set_tip (tips, entry,
  		      _("Specify a short name for the transaction type"), NULL);

  entry = gtk_entry_new ();
  gtk_table_attach_defaults (GTK_TABLE (table), entry, 1, 5, 0, 1);
  info->desc_entry = entry;

  gtk_signal_connect (GTK_OBJECT (entry), "changed",
  		      GTK_SIGNAL_FUNC (on_desc_changed), info);

  gtk_tooltips_set_tip (tips, entry,
  		      _("Specify a descriptive name for the transaction type (optional)"), NULL);

  button = gtk_check_button_new_with_label (_("Numbered"));
  gtk_table_attach_defaults (GTK_TABLE (table), button, 0, 1, 1, 2);
  info->numbered_button = button;

  gtk_signal_connect (GTK_OBJECT (button), "toggled",
  		      GTK_SIGNAL_FUNC (on_numbered_toggled), info);

  gtk_tooltips_set_tip (tips, button,
  		      _("Toggle numbered property"), NULL);

  button = gtk_check_button_new_with_label (_("Linked"));
  gtk_table_attach_defaults (GTK_TABLE (table), button, 1, 2, 1, 2);
  info->linked_button = button;

  gtk_signal_connect (GTK_OBJECT (button), "toggled",
  		      GTK_SIGNAL_FUNC (on_linked_toggled), info);

  gtk_tooltips_set_tip (tips, button,
  		      _("Toggle linked property (changes to a linked record "
		        "are automatically reflected in the peer record)"), NULL);

  hbox = gtk_hbox_new (FALSE, GNOME_PAD_SMALL);
  gtk_table_attach_defaults (GTK_TABLE (table), hbox, 2, 5, 1, 2);

  label = gtk_label_new (_("Sign:"));
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

  optmenu = gtk_option_menu_new ();
  gtk_box_pack_start (GTK_BOX (hbox), optmenu, TRUE, TRUE, 0);
  info->sign_menu = optmenu;

  gtk_tooltips_set_tip (tips, optmenu,
  		      _("Impose a restriction on the \"sign\" of "
		        "the entered amount"), NULL);

  return top;
}

static void
init_dialog (DialogInfo *info, GSList *node)
{
  GtkMenu *menu;
  RecordTypeContext *c;
  guint i;

  g_assert (NUMCOLS == 6);

  /* Add items to sign menu
   */
  menu = GTK_MENU (gtk_menu_new ());
  for (i=0; i < sizeof_array (signs); ++i)
  {
    GtkWidget *item = gtk_menu_item_new_with_label (_(signs[i].name));
    gtk_signal_connect (GTK_OBJECT (item), "activate",
    			GTK_SIGNAL_FUNC (on_sign_activated), info);
    gtk_widget_show (item);
    gtk_menu_append (menu, item);
  } 
  gtk_option_menu_set_menu (GTK_OPTION_MENU (info->sign_menu),
			    GTK_WIDGET (menu));
    
  /* Add items to clist
   */
  gtk_clist_freeze (info->clist);
  for (i=0; node; node=node->next, ++i)
  {
    gchar *text[NUMCOLS];

    c = LIST_DEREF (RecordTypeContext, node);

    text[0] = c->info.name;
    text[1] = c->info.description;
    text[2] = c->info.numbered ? _("x") : "";
    text[3] = c->info.linked ? _("x") : "";
    text[4] = (gchar *) stringize_sign (c->info.sign);
    text[5] = g_strdup_printf ("%d", c->info.usage_count);

    gtk_clist_insert (info->clist, i, text);
    gtk_clist_set_row_data (info->clist, i, c);

    g_free (text[5]);

  }
  gtk_clist_thaw (info->clist);
}

gboolean
dialog_edit_record_types (GtkWindow *parent, GSList **cts)
{
  GnomeDialog *dialog;
  DialogInfo info = {0};
  GtkWidget *widget;
  gboolean ret = FALSE;

  trace ("");

  dialog = GNOME_DIALOG (gnome_dialog_new (_("Define Transaction Types"),
  					   GNOME_STOCK_BUTTON_OK,
					   GNOME_STOCK_BUTTON_CANCEL,
					   NULL));
  gnome_dialog_set_parent (dialog, parent);
  gtk_window_set_policy (GTK_WINDOW (dialog), TRUE, TRUE, FALSE);

  info.ok_button = (GtkWidget *) dialog->buttons->data;
  gtk_widget_set_sensitive (info.ok_button, FALSE);

  widget = make_dialog (&info);
  gtk_widget_show_all (widget);

  init_dialog (&info, *cts);
  gtk_box_pack_start (GTK_BOX (dialog->vbox), widget, TRUE, TRUE, 0);

top:
  if (gnome_dialog_run (dialog) == 0)
  {
    /* Check to see if anything actually changed */
    if (info.mod_count > 0)
    {
      GSList *list = gather_results (&info);

      /* The list may be NULL if gather_results found an inconsistency
       * in the record types */
      if (list != NULL)
      {
	ret = TRUE;

	/* Free the old list */
	g_slist_free (*cts);

	/* Write the new one */
	*cts = list;
      }
      else
        goto top;
    }
  }
  else
  {
    /* We only have to worry about freeing the removed_list if
     * we are going to return FALSE */
    g_slist_free (info.removed_list);
  }

  gnome_dialog_close (dialog);
  return ret;
}
