/******************************************************************************\
 gnofin/merging.c   $Revision: 1.8 $
 Copyright (C) 2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include "common.h"
#include <ctype.h>
#include "merging.h"
#include "data-if.h"


/******************************************************************************
 * Helpers
 */

static gboolean
names_are_similar (const gchar *a, const gchar *b)
{
  trace ("");
  while (a[0] && b[0])
  {
    /* CHK and CHK1 are similar */
    if (a[0] == b[0] && (!a[1] || isdigit(a[1])) && (!b[1] || isdigit(b[1])))
      return TRUE;
    a++;
    b++;
  }
  return FALSE;
}

static inline gboolean
types_match (const RecordTypeInfo *a, const RecordTypeInfo *b)
{
  trace ("");
  return names_are_similar (a->name, b->name)
      && !strcmp (a->description, b->description)
      && (a->linked == b->linked)
      && (a->numbered == b->numbered)
      && (a->sign == b->sign);
}

static RecordType *
find_matching_type_by_info (const GList *types, const RecordTypeInfo *typ1)
{
  RecordTypeInfo typ2 = {0};
  RecordType *type;

  trace ("");

  for (; types; types=types->next)
  {
    type = LIST_DEREF (RecordType, types);
    if_record_type_get_info (type, 0, &typ2);
    if (types_match (typ1, &typ2))
      return type;
  }
  return NULL;
}

static inline RecordType *
find_matching_type (const GList *types, const RecordType *type)
{
  RecordTypeInfo typ1 = {0};

  trace ("");

  if_record_type_get_info (type, 0, &typ1);
  return find_matching_type_by_info (types, &typ1);
}

typedef gpointer (* ObjectNameLookupFunc) (const Bankbook *, const gchar *);

static gchar *
get_unique_object_name (const Bankbook *book, const gchar *name,
			ObjectNameLookupFunc lookup_func, gboolean force)
{
  trace ("");
  g_return_val_if_fail (book, NULL);
  g_return_val_if_fail (name, NULL);
  g_return_val_if_fail (lookup_func, NULL);

  if (lookup_func (book, name) != NULL)
  {
    gint counter = 1;
    gchar *buf = NULL;

    if (!force)
      return NULL;

    do {
      g_free (buf);
      buf = g_strdup_printf ("%s%d", name, counter);
      counter++;
    } while (lookup_func (book, buf) != NULL);

    return buf;
  }
  return g_strdup (name);
}

/******************************************************************************
 * Interface
 */

Record *
merge_record_into_account (Account *dest, const Record *src, gboolean force)
{
  RecordInfo rec = {0};
  RecordType *type;
  RecordTypeInfo typ = {0};
  Bankbook *book;
  GList *types;

  trace ("");
  g_return_val_if_fail (dest, NULL);
  g_return_val_if_fail (src, NULL);

  book = if_account_get_parent (dest);
  types = (GList *) if_bankbook_get_record_types (book);

  if_record_get_info (src, 0, &rec);
  type = find_matching_type (types, rec.type);
  if (type)
    rec.type = type;
  else
  {
    if (!(type = merge_record_type_into_bankbook (book, rec.type, force)))
      return NULL;

    /* Change the name of the type previously associated with this record
     * so that future merges from the same bankbook will find a matching
     * record type in the destination bankbook.
     */
    if_record_type_get_info (type, RECORD_TYPE_FIELD_NAME, &typ);
    if (strcmp (typ.name, if_record_type_get_name (rec.type)))
      if_record_type_set_info (rec.type, RECORD_TYPE_FIELD_NAME, &typ);
    rec.type = type;
  }
  return if_account_insert_record (dest, &rec);
}

gboolean
merge_account_into_account (Account *dest, const Account *src, gboolean force)
{
  Bankbook *book;
  const GList *node;
  gboolean ret = TRUE;

  trace ("");
  g_return_val_if_fail (dest, FALSE);
  g_return_val_if_fail (src, FALSE);

  book = if_account_get_parent (dest);
  if_bankbook_enter_batch_mode (book);

  node = if_account_get_records (src);
  for (; node; node=node->next)
    ret = ret && merge_record_into_account (dest, LIST_DEREF (Record, node), force);

  if_bankbook_leave_batch_mode (book);
  return ret;
}

Account *
merge_account_into_bankbook (Bankbook *dest, const Account *src, gboolean force)
{
  Account *account;
  AccountInfo acc = {0};
  gchar *name;

  trace ("");
  g_return_val_if_fail (dest, NULL);
  g_return_val_if_fail (src, NULL);

  /* Make sure we have a valid account name */
  name = get_unique_object_name (dest, if_account_get_name (src),
	  (ObjectNameLookupFunc) if_bankbook_get_account_by_name, force);

  if_bankbook_enter_batch_mode (dest);

  /* Create the new account */
  if_account_get_info (src, 0, &acc);
  acc.name = name;
  account = if_bankbook_insert_account (dest, &acc);

  if (account)
  {
    /* Now just merge source account into new account */
    merge_account_into_account (account, src, force);

    // FIXME: not sure what to do if there is an error here
  }

  if_bankbook_leave_batch_mode (dest);

  g_free (name);
  return account;
}

gboolean
merge_bankbook_into_bankbook (Bankbook *dest, const Bankbook *src, gboolean force)
{
  const GList *node;
  gboolean ret = TRUE;

  trace ("");
  g_return_val_if_fail (dest, FALSE);
  g_return_val_if_fail (src, FALSE);

  if_bankbook_enter_batch_mode (dest);

  node = if_bankbook_get_accounts (src);
  for (; node; node=node->next)
    ret = ret && merge_account_into_bankbook (dest, LIST_DEREF (Account, node), force);

  if_bankbook_leave_batch_mode (dest);
  return ret;
}

RecordType *
merge_record_type_into_bankbook (Bankbook *dest, const RecordType *src, gboolean force)
{
  RecordType *type;

  trace ("");
  g_return_val_if_fail (dest, NULL);
  g_return_val_if_fail (src, NULL);

  if_bankbook_enter_batch_mode (dest);

  /* Check to see if record type already exists */
  type = find_matching_type (if_bankbook_get_record_types (dest), src); 
  if (type == NULL)
  {
    RecordTypeInfo typ = {0};
    gchar *name;

    /* Figure out what name the new record should have */
    name = get_unique_object_name (dest, if_record_type_get_name (src),
	    (ObjectNameLookupFunc) if_bankbook_get_record_type_by_name, force);

    /* Create the new record type */
    if_record_type_get_info (src, 0, &typ);
    typ.name = name;
    type = if_bankbook_insert_record_type (dest, &typ);

    g_free (name);
  }

  if_bankbook_leave_batch_mode (dest);
  return type;
}
