/******************************************************************************\
 gnofin/money.c   $Revision: 1.12 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <libgnome/gnome-config.h>
#include <stdlib.h>
#include <locale.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include "money.h"
#include "notification-list.h"
#include "config-saver.h"
#include "numeric-parser.h"

#define MON_BUFSIZ 64


/* FIXME: what about using the strfmon function in the standard library?
 *        unfortunately i can't find any documentation on strfmon and there
 *        also doesn't appear to be a corresponding strpmon! */


/******************************************************************************
 * Format change notification
 */

NotificationList money_format_change_listeners = {0};


/******************************************************************************
 * Configuration
 *
 * By default we try to use the monetary format defined by the current
 * locale.  If these values aren't defined by the current locale then
 * we assume en_US defaults.
 */

static struct lconv *lconv = NULL;

static inline char
get_locale_thousands_sep ()
{
  trace ("");

  if (lconv == NULL)
    lconv = localeconv ();

  if (lconv->mon_thousands_sep[0] == '\0')
    return ',';
  else
    return lconv->mon_thousands_sep[0];
}

static inline char
get_locale_decimal_sep ()
{
  trace ("");

  if (lconv == NULL)
    lconv = localeconv ();

  if (lconv->mon_decimal_point[0] == '\0')
    return '.';
  else
    return lconv->mon_decimal_point[0];
}

/******************************************************************************/

typedef struct {
  gchar decisep;
  gchar thousep;
} Config;

#define CAT "Money"
#define KEY "/" PACKAGE "/" CAT "/"

static void
load_config (Config *config)
{
  gchar path[256];
  gchar *s;

  trace ("");

  g_snprintf (path, sizeof path, KEY "decimal_sep=%c",
  	      get_locale_decimal_sep ());
  s = gnome_config_get_string (path);
  config->decisep = s[0];

  g_snprintf (path, sizeof path, KEY "thousands_sep=%c",
  	      get_locale_thousands_sep ());
  s = gnome_config_get_string (path);
  config->thousep = s[0];
}

static void
save_config (Config *config)
{
  gchar b[2];

  trace ("");

  b[1] = '\0';

  b[0] = config->decisep;
  gnome_config_set_string (KEY "decimal_sep", b);

  b[0] = config->thousep;
  gnome_config_set_string (KEY "thousands_sep", b);
}

static Config *
get_config (void)
{
  static gboolean init = FALSE;
  static Config config = {0};

  if (!init)
  {
    init = TRUE;
    load_config (&config);
    config_saver_register (CAT, (ConfigSaveFunc) save_config, &config);
  }
  return &config;
}


/******************************************************************************
 * Helpers
 */

static gboolean
money_parse_num (const gchar *buf, money_t *n, int *sign)
{
  const gchar *p;

  trace ("");
  g_return_val_if_fail (n, FALSE);
  
  if (sign)
    *sign = +1;

  for (p=buf; *p; ++p)
  {
    if (isdigit (*p))
    {
#ifdef G_HAVE_GINT64
      return int64_parse (p, (gint64 *) n);
#else
      return int_parse (p, (gint *) n);
#endif
    }
    else if (*p == '-')
    {
      if (sign)
      	*sign = -1;
      if (*(p+1) == '\0')
      {
        *n = 0;
	return TRUE;
      }
    }
  }
  return FALSE;
}

static inline void
money_stringize_raw_1 (gchar *buf, int maxlen, money_t amount)
{
  trace ("");

#ifdef G_HAVE_GINT64
  /* Note: usage of %qd limits portability */
  g_snprintf (buf, maxlen, "%qd", amount);
#else
  g_snprintf (buf, maxlen, "%ld", amount);
#endif
}


/******************************************************************************
 * Interface
 */

gchar *
money_stringize (gchar *buf, int maxlen, money_t amount)
{
  Config *config = get_config ();

  trace ("");

  return money_stringize_f (buf, maxlen, amount,
			    config->decisep, config->thousep);
}

gchar *
money_stringize_f (gchar *buf, int maxlen, money_t amount,
		   gchar decisep, gchar thousep)
{
  gchar tmp[MON_BUFSIZ];
  gboolean neg = (amount < 0);
  gint i, k=0, n;

  trace ("");

  if (!buf)
  {
    buf = g_new0 (gchar, MON_BUFSIZ);
    maxlen = MON_BUFSIZ;
  }
  else
    memset (buf, 0, maxlen);

  if (neg)
  {
    buf[k] = '-';
    k++;
  }

  money_stringize_raw_1 (tmp, sizeof tmp, money_major (amount));
  n = strlen (tmp);

  for (i=0; i<n; ++i)
  {
    if ((thousep != '\0') && (i>0) && (((n - i) % 3) == 0))
    {
      buf[k] = thousep;
      k++;
    }
    buf[k] = tmp[i];
    k++;
  }

  if (decisep != '\0')
    g_snprintf (buf + k, maxlen - k - (neg ? 0 : 1),
		"%c%02ld", decisep, (long) money_minor (amount));

  return buf;
}

gchar *
money_stringize_raw (gchar *buf, int maxlen, money_t amount)
{
  trace ("");

  if (!buf)
  {
    buf = g_new0 (gchar, MON_BUFSIZ);
    maxlen = MON_BUFSIZ;
  }

  money_stringize_raw_1 (buf, maxlen, amount);
  return buf;
}

gboolean
money_parse (const gchar *inbuf, money_t *amount)
{
  Config *config = get_config ();

  trace ("");

  return money_parse_f (inbuf, amount, config->decisep, config->thousep);
}

gboolean
money_parse_f (const gchar *inbuf, money_t *amount, gchar decisep, gchar thousep)
{
  money_t major, minor;
  gchar *buf, *p;
  int sign = +1;

  trace ("");
  g_return_val_if_fail (inbuf, FALSE);
  g_return_val_if_fail (amount, FALSE);

  if (inbuf[0] == '\0')
  {
    trace ("no chars to convert!");
    *amount = 0;
    return TRUE;
  }

  buf = strdup (inbuf);

  /* Skip over thousands-place separator */
  if (thousep != '\0')
  {
    gchar *ptr = (gchar *)buf;
    gchar *savptr = (gchar *)buf;
    gchar ch;

    while (*buf)
    {
      ch = *buf++;
      if (ch == thousep)
        continue;
      *ptr++ = ch;
    }
    *ptr = '\0';
    buf = savptr;
  }

  if (decisep && (p = strchr (buf, decisep)))
  {
    int n;

    p[0] = '\0';
    if (strlen (buf) == 0)
      major = 0;
    else if (! money_parse_num (buf, &major, &sign))
      goto fail;

    n = strlen (p+1);
    if (n == 0)
      minor = 0;
    else
    {
      if (n > 2)
        *((p+1)+2) = '\0';  // FIXME: we should really be rounding here!!
      if (! money_parse_num (p+1, &minor, NULL))
	goto fail;
    }

    if (p[2] == '\0')
      minor *= 10;
  }
  else
  {
    minor = 0;
    if (! money_parse_num (buf, &major, &sign))
      goto fail;
  }

  *amount = ((money_t)(sign < 0 ? -1 : 1)) * ((money_abs (major) * 100) + minor);

  g_free (buf);
  return TRUE;

fail:
  g_free (buf);
  return FALSE;
}

gboolean
money_parse_raw (const gchar *inbuf, money_t *amount)
{
  gboolean res;
  int sign = 1;

  trace ("");
  g_return_val_if_fail (inbuf, FALSE);
  g_return_val_if_fail (amount, FALSE);

  res = money_parse_num (inbuf, amount, &sign);
  *amount = *amount * (money_t)(sign < 0 ? -1 : 1);
  return res;
}

gint
money_compare (money_t a, money_t b)
{
  a = money_sign (a);
  b = money_sign (b);

  /* Group positive amounts before negative amounts */

  if ((a > 0) && (b < 0))
    return -1;
  else if ((b > 0) && (a < 0))
    return +1;
  else
    return 0;
}

gchar
money_get_decimal_sep (void)
{
  trace ("");
  return get_config ()->decisep;
}

gchar
money_get_thousands_sep (void)
{
  trace ("");
  return get_config ()->thousep;
}

void
money_set_format (gchar dsep, gchar tsep)
{
  Config *config = get_config ();

  trace ("");

  if ((config->decisep != dsep) || (config->thousep != tsep))
  {
    config->decisep = dsep;
    config->thousep = tsep;

    notification_list_notify (&money_format_change_listeners);
  }
}

// vim: ts=8 sw=2
