/** -*- Mode: C++; tab-width: 4 -*-
 * vim: sw=4 ts=4:
 *
 * Gnome Apt package cache handling class
 *
 * 	(C) 1998 Havoc Pennington <hp@pobox.com>
 * 	    2003, 2004 Filip Van Raemdonck <mechanix@debian.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * 	$Id$
 *
 **/

#ifndef GNOME_APT_CACHE_H
#define GNOME_APT_CACHE_H

#include <apt-pkg/depcache.h>
#include <apt-pkg/pkgrecords.h>
#include <set>

using namespace std;

class pkgPolicy;
class GAProgress;

class GAptCache : public pkgDepCache {
public:
	GAptCache (pkgCache*, pkgPolicy*);
	~GAptCache (void);

	struct aptitude_state {
		bool new_package;
		pkgCache::State::PkgSelectedState selection_state;
	};

	void forget_new (void);
	void set_states (set<string>&);

	inline void mark_new (const PkgIterator& Pkg, bool newstate = false) {
		package_states[Pkg->ID].new_package = newstate;
	}

	/* Ideally, this would be dependent on the cache type. */
	typedef enum {
		StatusNowBroken,
		StatusNewInstall, StatusUpgrade, StatusDowngrade, StatusDelete, StatusInstall,
		StatusNotInstalled,
		StatusKept, StatusHeld,
		StatusOutdated, StatusUptodate,
		StatusTypeEnd
	} PkgStatusType;

	PkgStatusType pkgStatus (pkgCache::PkgIterator&);
	static const char* statusText (PkgStatusType);

	const char* priorityString (pkgCache::PkgIterator&, pkgCache::State::VerPriority*);
	pkgRecords::Parser* GAptCache::pkgParser (pkgCache::PkgIterator&);

private:
	aptitude_state* package_states;
	pkgRecords* records;
};

class GAptCacheFile
{  
  FileFd *file_;
  MMap *map_;
	GAptCache* cache_;

public:
	GAptCache* cache() { return cache_; }

	bool Open (OpProgress*);
  // Mark packages that can be upgraded without removals
  //  or installs (apt-get upgrade)
  void MarkUpgrades();

  // Fix broken packages
  void Fix();

	GAptCacheFile (void) : file_ (0), map_ (0), cache_ (0) {}
	~GAptCacheFile (void) { clear(); }

  class CacheView {
  public:
    virtual void set_cache(GAptCache* cache) = 0;
  };

  void add_view(CacheView* v) {
#ifdef GNOME_ENABLE_DEBUG
    set<CacheView*>::iterator i = views_.find(v);
    if (i != views_.end()) g_warning("Adding view twice!");
#endif
    views_.insert(v);
    v->set_cache(cache_);
  }
  
  void remove_view(CacheView* v) {
    set<CacheView*>::iterator i = views_.find(v);
    if (i != views_.end())
      views_.erase(i);
#ifdef GNOME_ENABLE_DEBUG
    else g_warning("View not in list!");
#endif
    v->set_cache(0);
  }

private:
	set<CacheView*> views_;
	set<string> pkg_names;

	void clear (bool = false);
};

GAptCacheFile* gnome_apt_cache_file_init (OpProgress*);

#endif
