//  
//  Copyright (C) 2009 GNOME Do
// 
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
// 
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
// 
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
// 

using System;
using System.Collections.Generic;

using Gtk;

using Do.Platform;

namespace WeatherDocklet
{
	[System.ComponentModel.Category("File")]
	[System.ComponentModel.ToolboxItem(true)]
	public partial class WeatherConfiguration : Bin
	{
		TreeStore locationTreeStore = new TreeStore (typeof (string));
		TreeStore searchTreeStore = new TreeStore (typeof (string), typeof (string));

		public WeatherConfiguration ()
		{			
			Build ();
			
			locationTreeView.Model = locationTreeStore;
			locationTreeView.Selection.Changed += OnLocationSelectionChanged;
			locationTreeView.AppendColumn ("Locations", new CellRendererText (), "text", 0);
			UpdateLocations ();
			
			searchTreeView.Model = searchTreeStore;
			searchTreeView.Selection.Changed += OnSearchSelectionChanged;
			searchTreeView.AppendColumn ("Search Results", new CellRendererText (), "text", 0);
			
			timeout.Value = Preferences.Timeout;
			metric.Active = Preferences.Metric;
			aboutLabel.Text = WeatherController.Weather.About;
			
			int selected = 0;
			foreach (AbstractWeatherSource aws in WeatherController.Service.WeatherSources)
			{
				source.AppendText (aws.Name);
				if (aws.Name.Equals(Preferences.Source))
					source.Active = selected;
				selected++;
			}
		}
		
		void UpdateLocations ()
		{
			locationTreeStore.Clear ();
			for (int i = 0; i < Preferences.Location.Length; i++)
				locationTreeStore.AppendValues (Preferences.Location [i]);
		}
		
		protected virtual void OnTimeoutFocusOut (object o, Gtk.FocusOutEventArgs args)
		{
			Preferences.Timeout = (uint) timeout.ValueAsInt;
		}
		
		protected virtual void OnMetricFocusOut (object o, Gtk.FocusOutEventArgs args)
		{
			Preferences.Metric = metric.Active;
		}
		
		protected virtual void OnSourceChanged (object sender, System.EventArgs e)
		{
			Preferences.Source = source.ActiveText;
			aboutLabel.Text = WeatherController.Weather.About;
			searchTreeStore.Clear ();
		}
		
		protected virtual void OnSearchClicked (object sender, System.EventArgs e)
		{
			searchTreeStore.Clear ();
			searchTreeStore.AppendValues ("Currently searching...", "");
			
			Services.Application.RunOnThread (() => {
				List<string> vals = new List<string> ();
	
				foreach (string s in WeatherController.Weather.SearchLocation (location.Text))
					vals.Add (s);
				
				Services.Application.RunOnMainThread (() => {
					searchTreeStore.Clear ();
					
					if (vals.Count > 0)
						for (int i = 0; i < vals.Count; i += 2)
							searchTreeStore.AppendValues (vals [i], vals [i + 1]);
					else
						searchTreeStore.AppendValues ("No results found", "");
				});
			});
		}

		protected virtual void OnLocationSelectionChanged (object o, System.EventArgs args)
		{
			TreeIter iter;
			TreeModel model;
			
			if (((TreeSelection)o).GetSelected (out model, out iter))
			{
				string selected = (string) model.GetValue (iter, 0);
				int index = FindLocationIndex (selected);
				
				location.Text = selected;
				btnMoveUp.Sensitive = index > 0;
				btnMoveDown.Sensitive = index < Preferences.Location.Length - 1;
				btnRemove.Sensitive = true;
			} else {
				btnMoveUp.Sensitive = false;
				btnMoveDown.Sensitive = false;
				btnRemove.Sensitive = false;
			}
		}
		
		protected virtual void OnSearchSelectionChanged (object o, System.EventArgs args)
		{
			TreeIter iter;
			TreeModel model;
			
			if (((TreeSelection)o).GetSelected (out model, out iter))
			{
				string val = (string) model.GetValue (iter, 1);
				location.Text = val;
			}
		}

		protected virtual void OnAddClicked (object sender, System.EventArgs e)
		{
			if (location.Text.Trim ().Length == 0)
				return;
			if (FindLocationIndex (location.Text) != -1)
				return;
			
			string[] locations = new string [Preferences.Location.Length + 1];
			Array.Copy (Preferences.Location, 0, locations, 0, Preferences.Location.Length);
			locations [Preferences.Location.Length] = location.Text.Trim ();
			
			Preferences.Location = locations;
			UpdateLocations ();
		}

		int FindLocationIndex (string location)
		{
			for (int i = 0; i < Preferences.Location.Length; i++)
				if (Preferences.Location [i].Equals (location))
					return i;
			
			return -1;
		}

		protected virtual void OnRemoveClicked (object sender, System.EventArgs e)
		{
			TreeIter iter;
			TreeModel model;
			
			if (locationTreeView.Selection.GetSelected (out model, out iter))
			{
				string removedLocation = (string) model.GetValue (iter, 0);
				int index = FindLocationIndex (removedLocation);
				
				string[] locations = new string [Preferences.Location.Length - 1];
				Array.Copy (Preferences.Location, 0, locations, 0, index);
				Array.Copy (Preferences.Location, index + 1, locations, index, Preferences.Location.Length - index - 1);
				
				Preferences.Location = locations;
				
				if (removedLocation.Equals (WeatherController.CurrentLocation))
					WeatherController.PreviousLocation ();
				
				UpdateLocations ();
			}
		}

		protected virtual void OnMoveDownClicked (object sender, System.EventArgs e)
		{
			TreeIter iter;
			TreeModel model;
			
			if (locationTreeView.Selection.GetSelected (out model, out iter))
			{
				int index = FindLocationIndex ((string) model.GetValue (iter, 0));
				
				string[] locations = Preferences.Location;
				string temp = locations [index];
				locations [index] = locations [index + 1];
				locations [index + 1] = temp;
				
				Preferences.Location = locations;
				UpdateLocations ();
			}
		}

		protected virtual void OnMoveUpClicked (object sender, System.EventArgs e)
		{
			TreeIter iter;
			TreeModel model;
			
			if (locationTreeView.Selection.GetSelected (out model, out iter))
			{
				int index = FindLocationIndex ((string) model.GetValue (iter, 0));
				
				string[] locations = Preferences.Location;
				string temp = locations [index];
				locations [index] = locations [index - 1];
				locations [index - 1] = temp;
				
				Preferences.Location = locations;
				UpdateLocations ();
			}
		}
	}
}
