/* gnome-gpg.c - wrapper around GNUPG to use GNOME Keyring

   Copyright (C) 2004 Colin Walters <walters@verbum.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <libgnomeui/libgnomeui.h>
#include <gnome-keyring.h>

#include "gnome-gpg.h"

/* Temporary */
#define _(STR) (STR)

static int status_fd, passphrase_fd;
static GString *status_commands;
static GHashTable *userid_hints;
static GHashTable *bad_passphrases;
static char *passphrase_keyid = NULL;

static void
error_dialog (const char *fmt, ...)
{
  GtkWidget *dialog;
  char buffer[1025];
  va_list args;

  va_start (args, fmt);
  
  g_vsnprintf (buffer, 1024, fmt, args);
  fprintf (stderr, "%s\n", buffer);

  va_end (args);
  
  dialog = gtk_message_dialog_new (NULL,
				   0,
				   GTK_MESSAGE_ERROR,
				   GTK_BUTTONS_OK,
				   "%s",
				   buffer);
  
  zenity_util_show_dialog (dialog);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_object_destroy (GTK_OBJECT (dialog));
}

/* send the passphrase for the given key ID to the passphrase FD */
static void
handle_get_passphrase (const char *keyid, const char *userid,
		       gboolean bad_passphrase)
{
  GnomeKeyringAttributeList *attributes;
  GnomeKeyringResult keyring_result;
  GList *found_items = NULL;
  char *keyring = NULL, *passphrase;
  gboolean keyring_available, save_password = FALSE;
  guint32 item_id;

  keyring_available = gnome_keyring_is_available();

  if (keyring_available) {
    attributes = gnome_keyring_attribute_list_new ();
    gnome_keyring_attribute_list_append_string (
        attributes, GNOME_GPG_KEYID_ATTR, keyid);
    keyring_result = gnome_keyring_find_items_sync (
        GNOME_KEYRING_ITEM_GENERIC_SECRET, attributes, &found_items);
    gnome_keyring_attribute_list_free (attributes);

    if (keyring_result != GNOME_KEYRING_RESULT_OK
	&& keyring_result != GNOME_KEYRING_RESULT_DENIED) {
      error_dialog (_("Couldn't search keyring (code %d)"),
		    (int) keyring_result);
      close (passphrase_fd);
      gtk_main_quit ();
      return;
    }
  }

  if (found_items == NULL) {
    char *message;
    const char *short_keyid;
    int keyid_len;
    GnomePasswordDialogRemember remember;
    GnomePasswordDialog *dialog;

    keyid_len = strlen(keyid);
    if (keyid_len > 8)
      short_keyid = &keyid[keyid_len - 8];
    else
      short_keyid = keyid;

    if (userid) {
      message = g_strdup_printf (
          _("Please enter the passphrase for \"%s\" (%s)"),
          userid, short_keyid);
    } else {
      message = g_strdup_printf(
          _("Please enter the passphrase for key %s"), short_keyid);
    }
    /* if the previous attempt gave a bad passphrase, try again. */
    if (bad_passphrase) {
      char *tmp = g_strconcat(_("Invalid passphrase; please try again.\n\n"),
			      message, NULL);
      g_free(message);
      message = tmp;
    }
    dialog = GNOME_PASSWORD_DIALOG (gnome_password_dialog_new (
        _("GNU Privacy Guard passphrase"),
	message,
	NULL,
	NULL,
	TRUE));
    g_free(message);
    gnome_password_dialog_set_show_userpass_buttons (dialog, FALSE);
    gnome_password_dialog_set_show_username (dialog, FALSE);
    gnome_password_dialog_set_show_domain (dialog, FALSE);
    gnome_password_dialog_set_show_remember (dialog, keyring_available);

    zenity_util_show_dialog (GTK_WIDGET (dialog));
    if (!gnome_password_dialog_run_and_block (dialog)) {
      gtk_object_destroy (GTK_OBJECT (dialog));
      close (passphrase_fd);
      gtk_main_quit ();
      return;
    }
      
    passphrase = gnome_password_dialog_get_password (dialog);
    remember = gnome_password_dialog_get_remember (dialog);
    gtk_object_destroy (GTK_OBJECT (dialog));

    if (remember == GNOME_PASSWORD_DIALOG_REMEMBER_SESSION) {
      save_password = TRUE;
      keyring = "session";
    } else if (remember == GNOME_PASSWORD_DIALOG_REMEMBER_FOREVER) {
      save_password = TRUE;
      keyring = NULL;
    } else {
      save_password = FALSE;
    }
  } else {
    GnomeKeyringFound *found;

    found = found_items->data;
    passphrase = g_strdup(found->secret);
    save_password = FALSE;
  }
  gnome_keyring_found_list_free(found_items);
    
  write (passphrase_fd, passphrase, strlen (passphrase));
  write (passphrase_fd, "\n", 1);

  if (keyring_available && save_password) {
    gchar *nice_name = g_strdup_printf (_("GNU Privacy Guard passphrase for key ID: %s"),
					keyid);
    attributes = gnome_keyring_attribute_list_new ();
    gnome_keyring_attribute_list_append_string (attributes,
						GNOME_GPG_KEYID_ATTR,
						keyid);
    if (userid) {
      gnome_keyring_attribute_list_append_string (attributes,
						  GNOME_GPG_USERID_ATTR,
						  userid);
    }
    keyring_result = gnome_keyring_item_create_sync (keyring,
						     GNOME_KEYRING_ITEM_GENERIC_SECRET,
						     nice_name,
						     attributes,
						     passphrase,
						     TRUE,
						     &item_id);
    gnome_keyring_attribute_list_free (attributes);
    g_free (nice_name);
    if (keyring_result != GNOME_KEYRING_RESULT_OK) {
      error_dialog (_("Couldn't store passphrase in keyring (code %d)"),
		    (int) keyring_result);
    }
  }
  gnome_keyring_free_password (passphrase);
}

/* a bad passphrase was entered in, so clear any secrets stored in the
 * keyring */
static void
handle_bad_passphrase (const char *keyid)
{
  GnomeKeyringAttributeList *attributes;
  GnomeKeyringResult keyring_result;
  GList *found_items = NULL, *tmp;

  /* if the keyring is not available, we can't invalidate the item */
  if (!gnome_keyring_is_available())
    return;
  
  attributes = gnome_keyring_attribute_list_new ();
  gnome_keyring_attribute_list_append_string (attributes,
					      GNOME_GPG_KEYID_ATTR,
					      keyid);
  keyring_result = gnome_keyring_find_items_sync (
      GNOME_KEYRING_ITEM_GENERIC_SECRET,
      attributes, &found_items);
  gnome_keyring_attribute_list_free (attributes);

  if (keyring_result != GNOME_KEYRING_RESULT_OK
      && keyring_result != GNOME_KEYRING_RESULT_DENIED) {
    error_dialog (_("Couldn't search keyring (code %d)"),
		  (int) keyring_result);
    close(passphrase_fd);
    gtk_main_quit();
    return;
  }

  for (tmp = found_items; tmp != NULL; tmp = tmp->next) {
    GnomeKeyringFound *found = tmp->data;

    gnome_keyring_item_delete_sync (found->keyring, found->item_id);
  }

  gnome_keyring_found_list_free(found_items);
}

static void
handle_command (const char *command)
{
  char *space, *keyid;

  /* ignore non-command line */
  if (strncmp(command, "[GNUPG:] ", 9) != 0)
    return;
  command += 9;

  if (!strncmp(command, "USERID_HINT ", 12)) {
    char *userid;
    command += 12;

    space = strchr(command, ' ');
    if (space != NULL) {
      keyid = g_strndup(command, space - command);
      userid = g_strdup(space + 1);
      g_hash_table_insert(userid_hints, keyid, userid);
    } 
  } else if (!strncmp(command, "NEED_PASSPHRASE ", 16)) {
    command += 16;
    
    g_free(passphrase_keyid);
    space = strchr(command, ' ');
    if (space != NULL) {
      passphrase_keyid = g_strndup(command, space - command);
    } else {
      passphrase_keyid = g_strdup(command);
    }
  } else if (!strcmp(command, "GET_HIDDEN passphrase.enter")) {
    char *userid;
    gboolean bad_passphrase;

    /* get the userid hint for this key, if any */
    userid = g_hash_table_lookup(userid_hints, passphrase_keyid);
    /* was a bad passphrase entered? */
    bad_passphrase = GPOINTER_TO_INT(g_hash_table_lookup(
        bad_passphrases, passphrase_keyid));
    g_hash_table_remove(bad_passphrases, passphrase_keyid);
    handle_get_passphrase(passphrase_keyid, userid, bad_passphrase);
  } else if (!strncmp(command, "GET", 3)) {
    fprintf(stderr, "unhandled GET issued: %s\n", command);
    close(passphrase_fd);
    gtk_main_quit();
  } else if (!strncmp(command, "BAD_PASSPHRASE ", 15)) {
    command += 15;

    space = strchr(command, ' ');
    if (space != NULL) {
      keyid = g_strndup(command, space - command);
    } else {
      keyid = g_strdup(command);
    }
    g_hash_table_insert (bad_passphrases, g_strdup (keyid),
			 GINT_TO_POINTER(TRUE));
    handle_bad_passphrase(keyid);
    g_free(keyid);
  } else {
    /* ignore */
  }
}

static gboolean
read_status (GIOChannel *channel, GIOCondition condition, gpointer data)
{
  char buf[4096];
  gssize n_read;
  char *newline;

  if ((condition & G_IO_IN) != 0) {
    n_read = read(status_fd, buf, 4096);
    if (n_read > 0) {
      g_string_append_len(status_commands, buf, n_read);
      /* do we have a full commands in the buffer? */
      while ((newline = strchr(status_commands->str, '\n')) != NULL) {
	char *command;
	GString *rest;

	command = g_strndup(status_commands->str,
			    newline - status_commands->str);
	handle_command(command);
	g_free(command);

	/* remove the command from the queue*/
	rest = g_string_new_len(newline + 1,
		  status_commands->len - (newline + 1 - status_commands->str));
	g_string_free(status_commands, TRUE);
	status_commands = rest;
      }
    }
  }

  /* if the status FD is closed, quit */
  if ((condition & G_IO_HUP) != 0) {
    gtk_main_quit ();
    return FALSE;
  }

  return TRUE;
}

int
main (int argc, char **argv)
{
  pid_t pid;
  int status_fds[2], passphrase_fds[2];
  int i;
  GIOChannel *channel;
  char *short_argv[2];

  if (pipe (passphrase_fds) < 0) {
    g_error ("Couldn't pipe: %s", g_strerror (errno));
    exit (1);
  }

  if (pipe (status_fds) < 0) {
    g_error ("Couldn't pipe: %s", g_strerror (errno));
    exit (1);
  }

  if ((pid = fork()) < 0) {
    g_error ("Couldn't fork: %s", g_strerror (errno));
    exit (1);
  } else if (pid != 0) {
    /* execute gpg in the parent process */
    GPtrArray *args;
    char *fd_str;

    /* child */

    close (passphrase_fds[1]);
    close (status_fds[0]);

    args = g_ptr_array_new ();

    g_ptr_array_add (args, "gpg");
    g_ptr_array_add (args, "--no-tty");
    g_ptr_array_add (args, "--status-fd");
    fd_str = g_strdup_printf ("%d", status_fds[1]);
    g_ptr_array_add (args, fd_str);
    g_ptr_array_add (args, "--command-fd");
    fd_str = g_strdup_printf ("%d", passphrase_fds[0]);
    g_ptr_array_add (args, fd_str);
    for (i = 1; i < argc; i++) {
      g_ptr_array_add (args, argv[i]);
    }
    g_ptr_array_add (args, NULL);

    if (execv (GPG, (char **) g_ptr_array_free (args, FALSE)) < 0) {
      fprintf (stderr, "Couldn't exec gpg: %s", g_strerror (errno));
    }
    return 1;
  }

  /* child process: handle passphrase entry */
  close (passphrase_fds[0]);
  passphrase_fd = passphrase_fds[1];
  close (status_fds[1]);
  status_fd = status_fds[0];

  /* since the arguments passed to gnome-gpg are actually intended for
   * gpg, we present a short argv to gnome_program_init(), containing
   * only the program name. */
  short_argv[0] = argv[0];
  short_argv[1] = NULL;

  gtk_set_locale ();
  gnome_program_init (PACKAGE_NAME, PACKAGE_VERSION,
		      LIBGNOMEUI_MODULE, 1, short_argv,
		      GNOME_PARAM_HUMAN_READABLE_NAME,
		      _("GNU Privacy Guard"),
		      GNOME_CLIENT_PARAM_SM_CONNECT, FALSE,
		      NULL);

  /* hints about particular keys*/
  userid_hints = g_hash_table_new_full (g_str_hash, g_str_equal,
					(GDestroyNotify)g_free,
					(GDestroyNotify)g_free);

  /* keys for which a bad passphrase has been given */
  bad_passphrases = g_hash_table_new_full (g_str_hash, g_str_equal,
					   (GDestroyNotify)g_free,
					   (GDestroyNotify)NULL);

  /* buffer of status commands from gpg */
  status_commands = g_string_new (NULL);

  /* watch the status fd */
  channel = g_io_channel_unix_new (status_fd);
  g_io_add_watch (channel, G_IO_IN | G_IO_HUP, read_status, NULL);

  gtk_main ();
  return 0;
}
