/* gnome-rdp
 * Copyright (C) 2005-2006 Balazs Varkonyi
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
using System;
using Gtk;
using Glade;
using System.Collections;
using Mono.Unix;

namespace GnomeRDP
{
	public class OptionsDialog : GladeDialog
	{
		[Widget]Entry eSessionName;
		[Widget]ComboBox cbProtocol;
		[Widget]Entry eComputer;
		[Widget]Entry eUser;
		[Widget]Entry ePassword;
		[Widget]Entry eDomain;
		[Widget]CheckButton cbSavePassword;
		[Widget]ComboBox cbSrvType;
		[Widget]Image iColors;
		[Widget]ComboBox cbSoundRedirection;
		[Widget]ComboBox cbColorDepth;
		[Widget]ComboBox cbScreenResolution;
		[Widget]ComboBox cbConnectionType;
		[Widget]ComboBox cbWindowMode;
		[Widget]ComboBox cbTerminalSize;
		[Widget]HBox hboxCustomResolution;
		[Widget]SpinButton sbWidth;
		[Widget]SpinButton sbHeight;
		[Widget]SpinButton sbCompressionLevel;
		[Widget]SpinButton sbImageQuality;
		[Widget]ScrolledWindow scrolledwindow2;
		[Widget]ScrolledWindow scrolledwindow3;
		[Widget]Button btOk;
		[Widget]IconView ivTabs;
		[Widget]Notebook nbTabs;
		private TreeStore store = new TreeStore(typeof(string));
		private ListStore storeIconList;
		private TreeView treeKeyboard;
		private TreeIter iter;
		private string[] keyboardList = {"ar", "da", "de", "de-ch", "en-gb", "en-us", "es", "et", +
			"fi", "fo", "fr", "fr-be", "fr-ca", "fr-ch", "hr", "hu", "is", "it", "ja", "ko", "lt", +
			"lv", "mk", "nl", "nl-be", "no", "pl", "pt", "pt-br", "ru", "sl", "sv", "th", "tr"};
		private Configuration config;
		private SessionTreeStore categoryTreeStore;
		private SessionTreeView categoryTreeView;
		private TreeIter iterSelected;
				
		public OptionsDialog (Configuration Config) : base ("dlgOptions")
		{
			TreeIter selIter;
			this.config = Config;
			
			ivTabs.SelectionChanged += new EventHandler(OnSelectionChanged);
			ivTabs.Columns = 1;
			categoryTreeStore = new SessionTreeStore();
			categoryTreeView = new SessionTreeView(categoryTreeStore, true);
			scrolledwindow3.Add(categoryTreeView);
			scrolledwindow3.ShowAll();
			this.RefreshCategoryList();
			categoryTreeView.ExpandAll();
			categoryTreeView.Selection.SelectIter(iterSelected);
			treeKeyboard = new TreeView();
			treeKeyboard.HeadersVisible = false;
			treeKeyboard.AppendColumn("KEY", new CellRendererText(), "text", 0);
			selIter = iter;
			foreach (string keys in keyboardList)
			{
				iter = store.AppendValues(keys);
				if (keys == config.KeyboardLang)
				{
					selIter = iter;
				}
			}
			treeKeyboard.Model = store;
			treeKeyboard.SetCursor(treeKeyboard.Model.GetPath(selIter), treeKeyboard.GetColumn(0), false); 	

			scrolledwindow2.Add(treeKeyboard);
			scrolledwindow2.ShowAll();
			eSessionName.Text = config.SessionName;
			cbProtocol.Active = config.Protocol;
			eComputer.Text = config.Computer;
			eUser.Text = config.User;
			ePassword.Text = config.Password;
			eDomain.Text = config.Domain;
			if (ePassword.Text.Length > 0)
			{
				cbSavePassword.Active = true;
			}
			cbSrvType.Active = config.SrvType;
			iColors.Pixbuf = Gdk.Pixbuf.LoadFromResource("colors_1.png");
			cbColorDepth.Changed += new EventHandler(OnColorDepthChanged);
			cbScreenResolution.Changed += new EventHandler(OnScreenResolutionChanged);
			cbProtocol.Changed += OnProtocolChanged;
			cbSoundRedirection.Active = config.SoundRedirection;
			cbColorDepth.Active = config.ColorDepth;
			if ((config.ScreenResolutionX == 640) && (config.ScreenResolutionY == 480))
			{
				cbScreenResolution.Active = 0;
				sbWidth.Value = 640;
				sbHeight.Value = 480;						
			}
			else if ((config.ScreenResolutionX == 800) && (config.ScreenResolutionY == 600))
			{
				cbScreenResolution.Active = 1;
				sbWidth.Value = 800;
				sbHeight.Value = 600;
			}
			else if ((config.ScreenResolutionX == 0) && (config.ScreenResolutionY == 0))
			{
				cbScreenResolution.Active = 2;
				sbWidth.Value = 0;
				sbHeight.Value = 0;
			}
			else
			{
				cbScreenResolution.Active = 3;
				sbWidth.Value = config.ScreenResolutionX;
				sbHeight.Value = config.ScreenResolutionY;			
			}
			cbConnectionType.Active = config.ConnectionType;
			cbWindowMode.Active = config.WindowMode;
			cbTerminalSize.Active = config.TerminalSize;
			sbCompressionLevel.Value = config.CompressionLevel;
			sbImageQuality.Value = config.ImageQuality;
			eSessionName.Changed += OnSessionNameChanged;
			eComputer.Changed += OnSessionNameChanged;
			categoryTreeView.WidgetEvent += OnSessionNameChanged;
			if (eSessionName.Text.Length > 0)
			{
				btOk.Sensitive = true;
			}
			else
			{
				btOk.Sensitive = false;
			}
			this.RefreshControlsSensitivity();
		}
		
		private ListStore FillIconView()
		{
			storeIconList = new ListStore(
				typeof(string),
				typeof(int), 
				typeof(Gdk.Pixbuf));
			storeIconList.Clear();
			storeIconList.AppendValues("General", 0, Gdk.Pixbuf.LoadFromResource("general.png"));
			switch (cbProtocol.Active)
			{
				case 0:
					storeIconList.AppendValues("RDP", 1, Gdk.Pixbuf.LoadFromResource("rdp.png"));
					break;
				case 1:
					storeIconList.AppendValues("VNC", 2, Gdk.Pixbuf.LoadFromResource("vnc.png"));
					break;
				case 2:
					storeIconList.AppendValues("SSH", 3, Gdk.Pixbuf.LoadFromResource("ssh.png"));
					break;
			}
			return storeIconList;
		}
		
		private void RefreshCategoryList()
		{
			Configuration category = new Configuration();
			
			categoryTreeStore.Clear();
			category.ParentId = 0;
			category.SessionName = Catalog.GetString("<ROOT>");
			category.Protocol = 99;

			TreeIter iter = categoryTreeStore.AppendValuesSession(category);
			iterSelected = iter;
			RefreshCategoryListRecursive(0, iter);
		}
		
		private void RefreshCategoryListRecursive(int parentId, TreeIter parentIter)
		{
			ArrayList catList;
			Configuration cf = new Configuration();
			
			catList = cf.LoadCategories();
			foreach (Configuration cfg in catList)
			{
				if (cfg.ParentId == parentId)
				{
					TreeIter iter = categoryTreeStore.AppendValuesSession(parentIter, cfg);
					if (config.ParentId == cfg.Id)
					{
						this.iterSelected = iter;
					}					
					this.RefreshCategoryListRecursive(cfg.Id, iter);
				}
			}
		}		
		
		public int Run()
		{
			int retval = this.Dialog.Run();
			if ( retval == (int)Gtk.ResponseType.Ok )
			{
				if (eSessionName.Text.Length > 0)
				{
					TreeModel model = treeKeyboard.Model;
					treeKeyboard.Selection.GetSelected(out model, out iter);
					config.ParentId = categoryTreeView.config.Id;
					config.KeyboardLang = (string)model.GetValue(iter, 0);
					config.Protocol = cbProtocol.Active;
					config.SrvType = cbSrvType.Active;
					config.SessionName = eSessionName.Text.Trim();
					config.Computer = eComputer.Text.Trim();
					config.User = eUser.Text.Trim();
					config.Password = ePassword.Text.Trim();
					config.Domain = eDomain.Text.Trim();
					config.ColorDepth = cbColorDepth.Active;
					config.ScreenResolutionX = Convert.ToInt32(sbWidth.Value);
					config.ScreenResolutionY = Convert.ToInt32(sbHeight.Value);
					config.SoundRedirection = cbSoundRedirection.Active;
					config.ConnectionType = cbConnectionType.Active;
					config.WindowMode = cbWindowMode.Active;
					config.TerminalSize = cbTerminalSize.Active;
					config.CompressionLevel = Convert.ToInt32(sbCompressionLevel.Text);
					config.ImageQuality = Convert.ToInt32(sbImageQuality.Text);	
					if (config.Id > 0)
					{
						config.UpdateSession(cbSavePassword.Active);
					}
					else
					{
						config.SaveSession(cbSavePassword.Active, false);
					}
				}
			}
			this.Dialog.Destroy();
			return retval;
		}
		
		private void RefreshControlsSensitivity()
		{
			TreeIter selIter;
			switch (cbProtocol.Active)
			{
				// RDP
				case 0:
					this.eUser.Sensitive 			= true;
					this.ePassword.Sensitive 		= true;
					this.eDomain.Sensitive 		= true;
					this.cbSavePassword.Sensitive 	= true;
					this.cbSrvType.Sensitive		= true;
					break;
				// VNC
				case 1:
					this.eUser.Sensitive 			= false;
					this.ePassword.Sensitive 		= true;
					this.eDomain.Sensitive 		= false;
					this.cbSavePassword.Sensitive 	= true;
					this.cbSrvType.Sensitive		= false;
					break;
				// SSH
				case 2:
					this.eUser.Sensitive 			= true;
					this.ePassword.Sensitive 		= false;
					this.eDomain.Sensitive 		= false;
					this.cbSavePassword.Sensitive 	= false;
					this.cbSrvType.Sensitive		= false;
					this.cbSavePassword.Active	 	= false;
					break;
			}
			ivTabs.Model = FillIconView();
			ivTabs.TextColumn = 0;
			ivTabs.PixbufColumn = 2;
			storeIconList.GetIterFirst(out selIter);
			ivTabs.SelectPath(storeIconList.GetPath(selIter));
			ivTabs.ShowAll();			
		}
		
		private void OnColorDepthChanged (object sender, EventArgs a)
		{
			switch (cbColorDepth.Active)
			{
				case 0:
					iColors.Pixbuf = Gdk.Pixbuf.LoadFromResource("colors_1.png");
					break;
				case 1:
					iColors.Pixbuf = Gdk.Pixbuf.LoadFromResource("colors_2.png");
					break;
				case 2:
					iColors.Pixbuf = Gdk.Pixbuf.LoadFromResource("colors_3.png");
					break;
				case 3:
					iColors.Pixbuf = Gdk.Pixbuf.LoadFromResource("colors_4.png");
					break;
					
			}
		}

		private void OnScreenResolutionChanged (object sender, EventArgs a)
		{
			switch (cbScreenResolution.Active)
			{
				case 0:
					hboxCustomResolution.Sensitive = false;
					sbWidth.Value = 640;
					sbHeight.Value = 480;
					break;
				case 1:
					hboxCustomResolution.Sensitive = false;
					sbWidth.Value = 800;
					sbHeight.Value = 600;
					break;
				case 2:
					hboxCustomResolution.Sensitive = false;
					sbWidth.Value = 0;
					sbHeight.Value = 0;					
					break;
				case 3:
					hboxCustomResolution.Sensitive = true;
					sbWidth.Value = 1024;
					sbHeight.Value = 768;					
					break;
			}
		}
		
		private void OnSessionNameChanged(object sender, EventArgs a)
		{
			if ((eSessionName.Text.Length > 0) &&
				(eComputer.Text.Length > 0) &&
				(categoryTreeView.config != null))
			{
				btOk.Sensitive = true;
			}
			else
			{
				btOk.Sensitive = false;
			}
		}
		
		private void OnProtocolChanged(object sender, EventArgs a)
		{
			this.RefreshControlsSensitivity();
		}
		
		private void OnSelectionChanged(object sender, EventArgs a)
		{
			TreeIter iter;
			TreePath[] path = ivTabs.SelectedItems;
			if (path.Length > 0)
			{
				storeIconList.GetIter(out iter, path[0]);
				nbTabs.Page = (int)storeIconList.GetValue(iter, 1);
			}
		}
	}
}
