/* GnomerMind - A MasterMind(R)-based game for Gnome
 * (C) 2001 Germano Rizzo
 *
 * cache.c - routines to cache the theme in memory
 * Author: Germano Rizzo
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <sys/stat.h>
#include "sounds.h"
#include "themes.h"
#include "cache.h"

const gchar *pix_base_name[] = {
  "tile", "tokh", "tok0", "tok1", "tok2", "tok3", "tok4", "tok5", "tok6",
  "tok7", "tok8", "tok9", "pin0", "pin1", "pin2", "stt0", "stt1", "stt2",
  "stt3", "u", "b", "l", "r", "ul", "ur", "bl", "br"
};

const gchar *bit_base_name[] = {
  "mtok", "mstt", "mpin0", "mpin1", "mpin2", "mpin3", "mpin4", "mpin5"
};

struct theme_cache stock;

#ifdef HAVE_ESD

#include <esd.h>

const gchar *wav_base_name[] = {
  "pickup", "putdown", "guess", "success", "fail", "reset"
};

struct sound_cache sndstock;

#endif

/**
 * loadimg
 * @name: the `base' name of the file of the pixmap
 * @pix: the pixmap to load
 * @bit_ the bitmap to load
 *
 * Loads a file into a GdkPixmap or GdkBitmap
 **/
static void
loadimg (const gchar * name, GdkPixmap ** pix, GdkBitmap ** bit)
{
  gchar *pat;
  GdkPixbuf *tmpimg;

  pat = getpath (theme, g_strdup (name));

  verify (pat);
  tmpimg = gdk_pixbuf_new_from_file (pat);

  if (!tmpimg)
    /* The 1st %s is the type, the 2nd the filename 
       please don't change the order, if possible   */
    g_error (_("can't load %s data from file %s"), "png", pat);

  gdk_pixbuf_render_pixmap_and_mask (tmpimg, pix, bit, 127);
  gdk_pixbuf_unref (tmpimg);
}

/**
 * theme_cache_load
 *
 * Load the theme data in the theme cache
 **/
void
theme_cache_load ()
{
  gint i;

  g_assert (validate (theme));

  for (i = 0; i < PIX; i++)
    loadimg (pix_base_name[i], &stock.pix[i], NULL);

  gdk_window_get_size (stock.pix[L], (gint *) & stock.ext,
		       (gint *) & stock.base);

  for (i = 0; i < BIT; i++)
    loadimg (bit_base_name[i], NULL, &stock.bit[i]);

  {
    struct stat buf;
    gchar *path;

    path = getabout (theme);
    stat (path, &buf);

    {
      gchar file_content[buf.st_size + 1];
      FILE *file;

      file = fopen (path, "r");
      fread (&file_content, buf.st_size, 1, file);
      file_content[buf.st_size] = '\0';
      fclose (file);

      stock.about = g_strdup (file_content);
    }
  }
}

/**
 * theme_cache_free
 *
 * Frees the theme cache
 **/
void
theme_cache_free ()
{
  gint i;

  for (i = 0; i < PIX; i++)
    gdk_pixmap_unref (stock.pix[i]);

  for (i = 0; i < BIT; i++)
    if (stock.bit[i])
      gdk_bitmap_unref (stock.bit[i]);

  stock.base = 0;
  stock.ext = 0;

  stock.about = NULL;
}

#ifdef HAVE_ESD

/**
 * sound_cache_load
 *
 * Loads the sound data in the sound cache
 **/
void
sound_cache_load ()
{
  gint i;

  sndstock.connection = gnome_sound_connection;

  if (!sndstock.connection)
    g_warning (_("can't extabilish connection to GNOME sound server"));


  for (i = 0; i < WAV; i++)
    sndstock.sample[i] = sound_load (wav_base_name[i]);
}

#endif
